///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
///
/// @file Filter.h

#ifndef OPENVDB_TOOLS_FILTER_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_FILTER_HAS_BEEN_INCLUDED

#include <tbb/parallel_reduce.h>
#include <tbb/parallel_for.h>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/scoped_ptr.hpp>
#include <openvdb/openvdb.h>
#include <openvdb/Types.h>
#include <openvdb/math/Math.h>
#include <openvdb/math/Stencils.h>
#include <openvdb/math/Transform.h>
#include <openvdb/Grid.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Filtering of VDB volumes
/// @todo This really only works for scalar, floating-point grids.
template<typename _GridType>
class Filter
{
public:
    typedef _GridType                              GridType;
    typedef typename GridType::TreeType            TreeType;
    typedef typename TreeType::LeafNodeType        LeafType;
    typedef typename LeafType::ValueType           ValueType;
    typedef typename tree::LeafArray<TreeType, 1>  ArrayType; // array of leaf nodes & 1 buffer
    typedef typename ArrayType::IterRangeType      RangeType;

    /// Constructor
    /// @throw ValueError if the grid pointer is null
    explicit Filter(GridType&);

    /// Shallow copy constructor called by tbb::parallel_for() threads during filtering
    Filter(const Filter&);

    /// Copy constructor called by tbb::parallel_reduce() threads during renormalization
    Filter(Filter&, tbb::split);

    /// One interation - no progress-bar!
    void median(int width = 1, bool serial = false);
    void mean(int width = 1, bool serial = false);
    void meanCurvature(bool serial = false);
    void laplacian(bool serial = false);
    void offset(float offset, bool serial = false);

    /// Iterative re-normalization
    template<bool UseWENO>
    float renormalize(size_t maxSteps, float epsilon, bool verbose = false, bool serial = false);

    /// Used internally by tbb::parallel_for()
    void operator()(const RangeType& r) const
    {
        if (mTask) mTask(const_cast<Filter*>(this), r);
        else OPENVDB_THROW(ValueError, "task is undefined - call median(), mean(), etc.");
    }

    /// Used internally by tbb::parallel_reduce()
    void operator()(const RangeType& r)
    {
        if (mTask) mTask(this, r);
        else OPENVDB_THROW(ValueError, "task is undefined - call median(), mean(), etc.");
    }

    /// This is only called by tbb::parallel_reduce() threads
    void join(const Filter& other) { mDelta = std::max(mDelta, other.mDelta); }

private:
    typedef typename boost::function<void (Filter*, const RangeType&)> FuncType;

    GridType*               mGrid;
    const float             mDx; // voxel size
    boost::scoped_ptr<ArrayType>
                            mArray; // array of leaf nodes and buffers
    float                   mDelta; // |Grad|-1|
    FuncType                mTask;

    // Private enum and cook method calling tbb::parallel_for or tbb::parallel_reduce
    enum ModeType { SERIAL, PARALLEL_FOR, PARALLEL_REDUCE }; // for internal use
    void cook(ModeType mode, int swapBuffer);

    // Private renormalization method called by tbb::parallel_reduce threads
    template<typename NormSqGradType> void doRenorm(const RangeType&);
    void doRenormUpwind(const RangeType& r) { this->doRenorm<math::GradStencil<GridType> >(r); }
    void doRenormWENO(const RangeType& r) { this->doRenorm<math::WenoStencil<GridType> >(r); }

    // Private filter methods called by tbb::parallel_for threads
    void doMedian(const RangeType&, int);
    void doMean(const RangeType&, int);
    void doMeanCurvature(const RangeType&);
    void doLaplacian(const RangeType&);
    void doOffset(const RangeType&, float);

}; // end of Filter class


////////////////////////////////////////


template<typename GridT>
Filter<GridT>::Filter(GridType& grid):
    mGrid(&grid),
    mDx(grid.voxelDimensions()[0]),
    mArray(new ArrayType(grid.tree())),
    mDelta(0),
    mTask(0)
{
}


template<typename GridT>
Filter<GridT>::Filter(const Filter& other):
    mGrid(other.mGrid),
    mDx(other.mDx),
    mArray(NULL),
    mDelta(other.mDelta),
    mTask(other.mTask)
{
}


template<typename GridT>
Filter<GridT>::Filter(Filter& other, tbb::split):
    mGrid(other.mGrid),
    mDx(other.mDx),
    mArray(NULL),
    mDelta(other.mDelta),
    mTask(other.mTask)
{
}


////////////////////////////////////////


template<typename GridT>
inline void
Filter<GridT>::median(int width, bool serial)
{
    mTask = boost::bind(&Filter::doMedian, _1, _2, width);
    this->cook(serial ? SERIAL : PARALLEL_FOR, 0);
}


template<typename GridT>
inline void
Filter<GridT>::mean(int width, bool serial)
{
    mTask = boost::bind(&Filter::doMean, _1, _2, std::max(1, width));
    this->cook(serial ? SERIAL : PARALLEL_FOR, 0);
}


template<typename GridT>
inline void
Filter<GridT>::meanCurvature(bool serial)
{
    mTask = boost::bind(&Filter::doMeanCurvature, _1, _2);
    this->cook(serial ? SERIAL : PARALLEL_FOR, 0);
}

template<typename GridT>
inline void
Filter<GridT>::laplacian(bool serial)
{
    mTask = boost::bind(&Filter::doLaplacian, _1, _2);
    this->cook(serial ? SERIAL : PARALLEL_FOR, 0);
}


template<typename GridT>
inline void
Filter<GridT>::offset(float value, bool serial)
{
    mTask = boost::bind(&Filter::doOffset, _1, _2, value);
    this->cook(serial ? SERIAL : PARALLEL_FOR, -1);
}


template<typename GridT>
template<bool UseWENO>
inline float
Filter<GridT>::renormalize(size_t maxSteps, float epsilon, bool verbose, bool serial)
{
    if (maxSteps == 0) return 0.0f;
    mTask = UseWENO ? boost::bind(&Filter::doRenormWENO, _1, _2)
        : boost::bind(&Filter::doRenormUpwind, _1, _2);
    for (size_t n = 0; n < maxSteps; ++n) {
        mDelta = 0.0f;
        if (verbose) std::cerr << "Renormalization #" << (n + 1) << std::flush;
        this->cook(serial ? SERIAL : PARALLEL_REDUCE, 0);
        if (verbose) std::cerr << " |Grad|-1| = " << mDelta << std::endl;
        if (mDelta < epsilon) break;
    }
    return mDelta;
}


////////////////////////////////////////


/// Private method to perform the task (serial or threaded) and
/// subsequently swap the leaf buffers.
template<typename GridT>
inline void
Filter<GridT>::cook(ModeType mode, int swapBuffer)
{
    switch (mode) {
        case SERIAL:
            (*this)(mArray->getRange()); break;
        case PARALLEL_FOR:
            tbb::parallel_for(mArray->getRange(), *this); break;
        case PARALLEL_REDUCE:
            tbb::parallel_reduce(mArray->getRange(), *this); break;
    }
    if (swapBuffer >= 0) mArray->swapBuffers(swapBuffer, mode == SERIAL);
}


/// Perform renormalization using WENO or single-sided upwinding.
template<typename GridT>
template<typename StencilType>
inline void
Filter<GridT>::doRenorm(const RangeType &range)
{
    StencilType stencil(*mGrid, mDx);//has local cache
    const float CFL = 1.0f, dt = CFL * mDx, dx2 = mDx * mDx;
    typename ArrayType::IterType aIter, aEnd = range.end();//leaf nodes & buffers
    typename LeafType::ValueOnCIter vIter;//leaf values
    for (aIter = range.begin(); aIter != aEnd; ++aIter) {
        for (vIter = aIter->leaf->cbeginValueOn(); vIter; ++vIter) {
            stencil.moveTo(vIter);
            const float normSqGradPhi = stencil.normSqGrad();
            const float phi = stencil.getValue();
            const float diff = math::Sqrt(normSqGradPhi) - 1.0f;
            const float S = phi / (math::Sqrt(math::Pow2(phi) + normSqGradPhi * dx2));
            mDelta = std::max(mDelta, std::abs(diff));
            aIter->buffer[0].setValue(vIter.pos(), phi - dt * S * diff);
        }
    }
}

/// Performs parabolic mean-curvature diffusion
template<typename GridT>
inline void
Filter<GridT>::doMeanCurvature(const RangeType& range)
{
    const float CFL = 0.9f, dt = CFL * mDx * mDx / 6.0f;
    math::CurvatureStencil<openvdb::FloatGrid> stencil(*mGrid, mDx);
    typename ArrayType::IterType aIter, aEnd = range.end();//leaf nodes & buffers
    typename LeafType::ValueOnCIter vIter;//leaf values
    for (aIter = range.begin(); aIter != aEnd; ++aIter) {
        for (vIter = aIter->leaf->cbeginValueOn(); vIter; ++vIter) {
            stencil.moveTo(vIter);
            const float K = stencil.meanCurvatureNormGrad();
            const float P = stencil.getValue();
            aIter->buffer[0].setValue(vIter.pos(), P + dt * K);
        }
    }
}

/// Performs laplacian diffusion. Note if the grids contains a true
/// signed distance field (e.g. a solution to the Eikonal equation)
/// Laplacian diffusions (e.g. geometric heat equation) is actually
/// identical to mean curvature diffusion, yet much less
/// computationally expensive! In other words if you're performing
/// renormalization anyway (i.e. solveing the Eikonal eq) you should
/// consider performing laplacian diffusion over mean curvature flow!
template<typename GridT>
inline void
Filter<GridT>::doLaplacian(const RangeType& range)
{
    const float CFL = 0.9f, half_dt = CFL * mDx * mDx / 12.0f;
    math::GradStencil<openvdb::FloatGrid> stencil(*mGrid, mDx);
    typename ArrayType::IterType aIter, aEnd = range.end();//leaf nodes & buffers
    typename LeafType::ValueOnCIter vIter;//leaf values
    for (aIter = range.begin(); aIter != aEnd; ++aIter) {
        for (vIter = aIter->leaf->cbeginValueOn(); vIter; ++vIter) {
            stencil.moveTo(vIter);
            const float L = stencil.laplacian();
            const float P = stencil.getValue();
            aIter->buffer[0].setValue(vIter.pos(), P + half_dt * L);
        }
    }
}

/// Performs simple but fast median-value diffusion
template<typename GridT>
inline void
Filter<GridT>::doMedian(const RangeType& range, int width)
{
    typename math::DenseStencil<GridType> stencil(*mGrid, width);//creates local cache!
    typename ArrayType::IterType aIter, aEnd=range.end();//leaf nodes & buffers
    typename LeafType::ValueOnCIter vIter;//leaf values
    for (aIter = range.begin(); aIter != aEnd; ++aIter) {
        for (vIter = aIter->leaf->cbeginValueOn(); vIter; ++vIter) {
            stencil.moveTo(vIter.getCoord());
            aIter->buffer[0].setValue(vIter.pos(), stencil.median());
        }
    }
}

/// Performs simple but fast mean-value diffusion
template<typename GridT>
inline void
Filter<GridT>::doMean(const RangeType& range, int width)
{
    math::DenseStencil<GridType> stencil(*mGrid, width);//creates local cache!
    typename ArrayType::IterType aIter, aEnd=range.end();//array of leaf nodes & buffers
    typename LeafType::ValueOnCIter vIter;//leaf values
    for (aIter = range.begin(); aIter != aEnd; ++aIter) {
        for (vIter = aIter->leaf->cbeginValueOn(); vIter; ++vIter) {
            stencil.moveTo(vIter.getCoord());
            aIter->buffer[0].setValue(vIter.pos(), stencil.mean());
        }
    }
}

/// Offsets the values by a constant
template<typename GridT>
inline void
Filter<GridT>::doOffset(const RangeType& range, float floatVal)
{
    const ValueType value = static_cast<ValueType>(floatVal);
    typename ArrayType::IterType iter, end=range.end();//array of leaf nodes & buffers
    for (iter = range.begin(); iter != end; ++iter) iter->leaf->addValue(value);
}

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_FILTER_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
