///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
/// @file Divergence.h

#ifndef OPENVDB_TOOLS_DIVERGENCE_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_DIVERGENCE_HAS_BEEN_INCLUDED

#include <openvdb/Grid.h>
#include <openvdb/math/Math.h>
#include <openvdb/math/Operators.h>
#include <openvdb/math/Stencils.h>
#include <openvdb/tree/ValueAccessor.h>
#include <tbb/parallel_for.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Computes the divergence of a vector-valued grid
///
/// @note The current implementation assumes all the input vector
/// values are represented by leaf voxels and not tiles. In the
/// future we will expand this class to also handle tile values.
template<typename InGridT, typename MapType, math::DScheme DiffScheme>
class DivergenceImpl
{
public:
    typedef InGridT                                                      InGridType;
    typedef typename InGridT::ValueType::value_type                      Real;
    typedef typename InGridT::ConstAccessor                              InAccessorType;
    typedef typename InGridT::TreeType                                   InTreeType;
    typedef typename InTreeType::template ValueConverter<Real>::Type     OutTreeType;
    typedef Grid<OutTreeType>                                            OutGridType;
    typedef typename tree::LeafArray<OutTreeType, 0>                     ArrayType;
    typedef typename ArrayType::IterRangeType                            RangeType;

    DivergenceImpl(const InGridType& grid, const MapType& map):
        mInputGrid(&grid),
        mInputAccessor(grid.getConstAccessor()),
        mMap(&map)
    {
        const Real backg = 0;// Divergence is zero in constant regions!
        // output tree = topology copy of input tree!
        typename OutTreeType::Ptr outTree(
            new OutTreeType(grid.tree(), backg, TopologyCopy()));
        // create grid with output tree and unit transform
        mOutputGrid = typename OutGridType::Ptr(new OutGridType(outTree));
        // transform of output grid = transform of input grid
        mOutputGrid->setTransform(grid.transform().copy());
    }

    /// Copy constructor called by tbb::parallel_for threads
    DivergenceImpl(const DivergenceImpl& other):
        mInputGrid(other.mInputGrid),//shallow copy
        mOutputGrid(other.mOutputGrid),//shallow copy
        mInputAccessor(mInputGrid->getConstAccessor()),//local instance
        mMap(other.mMap)//shallow copy
    {
    }

    virtual ~DivergenceImpl() {}

    typename OutGridType::Ptr process(bool threaded=true)
    {
        ArrayType leafArray(mOutputGrid->tree());
        if (threaded) {
            tbb::parallel_for(leafArray.getRange(), *this);
        } else {
            (*this)(leafArray.getRange());
        }
        return mOutputGrid;
    }

    /// Iterate sequentially over LeafNodes and voxels in the output
    /// grid and compute the divergence using a valueAccessor for the
    /// input grid
    void operator()(const RangeType& range) const
    {
        typedef math::Divergence<MapType, DiffScheme> DivT;

        typename ArrayType::IterType aIter, aEnd = range.end(); // leaf nodes
        typename OutTreeType::LeafNodeType::ValueOnIter vIter; // leaf values
        for (aIter = range.begin(); aIter != aEnd; ++aIter) {
            for (vIter = aIter->leaf->beginValueOn(); vIter; ++vIter) {
                vIter.setValue(DivT::result(*mMap, mInputAccessor, vIter.getCoord()));
            }
        }
    }

protected:
    const InGridType*           mInputGrid;
    typename OutGridType::Ptr   mOutputGrid;
    mutable InAccessorType      mInputAccessor;
    const MapType*              mMap;
}; // end of DivergenceImpl class


template<typename InGridT, math::DScheme DiffScheme = math::CD_2ND>
class DivFunctor
{
public:
    typedef InGridT                                                      InGridType;
    typedef typename InGridT::ValueType::value_type                      Real;
    typedef typename InGridT::TreeType                                   InTreeType;
    typedef typename InTreeType::template ValueConverter<Real>::Type     OutTreeType;
    typedef Grid<OutTreeType>                                            OutGridType;


    DivFunctor(const InGridType& grid, bool threaded):
        mInputGrid(&grid), mThreaded(threaded)
    {}

    template<typename MapType>
    void operator()(const MapType& map)
    {
        DivergenceImpl<InGridType, MapType, DiffScheme> divImpl(*mInputGrid, map);
        mOutputGrid = divImpl.process(mThreaded); // cache the result
    }

    // for output of the result
    typename OutGridType::Ptr result() { return mOutputGrid; }

private:
    const InGridType*           mInputGrid;
    bool                        mThreaded;
    typename OutGridType::Ptr   mOutputGrid;
}; // class DivFunctor


template<typename InGridT>
class Divergence
{
public:
    typedef InGridT                                                      InGridType;
    typedef typename InGridT::ValueType::value_type                      Real;
    typedef typename InGridT::TreeType                                   InTreeType;
    typedef typename InTreeType::template ValueConverter<Real>::Type     OutTreeType;
    typedef Grid<OutTreeType>                                            OutGridType;


    Divergence(const InGridType& grid): mInputGrid(&grid) {}

    /// Computes the divergence of a vector field. Second-order central
    /// differencing is used for collocated fields. First order forward
    /// differencing is used for staggered fields.
    ///
    /// @pre Staggered velocity fields must have their grid class set to
    /// GRID_STAGGERED. It is assumed that a staggered velocity value
    /// refers to the velocities at the -i,-j,-k faces.
    typename OutGridType::Ptr process(bool threaded = true)
    {
        if( mInputGrid->getGridClass() == GRID_STAGGERED ) {
            DivFunctor<InGridType, math::FD_1ST> divFunctor(*mInputGrid, threaded);
            processTypedMap(mInputGrid->transform(), divFunctor);
            return divFunctor.result();
        }
        else {
            DivFunctor<InGridType> divFunctor(*mInputGrid, threaded);
            processTypedMap(mInputGrid->transform(), divFunctor);
            return divFunctor.result();
        }
    }

protected:
    const InGridType* mInputGrid;
}; // end of DivergenceImpl class

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_DIVERGENCE_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
