///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
/// @file Curl.h

#ifndef OPENVDB_TOOLS_CURL_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_CURL_HAS_BEEN_INCLUDED

#include <openvdb/Types.h>
#include <openvdb/Grid.h>
#include <openvdb/math/Math.h>
#include <openvdb/math/Operators.h>
#include <openvdb/math/Stencils.h>
#include <openvdb/tree/Tree.h>
#include <tbb/parallel_for.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

template<typename GridT, typename MapType, math::DScheme DiffScheme>
class CurlImpl
{
public:
    typedef GridT                                 GridType;
    typedef typename GridType::TreeType           TreeType;
    typedef typename GridType::ConstAccessor      Accessor;
    typedef typename GridType::ValueType          Vec3Type;
    typedef typename Vec3Type::value_type         Real;
    typedef typename TreeType::LeafNodeType       LeafType;
    typedef GridT                                 OutGridType;
    typedef typename tree::LeafArray<TreeType,0>  ArrayType;
    typedef typename ArrayType::IterRangeType     RangeType;

    CurlImpl(const GridType& grid, const MapType& map):
        mInputGrid(&grid),
        mInputAccessor(mInputGrid->getConstAccessor()),
        mMap(&map)
    {
        const Vec3Type backg(0, 0, 0);// Curl is zero in constant regions!
        /// output tree = topology copy of input tree!
        typename TreeType::Ptr outTree(new TreeType(grid.tree(), backg, TopologyCopy()));
        /// create grid with output tree and unit transform
        mOutputGrid = typename GridType::Ptr(new GridType(outTree));
        /// transform of output grid = transform of input grid
        mOutputGrid->setTransform(grid.transform().copy());
    }

    /// Copy constructor called by tbb::parallel_for threads
    CurlImpl(const CurlImpl& other):
        mInputGrid(other.mInputGrid),
        mOutputGrid(other.mOutputGrid),
        mInputAccessor(mInputGrid->getConstAccessor()), // local accessor
        mMap(other.mMap)
    {
    }

    virtual ~CurlImpl() {}

    typename OutGridType::Ptr process(bool threaded = true)
    {
        ArrayType leafArray(mOutputGrid->tree());

        if (threaded) {
            tbb::parallel_for(leafArray.getRange(), *this);
        } else {
            (*this)(leafArray.getRange());
        }

        return mOutputGrid;
    }

    void operator()(const RangeType& range) const
    {
        typename ArrayType::IterType aIter, aEnd = range.end(); // leaf nodes
        typename LeafType::ValueOnIter vIter; // leaf values
        for (aIter = range.begin(); aIter != aEnd; ++aIter) {
            for (vIter = aIter->leaf->beginValueOn(); vIter; ++vIter) {
                vIter.setValue((*this)(vIter.getCoord()));
            }
        }
    }

protected:
    Vec3Type operator()(const Coord& ijk) const
    {
        return math::Curl<MapType, DiffScheme>::result(*mMap, mInputAccessor, ijk);
    }


    const GridType*         mInputGrid;
    typename GridType::Ptr  mOutputGrid;
    mutable Accessor        mInputAccessor;
    const MapType*          mMap;
}; // end of CurlImpl class


template<typename GridT>
class CurlFunctor
{
public:
    typedef GridT GridType;

    CurlFunctor(const GridType& grid, bool threaded):
        mInputGrid(&grid), mThreaded(threaded)
    {}

    template<typename MapType>
    void operator()(const MapType& map)
    {
        CurlImpl<GridType, MapType, math::CD_2ND> curlImpl(*mInputGrid, map);
        mOutputGrid = curlImpl.process(mThreaded); // cache the result
    }

    // for output of the result
    typename GridType::Ptr result() { return mOutputGrid; }

private:
    const GridType*         mInputGrid;
    bool                    mThreaded;
    typename GridType::Ptr  mOutputGrid;
}; // class CurlFunctor


/// @brief Compute the curl of a vector-valued grid.
///
/// @note The current implementation assumes all the input vector
/// values are represented by leaf voxels and not tiles. In the
/// future we will expand this class to also handle tile values.
template<typename GridT>
class Curl
{
public:
    typedef GridT GridType;
    typedef GridT OutGridType;

    Curl(const GridType& grid): mInputGrid(&grid) {}

    virtual ~Curl() {}

    typename OutGridType::Ptr process(bool threaded = true)
    {
        CurlFunctor<GridT> curlFunctor(*mInputGrid, threaded);
        processTypedMap(mInputGrid->transform(), curlFunctor);
        return curlFunctor.result();
    }

protected:
    const GridType* mInputGrid;
};

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_CURL_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
