///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @file Ray.h
/// @author Ken Museth

#ifndef OPENVDB_MATH_RAY_HAS_BEEN_INCLUDED
#define OPENVDB_MATH_RAY_HAS_BEEN_INCLUDED

#include <set>
#include <cassert>
#include <cstring>
#include <iostream>  // for cout
#include <fstream>   // for ofstream
#include <cmath>     // for round
#include "Math.h"    // for isExactlyEqual()
#include "Vec3.h"
#include <openvdb/math/Transform.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace math {

template<typename _Real>
class Ray
{
public:
    typedef _Real Real;
    typedef Vec3<Real> Vec3R;

    //un-initialized, i.e. not valid
    Ray(): mTime(-1),mMinTime(0), mMaxTime(std::numeric_limits<Real>::max())
    {
    }

    Ray(const Vec3R &origin, const Vec3R &direction,
        Real time = std::numeric_limits<Real>::epsilon(),
        Real start=0, Real end = std::numeric_limits<Real>::max())
        : mOrigin(origin), mDirection(direction),
          mInvDirection(1/direction[0],1/direction[1],1/direction[2]),
          mTime(time), mMinTime(start), mMaxTime(end)
    {
    }

    Vec3R getOrigin() const {return mOrigin;}
    Real  getOrigin(int i) const {return mOrigin[i];}

    Vec3R getDirection() const {return mDirection;}
    Real  getDirection(int i) const {return mDirection[i];}

    Vec3R getInvDirection() const {return mInvDirection;}
    Real  getInvDirection(int i) const {return mInvDirection[i];}

    Real  getTime() const {return mTime;}
    Real  getMinTime() const {return mMinTime;}
    Real  getMaxTime() const {return mMaxTime;}

    void setOrigin(const Vec3R &orig) {mOrigin=orig;}
    void setOrigin(Real x, Real y, Real z)
    {
        mOrigin[0]=x;
        mOrigin[1]=y;
        mOrigin[2]=z;
    }

    void setDirection(const Vec3R &dir)
    {
        mDirection=dir;
        this->setInvDirection();
    }

    void setDirection(Real x, Real y, Real z)
    {
        mDirection[0]=x;
        mDirection[1]=y;
        mDirection[2]=z;
        this->setInvDirection();
    }

    void setTime(Real time) {mTime=time;}

    void setMinTime(Real time)
    {
        mMinTime=time;
    }

    void setMaxTime(Real time)
    {
        mMaxTime=time;
    }
    void setMinMaxTimes(Real t0, Real t1)
    {
        mMinTime=t0;
        mMaxTime=t1;
    }

    // For set and get
    Real& timer() {return mTime;}
    Vec3R position(Real time) const { return mOrigin + mDirection * time; }
    Vec3R start() const { return mOrigin + mDirection * mMinTime; }
    Vec3R end() const { return mOrigin + mDirection * mMaxTime; }
    bool isValid() const { return (mTime>mMinTime && mTime<mMaxTime); }

    /// Transform this ray (origin and direction) from worldspace to indexspace.
    /// @param xform Transform definition.
    void worldToIndex(const Transform::Ptr& xform)
    {

        mOrigin = xform->worldToIndex(mOrigin);

        mDirection  = xform->worldToIndex(mDirection);
        mDirection -= xform->indexToWorld(Vec3R(0,0,0));//ignore translation!
        this->setInvDirection();
    }

    /// Transform this ray (origin and direction) from indexspace to worldspace.
    /// @param xform Transform definition.
    void indexToWorld( const Transform::Ptr& xform )
    {
        mOrigin = xform->indexToWorld( mOrigin );

        mDirection += xform->indexToWorld( Vec3R( 0,0,0 ) );
        mDirection = xform->indexToWorld( mDirection );
        this->setInvDirection();
    }

private:
    void setInvDirection()
    {
        mInvDirection[0] = 1/mDirection[0];//1/0=inf, should be safe!
        mInvDirection[1] = 1/mDirection[1];
        mInvDirection[2] = 1/mDirection[2];
    }

    Vec3R mOrigin;
    Vec3R mDirection;
    Vec3R mInvDirection;//for acceleration
    Real  mTime;
    Real  mMinTime, mMaxTime;
}; // end of Ray class


template<typename Real>
inline std::ostream &operator<<(std::ostream &os, const Ray<Real> &r) {
    os << "org=" << r.getOrigin() << " dir=" << r.getDirection() << " tmin=" <<
        r.getMinTime() << " tmax=" << r.getMaxTime() << " t=" << r.getTime();
    return os;
}


template<typename Real, typename ValueType>
class RayInterval
{
public:
    /// Dummy class to distinguish constructors
    class Hit {};

    enum Tag {EMPTY=0, CONSTANT, VARIABLE};

    Real t0, t1;
    Tag  tag;
    ValueType value;//only used with CONSTANT tag!!!

    // Constructs an EMPTY interval is maximum extent!
    RayInterval()
        : t0(std::numeric_limits<Real>::epsilon()),
          t1(std::numeric_limits<Real>::max()), tag(EMPTY)
    {
    }
    // Constructs an EMPTY interval with t0 and t1
    RayInterval(Real _t0, Real _t1)
       : t0(_t0), t1(_t1), tag(EMPTY)
    {
        assert(t0<=t1);
    }
    // Constructs an VARIABLE interval with t0 and t1
    RayInterval(Real _t0, Real _t1, Hit)
       : t0(_t0), t1(_t1), tag(VARIABLE)
    {
        assert(t0<=t1);
    }
    // Constructs a CONSTANT interval with t0 and t1
    RayInterval(Real _t0, Real _t1, const ValueType &_value)
        : t0(_t0), t1(_t1), tag(CONSTANT), value(_value)
    {
        assert(t0<=t1);
    }
    bool isEmpty()    const {return tag==EMPTY;}
    bool isConstant() const {return tag==CONSTANT;}
    bool isVariable() const {return tag==VARIABLE;}
    // redefines t0 and t1
    void set(Real _t0, Real _t1)
    {
        t0  = _t0;
        t1  = _t1;
        assert(t0<=t1);
    }
    // Only for EMPTY or VARIABLE intervals!
    void set(Real _t0, Real _t1, bool empty=false)
    {
        t0  = _t0;
        t1  = _t1;
        tag = empty ? EMPTY : VARIABLE;
        assert(t0<=t1);
    }
    // Only for CONSTANT intervals
    void set(Real _t0, Real _t1, const ValueType &_value)
    {
        t0   =  _t0;
        t1   =  _t1;
        tag  =  CONSTANT;
        value= _value;
        assert(t0<=t1);
    }
    /// @return true if this->t0 < other.t0
    bool operator < (const RayInterval &other) const
    {
        return t0<other.t0;
    }
    bool operator==(const RayInterval &other) const
    {
        if (tag != other.tag || !isExactlyEqual(t0, other.t0)
            || !isExactlyEqual(t1, other.t1)) return false;
        if (tag==CONSTANT) return !isExactlyEqual(value, other.value);
        return true;
    }
    /// we assume the two intervals are non-overlapping
    bool merge(const RayInterval &other, Real tolerance=1e-4)
    {
        if (this->tag != other.tag
            || (this->tag == CONSTANT && !isExactlyEqual(this->value, other.value)) )
        {
            return false;
        } else if (std::abs(this->t0-other.t1)<tolerance) {//other is before
            this->t0=other.t0;
            return true;
        } else if (std::abs(this->t1-other.t0)<tolerance) {//other is after
            this->t1=other.t1;
            return true;
        }
        return false;
    }

    void print() const
    {
        std::cerr << "t0=" << t0 << "\tt1=" << t1 << "\ttag="
            << (tag==0 ? "EMPTY" :  tag==1 ? "CONSTANT" : "VARIABLE");
        if (tag==1) std::cerr << "\tvalue=" << value;
        std::cerr << std::endl;
    }
}; // end of RayInterval class


template<typename Real, typename ValueType>
class RayIntervalSet: public std::set<RayInterval<Real,ValueType> >
{
public:
    typedef RayInterval<Real,ValueType> IntervalType;

    RayIntervalSet() : std::set<IntervalType>()
    {
    }
    /// @return true if intervals are merged
    /// @brief merges and padds input intersections. New intervals are tagged EMPTY
    /// We assume:
    /// 1) input RayIntervalSets are non-overlapping (this is a
    /// consequense of non-overlapping block in vdb!)
    /// 2) input RayIntervalSets does not contain any EMPTY
    /// intervals (these are ignored in the Grid::intersection)
    bool merge(const Ray<Real> &R, Real tolerance=1e-4)
    {
        if (this->size()==0) return false;//empty - nothing to do!
        RayIntervalSet merged;
        typename std::set<IntervalType>::const_iterator iter=this->begin(), iend=this->end();
        if (R.getMinTime() < iter->t0) // need to patch beginning?
            merged.insert(IntervalType(R.getMinTime(),iter->t0));//insert EMPTY
        IntervalType I = *iter;
        for ( ++iter; iter != iend; ++iter ) {
            if (I.merge(*iter,tolerance)) continue;
            merged.insert(I);
            if (std::abs(I.t1-iter->t0)>tolerance)//is there a gab?
                merged.insert(IntervalType(I.t1,iter->t0));//insert EMPTY
            I=*iter;
        }
        merged.insert(I);
        if (R.getMaxTime()>I.t1) //need to patch end?
            merged.insert(IntervalType(I.t1,R.getMaxTime()));//insert EMPTY
        *this = merged;
        return true;
    }
    bool isEmpty() const {return this->size()==0;}

    // Convenient for debugging!
    void print(const std::string &str=std::string("")) const
    {
        if (str!=std::string(""))
            std::cerr << "RayIntervalSet for \"" << str << "\"" << std::endl;
        if (this->isEmpty()) {
            std::cerr << "Empty RayIntervalSet!" << std::endl;
            return;
        }
        typename std::set<IntervalType>::const_iterator iter=this->begin();
        for (; iter!=this->end(); ++iter) iter->print();
    }
}; // end of RayIntervalSet class

} // namespace math
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_MATH_RAY_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
