///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @file Plane.h
/// @author Ken Museth

#ifndef OPENVDB_MATH_PLANE_HAS_BEEN_INCLUDED
#define OPENVDB_MATH_PLANE_HAS_BEEN_INCLUDED

#include <cassert>
#include <cstring>
#include <iostream>// for cout
#include <fstream>//for ofstream
#include <algorithm>//for min/max
#include <cmath>//for sqrt
#include "Math.h" // for Clamp01
#include "Vec3.h"
#include "Ray.h"


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace math {

/// @brief Plane Class
///
/// Implicit definition of plane with normal N through a point P0:
///
/// (P-P0).N = P.N-P0.N = 0
///
/// Nx*Px + Ny*Py + Nz*Pz - ( Nx*P0x + Ny*P0y + Nz*P0z ) = Nx*Px + Ny*Py + Nz*Pz - Dist = 0
///
/// ******************************
/// *Nx*Px + Ny*Py + Nz*Pz = Dist*
/// ******************************
///
template<typename Real>
class Plane
{
public:
    typedef Vec3<Real> Point;
    typedef Vec3<Real> Vector;

    /// Dummy class to distinguish constructors where the normal is
    /// normalized or not
    class Normalize {
    };

  private:

    Vector mNorm;//normal of plane
    Point  mPoint;// point on plane (useful for interpolation)
    Real   mDist;// = N.P0, i.e. normal dot any point in the plane

  public:

    /// Invalid plane since |N|=0
    Plane() : mNorm(0,0,0), mPoint(0,0,0), mDist(0)
      {
      }

    /// Define plane from a point in the pane and its normal
    /// Assumes N is already normalized!
    Plane(const Point &P0, const Vector &N)
        : mNorm(N), mPoint(P0), mDist(N.dot(P0))
      {
      }

    /// Define plane from a point in the pane and its normal
    Plane(const Point &P0, const Vector &N, Normalize)
        : mNorm(N), mPoint(P0)
      {
          mNorm *= Inv(mNorm.length());
          mDist = mNorm.dot(P0);
      }

    /// @brief Define half-plane as affine interpolation between two points
    /// Normal points from P1 to P2 and plane goes through P1 for alpha=0
    Plane(const Vector &P1, const Vector &P2, Real alpha)
      {
          assert(P1!=P2);
          mNorm = P2-P1;
          mNorm *= Inv(mNorm.length());
          const Real C1=Clamp01(alpha), C2=1-C1;
          mPoint= C2*P1+C1*P2;
          mDist = mNorm.dot(mPoint);
      }

    /// Define plane from three point in plane and normal is right-hand turn
    /// The point (used for interpolation) is the first, i.e. P0
    Plane(const Point &P0, const Point &P1, const Point &P2)
      {
          assert(P0!=P1);
          assert(P0!=P2);
          assert(P1!=P2);
          mNorm.cross(P1-P0,P2-P0);//right-hand turn!
          mNorm *= Inv(mNorm.length());
          assert(Abs(1.-mNorm.length())<0.001);
          mPoint = P0;
          mDist=mNorm.dot(P0);
      }
    /// Same as above, however this constructor allow the user the
    /// specify which of the three input points to use as P0
    Plane(const Point &P0, const Point &P1, const Point &P2, int n)
      {
          assert(P0!=P1);
          assert(P0!=P2);
          assert(P1!=P2);
          mNorm.cross(P1-P0,P2-P0);//right-hand turn!
          mNorm *= Inv(mNorm.length());
          assert(Abs(1.-mNorm.length())<0.001);
          mPoint = n==0 ? P0: n==1 ? P1 : P2;
          mDist=mNorm.dot(P0);
      }

    /// flips normal of plane. Note that the point is unchanged!
    inline void flip()
      {
          mNorm[0] = -mNorm[0];
          mNorm[1] = -mNorm[1];
          mNorm[2] = -mNorm[2];
          mDist    = -mDist;
      }

    /// @return Dist in Nx*Px + Ny*Py + Nz*Pz - Dist = 0
    /// This is useful for intersection tests
    inline Real getDist() const {return mDist;}

    /// @return the normal of the plane
    inline Vector getNorm() const {return mNorm;}
    inline Vector getNormal() const {return this->getNorm();}

    /// @return the point on the plane used for the construction
    inline Point getPoint() const {return mPoint;}

    /// @return component of the point used for the construction
    inline Real getPoint(int i) const {return mPoint[i];}

    /// @return component of the plane normal
    inline Real getNorm(int i) const {return mNorm[i];}

    /// @return the point in the plane that intersects a line through
    /// the origin and a direction defined by the plane normal
    inline Point  getNormalPoint() const {return mDist*mNorm;}

    /// redefines normal of plane but assumes the plane is still
    /// going through the intersection point of the old plan normal.
    /// Assumes N is normalized
    inline void setNorm(const Vector &N)
      {
          mDist = N.dot(mPoint);
          mNorm = N;
      }

    /// redefines plane position but maintains the old normal
    inline void setPoint(const Point &P0)
      {
          mPoint= P0;
          mDist = mNorm.dot(P0);
      }

    /// redefines plane. Assumes N is normalized
    inline void set(const Point &P0, const Vector &N)
      {
          mNorm = N;
          mPoint= P0;
          mDist = N.dot(P0);
      }

    /// redefines plane. Does not assume N is normalized
    inline void set(const Point &P0, const Vector &N, Normalize)
      {
          mNorm = N;
          mNorm *= Inv(mNorm.length());
          mPoint= P0;
          mDist = N.dot(P0);
      }

    /// @return true if Planes are identical. Note that strictly
    /// speaking two Planes can have different Points but still
    /// define identical half-planes. If you only care about the
    /// actual half-planes explicitly test the Norm and Dist!
    /// the .eq operator applies a tolerance to the equality 
    inline bool operator==(const Plane &other) const
      {
          return mNorm.eq(other.mNorm) && mPoint.eq(other.mPoint);
      }

    /// @return true if Planes are different. Note that strictly
    /// speaking two Planes can have different Points but still
    /// define identical half-planes. If you only care about the
    /// actual half-planes explicitly test the Norm and Dist!
    inline bool operator!=(const Plane &other) const
      {
          return  !(*this == other );
      }

    /// @return TRUE if the plane is well defined, i.e. the normal
    /// is non-zero!
    inline bool isValid() const {return (mNorm[0] || mNorm[1] || mNorm[2]);}

    /// @return TRUE is the point, P, is inside the half-plane,
    /// i.e. on the opposit side of the plane normal!
    inline bool isInside(const Point &P) const
      {
          return mNorm.dot(P) < mDist;
      }

    /// Convenient for checking vertices of an edge or ray
    inline bool isInside(const Point &P1, const Point &P2) const
      {
          return mNorm.dot(P1)<mDist && mNorm.dot(P2)<mDist;
      }

    /// Convenient for checking vertices of a trinagle
    inline bool isInside(const Point &P1, const Point &P2, const Point &P3) const
      {
          return mNorm.dot(P1)<mDist && mNorm.dot(P2)<mDist && mNorm.dot(P3)<mDist;
      }

    /// @return true if point is out the plane defined by the normal
    inline bool isOutside(const Point &P) const
      {
          return mNorm.dot(P) > mDist;
      }

    /// Convenient for checking vertices of an edge or ray
    inline bool isOutside(const Point &P1, const Point &P2) const
      {
          return mNorm.dot(P1)>mDist && mNorm.dot(P2)>mDist;
      }

    // Convenient for checking vertices of a trinagle
    inline bool isOutside(const Point &P1, const Point &P2, const Point &P3) const
      {
          return mNorm.dot(P1)>mDist && mNorm.dot(P2)>mDist && mNorm.dot(P3)>mDist;
      }

    /// @return ture if point is on the plane
    inline bool isOn(const Point &pt, Real tolerance=1e-6) const
      {
          return std::abs(mNorm.dot(pt) - mDist) < tolerance;
      }

    /// Convenient for checking vertices of an edge or ray
    inline bool intersects(const Point &P1, const Point &P2) const
      {
          return (mNorm.dot(P1)-mDist)*(mNorm.dot(P2)-mDist)<0;
      }

    /// @return true if the ray intersects the plane
    inline bool intersects(const Ray<Real> &ray) const
      {
          return this->intersects(ray.start(),ray.end());
      }

    /// @return true if the eay intersects the plane. In this case
    /// t takes the value of the intersection point
    inline bool intersects(const Ray<Real> &ray, Real &t) const
      {
          const Real cosAngle = mNorm.dot(ray.getDirection());//= PlaneNorm . RayNorm
          if (isExactlyEqual(cosAngle, 0.0)) return false;//parallel
          t = (mDist-mNorm.dot(ray.getOrigin()))/cosAngle;
          if (t<ray.getMinTime() || t>ray.getMaxTime()) return false;//out of range
          return true;
      }

    /// @return angle between planes in radians
    inline Real angle(const Plane &other) const {return acos(mNorm.dot(other.mNorm));}

    /// @return plane from spherical interpolation of normals and linear
    /// interpolation of the positions. Note that t is clamped to
    /// [0;1] and t=0 returns this and t=1 returns other!
    Plane slerp(const Plane &other, Real t) const {
        const Real t1 = Clamp01(t), t0 = 1-t1;
        const Real NdotN = mNorm.dot(other.mNorm);
        if (NdotN > 0.9995) {//normals are almost parallel so do lerp!
            return Plane(t0*mPoint + t1*other.mPoint,
                         t0*mNorm  + t1*other.mNorm,Normalize());//slerp normals
        }
        const Real angle=acos(NdotN), inv = Real(1)/sin(angle);
        const Real c0 = sin(t0*angle)*inv, c1 = sin(t1*angle)*inv;
        return Plane(t0*mPoint + t1*other.mPoint,//lerp positions
                     c0*mNorm  + c1*other.mNorm);//slerp normals

    }

    /// @return linear interpolation of planes (both normals and
    /// positions). Note that t is clamped to [0;1] and t=0
    /// returns this and t=1 returns other!
    Plane lerp(const Plane &other, Real t) const {
        const Real t1 = Clamp01(t), t0 = 1-t1;
        return Plane(t0*mPoint + t1*other.mPoint,//lerp positions
                     t0*mNorm  + t1*other.mNorm,Normalize());//lerp normals
    }
    /// @return linear interpolation of planes (ONLY positions).
    /// Note that t is clamped to [0;1] and t=0 returns this and t=1 returns other!
    Plane lerpPoints(const Plane &other, Real t) const
      {
          const Real t1 = Clamp01(t), t0 = 1-t1;
          return Plane(t0*mPoint + t1*other.mPoint,mNorm);//lerp positions
      }
};//end of Plane class


/// Simple output streaming of a Plane. Writes normal and position
template<typename Real>
inline std::ostream& operator<<(std::ostream& os, const Plane<Real>& p) {
    os << "Plane: N=" << p.getNorm() << " P=" << p.getPoint() << " D=" << p.getDist();
    return os;
}

} // namespace math
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_MATH_PLANE_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
