//                                               -*- C++ -*-
/**
 *  @file  t_RandomMixture_std.cxx
 *  @brief The test file of class RandomMixture for standard methods
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-07-03 08:31:34 +0200 (jeu, 03 jui 2008) $
 *  Id:      $Id: t_RandomMixture_std.cxx 867 2008-07-03 06:31:34Z lebrun $
 */
#include <iostream>
#include <sstream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "RandomMixture.hxx"
#include "Distribution.hxx"
#include "Uniform.hxx"
#include "Triangular.hxx"
#include "Normal.hxx"
#include "Exponential.hxx"
#include "Gamma.hxx"
#include "Collection.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Distribution;
using namespace OT::Uncertainty::Model;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    // Create a collection of test-cases and the associated references
    UnsignedLong numberOfTests(3);
    Collection< Collection< Distribution > > testCases(numberOfTests);
    Collection< Distribution > references(numberOfTests);
    testCases[0] = Collection<Distribution>(2);
    testCases[0][0] = Uniform(-1.0, 3.0);
    testCases[0][1] = Uniform(-1.0, 3.0);
    references[0] = Triangular(-2.0, 2.0, 6.0);
    testCases[1] = Collection<Distribution>(3);
    testCases[1][0] = Normal();
    testCases[1][1] = Normal(1.0, 2.0);
    testCases[1][2] = Normal(-2.0, 2.0);
    references[1] = Normal(-1.0, 3.0);
    testCases[2] = Collection<Distribution>(3);
    testCases[2][0] = Exponential();
    testCases[2][1] = Exponential();
    testCases[2][2] = Exponential();
    references[2] = Gamma(3.0, 1.0, 0.0);
    fullprint << "testCases=" << testCases << std::endl;
    fullprint << "references=" << references << std::endl;
    for (UnsignedLong testIndex = 0; testIndex < testCases.getSize(); ++testIndex)
      {
	// Instanciate one distribution object
	RandomMixture distribution(testCases[testIndex]);
	distribution.setBlockMin(5);
	distribution.setBlockMax(20);
	Distribution distributionReference(references[testIndex]);
	fullprint << "Distribution " << distribution << std::endl;

	// Is this distribution elliptical ?
	fullprint << "Elliptical = " << (distribution.isElliptical() ? "true" : "false") << std::endl;
	
	// Is this distribution continuous ?
	fullprint << "Continuous = " << (distribution.isContinuous() ? "true" : "false") << std::endl;
	
	// Test for realization of distribution
	NumericalPoint oneRealization = distribution.getRealization();
	fullprint << "oneRealization=" << oneRealization << std::endl;
	
	// Test for sampling
	UnsignedLong size = 10000;
	NumericalSample oneSample = distribution.getNumericalSample( size );
	fullprint << "oneSample first=" << oneSample[0] << " last=" << oneSample[size - 1] << std::endl;
	fullprint << "mean=" << oneSample.computeMean() << std::endl;
	fullprint << "covariance=" << oneSample.computeCovariance() << std::endl;
	
	// Define a point
	NumericalPoint point(distribution.getDimension(), 0.5);
	fullprint << "Point= " << point << std::endl;
	
	// Show PDF and CDF of point
	NumericalScalar eps(1e-5);
	NumericalPoint DDF = distribution.computeDDF(point);
	fullprint << "ddf      =" << DDF << std::endl;
	fullprint << "ddf  (FD)=" << NumericalPoint(1, (distribution.computePDF( point + NumericalPoint(1, eps) ) - distribution.computePDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps)) << std::endl;
	fullprint << "ddf (ref)=" << distributionReference.computeDDF(point) << std::endl;
	NumericalScalar PDF = distribution.computePDF(point);
	fullprint << "pdf      =" << PDF << std::endl;
	fullprint << "pdf  (FD)=" << (distribution.computeCDF( point + NumericalPoint(1, eps) ) - distribution.computeCDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps) << std::endl;
	fullprint << "pdf (ref)=" << distributionReference.computePDF(point) << std::endl;
	NumericalScalar CDF = distribution.computeCDF( point );
	fullprint << "cdf      =" << CDF << std::endl;
	fullprint << "cdf (ref)=" << distributionReference.computeCDF(point) << std::endl;
	NumericalPoint quantile = distribution.computeQuantile( 0.95 );
	fullprint << "quantile      =" << quantile << std::endl;
	fullprint << "quantile (ref)=" << distributionReference.computeQuantile(0.95) << std::endl;
	fullprint << "cdf(quantile)=" << distribution.computeCDF(quantile) << std::endl;
	NumericalPoint mean = distribution.getMean();
	fullprint << "mean      =" << mean << std::endl;
	fullprint << "mean (ref)=" << distributionReference.getMean() << std::endl;
	NumericalPoint standardDeviation = distribution.getStandardDeviation();
	fullprint << "standard deviation      =" << standardDeviation << std::endl;
	fullprint << "standard deviation (ref)=" << distributionReference.getStandardDeviation() << std::endl;
	NumericalPoint skewness = distribution.getSkewness();
	fullprint << "skewness      =" << skewness << std::endl;
	fullprint << "skewness (ref)=" << distributionReference.getSkewness() << std::endl;
	NumericalPoint kurtosis = distribution.getKurtosis();
	fullprint << "kurtosis      =" << kurtosis << std::endl;
	fullprint << "kurtosis (ref)=" << distributionReference.getKurtosis() << std::endl;
	CovarianceMatrix covariance = distribution.getCovariance();
	fullprint << "covariance      =" << covariance << std::endl;
	fullprint << "covariance (ref)=" << distributionReference.getCovariance() << std::endl;
	RandomMixture::NumericalPointWithDescriptionCollection parameters = distribution.getParametersCollection();
	fullprint << "parameters=" << parameters << std::endl;
	
	// Specific to this distribution
	NumericalPoint weights(distribution.getWeights());
	fullprint << "weights=" << weights << std::endl;
	distribution.setWeights(2.0 * weights);
	fullprint << "new weights=" << distribution.getWeights() << std::endl;
	fullprint << "blockMin=" << distribution.getBlockMin() << std::endl;
	fullprint << "blockMax=" << distribution.getBlockMax() << std::endl;
	fullprint << "maxSize=" << distribution.getMaxSize() << std::endl;
	fullprint << "alpha=" << distribution.getAlpha() << std::endl;
	fullprint << "beta=" << distribution.getBeta() << std::endl;
      }
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
