//                                               -*- C++ -*-
/**
 *  @file  RiskyAndFast.cxx
 *  @brief Find the roots in a given direction according to the Proban
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: RiskyAndFast.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "RiskyAndFast.hxx"
#include "NumericalPoint.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      typedef Base::Type::NumericalPoint                      NumericalPoint;
      typedef RootStrategyImplementation::NotDefinedException NotDefinedException;

      /**
       * @class RiskyAndFast
       */

      CLASSNAMEINIT(RiskyAndFast);

      /* Constructor with parameters */
      RiskyAndFast::RiskyAndFast():
	RootStrategyImplementation()
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      RiskyAndFast::RiskyAndFast(const Solver & solver):
	RootStrategyImplementation(solver)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      RiskyAndFast::RiskyAndFast(const Solver & solver,
				 const NumericalScalar maximumDistance):
	RootStrategyImplementation(solver, maximumDistance, 0.0)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      RiskyAndFast * RiskyAndFast::clone() const
      {
	return new RiskyAndFast(*this);
      }

      /* Solve gives all the roots found applying the root strategy */
      RiskyAndFast::NumericalScalarCollection RiskyAndFast::solve(const NumericalMathFunction & function,
								  const NumericalScalar value)
      {
	NumericalScalarCollection result(0);
	NumericalScalar infPoint(0.0);
	NumericalScalar infValue;
	// Get the value of the function at the origin
	try
	  {
	    infValue = getOriginValue();
	  }
	// If it has not yet been computed, compute it and store it
	catch (NotDefinedException & ex)
	  {
	    infValue = function(NumericalPoint(1, infPoint))[0];
	    setOriginValue(infValue);
	  }
	NumericalScalar supPoint(getMaximumDistance());
	NumericalScalar supValue(function(NumericalPoint(1, supPoint))[0]);
	Solver solver(getSolver());
	// If there is a sign change on [0, maximumDistance]
	if ((infValue - value) * (supValue - value) < 0.0)
	  {
	    result.add(solver.solve(function, value, infPoint, supPoint, infValue, supValue));
	  }
	return result;
      }

      /* String converter */
      String RiskyAndFast::__repr__() const
      {
	OSS oss;
	oss << "class=" << RiskyAndFast::GetClassName()
	    << " derived from " << RootStrategyImplementation::__repr__();
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
