//                                               -*- C++ -*-
/**
 *  @file  QuadraticCumul.hxx
 *  @brief 
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: QuadraticCumul.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_QUADRATICCUMUL_HXX
#define OPENTURNS_QUADRATICCUMUL_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "NumericalPointWithDescription.hxx"
#include "RandomVector.hxx"
#include "Exception.hxx"
#include "SymmetricTensor.hxx"
#include "CovarianceMatrix.hxx"
#include "Matrix.hxx"
#include "Graph.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {


      /**
       * @class QuadraticCumul
       * QuadraticCumul implements the mean and covariance of a random vector
       Y=G(X) by implementing the Taylor approximation of G
      */

      class QuadraticCumul
	: public Base::Common::PersistentObject
      {

	CLASSNAME;
      public:

	typedef Base::Type::NumericalPoint                          NumericalPoint;
	typedef Base::Type::NumericalPointWithDescription           NumericalPointWithDescription;
	typedef Base::Type::SymmetricTensor                         SymmetricTensor;
	typedef Base::Type::Matrix                                  Matrix;
	typedef Base::Stat::CovarianceMatrix                        CovarianceMatrix;
	typedef Base::Common::InvalidArgumentException              InvalidArgumentException;
	typedef Base::Common::InternalException                     InternalException;
	typedef Base::Common::NotDefinedException                   NotDefinedException;
	typedef Base::Common::NotSymmetricDefinitePositiveException NotSymmetricDefinitePositiveException;
	typedef Base::Common::InvalidDimensionException             InvalidDimensionException;
	typedef Model::RandomVector                                 RandomVector;
	typedef Base::Graph::Graph                                  Graph;
	typedef Base::Common::StorageManager                        StorageManager;

	/** Default constructor */
	QuadraticCumul() {};

	/** Constructor with parameters */
	explicit QuadraticCumul(const RandomVector & limitStateVariable, const String & name = OT::DefaultName)
	  /* throw(InvalidArgumentException) */;


	/** Virtual constructor */
	virtual QuadraticCumul * clone() const;

	/** String converter */
	String __repr__() const;

	/** limitStateVariable accessor */
	RandomVector getLimitStateVariable() const;

	/** meanFirstOrder accessor */
	NumericalPoint getMeanFirstOrder();

	/** meanSecondOrder accessor */
	NumericalPoint getMeanSecondOrder();

	/** covariance accessor */
	CovarianceMatrix getCovariance();

	/** importance factors accessor */
	NumericalPointWithDescription getImportanceFactors();

	/** ImportanceFactors graph */
	Graph drawImportanceFactors();

 	/** Method save() stores the object through the StorageManager */
	void save(StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(StorageManager::Advocate & adv);

      protected:

	friend class Base::Common::Factory<QuadraticCumul>;

      private:

	/** the function that computes the first order evaluation of the mean vector  */
	void computeMeanFirstOrder()
	  /* throw(InvalidArgumentException, InternalException, NotDefinedException) */;

	/** the function that computes the second order evaluation of the mean vector */
	void computeMeanSecondOrder()
	  /* throw(InvalidArgumentException, InternalException, NotDefinedException) */;

	/** the function that computes the matrix covariance */
	void computeCovariance()
	  /* throw(NotSymmetricDefinitePositiveException, InvalidArgumentException, InternalException, NotDefinedException) */;

	/** the function that computes the importance factors only in the scalar case */
	void computeImportanceFactors ()
	  /* throw(NotSymmetricDefinitePositiveException, InvalidDimensionException, InvalidArgumentException, InternalException, NotDefinedException) */;
 
	RandomVector limitStateVariable_;
	NumericalPoint meanInputVector_;
	NumericalPoint valueAtMean_;
	Matrix gradientAtMean_; 
	SymmetricTensor hessianAtMean_; 
	Bool isAlreadyComputedValue_;
	Bool isAlreadyComputedGradient_;
	Bool isAlreadyComputedHessian_;
	Bool isAlreadyComputedMeanFirstOrder_;
	Bool isAlreadyComputedMeanSecondOrder_;
	Bool isAlreadyComputedCovariance_;
	Bool isAlreadyComputedImportanceFactors_;
	CovarianceMatrix inputCovariance_;
	NumericalPoint meanFirstOrder_;
	NumericalPoint meanSecondOrder_;
	CovarianceMatrix covariance_;
	NumericalPointWithDescription importanceFactors_;

      } ; /* class QuadraticCumul */

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_QUADRATICCUMUL_HXX */
