//                                               -*- C++ -*-
/**
 *  @file  FunctionalChaosRandomVector.cxx
 *  @brief An implementation class for functionalChaos random vectors
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-07-03 08:31:34 +0200 (jeu. 03 juil. 2008) $
 *  Id:      $Id: FunctionalChaosRandomVector.cxx 867 2008-07-03 06:31:34Z lebrun $
 */
#include "PersistentObjectFactory.hxx"
#include "FunctionalChaosRandomVector.hxx"
#include "UsualRandomVector.hxx"
#include "EnumerateFunction.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      CLASSNAMEINIT(FunctionalChaosRandomVector);

      static Base::Common::Factory<FunctionalChaosRandomVector> RegisteredFactory("FunctionalChaosRandomVector");

      typedef Algorithm::EnumerateFunction EnumerateFunction;

      /* Default constructor */
      FunctionalChaosRandomVector::FunctionalChaosRandomVector(const FunctionalChaosResult & functionalChaosResult)
	: CompositeRandomVector(functionalChaosResult.getMetaModel(), UsualRandomVector(functionalChaosResult.getDistribution()).clone()),
	  functionalChaosResult_(functionalChaosResult),
	  covariance_(0.0),
	  isAlreadyComputedCovariance_(false)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      FunctionalChaosRandomVector * FunctionalChaosRandomVector::clone() const
      {
	return new FunctionalChaosRandomVector(*this);
      }

      /* String converter */
      String FunctionalChaosRandomVector::__repr__() const {
	OSS oss;
	oss << "class=" << FunctionalChaosRandomVector::GetClassName()
	    << " functional chaos result=" << functionalChaosResult_;
	return oss;
      }
  

      /* Mean accessor */
      FunctionalChaosRandomVector::NumericalPoint FunctionalChaosRandomVector::getMean() const
      {
	return NumericalPoint(1, functionalChaosResult_.getCoefficients()[0]);
      }

      /* Covariance accessor */
      FunctionalChaosRandomVector::CovarianceMatrix FunctionalChaosRandomVector::getCovariance() const
      {
	CovarianceMatrix covariance(1);
	if (!isAlreadyComputedCovariance_)
	  {
	    computeCovariance();
	  }
	// Put the scalar covariance into a CovarianceMatrix
	covariance(0, 0) = covariance_;
	return covariance;
      }

      /* Compute the scalar covariance */
      void FunctionalChaosRandomVector::computeCovariance() const
      {
	covariance_ = 0.0;
	const NumericalPoint coefficients(functionalChaosResult_.getCoefficients());
	const UnsignedLong size(coefficients.getDimension());
	// The covariance is equal to the sum of the coefficients squared excluding the first coefficient. It is computed this way to avoid roundoff error.
	for (UnsignedLong i = 1; i < size; ++i)
	  {
	    covariance_ += coefficients[i] * coefficients[i];
	  }
	isAlreadyComputedCovariance_ = true;
      }

      /* Sobol index accessor */
      NumericalScalar FunctionalChaosRandomVector::getSobolIndex(const Indices & variableIndices) const
      {
      	const UnsignedLong inputDimension(getAntecedent()->getDimension());
	if (!variableIndices.check(inputDimension - 1)) throw InvalidArgumentException(HERE) << "The variable indices of a Sobol indice must be in the range [0, dim-1] and  must be different.";
	const UnsignedLong orderSobolIndice(variableIndices.getSize());
	const NumericalPoint coefficients(functionalChaosResult_.getCoefficients());
	const Indices coefficientIndices(functionalChaosResult_.getIndices());
	const UnsignedLong size(coefficients.getDimension());
	NumericalScalar covarianceVariables(0.0);
	const EnumerateFunction enumerateFunction(functionalChaosResult_.getOrthogonalBasis().getEnumerateFunction());
	// Sum the contributions to all the coefficients associated to a basis vector involving only the needed variables
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    Indices multiIndices(enumerateFunction(coefficientIndices[i]));
	    Bool isProperSubset(true);
	    // First check that the exponents associated with the selected variables are > 0
	    for (UnsignedLong j = 0; j < orderSobolIndice; ++j)
	      {
		const UnsignedLong varJ(variableIndices[j]);
		isProperSubset = isProperSubset && (multiIndices[varJ] > 0);
		multiIndices[varJ] = 0;
	      }
	    // Second, check that the other coefficients are 0
	    for (UnsignedLong j = 0; j < inputDimension; ++j)
	      {
		isProperSubset = isProperSubset && (multiIndices[j] == 0);
	      }
	    if (isProperSubset) covarianceVariables += coefficients[i] * coefficients[i];
	  }
	return covarianceVariables / getCovariance()(0, 0);
      }

      /* Sobol index accessor */
      NumericalScalar FunctionalChaosRandomVector::getSobolIndex(const UnsignedLong variableIndex) const
      {
	Indices index(1);
	index[0] = variableIndex;
	return getSobolIndex(index);
      }

      /* Sobol total index accessor */
      NumericalScalar FunctionalChaosRandomVector::getSobolTotalIndex(const Indices & variableIndices) const
      {
      	const UnsignedLong inputDimension(getAntecedent()->getDimension());
	if (!variableIndices.check(inputDimension - 1)) throw InvalidArgumentException(HERE) << "The variable indices of a Sobol indice must be in the range [0, dim-1] and  must be different.";
	const UnsignedLong orderSobolIndice(variableIndices.getSize());
	const NumericalPoint coefficients(functionalChaosResult_.getCoefficients());
	const Indices coefficientIndices(functionalChaosResult_.getIndices());
	const UnsignedLong size(coefficients.getDimension());
	NumericalScalar covarianceVariables(0.0);
	const EnumerateFunction enumerateFunction(functionalChaosResult_.getOrthogonalBasis().getEnumerateFunction());
	// Sum the contributions to all the coefficients associated to a basis vector involving at least the variable i
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    const Indices multiIndices(enumerateFunction(coefficientIndices[i]));
	    Bool isProperSubset(true);
	    // Check that the exponents associated with the selected variables are > 0
	    for (UnsignedLong j = 0; j < orderSobolIndice; ++j)
	      {
		const UnsignedLong varJ(variableIndices[j]);
		isProperSubset = isProperSubset && (multiIndices[varJ] > 0);
	      }
	    if (isProperSubset) covarianceVariables += coefficients[i] * coefficients[i];
	  }
	return covarianceVariables / getCovariance()(0, 0);
      }

      /* Sobol total index accessor */
      NumericalScalar FunctionalChaosRandomVector::getSobolTotalIndex(const UnsignedLong variableIndex) const
      {
	Indices index(1);
	index[0] = variableIndex;
	return getSobolTotalIndex(index);
      }

      /* Functional chaos result accessor */
      FunctionalChaosRandomVector::FunctionalChaosResult FunctionalChaosRandomVector::getFunctionalChaoResult() const
      {
	return functionalChaosResult_;
      }

      /* Method save() stores the object through the StorageManager */
      void FunctionalChaosRandomVector::save(const StorageManager::Advocate & adv) const
      {
	RandomVectorImplementation::save(adv);
	adv.writeValue(functionalChaosResult_, StorageManager::MemberNameAttribute, "functionalChaosResult_");
      }

      /* Method load() reloads the object from the StorageManager */
      void FunctionalChaosRandomVector::load(const StorageManager::Advocate & adv)
      {
	RandomVectorImplementation::load(adv);
	adv.readValue(functionalChaosResult_, StorageManager::MemberNameAttribute, "functionalChaosResult_");
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
