//                                               -*- C++ -*-
/**
 *  @file  Weibull.cxx
 *  @brief The Weibull distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Weibull.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cmath>
#include "Weibull.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Weibull);

      /* Default constructor */
      Weibull::Weibull()
	: NonEllipticalDistribution("Weibull"),
	  alpha_(1.0), beta_(1.0), gamma_(0.0)
      {
	setDimension(1);
	computeRange();
      }

      /* Parameters constructor */
      Weibull::Weibull(const NumericalScalar arg1,
		       const NumericalScalar arg2,
		       const NumericalScalar gamma,
		       const ParameterSet set)
	throw (InvalidArgumentException)
	: NonEllipticalDistribution("Weibull"),
	  alpha_(0.0), beta_(0.0), gamma_(gamma)
      {
	switch (set) {
	case ALPHABETA:
	  setAlpha(arg1);
	  setBeta(arg2);
	  break;
	  
	case MUSIGMA:
	  setMuSigma(arg1, arg2);
	  break;

	default:
	  throw InvalidArgumentException(HERE) << "Invalid parameter set argument";

	} /* end switch */

	setDimension(1);
	computeRange();
      }

      /* Comparison operator */
      Bool Weibull::operator ==(const Weibull & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO: Write Weibull::operator ==(...)
	  sameObject = (alpha_ == other.alpha_) && (beta_ == other.beta_) && (gamma_ == other.gamma_);
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Weibull::__repr__() const
      {
	OSS oss;
	oss << "class=" << Weibull::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " alpha=" << alpha_
	    << " beta=" << beta_
	    << " gamma=" << gamma_;
	return oss;
      }
  
      /* Virtual constructor */
      Weibull * Weibull::clone() const
      {
	return new Weibull(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Weibull::computeRange()
      {
	const NumericalPoint lowerBound(1, gamma_);
	const NumericalPoint upperBound(1, gamma_ + alpha_ * pow(SpecFunc::LogMaxNumericalScalar, 1.0 / beta_));
	const Interval::BoolCollection finiteLowerBound(1, true);
	const Interval::BoolCollection finiteUpperBound(1, false);
	setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }


      /* Get one realization of the distribution */
      Weibull::NumericalPoint Weibull::getRealization() const
      {
	return NumericalPoint(1, gamma_ + alpha_ * pow(-log(1.0 - RandomGenerator::Generate()), 1.0 / beta_));
      }
     

      /* Get the DDF of the distribution */
      Weibull::NumericalPoint Weibull::computeDDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return NumericalPoint(1, 0.0);
	const NumericalScalar powX(pow(x / alpha_, beta_));
	return NumericalPoint(1, (beta_ * (1.0 - powX) - 1.0) / (x * x) * beta_ * powX * exp(-powX));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Weibull::computePDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return 0.0;
	const NumericalScalar powX(pow(x / alpha_, beta_));
	return beta_ / x * powX * exp(-powX);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Weibull::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	const NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return (tail ? 1.0 : 0.0);
	if (tail) return exp(-pow(x / alpha_, beta_));
	return 1.0 - exp(-pow(x / alpha_, beta_));
      }

      /* Get the PDFGradient of the distribution */
      Weibull::NumericalPoint Weibull::computePDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	NumericalPoint pdfGradient(3, 0.0);
	if (x <= 0.0) return pdfGradient;
	const NumericalScalar powX(pow(x / alpha_, beta_));
	const NumericalScalar factor(powX / x * exp(-powX));
	pdfGradient[0] = factor * (powX - 1.0) * beta_ * beta_ / alpha_;
	pdfGradient[1] = factor * (1.0 + (1.0 - powX) * log(powX));
	pdfGradient[2] = factor * (1.0 - beta_ + beta_ * powX) / x * beta_;
	return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Weibull::NumericalPoint Weibull::computeCDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	NumericalPoint cdfGradient(3, 0.0);
	if (x <= 0.0) return cdfGradient;
	const NumericalScalar powX(pow(x / alpha_, beta_));
	const NumericalScalar factor(powX * exp(-powX));
	cdfGradient[0] = -factor * beta_ / alpha_;
	cdfGradient[1] = factor * log(x / alpha_);
	cdfGradient[2] = -factor * beta_ / x;
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Weibull::computeScalarQuantile(const NumericalScalar prob,
						     const NumericalScalar initialGuess,
						     const NumericalScalar initialStep,
						     const NumericalScalar precision) const
      {
	return gamma_ + alpha_ * pow(-log(1.0 - prob), 1.0 / beta_);
      }

      /* Get the mean of the distribution */
      Weibull::NumericalPoint Weibull::getMean() const throw(NotDefinedException)
      {
	return NumericalPoint(1, getMu());
      }

      /* Get the standard deviation of the distribution */
      Weibull::NumericalPoint Weibull::getStandardDeviation() const throw(NotDefinedException)
      {
	return NumericalPoint(1, getSigma());
      }

      /* Get the skewness of the distribution */
      Weibull::NumericalPoint Weibull::getSkewness() const throw(NotDefinedException)
      {
	const NumericalScalar gamma1(SpecFunc::Gamma(1.0 + 1.0 / beta_));
	const NumericalScalar gamma2(SpecFunc::Gamma(1.0 + 2.0 / beta_));
	const NumericalScalar gamma3(SpecFunc::Gamma(1.0 + 3.0 / beta_));
	return NumericalPoint(1, (2.0 * gamma1 * gamma1 * gamma1 - 3.0 * gamma1 * gamma2 + gamma3) * pow(alpha_ / (gamma2 - gamma1 * gamma1), 1.5));
      }

      /* Get the kurtosis of the distribution */
      Weibull::NumericalPoint Weibull::getKurtosis() const throw(NotDefinedException)
      {
	const NumericalScalar gamma1(SpecFunc::Gamma(1.0 + 1.0 / beta_));
	const NumericalScalar gamma12(gamma1 * gamma1);
	const NumericalScalar gamma2(SpecFunc::Gamma(1.0 + 2.0 / beta_));
	const NumericalScalar gamma3(SpecFunc::Gamma(1.0 + 3.0 / beta_));
	const NumericalScalar gamma4(SpecFunc::Gamma(1.0 + 4.0 / beta_));
	return NumericalPoint(1, (6.0 * gamma12 * gamma2 + gamma4 - 4.0 * gamma1 * gamma3 - 3.0 * gamma12 * gamma12) * pow(alpha_ / (gamma2 - gamma12), 2.0));
      }

      /* Get the covariance of the distribution */
      Weibull::CovarianceMatrix Weibull::getCovariance() const throw(NotDefinedException)
      {
	CovarianceMatrix covariance(1);
	const NumericalScalar gammaOneOverBeta(SpecFunc::Gamma(1.0 + 1.0 / beta_));
	covariance(0, 0) = alpha_ * (SpecFunc::Gamma(1.0 + 2.0 / beta_) - gammaOneOverBeta * gammaOneOverBeta);
	return covariance;
      }

      /* Parameters value and description accessor */
      Weibull::NumericalPointWithDescriptionCollection Weibull::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(3);
        Description description(point.getDimension());
	point[0] = alpha_;
	point[1] = beta_;
	point[2] = gamma_;
	description[0] = "alpha";
	description[1] = "beta";
	description[2] = "gamma";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }

      void Weibull::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
	*this = Weibull(parametersCollection[0][0], parametersCollection[0][1], parametersCollection[0][2]);
      }


      /* Beta accessor */
      void Weibull::setBeta(const NumericalScalar beta)
	throw(InvalidArgumentException)
      {
	if (beta <= 0.) throw InvalidArgumentException(HERE) << "Beta MUST be positive";
	beta_ = beta;
	computeRange();
      }

      NumericalScalar Weibull::getBeta() const
      {
	return beta_;
      }


      /* Alpha accessor */
      void Weibull::setAlpha(const NumericalScalar alpha)
	throw(InvalidArgumentException)
      {
	if (alpha <= 0.) throw InvalidArgumentException(HERE) << "Alpha MUST be positive";
	alpha_ = alpha;
	computeRange();
      }

      NumericalScalar Weibull::getAlpha() const
      {
	return alpha_;
      }


      /* MuSigma accessor */
      /* To compute (alpha, beta) from (mu, sigma), we proceed as follows:
         var_ / (mu_ - gamma_) = GAMMA(1+2/beta_)/GAMMA(1+1/beta_)-GAMMA(1+1/beta_)
	 = t(beta_)
         This function is strictly decreasing and takes arbitrary large values near 0.
	 We solve the equation t(beta) = var_ / (mu_ - gamma_) using Bisection's method, as it is very robust and we don't really have a speed constraint here. Then, we find alpha_ thanks to:
         alpha_ = (mu_ - gamma_) / GAMMA(1 + 1 / beta_)
      */
      void Weibull::setMuSigma(const NumericalScalar mu,
			       const NumericalScalar sigma)
	throw(InvalidArgumentException)
      {
	if (mu <= gamma_) throw InvalidArgumentException(HERE) << "Mu MUST be > gamma, here mu=" << mu << " and gamma=" << gamma_;
	if (sigma <= 0.0) throw InvalidArgumentException(HERE) << "Sigma MUST be > 0.0, here sigma=" << sigma;
	const NumericalScalar ratio(sigma * sigma / (mu - gamma_));
	NumericalScalar t;
	NumericalScalar betaMin(1.0);
	NumericalScalar betaMax(1.0);
	NumericalScalar step(0.5);
	NumericalScalar lng;
	// Bracketing interval
	// Case beta < 1, i.e. ratio > 1
	if (ratio > 1)
	  {
	    do
	      {
		betaMin -= step;
		step *= 0.5;
		lng = SpecFunc::LnGamma(1.0 + 1.0 / betaMin);
		t = exp(SpecFunc::LnGamma(1.0 + 2.0 / betaMin) - lng) - exp(lng);
	      } while (t < ratio);
	    // Here, we know that betaMin <= beta < betaMin + 2.0 * step
	    betaMax = betaMin + 2.0 * step;
	  }
	// Case beta >= 1, i.e. ratio <= 1
	else
	  {
	    do
	      {
		betaMax += step;
		step *= 2.0;
		lng = SpecFunc::LnGamma(1.0 + 1.0 / betaMax);
		t = exp(SpecFunc::LnGamma(1.0 + 2.0 / betaMax) - lng) - exp(lng);
	      } while (t >= ratio);
	    // Here, we know that betaMax - 0.5 * step <= beta < betaMax
	    betaMin = betaMax - 0.5 * step;
	  }
	// Bisection loop
	for (;;)
	  {
	    beta_ = 0.5 * (betaMin + betaMax);
	    lng = SpecFunc::LnGamma(1.0 + 1.0 / beta_);
	    // Convergence
	    if (betaMax - betaMin <= DefaultQuantileEpsilon * (1.0 + fabs(betaMax + betaMin)))
	      {
		alpha_ = (mu - gamma_) / exp(lng);
		return;
	      }
	    // Non convergence, one step further
	    t = exp(SpecFunc::LnGamma(1.0 + 2.0 / beta_) - lng) - exp(lng);
	    if (t < ratio)
	      {
		betaMax = beta_;
	      }
	    else
	      {
		betaMin = beta_;
	      }
	  }
	computeRange();
      }

      NumericalScalar Weibull::getMu() const
      {
	return gamma_ + alpha_ * SpecFunc::Gamma(1.0 + 1.0 / beta_);
      }


      NumericalScalar Weibull::getSigma() const
      {
	const NumericalScalar gammaOneOverBeta(SpecFunc::Gamma(1.0 + 1.0 / beta_));
	return sqrt(alpha_ * (SpecFunc::Gamma(1.0 + 2.0 / beta_) - gammaOneOverBeta * gammaOneOverBeta));
      }


      /* Gamma accessor */
      void Weibull::setGamma(const NumericalScalar gamma)
      {
	gamma_ = gamma;
	computeRange();
      }

      NumericalScalar Weibull::getGamma() const
      {
	return gamma_;
      }


      /* Method save() stores the object through the StorageManager */
      void Weibull::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue("alpha_", alpha_);
	adv.writeValue("beta_", beta_);
	adv.writeValue("gamma_", gamma_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Weibull::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "alpha_") alpha_ = value;
	    if (name == "beta_") beta_ = value;
	    if (name == "gamma_") gamma_ = value;
	  }
	}
	computeRange();
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
