//                                               -*- C++ -*-
/**
 *  @file  Laplace.cxx
 *  @brief The Laplace distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-11-23 16:03:50 +0100 (dim, 23 nov 2008) $
 *  Id:      $Id: Laplace.cxx 1026 2008-11-23 15:03:50Z lebrun $
 */
#include <cmath>
#include "Laplace.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Laplace);

      static Base::Common::Factory<Laplace> RegisteredFactory("Laplace");

      /* Default constructor */
      Laplace::Laplace()
        : NonEllipticalDistribution("Laplace"),
          lambda_(1.0),
          mu_(0.0)
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Laplace::Laplace(const NumericalScalar lambda,
                       const NumericalScalar mu)
        throw(InvalidArgumentException)
        : NonEllipticalDistribution("Laplace"),
          lambda_(lambda),
          mu_(mu)
      {
        // We set the dimension of the Laplace distribution
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool Laplace::operator ==(const Laplace & other) const
      {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          if ( (mu_ == other.mu_) && (lambda_ == other.lambda_) )
            sameObject = true;

        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String Laplace::__repr__() const
      {
        OSS oss;
        oss << "class=" << Laplace::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " lambda=" << lambda_
            << " mu=" << mu_;
        return oss;
      }

      /* Virtual constructor */
      Laplace * Laplace::clone() const
      {
        return new Laplace(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Laplace::computeRange()
      {
        const NumericalPoint lowerBound(1, mu_ - (log(lambda_ - SpecFunc::LogMinNumericalScalar) / lambda_));
        const NumericalPoint upperBound(1, mu_ + (log(lambda_ - SpecFunc::LogMinNumericalScalar) / lambda_));
        const Interval::BoolCollection finiteLowerBound(1, false);
        const Interval::BoolCollection finiteUpperBound(1, false);
        setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Get one realization of the distribution */
      Laplace::NumericalPoint Laplace::getRealization() const
      {
        const NumericalScalar d(RandomGenerator::Generate() - 0.5);
        if (d < 0.0) return NumericalPoint(1, mu_ + lambda_ * log(1.0 + 2.0 * d));
        return NumericalPoint(1, mu_ - lambda_ * log(1.0 - 2.0 * d));
      }


      /* Get the DDF of the distribution */
      Laplace::NumericalPoint Laplace::computeDDF(const NumericalPoint & point) const
      {
        const NumericalScalar value(computePDF(point) / lambda_);
        return (point[0] < mu_ ? NumericalPoint(1, value) : NumericalPoint(1, -value));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Laplace::computePDF(const NumericalPoint & point) const
      {
        return exp(-fabs(point[0] - mu_) / lambda_) / (2.0 * lambda_);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Laplace::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        const NumericalScalar u((point[0] - mu_) / lambda_);
        if (u < 0.0) return 0.5 * exp(u);
        return 1.0 - 0.5 * exp(-u);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Laplace::computeCharacteristicFunction(const NumericalScalar x,
                                                              const Bool logScale) const
      {
        if (logScale) return NumericalComplex(0.0, mu_ * x) - log(1.0 + lambda_ * lambda_ * x * x);
        return exp(NumericalComplex(0.0, mu_ * x)) / (1.0 + lambda_ * lambda_ * x * x);
      }

      /* Get the PDFGradient of the distribution */
      Laplace::NumericalPoint Laplace::computePDFGradient(const NumericalPoint & point) const
      {
        NumericalPoint pdfGradient(2, 0.0);
        const NumericalScalar pdf(computePDF(point));
        const NumericalScalar factor(fabs(point[0] - mu_) / lambda_);
        pdfGradient[0] = (factor - 1.0) * pdf / lambda_;
        pdfGradient[1] = (point[0] > mu_ ? pdf / lambda_ : -pdf / lambda_);
        return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Laplace::NumericalPoint Laplace::computeCDFGradient(const NumericalPoint & point) const
      {
        NumericalPoint cdfGradient(2, 0.0);
        const NumericalScalar pdf(computePDF(point));
        const NumericalScalar factor(fabs(point[0] - mu_) / lambda_);
        cdfGradient[0] = -factor * pdf;
        cdfGradient[1] = -pdf;
        return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Laplace::computeScalarQuantile(const NumericalScalar prob,
                                                     const NumericalScalar initialGuess,
                                                     const NumericalScalar initialStep,
                                                     const NumericalScalar precision) const
      {
        const NumericalScalar d(prob - 0.5);
        if (d < 0.0) return mu_ + lambda_ * log(1.0 + 2.0 * d);
        return mu_ - lambda_ * log(1.0 - 2.0 * d);
      }

      /* Get the mean of the distribution */
      Laplace::NumericalPoint Laplace::getMean() const throw(NotDefinedException)
      {
        return NumericalPoint(1, mu_);
      }

      /* Get the standard deviation of the distribution */
      Laplace::NumericalPoint Laplace::getStandardDeviation() const throw(NotDefinedException)
      {
        return NumericalPoint(1, sqrt(2.0) * lambda_);
      }

      /* Get the skewness of the distribution */
      Laplace::NumericalPoint Laplace::getSkewness() const throw(NotDefinedException)
      {
        return NumericalPoint(1, 0.0);
      }

      /* Get the kurtosis of the distribution */
      Laplace::NumericalPoint Laplace::getKurtosis() const throw(NotDefinedException)
      {
        return NumericalPoint(1, 6.0);
      }

      /* Get the covariance of the distribution */
      Laplace::CovarianceMatrix Laplace::getCovariance() const throw(NotDefinedException)
      {
        CovarianceMatrix covariance(1);
        covariance(0, 0) = 2.0 * lambda_ * lambda_;
        return covariance;
      }

      /* Parameters value and description accessor */
      Laplace::NumericalPointWithDescriptionCollection Laplace::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(2);
        Description description(point.getDimension());
        point[0] = lambda_;
        point[1] = mu_;
        description[0] = "lambda";
        description[1] = "mu";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void Laplace::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = Laplace(parametersCollection[0][0], parametersCollection[0][1]);
      }

      /* Mu accessor */
      void Laplace::setMu(const NumericalScalar mu)
        throw(InvalidArgumentException)
      {
        mu_ = mu;
        computeRange();
      }

      /* Mu accessor */
      NumericalScalar Laplace::getMu() const
      {
        return mu_;
      }


      /* Lambda accessor */
      void Laplace::setLambda(const NumericalScalar lambda)
      {
        if (lambda_ <= 0.) throw InvalidArgumentException(HERE) << "Lambda MUST be positive";
        lambda_ = lambda;
        computeRange();
      }

      /* Lambda accessor */
      NumericalScalar Laplace::getLambda() const
      {
        return lambda_;
      }

      /* Method save() stores the object through the StorageManager */
      void Laplace::save(const StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.writeValue("lambda_", lambda_);
        adv.writeValue("mu_", mu_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Laplace::load(const StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);

        String name;
        NumericalScalar value;
        StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
        for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
          if (objList.readValue(name, value)) {
            if (name == "lambda_") lambda_ = value;
            if (name == "mu_") mu_ = value;
          }
        }
        computeRange();
      }




    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
