//                                               -*- C++ -*-
/**
 *  @file  Simulation.cxx
 *  @brief Simulation is a generic view of simulation methods for computing
 * probabilities and related quantities by sampling and estimation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Simulation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cstdlib>
#include <cmath>

#include "Simulation.hxx"
#include "Log.hxx"
#include "Curve.hxx"
#include "NumericalPoint.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      typedef Base::Common::Log          Log;
      typedef Base::Common::ResourceMap  ResourceMap;
      typedef Base::Graph::Curve         Curve;
      typedef Base::Type::NumericalPoint NumericalPoint;

      /*
       * @class Simulation
       */

      const UnsignedLong    Simulation::DefaultMaximumOuterSampling = 1000;//strtoul(ResourceMap::GetInstance().get("Simulation-DefaultMaximumOuterSampling").c_str(), NULL, 0);
      const NumericalScalar Simulation::DefaultMaximumCoefficientOfVariation = 1.0e-1;//strtod(ResourceMap::GetInstance().get("Simulation-DefaultMaximumCoefficientOfVariation").c_str(), NULL);
      const UnsignedLong    Simulation::DefaultBlockSize = 1;//strtoul(ResourceMap::GetInstance().get("Simulation-DefaultBlockSize").c_str(), NULL, 0);

      CLASSNAMEINIT(Simulation);

      /* Constructor with parameters */
      Simulation::Simulation(const Event & event,
			     const Bool verbose,
			     const HistoryStrategy & convergenceStrategy,
			     const HistoryStrategy & inputOutputStrategy)
	: Base::Common::PersistentObject(),
	  convergenceStrategy_(convergenceStrategy),
	  inputStrategy_(inputOutputStrategy),
	  outputStrategy_(inputOutputStrategy),
	  blockSize_(DefaultBlockSize),
	  event_(event),
	  result_(),
	  maximumOuterSampling_(DefaultMaximumOuterSampling),
	  maximumCoefficientOfVariation_(DefaultMaximumCoefficientOfVariation),
	  verbose_(verbose)
      {
	// Nothing to do
      }

      /*  Event accessor */
      Simulation::Event Simulation::getEvent() const
      {
	return event_;
      }

      /* Result accessor */
      void Simulation::setResult(const SimulationResult & result)
      {
	result_ = result;
      }

      /* Result accessor */
      SimulationResult Simulation::getResult() const
      {
	return result_;
      }

      /* Maximum sample size accessor */
      void Simulation::setMaximumOuterSampling(const UnsignedLong maximumOuterSampling)
      {
	maximumOuterSampling_ = maximumOuterSampling;
      }

      /* Maximum sample size accessor */
      UnsignedLong Simulation::getMaximumOuterSampling() const
      {
	return maximumOuterSampling_;
      }

      /* Maximum coefficient of variation accessor */
      void Simulation::setMaximumCoefficientOfVariation(const NumericalScalar maximumCoefficientOfVariation) throw(InvalidArgumentException)
      {
	// Check if the given coefficient of variation is >= 0
	//	if (maximumCoefficientOfVariation < 0.0) throw InvalidArgumentException(HERE) << "The maximum coefficient of variation must be >= 0.0";
	maximumCoefficientOfVariation_ = maximumCoefficientOfVariation;
      }

      /* Maximum coefficient of variation accessor */
      NumericalScalar Simulation::getMaximumCoefficientOfVariation() const
      {
	return maximumCoefficientOfVariation_;
      }

      /* Block size accessor */
      void Simulation::setBlockSize(const UnsignedLong blockSize) throw(InvalidArgumentException)
      {
	// Check if the given block size is >= 1
	if (blockSize < 1) throw InvalidArgumentException(HERE) << "The block size must be >= 1";
	blockSize_ = blockSize;
      }

      /* Block size accessor */
      UnsignedLong Simulation::getBlockSize() const
      {
	return blockSize_;
      }

      /* Verbosity accessor */
      void Simulation::setVerbose(const Bool verbose)
      {
	verbose_ = verbose;
      }

      /* Verbosity accessor */
      Bool Simulation::getVerbose() const
      {
	return verbose_;
      }

      /* String converter */
      String Simulation::__repr__() const
      {
	OSS oss;
	oss << "class=" << Simulation::GetClassName()
	    << " event=" << event_
	    << " maximumOuterSampling=" << maximumOuterSampling_
	    << " maximumCoefficientOfVariation=" << maximumCoefficientOfVariation_
	    << " blockSize=" << blockSize_;
	return oss;
      }

      /* Performs the actual computation. */
      void Simulation::run()
      {
	/* We want to compute the probability of occurence of the given event
	 *  We estimate this probability by computing the empirical mean of a
	 * sample of size at most outerSampling * blockSize, this sample being
         * built by blocks of size blockSize. It allows to use efficiently the
	 * distribution of the computation as well as it allows to deal with
	 * a sample size > 2^32 by a combination of blockSize and outerSampling
	 */
	UnsignedLong outerSampling(0);
	NumericalScalar coefficientOfVariation(-1.0);
	NumericalScalar probabilityEstimate(0.0);
	NumericalScalar varianceEstimate(0.0);
	UnsignedLong blockSize(getBlockSize());
	// Initialize the result
	result_.setProbabilityEstimate(0.0);
	result_.setVarianceEstimate(0.0);
	result_.setOuterSampling(outerSampling);
	result_.setBlockSize(blockSize);
	// Desactivate the cache mechanism, as it will be of no use in the context of simulation algorithms
	Bool cacheStatus(event_.getImplementation()->getFunction().isCacheEnabled());
	event_.getImplementation()->getFunction().disableCache();
	// We loop if there remains some outer sampling and the coefficient of variation is greater than the limit or has not been computed yet.
	while ((outerSampling < getMaximumOuterSampling()) && ((coefficientOfVariation == -1.0) || (coefficientOfVariation > getMaximumCoefficientOfVariation())))
	  {
	    NumericalSample blockSample(computeBlockSample());
	    ++outerSampling;
	    // Then, actualize the estimates
	    NumericalScalar meanBlock(blockSample.computeMean()[0]);
	    NumericalScalar varianceBlock(blockSample.computeCovariance()(0, 0));
	    // Let Skp be the empirical variance of a sample of size k*p
	    // Let Mkp be the empirical mean of a sample of size k*p
	    // Let Sp be the empirical variance of a sample of size p
	    // Let Mp be the empirical mean of a sample of size p
	    // Then, the empirical variance of the concatenated sample of size (k+1)*p is
	    // S(k+1)p = (Sp + k * Skp) / (k + 1) + k * (Mkp - Mp)^2 / (k + 1)^2
	    // and the empirical mean of the concatenated sample of size (k+1)*p is
	    // M(k+1)p = (Mp + k * Mkp) / (k + 1)
	    // To avoid integer overflow and double loss of precision, the formulas have to be written the way they are
	    NumericalScalar size(outerSampling);
	    varianceEstimate = (varianceBlock + (size - 1.0) * varianceEstimate) / size + (1.0 - 1.0 / size) * (probabilityEstimate - meanBlock) * (probabilityEstimate - meanBlock) / size;
	    probabilityEstimate = (meanBlock + (size - 1.0) * probabilityEstimate) / size;
	    NumericalScalar reducedVarianceEstimate(varianceEstimate / (size * blockSize));
	    // Update the history
	    NumericalPoint convergencePoint(2);
	    convergencePoint[0] = probabilityEstimate;
	    convergencePoint[1] = reducedVarianceEstimate;
	    convergenceStrategy_.store(convergencePoint);
	    // Update result
	    result_.setProbabilityEstimate(probabilityEstimate);
	    result_.setVarianceEstimate(reducedVarianceEstimate);
	    result_.setOuterSampling(outerSampling);
	    // Display the result at each outer sample
	    if (verbose_) Log::Info(result_.__repr__());
	    // Get the coefficient of variation back
	    coefficientOfVariation = result_.getCoefficientOfVariation();
	  }
	if (cacheStatus) event_.getImplementation()->getFunction().enableCache();
      }

      /* Convergence strategy accessor */
      void Simulation::setConvergenceStrategy(const HistoryStrategy & convergenceStrategy)
      {
	convergenceStrategy_ = convergenceStrategy;
      }
      
      Simulation::HistoryStrategy Simulation::getConvergenceStrategy() const
      {
	return convergenceStrategy_;
      }

      /* Input output strategy accessor */
      void Simulation::setInputOutputStrategy(const HistoryStrategy & inputOutputStrategy)
      {
	inputStrategy_ = inputOutputStrategy;
	outputStrategy_ = inputOutputStrategy;
	// Make sure the strategies do not carry old values.
	inputStrategy_.reset();
	outputStrategy_.reset();
      }

      Simulation::HistoryStrategy Simulation::getInputStrategy() const
      {
	return inputStrategy_;
      }

      Simulation::HistoryStrategy Simulation::getOutputStrategy() const
      {
	return outputStrategy_;
      }

      /* Draw the probability convergence at the given level */
      Simulation::Graph Simulation::drawProbabilityConvergence(const NumericalScalar level) const
      {
	NumericalSample convergenceSample(convergenceStrategy_.getSample());
	UnsignedLong size(convergenceSample.getSize());
	NumericalSample dataEstimate(size, 2);
	NumericalSample dataLowerBound(size, 2);
	NumericalSample dataUpperBound(size, 2);
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    NumericalScalar probabilityEstimate = convergenceSample[i][0];
	    NumericalScalar varianceEstimate    = convergenceSample[i][1];
	    dataEstimate[i][0] = i + 1;
	    dataEstimate[i][1] = probabilityEstimate;
	    NumericalScalar confidenceLength = SimulationResult(probabilityEstimate, varianceEstimate, i + 1, blockSize_).getConfidenceLength(level);
	    dataLowerBound[i][0] = i + 1;
	    dataLowerBound[i][1] = probabilityEstimate - 0.5 * confidenceLength;
	    dataUpperBound[i][0] = i + 1;
	    dataUpperBound[i][1] = probabilityEstimate + 0.5 * confidenceLength;
	  }
	Curve estimateCurve(dataEstimate, "red", "solid", 2, "probability estimate");
	Curve lowerBoundCurve(dataLowerBound, "green", "solid", 1, "lower bound");
	Curve upperBoundCurve(dataUpperBound, "green", "solid", 1, "upper bound");
	OSS oss;
	oss << getClassName() << " convergence graph at level " << level;
	Graph convergenceGraph(oss, "outer iteration", "estimate", true, "topright");
	convergenceGraph.addDrawable(estimateCurve);
	convergenceGraph.addDrawable(lowerBoundCurve);
	convergenceGraph.addDrawable(upperBoundCurve);
	return convergenceGraph;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
