//                                               -*- C++ -*-
/**
 *  @file  CleaningStrategy.cxx
 *  @brief An implementation returning the set of polynomials in sequence
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include <cstdlib>

#include "CleaningStrategy.hxx"
#include "Log.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "NumericalSample.hxx"
#include "Collection.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(CleaningStrategy);

      typedef Base::Common::Log                       Log;
      typedef Base::Common::ResourceMap               ResourceMap;
      typedef Base::Stat::NumericalSample             NumericalSample;
      typedef Base::Type::Collection<NumericalScalar> NumericalScalarCollection;

      static Base::Common::Factory<CleaningStrategy> RegisteredFactory("CleaningStrategy");

      const UnsignedLong CleaningStrategy::DefaultMaximumSize = 20;//strtoul(ResourceMap::GetInstance().get("CleaningStrategy-DefaultMaximumSize").c_str(), NULL, 0);
      const NumericalScalar CleaningStrategy::DefaultSignificanceFactor = 1.0e-4;//strtod(ResourceMap::GetInstance().get("CleaningStrategy-DefaultSignificanceFactor").c_str(), NULL);

      /* Default constructor */
      CleaningStrategy::CleaningStrategy()
        : AdaptiveStrategyImplementation(),
          currentVectorIndex_(0),
          maximumSize_(0),
          significanceFactor_(0.0),
          verbose_(false)
      {
        // Nothing to do
      }


      /* Constructor from an orthogonal basis */
      CleaningStrategy::CleaningStrategy(const OrthogonalBasis & basis,
                                         const UnsignedLong maximumDimension,
                                         const Bool verbose)
        : AdaptiveStrategyImplementation(basis, maximumDimension),
          currentVectorIndex_(0),
          maximumSize_(DefaultMaximumSize),
          significanceFactor_(DefaultSignificanceFactor),
          verbose_(verbose)
      {
        // Nothing to do
      }


      /* Constructor from an orthogonal basis */
      CleaningStrategy::CleaningStrategy(const OrthogonalBasis & basis,
                                         const UnsignedLong maximumDimension,
                                         const UnsignedLong maximumSize,
                                         const NumericalScalar significanceFactor,
                                         const Bool verbose)
        : AdaptiveStrategyImplementation(basis, maximumDimension),
          currentVectorIndex_(0),
          maximumSize_(maximumSize),
          significanceFactor_(significanceFactor),
          verbose_(verbose)
      {
        // Nothing to do
      }


      /* Compute initial basis for the approximation */
      void CleaningStrategy::computeInitialBasis()
      {
        // Start with the first vector
        Psi_k_p_ = NumericalMathFunctionCollection(maximumSize_);
        I_p_ = Indices(maximumSize_);
        for (UnsignedLong i = 0; i < maximumSize_; ++i)
          {
            Psi_k_p_[i] = basis_.build(i);
            I_p_[i] = i;
          }
        currentVectorIndex_ = maximumSize_;
      }

      /* Update the basis for the next iteration of approximation */
      void CleaningStrategy::updateBasis(const NumericalPoint & alpha_k_p_, const NumericalScalar residual_p_)
      {
        // The dimension will be adapted, so it is not const
        UnsignedLong dimension(alpha_k_p_.getDimension());
        NumericalScalarCollection coefficients(dimension);
        for (UnsignedLong i = 0; i < dimension; ++i) coefficients[i] = alpha_k_p_[i];
        if (verbose_)
          {
            Log::Info(OSS() << "initial state:");
            Log::Info(OSS() << "  vector index=" << currentVectorIndex_);
            Log::Info(OSS() << "  coeffs  size=" << coefficients.getSize());
            Log::Info(OSS() << "  coeffs      =" << coefficients);
            Log::Info(OSS() << "  I_p     size=" << I_p_.getSize());
            Log::Info(OSS() << "  I_p         =" << I_p_);
          }
        removedPsi_k_indices_ = Indices(0);
        NumericalScalarCollection removedCoefficients(0);

        // We keep at most maximumSize_ elements, the ones that have the largest magnitude and have a magnitude larger or equal to maximumMagnitude * significanceFactor
        // First, find the extrem magnitudes
        NumericalScalar largest(fabs(coefficients[0]));
        NumericalScalar smallest(largest);
        NumericalScalar secondSmallest(smallest);
        UnsignedLong indexSmallest(0);
        for (UnsignedLong i = 1; i < dimension; ++i)
          {
            const NumericalScalar tmp(fabs(coefficients[i]));
            if (tmp > largest) largest = tmp;
            if (tmp < smallest)
              {
                secondSmallest = smallest;
                smallest = tmp;
                indexSmallest = i;
              }
          }
        // Second, if the coefficient list is too large (it can be by only one term), remove the smallest term to free a place for the next vector
        UnsignedLong shift(0);
        if ((dimension > maximumSize_) && (currentVectorIndex_ < maximumDimension_))
          {
            // Add the smallest element to the removed list
            removedPsi_k_indices_.add(indexSmallest);
            if (verbose_) removedCoefficients.add(coefficients[indexSmallest]);
            // Compact Psi_k_p_ and I_p_
            for (UnsignedLong i = indexSmallest; i < dimension - 1; ++i)
              {
                Psi_k_p_[i] = Psi_k_p_[i + 1];
                I_p_[i] = I_p_[i + 1];
                coefficients[i] = coefficients[i + 1];
              }
            Psi_k_p_.resize(maximumSize_);
            I_p_.resize(maximumSize_);
            coefficients.resize(maximumSize_);
            if (verbose_)
              {
                Log::Info(OSS() << "intermediate state:");
                Log::Info(OSS() << "  coeffs  size=" << coefficients.getSize());
                Log::Info(OSS() << "  coeffs      =" << coefficients);
                Log::Info(OSS() << "  rem coeffs  =" << removedCoefficients);
                Log::Info(OSS() << "  I_p     size=" << I_p_.getSize());
                Log::Info(OSS() << "  I_p         =" << I_p_);
              }
            // The smallest remaining element is now the second smallest one
            smallest = secondSmallest;
            dimension = maximumSize_;
            shift = 1;
          } // Too much elements
        // Now, Psi_k_p_ contains at most maximumSize_ elements, associated with the coefficients the largest in magnitude
        // Third, remove all the elements with a magnitude less than largest * significanceFactor_
        // Quick rejection test: nothing to do if smallest >= largest * significanceFactor_
        if (smallest < largest * significanceFactor_)
          {
            UnsignedLong currentIndex(0);
            for (UnsignedLong i = 0; i < dimension; ++i)
              {
                if (fabs(coefficients[i]) >= largest * significanceFactor_)
                  {
                    Psi_k_p_[currentIndex] = Psi_k_p_[i];
                    I_p_[currentIndex] = I_p_[i];
                    if (verbose_) coefficients[currentIndex] = coefficients[i];
                    ++currentIndex;
                  }
                else
                  {
                    // Remove the vector
                    // As the removedPsi_k_indices_ stores the indices of the removed vectors in the *previous* partial basis Psi_k_p_,
                    // we must keep track to any shift in the indices
                    // If there is no shift, use directly the current counter
                    if (shift == 0) removedPsi_k_indices_.add(i);
                    // else we must take the shift into account
                    else
                      {
                        // If we are stricly before the index of the smallest coefficient, no shift
                        if (i < indexSmallest) removedPsi_k_indices_.add(i);
                        // Else take the shift into account
                        else removedPsi_k_indices_.add(i + shift);
		      }
                        if (verbose_) removedCoefficients.add(coefficients[i]);
		  }
	      }
	    Psi_k_p_.resize(currentIndex);
	    I_p_.resize(currentIndex);
	    if (verbose_) coefficients.resize(currentIndex);
	  }
	// If we have not generated all the vectors, go to the next one
	if (currentVectorIndex_ < maximumDimension_)
	  {
	    const NumericalMathFunction newVector(basis_.build(currentVectorIndex_));
	    Psi_k_p_.add(newVector);
	    I_p_.add(currentVectorIndex_);
	    addedPsi_k_indices_ = Indices(1, I_p_.getSize() - 1);
	    ++currentVectorIndex_;
	  }
	else
	  {
	    addedPsi_k_indices_ = Indices(0);
	  }
	if (verbose_)
	  {
	    Log::Info(OSS() << "final state:");
	    Log::Info(OSS() << "  vector index=" << currentVectorIndex_);
	    Log::Info(OSS() << "  coeffs  size=" << coefficients.getSize());
	    Log::Info(OSS() << "  coeffs      =" << coefficients);
	    Log::Info(OSS() << "  rem coeffs  =" << removedCoefficients);
	    Log::Info(OSS() << "  I_p     size=" << I_p_.getSize());
	    Log::Info(OSS() << "  I_p         =" << I_p_);
	  }
      }


      /* Virtual constructor */
      CleaningStrategy * CleaningStrategy::clone() const
      {
        return new CleaningStrategy(*this);
      }


      /* String converter */
      String CleaningStrategy::__repr__() const
      {
        return OSS() << "class=" << getClassName()
                     << " maximum size=" << maximumSize_
                     << " significance factor=" << significanceFactor_;
      }


      /* Current vector index accessor */
      UnsignedLong CleaningStrategy::getCurrentVectorIndex() const
      {
        return currentVectorIndex_;
      }

      /* Maximum size accessor */
      UnsignedLong CleaningStrategy::getMaximumSize() const
      {
        return maximumSize_;
      }

      void CleaningStrategy::setMaximumSize(const UnsignedLong maximumSize)
      {
        maximumSize_ = maximumSize;
      }

      /* Significance factor */
      NumericalScalar CleaningStrategy::getSignificanceFactor() const
      {
        return significanceFactor_;
      }

      void CleaningStrategy::setSignificanceFactor(const NumericalScalar significanceFactor)
      {
        significanceFactor_ = significanceFactor;
      }

      /* Verbosity accessor */
      Bool CleaningStrategy::getVerbose() const
      {
        return verbose_;
      }

      void CleaningStrategy::setVerbose(const Bool verbose)
      {
        verbose_ = verbose;
      }


      /* Method save() stores the object through the StorageManager */
      void CleaningStrategy::save(const StorageManager::Advocate & adv) const
      {
        AdaptiveStrategyImplementation::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void CleaningStrategy::load(const StorageManager::Advocate & adv)
      {
        AdaptiveStrategyImplementation::load(adv);
      }



    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
