//                                               -*- C++ -*-
/**
 *  @file  SymmetricMatrix.cxx
 *  @brief SymmetricMatrix implements the classical mathematical symmetric matrix
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: SymmetricMatrix.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "SymmetricMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(SymmetricMatrix);

      /* Default constructor */
      SymmetricMatrix::SymmetricMatrix()
	: SquareMatrix(0),
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }

      
      /* Constructor with size (dim, which is the same for nbRows_ and nbColumns_ )*/
      SymmetricMatrix::SymmetricMatrix(const UnsignedLong dim)
	: SquareMatrix(dim),
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }
      
      /* Constructor from external collection */
      /* If the dimensions of the matrix and of the collection */
      /* do not match, either the collection is truncated */
      /* or the rest of the matrix is filled with zeros */
      SymmetricMatrix::SymmetricMatrix(const UnsignedLong dim,
	                               const Collection<NumericalScalar> & elementsValues)
	: SquareMatrix(dim, elementsValues),
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }
      
      
      /* Constructor with implementation */
      SymmetricMatrix::SymmetricMatrix(const Implementation & i)
	: SquareMatrix(i) ,
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }
      
      
      /* Check if the internal representation is really symmetric */
      void SymmetricMatrix::checkSymmetry() const
      {
	if (!hasBeenSymmetrized_)
	  {
	    getImplementation()->symmetrize();
	    hasBeenSymmetrized_ = true;
	  }
      }

      /* String converter */
      String SymmetricMatrix::__repr__() const
      {
	checkSymmetry();
	return OSS() << "class=" << getClassName()
		     << " dimension=" << this->getDimension()
		     << " implementation=" << getImplementation()->__repr__();
      }

      /* Operator () gives access to the elements of the matrix (to modify these elements) */
      /* The element of the matrix is designated by its row number i and its column number j */
      /* the first element of the matrix is m(0,0) */
      NumericalScalar & SymmetricMatrix::operator() (const UnsignedLong i,
						     const UnsignedLong j) throw(InvalidDimensionException)
      {
	copyOnWrite();
	hasBeenSymmetrized_ = false;
	
	return (i < j) ? (*getImplementation())(i, j) : (*getImplementation())(j, i) ;
      }
	
      /* Operator () gives access to the elements of the matrix (read only) */
      /* The element of the matrix is designated by its row number i and its column number j */
      const NumericalScalar & SymmetricMatrix::operator() (const UnsignedLong i,
							   const UnsignedLong j)  const throw(InvalidDimensionException)
      {
	return (i < j) ? (*getImplementation())(i, j) : (*getImplementation())(j, i) ;
      }
      
      /* SymmetricMatrix transpose */
      SymmetricMatrix SymmetricMatrix::transpose () const 
      { 
        return *this;
      }

      /* SymmetricMatrix additions (must have the same dimensions) */
      Matrix SymmetricMatrix::operator + (const Matrix & m) const throw(InvalidDimensionException)
      {
	checkSymmetry();
        return Matrix(new MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }
      
      SquareMatrix SymmetricMatrix::operator + (const SquareMatrix & m) const throw(InvalidDimensionException)
      {
	checkSymmetry();
        return SquareMatrix(new MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }
      
      SymmetricMatrix SymmetricMatrix::operator + (const SymmetricMatrix & m) const throw(InvalidDimensionException)
      {
        return SymmetricMatrix(new MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }

      /* SymmetricMatrix substractions (must have the same dimensions) */
      Matrix SymmetricMatrix::operator - (const Matrix & m) const throw(InvalidDimensionException)
      {
	checkSymmetry();
        return SymmetricMatrix(new MatrixImplementation(*getImplementation() - *(m.getImplementation()) ));
      }
      
      SquareMatrix SymmetricMatrix::operator - (const SquareMatrix & m) const throw(InvalidDimensionException)
      {
	checkSymmetry();
        return SymmetricMatrix(new MatrixImplementation(*getImplementation() - *(m.getImplementation()) ));
      }
      
      SymmetricMatrix SymmetricMatrix::operator - (const SymmetricMatrix & m) const throw(InvalidDimensionException)
      {
        return SymmetricMatrix(new MatrixImplementation(*getImplementation() - *(m.getImplementation()) ));
      }

      /* SymmetricMatrix multiplications (must have consistent dimensions) */
      Matrix SymmetricMatrix::operator * (const Matrix & m) const throw(InvalidDimensionException)
      {
        return SymmetricMatrix(new MatrixImplementation(getImplementation()->symProd(*(m.getImplementation()), 'L') ));
      }
      
      SquareMatrix SymmetricMatrix::operator * (const SquareMatrix & m) const throw(InvalidDimensionException)
      {
        return SymmetricMatrix(new MatrixImplementation(getImplementation()->symProd(*(m.getImplementation()), 'L') ));
      }
      
      SymmetricMatrix SymmetricMatrix::operator * (const SymmetricMatrix & m) const throw(InvalidDimensionException)
      {
	m.checkSymmetry();
        return SymmetricMatrix(new MatrixImplementation(getImplementation()->symProd(*(m.getImplementation()), 'L') ));
      }
      
      SymmetricMatrix SymmetricMatrix::operator * (const IdentityMatrix & m) const throw(InvalidDimensionException)
      {
        return *this;
      }
      
      
      /* Multiplication with a NumericalPoint (must have consistent dimensions) */
      NumericalPoint SymmetricMatrix::operator * (const NumericalPoint & pt) const throw(InvalidDimensionException)
      {
        return getImplementation()->symVectProd(pt) ;
      }
      
      
      /* Multiplication with a NumericalScalar */
      SymmetricMatrix SymmetricMatrix::operator * (const NumericalScalar & s) const 
      {
        return SymmetricMatrix(new MatrixImplementation(*getImplementation() * s ));
      }

      /* Division by a NumericalScalar*/
      SymmetricMatrix SymmetricMatrix::operator / (const NumericalScalar & s) const throw(InvalidArgumentException)
      {
        return SymmetricMatrix(new MatrixImplementation(*getImplementation() / s ));
      }
      
      /* SquareMatrix integer power */
      SymmetricMatrix SymmetricMatrix::power(const UnsignedLong n) const
      {
	return SymmetricMatrix(new MatrixImplementation(getImplementation()->symPower(n)));
      }
      
      /* Resolution of a linear system */
      NumericalPoint SymmetricMatrix::solveLinearSystem(const NumericalPoint & b) const throw(InvalidDimensionException)
      {
        return getImplementation()->solveLinearSystemSym(b);
      }
  
      MatrixImplementation SymmetricMatrix::solveLinearSystem(const MatrixImplementation & b) const throw(InvalidDimensionException)
      {
        return getImplementation()->solveLinearSystemSym(b);
      }
  
      /* Compute determinant */
      NumericalScalar SymmetricMatrix::computeDeterminant() const
      {
        return getImplementation()->computeDeterminantSym();
      }    
      
      /* Compute eigenvalues */
      NumericalPoint SymmetricMatrix::computeEigenValues() const
      {
        return getImplementation()->computeEigenValuesSym();
      }


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
