//                                               -*- C++ -*-
/**
 *  @file  NumericalMathFunction.hxx
 *  @brief The class that implements numerical math functions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: NumericalMathFunction.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_NUMERICALMATHFUNCTION_HXX
#define OPENTURNS_NUMERICALMATHFUNCTION_HXX

#include "TypedInterfaceObject.hxx"
#include "NumericalMathFunctionImplementation.hxx"
#include "ComparisonOperator.hxx"
#include "Collection.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class NumericalMathFunction
       *
       * The class that simulates a numerical math function,
       * its gradient and its hessian. This class is just an interface
       * to actual implementation objects that can be hot-replaced
       * during computation. Each implementation object refers to
       * the function, the gradient or the hessian.
       */
      class NumericalMathFunction 
	: public Common::TypedInterfaceObject<NumericalMathFunctionImplementation>
      {
	CLASSNAME;
      public:

	/* Some typedefs for easy reading */
	typedef NumericalMathFunctionImplementation::Implementation                Implementation;
	typedef NumericalMathFunctionImplementation::EvaluationImplementation      EvaluationImplementation;
	typedef NumericalMathFunctionImplementation::GradientImplementation        GradientImplementation;
	typedef NumericalMathFunctionImplementation::HessianImplementation         HessianImplementation;
	typedef NumericalMathFunctionImplementation::NumericalPoint                NumericalPoint;
	typedef NumericalMathFunctionImplementation::NumericalPointWithDescription NumericalPointWithDescription;
	typedef NumericalMathFunctionImplementation::NumericalSample               NumericalSample;
	typedef NumericalMathFunctionImplementation::Matrix                        Matrix;
	typedef NumericalMathFunctionImplementation::SymmetricTensor               SymmetricTensor;
	typedef NumericalMathFunctionImplementation::Indices                       Indices;
	typedef NumericalMathFunctionImplementation::Description                   Description;
	typedef NumericalMathFunctionImplementation::InvalidArgumentException      InvalidArgumentException;
	typedef NumericalMathFunctionImplementation::InternalException             InternalException;
	typedef Common::ComparisonOperator                                         ComparisonOperator;
	typedef Type::Collection<NumericalMathFunction>                            NumericalMathFunctionCollection;

	/** Default constructor */
	NumericalMathFunction();

	/** Default constructor */
	NumericalMathFunction(const String & name);

	/** Constructor from NumericalMathFunctionImplementation */
	NumericalMathFunction(const NumericalMathFunctionImplementation & implementation);

#ifndef SWIG
	/** Constructor from implementation */
	NumericalMathFunction(const Implementation & p_implementation);

	/** Constructor from implementation pointer */
	NumericalMathFunction(NumericalMathFunctionImplementation * p_implementation);
#endif

	/** Composition constructor */
	NumericalMathFunction(const NumericalMathFunction & left,
			      const NumericalMathFunction & right);

	/** Analytical formula constructor */
	NumericalMathFunction(const Description & inputVariablesNames,
			      const Description & outputVariablesNames,
			      const Description & formulas);

	/** Indicator function constructor */
	NumericalMathFunction(const NumericalMathFunction & function,
			      const ComparisonOperator & comparisonOperator,
			      const NumericalScalar threshold);

	/** Linear combination function constructor */
	NumericalMathFunction(const NumericalMathFunctionCollection & functionCollection,
			      const NumericalPoint & coefficients);

	/** Simplified analytical formula constructor */
	NumericalMathFunction(const String & inputVariableName,
			      const String & formula,
			      const String & outputVariableName = "outputVariable");

#ifndef SWIG
	/** Constructor from implementations */
	NumericalMathFunction(const EvaluationImplementation & evaluationImplementation,
			      const GradientImplementation & gradientImplenmentation,
			      const HessianImplementation  & hessianImplementation);
#endif

	/** Constructor from a wrapper file */
	NumericalMathFunction(const WrapperFile & wrapperFile);

	/** Comparison operator */
	Bool operator ==(const NumericalMathFunction & other) const;

	/** String converter */
	virtual String __repr__() const;



	/** Enable or disable the internal cache */
	void enableCache() const;
	void disableCache() const;
	Bool isCacheEnabled() const;


	/* Function implementation accessors */
	void setEvaluationImplementation(const EvaluationImplementation & functionImplementation);
	const EvaluationImplementation & getEvaluationImplementation() const;

	/* Gradient implementation accessors */
	void setGradientImplementation(const NumericalMathGradientImplementation & gradientImplementation);
#ifndef SWIG
	/* Gradient implementation accessors */
	void setGradientImplementation(const GradientImplementation & gradientImplementation);
#endif
	const GradientImplementation & getGradientImplementation() const;

	/* Hessian implementation accessors */
	void setHessianImplementation(const NumericalMathHessianImplementation & hessianImplementation);
#ifndef SWIG
	/* Hessian implementation accessors */
	void setHessianImplementation(const HessianImplementation & hessianImplementation);
#endif
	const HessianImplementation & getHessianImplementation() const;

	/* Initial function implementation accessors */
	const EvaluationImplementation & getInitialEvaluationImplementation() const;

	/* Initial gradient implementation accessors */
	const GradientImplementation & getInitialGradientImplementation() const;

	/* Initial hessian implementation accessors */
	const HessianImplementation & getInitialHessianImplementation() const;




	/** Multiplication of two 1D output functions with the same input dimension */
	virtual NumericalMathFunction operator * (const NumericalMathFunction & right) const;

	/** Operator () */
	NumericalPoint operator() (const NumericalPoint & in) const
	  throw(InvalidArgumentException,InternalException);
	NumericalSample operator() (const NumericalSample & in) const
	  throw(InvalidArgumentException,InternalException);

	/** Method gradient() returns the Jacobian transposed matrix of the function at point */
	Matrix gradient(const NumericalPoint & in) const
	  throw(InvalidArgumentException,InternalException);

	/** Method hessian() returns the symmetric tensor of the function at point */
	SymmetricTensor hessian(const NumericalPoint & in) const
	  throw(InvalidArgumentException,InternalException);

	/** Gradient according to the marginal parameters */
	virtual Matrix parametersGradient(const NumericalPoint & in) const;

	/** Parameters value and description accessor */
	virtual NumericalPointWithDescription getParameters() const;
	virtual void setParameters(const NumericalPointWithDescription & parameters);

	/** Accessor for input point dimension */
	UnsignedLong getInputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for output point dimension */
	UnsignedLong getOutputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for input point dimension */
	UnsignedLong getInputDimension() const
	  throw(InternalException);

	/** Accessor for output point dimension */
	UnsignedLong getOutputDimension() const
	  throw(InternalException);

	/** Description Accessor, i.e. the names of the input and output parameters */
	void setDescription(const Description & description);
	Description getDescription() const;

	/** Input description Accessor, i.e. the names of the input parameters */
	Description getInputDescription() const;

	/** Output description Accessor, i.e. the names of the Output parameters */
	Description getOutputDescription() const;

	/** Get the i-th marginal function */
        NumericalMathFunction getMarginal(const UnsignedLong i) const throw(InvalidArgumentException);

	/** Get the function corresponding to indices components */
        NumericalMathFunction getMarginal(const Indices & indices) const throw(InvalidArgumentException);

	/** Number of calls to the evaluation */
	UnsignedLong getEvaluationCallsNumber() const;

	/** Number of calls to the gradient */
	UnsignedLong getGradientCallsNumber() const;

	/** Number of calls to the hessian */
	UnsignedLong getHessianCallsNumber() const;

	/** Static methods for documentation of analytical fnctions */
	static Description GetValidConstants();
	static Description GetValidFunctions();
	static Description GetValidOperators();

      }; /* class NumericalMathFunction */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALMATHFUNCTION_HXX */
