//                                               -*- C++ -*-
/**
 *  @file  ConstantNumericalMathHessianImplementation.cxx
 *  @brief Class for a constant numerical math hessian implementation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: ConstantNumericalMathHessianImplementation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "ConstantNumericalMathHessianImplementation.hxx"
#include "PersistentObjectFactory.hxx"


namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ConstantNumericalMathHessianImplementation);

      static Common::Factory<ConstantNumericalMathHessianImplementation> RegisteredFactory("ConstantNumericalMathHessianImplementation");

      /* Default constructor */
      ConstantNumericalMathHessianImplementation::ConstantNumericalMathHessianImplementation()
	: NumericalMathHessianImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor */
      ConstantNumericalMathHessianImplementation::ConstantNumericalMathHessianImplementation(const SymmetricTensor & constant)
        throw(InvalidDimensionException) :
        NumericalMathHessianImplementation(),
        constant_(constant)
      {
	/* Check if the dimensions of the constant term is compatible with the linear term */
      }

      /* Virtual constructor */
      ConstantNumericalMathHessianImplementation * ConstantNumericalMathHessianImplementation::clone() const
      {
	return new ConstantNumericalMathHessianImplementation(*this);
      }

      /* Comparison operator */
      Bool ConstantNumericalMathHessianImplementation::operator ==(const ConstantNumericalMathHessianImplementation & other) const
      {
	return (constant_ == other.constant_);
      }
  
      /* String converter */
      String ConstantNumericalMathHessianImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << ConstantNumericalMathHessianImplementation::GetClassName()
	    << " name=" << getName()
            << " constant=" << constant_;
	return oss;
      }
  
      /* Accessor for the constant term */
      ConstantNumericalMathHessianImplementation::SymmetricTensor ConstantNumericalMathHessianImplementation::getConstant() const
      {
	return constant_;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Hessian () */
      ConstantNumericalMathHessianImplementation::SymmetricTensor ConstantNumericalMathHessianImplementation::hessian(const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	callsNumber_++;
	if (in.getDimension() != constant_.getNbRows()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
	return constant_;
      }

      /* Accessor for input point dimension */
      UnsignedLong ConstantNumericalMathHessianImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return constant_.getNbRows();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ConstantNumericalMathHessianImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return constant_.getNbSheets();
      }
      
      /* Method save() stores the object through the StorageManager */
      void ConstantNumericalMathHessianImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathHessianImplementation::save(adv);
	adv.writeValue(constant_, StorageManager::MemberNameAttribute, "constant_");
      }

      /* Method load() reloads the object from the StorageManager */
      void ConstantNumericalMathHessianImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathHessianImplementation::load(adv);
	adv.readValue(constant_, StorageManager::MemberNameAttribute, "constant_");
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
