/*
 *
 * (C) Copyright 2009 The OpenSAF Foundation
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. This file and program are licensed
 * under the GNU Lesser General Public License Version 2.1, February 1999.
 * The complete license can be accessed from the following location:
 * http://opensource.org/licenses/lgpl-license.php
 * See the Copying file included with the OpenSAF distribution for full
 * licensing terms.
 *
 * Author(s): Ericsson AB
 *
 */

/* ========================================================================
 *   INCLUDE FILES
 * ========================================================================
 */
#include <string>
#include <sstream>
#include <assert.h>

#include <saImmOm.h>
#include <saImmOi.h>
#include <immutil.h>

#include "stdio.h"
#include "logtrace.h"
#include "SmfUpgradeCampaign.hh"
#include "SmfUpgradeProcedure.hh"
#include "SmfProcedureThread.hh"
#include "SmfCampaignThread.hh"
#include "SmfProcState.hh"
#include "SmfUpgradeStep.hh"
#include "SmfUpgradeMethod.hh"
#include "SmfUpgradeAction.hh"
#include "SmfUtils.hh"

/* ========================================================================
 *   DEFINITIONS
 * ========================================================================
 */

/* ========================================================================
 *   TYPE DEFINITIONS
 * ========================================================================
 */

/* ========================================================================
 *   DATA DECLARATIONS
 * ========================================================================
 */

/* ========================================================================
 *   FUNCTION PROTOTYPES
 * ========================================================================
 */

//================================================================================
// Class SmfUpgradeProcedureg71
// Purpose:
// Comments:
//================================================================================

 SmfUpgradeProcedure::SmfUpgradeProcedure():
    m_state(SmfProcStateInitial::instance()),
    m_procState(SA_SMF_PROC_INITIAL), 
    m_procedureThread(0), 
    m_name(""),
    m_time(0),
    m_execLevel(0), 
    m_dn(""),
//   m_upgradeMethod(0),
    m_procInitAction(0), 
    m_procWrapupAction(0)
{

}

// ------------------------------------------------------------------------------
// ~SmfUpgradeProcedure()
// ------------------------------------------------------------------------------
SmfUpgradeProcedure::~SmfUpgradeProcedure()
{
	if (m_procedureThread != NULL) {
		m_procedureThread->stop();
		/* The thread deletes it's own object when terminating */
		m_procedureThread = NULL;
	}
	delete m_upgradeMethod;

	std::vector < SmfUpgradeAction * >::iterator it;

	/* Delete procedure initialization */
	for (it = m_procInitAction.begin(); it != m_procInitAction.end(); ++it) {
		delete(*it);
	}

	/* Delete procedure wrapup */
	for (it = m_procWrapupAction.begin(); it != m_procWrapupAction.end(); ++it) {
		delete(*it);
	}

	std::vector < SmfUpgradeStep * >::iterator stepit;

	/* Delete upgrade steps */
	for (stepit = m_procSteps.begin(); stepit != m_procSteps.end(); ++stepit) {
		delete(*stepit);
	}
}

// ------------------------------------------------------------------------------
// init()
// ------------------------------------------------------------------------------
SaAisErrorT 
SmfUpgradeProcedure::init(const SaImmAttrValuesT_2 ** attrValues)
{
	const SaImmAttrValuesT_2 **attribute;

	for (attribute = attrValues; *attribute != NULL; attribute++) {
		void *value;

		if ((*attribute)->attrValuesNumber != 1) {
			LOG_ER("invalid number of values %u for %s", (*attribute)->attrValuesNumber,
			       (*attribute)->attrName);
			continue;
		}

		value = (*attribute)->attrValues[0];

		if (strcmp((*attribute)->attrName, "safSmfProcedure") == 0) {
			char *rdn = *((char **)value);
			TRACE("init safSmfProcedure = %s", rdn);
		} else if (strcmp((*attribute)->attrName, "saSmfProcState") == 0) {
			unsigned int state = *((unsigned int *)value);

			if ((state >= SA_SMF_PROC_INITIAL) && (state <= SA_SMF_PROC_ROLLBACK_FAILED)) {
				setProcState((SaSmfProcStateT) state);
			} else {
				LOG_ER("invalid proc state %u", state);
				setProcState(SA_SMF_PROC_INITIAL);
			}
			TRACE("init saSmfProcState = %u", (int)m_procState);
		} else {
			TRACE("init unhandled attribute = %s", (*attribute)->attrName);
		}
	}

	return SA_AIS_OK;
}

// ------------------------------------------------------------------------------
// changeState()
// ------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::changeState(const SmfProcState * i_state)
{
	TRACE_ENTER();
	//Change state class pointer
	m_state = const_cast < SmfProcState * >(i_state);

	//Set stat in IMM procedure object and send state change notification
	setImmStateAndSendNotification(m_state->getState());

	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// getState()
//------------------------------------------------------------------------------
SaSmfProcStateT SmfUpgradeProcedure::getState() const
{
	return m_state->getState();
}

//------------------------------------------------------------------------------
// setProcState()
//------------------------------------------------------------------------------ 
void 
SmfUpgradeProcedure::setProcState(SaSmfProcStateT i_state)
{
	TRACE_ENTER();

	switch (i_state) {
	case SA_SMF_PROC_INITIAL:
		{
			m_state = SmfProcStateInitial::instance();
			break;
		}

	case SA_SMF_PROC_EXECUTING:
		{
			m_state = SmfProcStateExecuting::instance();
			break;
		}

	case SA_SMF_PROC_SUSPENDED:
		{
			m_state = SmfProcStateExecSuspended::instance();
			break;
		}

	case SA_SMF_PROC_COMPLETED:
		{
			m_state = SmfProcStateExecutionCompleted::instance();
			break;
		}

	case SA_SMF_PROC_FAILED:
		{
			m_state = SmfProcStateExecFailed::instance();
			break;
		}

#if 0
		SA_SMF_PROC_STEP_UNDONE = 5, SA_SMF_PROC_ROLLING_BACK = 7, SA_SMF_PROC_ROLLBACK_SUSPENDED =
		    8, SA_SMF_PROC_ROLLED_BACK = 9, SA_SMF_PROC_ROLLBACK_FAILED = 10
#endif
	default:
		{
			LOG_ER("unknown state");
		}
	}

	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// setProcName()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::setProcName(std::string i_name)
{
	m_name = i_name;
}

//------------------------------------------------------------------------------
// getProcName()
//------------------------------------------------------------------------------
const std::string & 
SmfUpgradeProcedure::getProcName()
{
	return m_name;
}

//------------------------------------------------------------------------------
// setExecLevel()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::setExecLevel(std::string i_level)
{
	m_execLevel = atoi(i_level.c_str());
}

//------------------------------------------------------------------------------
// getExecLevel()
//------------------------------------------------------------------------------
const int &
SmfUpgradeProcedure::getExecLevel()
{
	return m_execLevel;
}

//------------------------------------------------------------------------------
// setProcedurePeriod()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::setProcedurePeriod(SaTimeT i_time)
{
	m_time = i_time;
}

//------------------------------------------------------------------------------
// getProcedurePeriod()
//------------------------------------------------------------------------------
const SaTimeT &
SmfUpgradeProcedure::getProcedurePeriod()
{
	return m_time;
}

//------------------------------------------------------------------------------
// setDn()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::setDn(const std::string & i_dn)
{
	m_dn = i_dn;
}

//------------------------------------------------------------------------------
// getDn()
//------------------------------------------------------------------------------
const std::string & 
SmfUpgradeProcedure::getDn()
{
	return m_dn;
}

//------------------------------------------------------------------------------
// setImmStateAndSendNotification()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::setImmStateAndSendNotification(SaSmfProcStateT i_state)
{
	TRACE_ENTER();

	TRACE("Set procedure state = %u", i_state);
	m_procState = i_state;
	getProcThread()->updateImmAttr(this->getDn().c_str(), "saSmfProcState", SA_IMM_ATTR_SAUINT32T, &m_procState);
	SmfCampaignThread::instance()->sendStateNotification(m_dn, 0x66, SA_NTF_MANAGEMENT_OPERATION,
							     SA_SMF_PROCEDURE_STATE, i_state);
	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// setUpgradeMethod()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::setUpgradeMethod(SmfUpgradeMethod * i_method)
{
	m_upgradeMethod = i_method;
}

//------------------------------------------------------------------------------
// getUpgradeMethod()
//------------------------------------------------------------------------------
SmfUpgradeMethod *
SmfUpgradeProcedure::getUpgradeMethod(void)
{
	return m_upgradeMethod;
}

//------------------------------------------------------------------------------
// addProcInitAction()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::addProcInitAction(SmfUpgradeAction * i_action)
{
	m_procInitAction.push_back(i_action);
}

//------------------------------------------------------------------------------
// addProcWrapupAction()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::addProcWrapupAction(SmfUpgradeAction * i_action)
{
	m_procWrapupAction.push_back(i_action);
}

//------------------------------------------------------------------------------
// setProcThread()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::setProcThread(SmfProcedureThread * i_procThread)
{
	m_procedureThread = i_procThread;
}

//------------------------------------------------------------------------------
// getProcThread()
//------------------------------------------------------------------------------
SmfProcedureThread *
SmfUpgradeProcedure::getProcThread()
{
	return m_procedureThread;
}

//------------------------------------------------------------------------------
// calculateSteps()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::calculateSteps()
{
	SmfUpgradeMethod *upgradeMethod = NULL;

	TRACE_ENTER();
	//TBD, this is just for testing purposes

	upgradeMethod = getUpgradeMethod();

	if (upgradeMethod == NULL) {
		LOG_ER("calculateSteps no upgrade method found");
		return false;
	}

	switch (upgradeMethod->getUpgradeMethod()) {
	case SA_SMF_ROLLING:
		{
			SmfRollingUpgrade *rollingUpgrade = (SmfRollingUpgrade *) upgradeMethod;
			if ( !calculateRollingSteps(rollingUpgrade)) {
                                LOG_ER("SmfUpgradeProcedure::calculateSteps:calculateRollingSteps failed");
                                return false;
                        }
			break;
		}

	case SA_SMF_SINGLE_STEP:
		{
			if ( !calculateSingleStep((SmfSinglestepUpgrade*)upgradeMethod)) {
                                LOG_ER("SmfUpgradeProcedure::calculateSteps:calculateSingleStep failed");
                                return false;
                        }
			break;
		}

	default:
		{
			LOG_ER("calculateSteps unknown upgrade method found %d", upgradeMethod->getUpgradeMethod());
			return false;
		}
	}

	TRACE_LEAVE();
        return true;
}

//------------------------------------------------------------------------------
// calculateRollingSteps()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::calculateRollingSteps(SmfRollingUpgrade * i_rollingUpgrade)
{
	TRACE_ENTER();
	const SmfByTemplate *byTemplate = (const SmfByTemplate *)i_rollingUpgrade->getUpgradeScope();

	if (byTemplate == NULL) {
		LOG_ER("SmfUpgradeProcedure::calculateRollingSteps:calculateRollingSteps no upgrade scope by template found");
		return false;
	}

	const SmfTargetNodeTemplate *nodeTemplate = byTemplate->getTargetNodeTemplate();

	if (nodeTemplate == NULL) {
		LOG_ER("SmfUpgradeProcedure::calculateRollingSteps:calculateRollingSteps no node template found");
		return false;
	}

	const std::string & objectDn = nodeTemplate->getObjectDn();
	std::list < std::string > nodeList;

	if ( !calculateNodeList(objectDn, nodeList) ){
		LOG_ER("SmfUpgradeProcedure::calculateRollingSteps:calculateRollingSteps no nodes found");
                return false;
        }

	if (nodeList.size() == 0) {
		LOG_ER("SmfUpgradeProcedure::calculateRollingSteps:calculateRollingSteps no nodes found");
		return false;
	}

	const std::list < SmfParentType * >&actUnitTemplates = nodeTemplate->getActivationUnitTemplateList();
	std::list < std::string > activationUnitList;

	if (actUnitTemplates.size() == 0) {
		/* No activation unit templates, use node list as activation/deactivation units */
		std::list < std::string >::const_iterator it;
		int stepCntr = 0;

		for (it = nodeList.begin(); it != nodeList.end(); ++it) {
			std::ostringstream ost;
			stepCntr++;
			ost << stepCntr;

			SmfUpgradeStep *newStep = new SmfUpgradeStep();
			newStep->setRdn("safSmfStep=" + ost.str());
			newStep->setDn(newStep->getRdn() + "," + getDn());
			newStep->addActivationUnit(*it);
			newStep->addDeactivationUnit(*it);
			newStep->setMaxRetry(i_rollingUpgrade->getStepMaxRetryCount());
			newStep->setRestartOption(i_rollingUpgrade->getStepRestartOption());
			newStep->addSwRemove(nodeTemplate->getSwRemoveList());
			newStep->addSwAdd(nodeTemplate->getSwInstallList());
			newStep->setSwNode(*it);

			TRACE("SmfUpgradeProcedure::calculateRollingSteps:calculateRollingSteps new step added %s with activation/deactivation unit %s",
			      newStep->getRdn().c_str(), (*it).c_str());

			/* TODO: Update objects to be modified by step */
			if ( !addStepModifications(newStep, byTemplate->getTargetEntityTemplate(), SMF_AU_AMF_NODE)){
                                LOG_ER("SmfUpgradeProcedure::calculateRollingSteps:addStepModifications failed");
                                return false;
                        }

			addProcStep(newStep);
		}
	} else {
		/* We have activation unit templates, calculate activation/deactivation units */
                std::list < std::string > actDeactUnits;

                //Find out all objects in the system pointed out by the templates
		std::list < SmfParentType * >::const_iterator it;
		for (it = actUnitTemplates.begin(); it != actUnitTemplates.end(); ++it) {
                        calcActivationUnitsFromTemplate((*it), nodeList, actDeactUnits);
		}

//TODO: The actDeactUnits list is a straight list of singular act/deact units calculated from 
//      the templates so the number of steps may be less since 
//      each step may contain several act/deact units if located in different SGs on the same node,
//       
//      Currently only one act/deact entity is specified for each step.

                TRACE("Create steps for all found act/deac units calculated from templates");
		int stepCntr = 0;
		std::list < std::string >::const_iterator itActDeact;
                for (itActDeact = actDeactUnits.begin(); itActDeact != actDeactUnits.end(); ++itActDeact) {
			TRACE("Create step %d for %s", stepCntr, (*itActDeact).c_str());

                        std::ostringstream ost;
                        stepCntr++;
                        ost << stepCntr;

                        SmfUpgradeStep *newStep = new(std::nothrow) SmfUpgradeStep();
                        assert(newStep != NULL);
                        newStep->setRdn("safSmfStep=" + ost.str());
                        newStep->setDn(newStep->getRdn() + "," + getDn());
                        newStep->addActivationUnit(*itActDeact);
                        newStep->addDeactivationUnit(*itActDeact);
                        newStep->setMaxRetry(i_rollingUpgrade->getStepMaxRetryCount());
                        newStep->setRestartOption(i_rollingUpgrade->getStepRestartOption());
                        newStep->addSwRemove(nodeTemplate->getSwRemoveList());
                        newStep->addSwAdd(nodeTemplate->getSwInstallList());
                        std::string node = getNodeForCompSu(*itActDeact);
                        if (node.length() == 0) {
                                LOG_ER("SmfUpgradeProcedure::calculateRollingSteps:getNodeForCompSu fails");
                                return false;
                        }
                        newStep->setSwNode(node);

                        TRACE("SmfUpgradeProcedure::calculateRollingSteps:calculateRollingSteps new step added %s with activation/deactivation unit %s",
                              newStep->getRdn().c_str(), (*itActDeact).c_str());

			if ( !addStepModifications(newStep, byTemplate->getTargetEntityTemplate(), SMF_AU_SU)){
                                LOG_ER("SmfUpgradeProcedure::calculateRollingSteps:addStepModifications failed");
                                return false;
                        }

                        addProcStep(newStep);
                }
	}

        TRACE_LEAVE();
        return true;
}

//------------------------------------------------------------------------------
// getNodeForCompSu()
//------------------------------------------------------------------------------
std::string 
SmfUpgradeProcedure::getNodeForCompSu(const std::string & i_objectDn)
{
        TRACE_ENTER();
	SmfImmUtils immUtil;
        SaImmAttrValuesT_2 **attributes;
	const char *className;

        /* First find out if it is a SU or components version type */
        if (immUtil.getObject(i_objectDn, &attributes) == false) {
                LOG_ER("SmfUpgradeProcedure::getNodeForCompSu:failed to get imm object %s", i_objectDn.c_str());
                TRACE_LEAVE();
                return "";
        }

        className = immutil_getStringAttr((const SaImmAttrValuesT_2 **)
                                          attributes, SA_IMM_ATTR_CLASS_NAME, 0);
        if (className == NULL) {
                LOG_ER("SmfUpgradeProcedure::getNodeForCompSu:class name not found for version type %s", i_objectDn.c_str());
                TRACE_LEAVE();
                return "";
        }

        if (strcmp(className, "SaAmfSU") == 0) {
                const SaNameT *hostedByNode = immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                  "saAmfSUHostedByNode",
                                                                  0);
                if (hostedByNode != NULL) {
                        return (char *)hostedByNode->value;
                } else {
                        LOG_ER("SmfUpgradeProcedure::getNodeForCompSu:No hostedByNode attr set for %s", i_objectDn.c_str());  
                        TRACE_LEAVE();
                        return "";
                }

        } else if (strcmp(className, "SaAmfComp") == 0) {
                /* Find the parent SU to this component */
                if (immUtil.getParentObject((i_objectDn), &attributes) == true) {
                        const SaNameT *hostedByNode = immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                          "saAmfSUHostedByNode", 
                                                                          0);
                        if (hostedByNode != NULL) {
                                TRACE_LEAVE();
                                return (char *)hostedByNode->value;
                        } else {
                                LOG_ER("SmfUpgradeProcedure::getNodeForCompSu:No hostedByNode attr set for %s", i_objectDn .c_str());  
                                TRACE_LEAVE();
                                return "";
                        }
                } else {
                        LOG_ER("SmfUpgradeProcedure::getNodeForCompSu:Fails to get parent to %s", i_objectDn.c_str());  
                        TRACE_LEAVE();
                        return "";
                }
        }

        LOG_ER("SmfUpgradeProcedure::getNodeForCompSu:The DN does not refere to a  SaAmfSUT or SaAmfComp class, DN=%s", i_objectDn.c_str());  
        TRACE_LEAVE();
        return "";
}

//------------------------------------------------------------------------------
// calculateSingleStep()
//------------------------------------------------------------------------------
bool SmfUpgradeProcedure::calculateSingleStep(SmfSinglestepUpgrade* i_upgrade)
{
        TRACE_ENTER();
	SmfUpgradeStep *newStep = new SmfUpgradeStep();
	const SmfUpgradeScope* scope = i_upgrade->getUpgradeScope();
	const SmfForAddRemove* forAddRemove = 
		dynamic_cast<const SmfForAddRemove*>(scope);

	newStep->setRdn("safSmfStep=1");
	newStep->setDn(newStep->getRdn() + "," + getDn());
	newStep->setMaxRetry(i_upgrade->getStepMaxRetryCount());
	newStep->setRestartOption(i_upgrade->getStepRestartOption());

	if (forAddRemove != NULL) {
		TRACE("SmfUpgradeProcedure::calculateSingleStep:calculateSingleStep: SmfForAddRemove");
		const SmfActivationUnitType* aunit;
		std::list<SmfEntity>::const_iterator e;
		std::list<SmfImmCreateOperation>::const_iterator o;

		/*
		 * Handle the activation-unit.
		 */
		aunit = forAddRemove->getActivationUnit();
		assert(aunit != NULL);
		assert(aunit->getRemoved().size() == 0);
		assert(aunit->getSwRemove().size() == 0);
		for (e = aunit->getActedOn().begin(); e != aunit->getActedOn().end(); e++) {
			if (e->getParent().length() > 0 || e->getType().length() > 0) {
				LOG_ER("SmfUpgradeProcedure::calculateSingleStep:calculateSingleStep: (activation) byTemplate not implemented");
				return false;
			}
			assert(e->getName().length() > 0);
			newStep->addActivationUnit(e->getName());
		}
		newStep->addSwAdd(aunit->getSwAdd());
		for (o = aunit->getAdded().begin(); o != aunit->getAdded().end(); o++) {
			newStep->addImmOperation(new SmfImmCreateOperation(*o));
		}

		/*
		 * Handle the deactivation-unit.
		 */
		aunit = forAddRemove->getDeactivationUnit();
		assert(aunit != NULL);
		assert(aunit->getAdded().size() == 0);
		assert(aunit->getSwAdd().size() == 0);
		for (e = aunit->getActedOn().begin(); e != aunit->getActedOn().end(); e++) {
			if (e->getParent().length() > 0 || e->getType().length() > 0) {
				LOG_ER("SmfUpgradeProcedure::calculateSingleStep:calculateSingleStep: (deactivation) byTemplate not implemented");
                                delete newStep;
				return false;
			}
			assert(e->getName().length() > 0);
			newStep->addDeactivationUnit(e->getName());
		}
		newStep->addSwRemove(aunit->getSwRemove());
		for (e = aunit->getRemoved().begin(); e != aunit->getRemoved().end(); e++) {
			if (e->getParent().length() > 0 || e->getType().length() > 0) {
				LOG_ER("SmfUpgradeProcedure::calculateSingleStep:calculateSingleStep: (removed) byTemplate not implemented");
                                delete newStep;
				return false;
			}
			assert(e->getName().length() > 0);
			SmfImmDeleteOperation* deleteop = new SmfImmDeleteOperation;
			deleteop->setDn(e->getName());
			newStep->addImmOperation(deleteop);
		}

		addProcStep(newStep);
		return true;		// <--------- RETURN HERE
	}

	const SmfForModify* forModify = dynamic_cast<const SmfForModify*>(scope);
	if (forModify != NULL) {
		LOG_ER("SmfUpgradeProcedure::calculateSingleStep:calculateSingleStep: UpgradeScope SmfForModify not implemented");
                return false;
	}

	delete newStep;
	LOG_ER("SmfUpgradeProcedure::calculateSingleStep:calculateSingleStep: Unknown upgradeScope");
        return false;
}

//------------------------------------------------------------------------------
// calculateNodeList()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::calculateNodeList(const std::string & i_objectDn, std::list < std::string > &o_nodeList)
{
	SmfImmUtils immUtil;
	SaImmAttrValuesT_2 **attributes;
	const char *className;

	if (immUtil.getObject(i_objectDn, &attributes) == false) {
		LOG_ER("SmfUpgradeProcedure::calculateNodeList:failed to get imm object %s", i_objectDn.c_str());
		return false;
	}

	className = immutil_getStringAttr((const SaImmAttrValuesT_2 **)attributes, SA_IMM_ATTR_CLASS_NAME, 0);
	if (className == NULL) {
		LOG_ER("SmfUpgradeProcedure::calculateNodeList:class name not found for %s", i_objectDn.c_str());
		return false;
	}

	if (strcmp(className, "SaAmfCluster") == 0) {
		/* object DN is of class SaAmfCluster, find all SaAmfNode children */
		(void)immUtil.getChildren(i_objectDn, o_nodeList, SA_IMM_SUBLEVEL, "SaAmfNode");
	} else if (strcmp(className, "SaAmfNodeGroup") == 0) {
		/* object DN is of class SaAmfNodeGroup, get saAmfNGNodeList attribute */
		unsigned int i;

		for (i = 0; attributes[i] != NULL; i++) {
			if (strcmp(attributes[i]->attrName, "saAmfNGNodeList")
			    == 0) {
				unsigned int j;
				for (j = 0; j < attributes[i]->attrValuesNumber; j++) {
					SaNameT *amfNode = (SaNameT *) attributes[i]->attrValues[j];
					std::string amfNodeDn;
					amfNodeDn.append((char *)amfNode->value, amfNode->length);
					TRACE("calculateNodeList adding amf group node %s to node list",
					      amfNodeDn.c_str());
					o_nodeList.push_back(amfNodeDn);
				}
			}
		}
	} else {
		LOG_ER("SmfUpgradeProcedure::calculateNodeList:class name %s for %s unknown as node template", className, i_objectDn.c_str());
		return false;
	}

        return true;
}

//------------------------------------------------------------------------------
// calculateActivationUnits()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::calculateActivationUnits(const std::list < std::string > &i_nodeList,
						   const std::list < SmfParentType * >&i_actUnitTemplates,
						   std::list < std::string > &o_activationUnitList)
{
	if (i_actUnitTemplates.size() == 0) {
		/* No activation unit templates, use node list as activation/deactivation units */
		o_activationUnitList = i_nodeList;
	} else {
		/* We have activation unit templates, calculate activation/deactivation units */
		std::list < SmfParentType * >::const_iterator it = i_actUnitTemplates.begin();

		while (it != i_actUnitTemplates.end()) {
			SmfParentType *parentType = *it;
			if (parentType->getParentDn().size() == 0) {
				if (parentType->getTypeDn().size() > 0) {
					/* Only type is set */
				}
			} else {
				if (parentType->getTypeDn().size() == 0) {
					/* Only parent is set */
				} else {
					/* Both type and parent is set */
				}
			}
			it++;
		}
	}

        return true;
}

//------------------------------------------------------------------------------
// calcActivationUnitsFromTemplate()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::calcActivationUnitsFromTemplate(SmfParentType * i_parentType, 
                                                     const std::list < std::string >&i_nodeList,
                                                     std::list < std::string > &o_actDeactUnits)
{
        TRACE_ENTER();
        SmfImmUtils immUtil;
        const char *className;
        SaImmAttrValuesT_2 **attributes;

//TODO: If the components are of non restartable type the SU shall be the act/deact unit. 
//      If the restart flag is set in the campaign and the components are non restartable
//      it is considered an error and the campaign fails. The oposit way if restartable
//      components are found and the restart flag is not set it is considered an error and 
//      the campaign shall fail.

        if ( (i_parentType->getTypeDn().size() == 0) && (i_parentType->getParentDn().size() == 0))
        {
                LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:No value for parent nor type found in activationUnitTemplate SmfParentType");
                return false;
        }

        if (i_parentType->getTypeDn().size() > 0) { /* Type is set, if parent is set it will just */
                /* narrow the search for SUs or Components   */
                                   
                /* First find out if it is a SU or components version type */
                if (immUtil.getObject(i_parentType->getTypeDn(), &attributes) == false) {
                        LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:failed to get version type imm object %s", i_parentType->getTypeDn().c_str());
                        return false;
                }

                className = immutil_getStringAttr((const SaImmAttrValuesT_2 **)
                                                  attributes, SA_IMM_ATTR_CLASS_NAME, 0);
                if (className == NULL) {
                        LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:class name not found for version type %s", i_parentType->getTypeDn().c_str());
                        return false;
                }

                if (strcmp(className, "SaAmfSUType") == 0) {
                        TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Check SU type %s for modifications", i_parentType->getTypeDn().c_str());
                        std::list < std::string > foundObjs; //All found objects of type SaAmfSUType in the system

                        /* type DN is of class SaAmfSUType, find all SU's of this version type */
                        (void)immUtil.getChildren(i_parentType->getParentDn(), foundObjs, SA_IMM_SUBTREE, "SaAmfSU");
                                               
                        std::list < std::string >::const_iterator objit;
                        //For each found object check the versioned type
                        for (objit = foundObjs.begin(); objit != foundObjs.end(); ++objit) {
                                TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Check SU %s for modifications", (*objit).c_str());
                                if (immUtil.getObject((*objit), &attributes) == true) {
                                        const SaNameT *typeRef = immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes, 
                                                                                     "saAmfSUType",
                                                                                     0);
                                        if ((typeRef != NULL) && 
                                            (strcmp(i_parentType->getTypeDn().c_str(), (char *)typeRef->value) == 0)) {
                                                                      
                                                /* This SU is of the correct version type. Check if it is hosted by any node in the node list */
                                                const SaNameT *hostedByNode =
                                                        immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                            "saAmfSUHostedByNode",
                                                                            0);
                                                if (hostedByNode != NULL) {
                                                        std::list < std::string >::const_iterator it;
                                                        for (it = i_nodeList.begin(); it != i_nodeList.end(); ++it) {
                                                                if (strcmp((*it).c_str(), (char *)hostedByNode->value) == 0) {
                                                                        /* The SU is hosted by the node */
                                                                        TRACE("SU %s hosted by %s, add to list", (*objit).c_str(), (char *)hostedByNode->value);
                                                                        o_actDeactUnits.push_back(*objit);
                                                                        break;
                                                                }
                                                        }
                                                } else {
                                                        LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:No hostedByNode attr set for %s", (*objit).c_str());  
                                                        return false;
                                                }
                                        }
                                } else {
                                        LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Fails to get object %s", (*objit).c_str());  
                                        return false;
                                }
                        }
                } else if (strcmp(className, "SaAmfCompType") == 0) {
                        TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Check Comp type %s for modifications", i_parentType->getTypeDn().c_str());
                        std::list < std::string > foundObjs; //All found objects of type SaAmfCompType in the system

                        /* type DN is of class SaAmfCompType, find all components's of this version type */
                        (void)immUtil.getChildren(i_parentType->getParentDn(), foundObjs, SA_IMM_SUBTREE, "SaAmfComp");

                        std::list < std::string >::const_iterator objit;
                        for (objit = foundObjs.begin(); objit != foundObjs.end(); ++objit) {
                                TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Check Comp %s for modifications", (*objit).c_str());
                                if (immUtil.getObject((*objit), &attributes) == true) {
                                        const SaNameT *typeRef =
                                                immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                    "saAmfCompType", 0);

                                        if ((typeRef != NULL)
                                            && (strcmp(i_parentType->getTypeDn().c_str(), (char *)typeRef->value) == 0)) {
                                                /* This component is of the correct version type. Check if it is hosted by any node in the node list */
                                                /* Find the parent SU to this component */
                                                if (immUtil.getParentObject((*objit), &attributes) == true) {
                                                        const SaNameT *hostedByNode = immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                                                          "saAmfSUHostedByNode", 
                                                                                                          0);
                                                        if (hostedByNode != NULL){
                                                                std::list < std::string >::const_iterator it;
                                                                for (it = i_nodeList.begin(); it != i_nodeList.end(); ++it) {
                                                                        if (strcmp((*it).c_str(), (char *)hostedByNode->value) == 0) {
                                                                                /* The SU is hosted by the node */
                                                                                TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Component %s hosted by %s, add to list", (*objit).c_str(), (char *)hostedByNode->value);
                                                                                o_actDeactUnits.push_back(*objit);
                                                                                break;
                                                                        }
                                                                }
                                                        } else {
                                                                LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:No hostedByNode attr set for %s", (*objit).c_str());  
                                                                return false;
                                                        }
                                                }
                                        }
                                } else {
                                        LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Fails to get object %s", (*objit).c_str());  
                                        return false;
                                }
                        }
                }
        }
        if (i_parentType->getTypeDn().size() == 0) {
                /* Only parent is set */
                TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Find SUs for parent SG %s for modifications", i_parentType->getTypeDn().c_str());
                std::list < std::string > foundObjs; //All found SUs which are children to the parent SG

                /* type DN is of class SaAmfSUType, find all SU's of this version type */
                (void)immUtil.getChildren(i_parentType->getParentDn(), foundObjs, SA_IMM_SUBTREE, "SaAmfSU");
                                               
                std::list < std::string >::const_iterator objit;
                //For each found object check the versioned type
                for (objit = foundObjs.begin(); objit != foundObjs.end(); ++objit) {
                        TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Check SU %s for modifications", (*objit).c_str());
                        if (immUtil.getObject((*objit), &attributes) == true) {
                                /* Type does not matter. Check if the SU is hosted by any node in the node list */
                                const SaNameT *hostedByNode = immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                                  "saAmfSUHostedByNode",
                                                                                  0);
                                if (hostedByNode != NULL) {
                                        std::list < std::string >::const_iterator it;
                                        for (it = i_nodeList.begin(); it != i_nodeList.end(); ++it) {
                                                if (strcmp((*it).c_str(), (char *)hostedByNode->value) == 0) {
                                                        /* The SU is hosted by the node */
                                                        TRACE("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:SU %s hosted by %s, add to list", (*objit).c_str(), (char *)hostedByNode->value);
                                                        o_actDeactUnits.push_back(*objit);
                                                        break;
                                                }
                                        }
                                }else {
                                        LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:No hostedByNode attr set for %s", (*objit).c_str());  
                                        return false;
                                }
                        } else {
                                LOG_ER("SmfUpgradeProcedure::calcActivationUnitsFromTemplate:Fails to get object %s", (*objit).c_str());  
                                return false;
                        }
                }
        }

        TRACE_LEAVE();
        return true;
}

//------------------------------------------------------------------------------
// addProcStep()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::addProcStep(SmfUpgradeStep * i_step)
{
	std::vector < SmfUpgradeStep * >::iterator iter;
	iter = m_procSteps.begin();

	i_step->setProcedure(this);

	while (iter != m_procSteps.end()) {
		if ((*iter)->getRdn() > i_step->getRdn()) {
			/* Insert new step before this one */
			m_procSteps.insert(iter, i_step);
			return;
		}
		iter++;
	}

	m_procSteps.push_back(i_step);
}

//------------------------------------------------------------------------------
// addStepModifications()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::addStepModifications(SmfUpgradeStep * i_newStep,
                                          const std::list < SmfTargetEntityTemplate * >&i_targetEntityTemplate,
                                          SmfAuT i_auType)
{
	std::list < SmfTargetEntityTemplate * >::const_iterator it;

	for (it = i_targetEntityTemplate.begin(); it != i_targetEntityTemplate.end(); ++it) {
		SmfTargetEntityTemplate *targetEntity = *it;
		const SmfParentType *parentType = targetEntity->getEntityTemplate();
		const std::list < SmfImmModifyOperation * >&modificationList = targetEntity->getModifyOperationList();

		switch (i_auType) {
		case SMF_AU_AMF_NODE:
			{
				if (!addStepModificationsNode(i_newStep, parentType, modificationList)) {
                                        LOG_ER("SmfUpgradeProcedure::addStepModifications: addStepModificationsNode failed");
                                        return false;
                                }
				break;
			}
		case SMF_AU_SU:
			{
				if ( !addStepModificationsSu(i_newStep, parentType, modificationList)) {
                                        LOG_ER("SmfUpgradeProcedure::addStepModifications: addStepModificationsSu failed");
                                        return false;
                                }
				break;
			}
		case SMF_AU_COMP:
			{
				if (!addStepModificationsComp(i_newStep, parentType, modificationList)) {
                                        LOG_ER("SmfUpgradeProcedure::addStepModifications: addStepModificationsComp failed");
                                        return false;
                                }
				break;
			}
		default:
                        
                        LOG_ER("SmfUpgradeProcedure::addStepModifications: unknown auType");
                        return false;
			break;
		}
	}

        return true;
}

//------------------------------------------------------------------------------
// addStepModificationsNode()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::addStepModificationsNode(SmfUpgradeStep * i_newStep, const SmfParentType * i_parentType,
                                              const std::list < SmfImmModifyOperation * >&i_modificationList)
{
        TRACE_ENTER();
	SmfImmUtils immUtil;
	SaImmAttrValuesT_2 **attributes;
	const char *className;
	std::list < std::string > objectList;
	std::list < std::string >::const_iterator objit;
	const std::string & auNodeName = i_newStep->getActivationUnitList().front();

	if (i_parentType->getTypeDn().size() > 0) {
		/* Type was specified, find all objects of this version type */
		TRACE("SmfUpgradeProcedure::addStepModificationsNode:Check type %s for modifications", i_parentType->getTypeDn().c_str());

		/* First find out if it's a SU or components version type */
		if (immUtil.getObject(i_parentType->getTypeDn(), &attributes) == false) {
			LOG_ER("SmfUpgradeProcedure::addStepModificationsNode:failed to get version type imm object %s", i_parentType->getTypeDn().c_str());
			return false;
		}

		className = immutil_getStringAttr((const SaImmAttrValuesT_2 **)
						  attributes, SA_IMM_ATTR_CLASS_NAME, 0);
		if (className == NULL) {
			LOG_ER("SmfUpgradeProcedure::addStepModificationsNode:class name not found for version type %s", i_parentType->getTypeDn().c_str());
			return false;
		}

		if (strcmp(className, "SaAmfSUType") == 0) {
			TRACE("SmfUpgradeProcedure::addStepModificationsNode:Check SU type %s for modifications", i_parentType->getTypeDn().c_str());

			/* type DN is of class SaAmfSUType, find all SU's of this version type */
			(void)immUtil.getChildren(i_parentType->getParentDn(), objectList, SA_IMM_SUBTREE, "SaAmfSU");

			for (objit = objectList.begin(); objit != objectList.end(); ++objit) {
				TRACE("SmfUpgradeProcedure::addStepModificationsNode:Check SU %s for modifications", (*objit).c_str());
				if (immUtil.getObject((*objit), &attributes) == true) {
					const SaNameT *typeRef =
                                                immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes, "saAmfSUType",
                                                                    0);

					if ((typeRef != NULL)
					    && (strcmp(i_parentType->getTypeDn().c_str(), (char *)typeRef->value) == 0)) {
						/* This SU is of the correct version type */

						/* Check if the found SU is hosted by the activation unit (node) */
						const SaNameT *hostedByNode =
                                                        immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                            "saAmfSUHostedByNode",
                                                                            0);
						if ((hostedByNode != NULL)
						    && (strcmp(auNodeName.c_str(), (char *)hostedByNode->value) == 0)) {
							/* The SU is hosted by the AU node */
							TRACE("SmfUpgradeProcedure::addStepModificationsNode:SU %s hosted by %s, add modifications", (*objit).c_str(),
							      auNodeName.c_str());
							if (!addStepModificationList(i_newStep, *objit, i_modificationList)) {
                                                                LOG_ER("SmfUpgradeProcedure::addStepModificationsNode:addStepModificationList fails");
                                                                return false;
                                                        }
						}
					}
				}
			}
		} else if (strcmp(className, "SaAmfCompType") == 0) {
			TRACE("SmfUpgradeProcedure::addStepModificationsNode:Check Comp type %s for modifications", i_parentType->getTypeDn().c_str());

			/* type DN is of class SaAmfCompType, find all components's of this version type */
			(void)immUtil.getChildren(i_parentType->getParentDn(), objectList, SA_IMM_SUBTREE, "SaAmfComp");

			for (objit = objectList.begin(); objit != objectList.end(); ++objit) {
				TRACE("SmfUpgradeProcedure::addStepModificationsNode:Check Comp %s for modifications", (*objit).c_str());
				if (immUtil.getObject((*objit), &attributes) == true) {
					const SaNameT *typeRef =
                                                immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                    "saAmfCompType", 0);

					if ((typeRef != NULL)
					    && (strcmp(i_parentType->getTypeDn().c_str(), (char *)typeRef->value) == 0)) {
						/* This component is of the correct version type */

						/* Check if the found component is hosted by the activation unit (node) */

						/* Find the parent SU to this component */
						if (immUtil.getParentObject((*objit), &attributes) == true) {
							const SaNameT *hostedByNode =
                                                                immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                                                    "saAmfSUHostedByNode", 0);
							if ((hostedByNode != NULL)
							    && (strcmp(auNodeName.c_str(), (char *)hostedByNode->value)
								== 0)) {
								/* The component is hosted by the AU node */
								TRACE("Component %s hosted by %s, add modifications",
								      (*objit).c_str(), auNodeName.c_str());
                                                                if (!addStepModificationList(i_newStep, *objit, i_modificationList)) {
                                                                        LOG_ER("SmfUpgradeProcedure::addStepModificationsNode:addStepModificationList fails");
                                                                        return false;
                                                                }
							}
						}
					}
				}
			}
		} else {
			LOG_ER("SmfUpgradeProcedure::addStepModificationsNode:class name %s for type %s not valid as type", className,
			       i_parentType->getTypeDn().c_str());
			return false;
		}
	} else if (i_parentType->getParentDn().size() > 0) {	
                /* Parent was specified */
                TRACE("SmfUpgradeProcedure::addStepModificationsNode:Check SU children to parent %s for modifications", i_parentType->getParentDn().c_str());

                /* Parent SG but no type specified, find all child SU's to the parent SG */
                (void)immUtil.getChildren(i_parentType->getParentDn(), objectList, SA_IMM_SUBLEVEL, "SaAmfSU");

                for (objit = objectList.begin(); objit != objectList.end(); ++objit) {
                        TRACE("SmfUpgradeProcedure::addStepModificationsNode:Check child SU %s for modifications", (*objit).c_str());
                        if (immUtil.getObject((*objit), &attributes) == true) {
                                /* Check if the found SU is hosted by the activation unit (node) */
                                const SaNameT *hostedByNode =
                                        immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes,
                                                            "saAmfSUHostedByNode",
                                                            0);
                                if ((hostedByNode != NULL)
                                    && (strcmp(auNodeName.c_str(), (char *)hostedByNode->value) == 0)) {
                                        /* The SU is hosted by the AU node */
                                        TRACE("SmfUpgradeProcedure::addStepModificationsNode:SU %s hosted by %s, add modifications", (*objit).c_str(),
                                              auNodeName.c_str());
                                        if (!addStepModificationList(i_newStep, *objit, i_modificationList)) {
                                                LOG_ER("SmfUpgradeProcedure::addStepModificationsNode:addStepModificationList fails");
                                                return false;
                                        }
                                }
                        }
                }
        } else {                          
                //No parent type was set, apply the modifications on the entity pointed out by the activation unit.
                //The optional modifyOperation RDN attribute if added when applying the modification
 
                if (!addStepModificationList(i_newStep, auNodeName, i_modificationList)) {
                        LOG_ER("SmfUpgradeProcedure::addStepModificationsNode:addStepModificationList fails");
                        return false;
                }
        }

        TRACE_LEAVE();
        return true;
}

//------------------------------------------------------------------------------
// addStepModificationsSu()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::addStepModificationsSu(SmfUpgradeStep * i_newStep, const SmfParentType * i_parentType,
                                            const std::list < SmfImmModifyOperation * >&i_modificationList)
{
        TRACE_ENTER();
	SmfImmUtils immUtil;
	SaImmAttrValuesT_2 **attributes;
	const char *className;
	std::list < std::string > objectList;
	std::list < std::string >::const_iterator objit;
        std::list < std::string >::const_iterator auEntityIt;

//	const std::string & auEntity = i_newStep->getActivationUnitList().front();
        const std::list < std::string > auEntityList = i_newStep->getActivationUnitList();

        if (i_parentType->getTypeDn().size() > 0) { //Type is set, parent may be set to limit the scope
                /* Type was specified, find all objects of this version type */
                TRACE("SmfUpgradeProcedure::addStepModificationsSu:Check type %s for modifications", i_parentType->getTypeDn().c_str());

                /* First find out if it's a SU or components version type */
                if (immUtil.getObject(i_parentType->getTypeDn(), &attributes) == false) {
                        LOG_ER("SmfUpgradeProcedure::addStepModificationsSu:failed to get version type imm object %s", i_parentType->getTypeDn().c_str());
                        return false;
                }

                className = immutil_getStringAttr((const SaImmAttrValuesT_2 **)
                                                  attributes, SA_IMM_ATTR_CLASS_NAME, 0);
                if (className == NULL) {
                        LOG_ER("SmfUpgradeProcedure::addStepModificationsSu:class name not found for version type %s", i_parentType->getTypeDn().c_str());
                        return false;
                }

                if (strcmp(className, "SaAmfSUType") == 0) {
                        TRACE("SmfUpgradeProcedure::addStepModificationsSu:Check SU type %s for modifications", i_parentType->getTypeDn().c_str());

                        /* type DN is of class SaAmfSUType, find all SU's of this version type */
                        (void)immUtil.getChildren(i_parentType->getParentDn(), objectList, SA_IMM_SUBTREE, "SaAmfSU");

			for (objit = objectList.begin(); objit != objectList.end(); ++objit) {
				TRACE("SmfUpgradeProcedure::addStepModificationsSu:Check SU %s for modifications", (*objit).c_str());
				if (immUtil.getObject((*objit), &attributes) == true) {
					const SaNameT *typeRef =
					    immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes, "saAmfSUType", 0);

					if ((typeRef != NULL)
					    && (strcmp(i_parentType->getTypeDn().c_str(), (char *)typeRef->value) == 0)) {
						/* This SU is of the correct version type */
                                                //For all AU/DU in the step, check if the SU DN is within the AU/DU DN
                                                for (auEntityIt = auEntityList.begin(); auEntityIt != auEntityList.end(); ++auEntityIt) {
                                                        if((*objit).find(*auEntityIt) != std::string::npos) {
                                                                //The auEntityIt DN is a substring of objit i.e. the object 
                                                                //to modify within the AU/DU domain
                                                                //Add object modifications

                                                                TRACE("SmfUpgradeProcedure::addStepModificationsSu:Modification DN=%s, is within AU/DU=%s, add modifications", (*objit).c_str(),(*auEntityIt).c_str());
                                                   
                                                                if (!addStepModificationList(i_newStep, *objit, i_modificationList)) {
                                                                        LOG_ER("SmfUpgradeProcedure::addStepModificationsSu:addStepModificationList fails");
                                                                        return false;
                                                                }
                                                                break;

                                                        } else {
                                                                TRACE("SmfUpgradeProcedure::addStepModificationsSu:NO MATCH Modification DN=%s, AU/DU=%s", (*objit).c_str(),(*auEntityIt).c_str());
                                                        }
                                                }
					}
				}
			}
                } else if (strcmp(className, "SaAmfCompType") == 0) {
                        TRACE("SmfUpgradeProcedure::addStepModificationsSu:Check Comp type %s for modifications", i_parentType->getTypeDn().c_str());

                        /* type DN is of class SaAmfCompType, find all components's of this version type */
                        (void)immUtil.getChildren(i_parentType->getParentDn(), objectList, SA_IMM_SUBTREE, "SaAmfComp");

                        for (objit = objectList.begin(); objit != objectList.end(); ++objit) {
				TRACE("SmfUpgradeProcedure::addStepModificationsSu:Check Comp %s for modifications", (*objit).c_str());
				if (immUtil.getObject((*objit), &attributes) == true) {
					const SaNameT *typeRef =
                                                immutil_getNameAttr((const SaImmAttrValuesT_2 **)attributes, "saAmfCompType", 0);

					if ((typeRef != NULL)
					    && (strcmp(i_parentType->getTypeDn().c_str(), (char *)typeRef->value) == 0)) {
						/* This component is of the correct version type */
                                                //For all objects found of this type, match DNs to see if it is within the act/deact unit
                                                for (objit = objectList.begin(); objit != objectList.end(); ++objit) {
                                                        for (auEntityIt = auEntityList.begin(); auEntityIt != auEntityList.end(); ++auEntityIt) {
                                                                if((*objit).find(*auEntityIt) != std::string::npos) {
                                                                        //The auEntityIt DN is a substring of objit i.e. the object 
                                                                        //to modify within the AU/DU domain
                                                                        //Add object modifications

                                                                        TRACE("SmfUpgradeProcedure::addStepModificationsSu:Modification DN=%s, is within AU/DU=%s, add modifications", (*objit).c_str(),(*auEntityIt).c_str());

                                                                        if (!addStepModificationList(i_newStep, *objit, i_modificationList)) {
                                                                                LOG_ER("SmfUpgradeProcedure::addStepModificationsSu:addStepModificationList fails");
                                                                                return false;
                                                                        }

                                                                        break;
                                                                } else {
                                                                        TRACE("SmfUpgradeProcedure::addStepModificationsSu:NO MATCH Modification DN=%s, AU/DU=%s", (*objit).c_str(),(*auEntityIt).c_str());
                                                                }
                                                        }
                                                }
					}
				}
			}
                } else {
                        LOG_ER("SmfUpgradeProcedure::addStepModificationsSu:class name %s for type %s not valid as type",
                               className,
                               i_parentType->getTypeDn().c_str());
                        return false;
                }
        } else if (i_parentType->getParentDn().size() > 0) { //Parent only is set, no type
                /* Parent was specified */
                TRACE("SmfUpgradeProcedure::addStepModificationsSu:Check SU children to parent %s for modifications", i_parentType->getParentDn().c_str());

                /* Parent SG but no type specified, find all child SU's to the parent SG */
                (void)immUtil.getChildren(i_parentType->getParentDn(), objectList, SA_IMM_SUBLEVEL, "SaAmfSU");

                for (objit = objectList.begin(); objit != objectList.end(); ++objit) {
                        for (auEntityIt = auEntityList.begin(); auEntityIt != auEntityList.end(); ++auEntityIt) {
                                if((*objit).find(*auEntityIt) != std::string::npos) {
                                        //The auEntityIt DN is a substring of objit i.e. the object 
                                        //to modify within the AU/DU domain
                                        //Add object modifications

                                        TRACE("SmfUpgradeProcedure::addStepModificationsSu:Modification DN=%s, is within AU/DU=%s, add modifications", (*objit).c_str(),(*auEntityIt).c_str());

                                        if (!addStepModificationList(i_newStep, *objit, i_modificationList)) {
                                                LOG_ER("SmfUpgradeProcedure::addStepModificationsSu:addStepModificationList fails");
                                                return false;
                                        }

                                        break;
                                }

                        }
                }
        } else {

                //No parent type was set, apply the modifications on the entity pointed out by the activation units.
                //The optional modifyOperation RDN attribute if added when applying the modification
                std::list < std::string >::const_iterator auEntityIt;
                for (auEntityIt = auEntityList.begin(); auEntityIt != auEntityList.end(); ++auEntityIt) {
                        if (!addStepModificationList(i_newStep, *auEntityIt, i_modificationList)) {
                                LOG_ER("SmfUpgradeProcedure::addStepModificationsSu:addStepModificationList fails");
                                return false;
                        }
                }
        }

        TRACE_LEAVE();
        return true;
}

//------------------------------------------------------------------------------
// addStepModificationsComp()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::addStepModificationsComp(SmfUpgradeStep * i_newStep, const SmfParentType * i_parentType,
						   const std::list < SmfImmModifyOperation * >&i_modificationList)
{
        return true;
}

//------------------------------------------------------------------------------
// addStepModificationList()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::addStepModificationList(SmfUpgradeStep * i_newStep, const std::string & i_dn,
						  const std::list < SmfImmModifyOperation * >&i_modificationList)
{
        TRACE_ENTER();
	SmfImmUtils immUtil;

        //The modifications must be copied, since several steps can use the same modification list
        //but with different DNs. Deep copy not neccessary since only DN is changed.

        //If RND is set, modifications shall be applied to entities matching DN + RDN. This is used to
        //apply changes to a certain Component within a SU, where DN is the SU and RDN is the component.

	std::list < SmfImmModifyOperation * >::const_iterator modit;
	for (modit = i_modificationList.begin(); modit != i_modificationList.end(); ++modit) {
		if ((*modit)->getRdn().size() > 0) {
			std::string dn = (*modit)->getRdn(); 
			dn += "," + i_dn;
                        
                        //Check if object exist
                        SaImmAttrValuesT_2 **attributes;
                        if (immUtil.getObject(dn, &attributes) == true) {
                                SmfImmModifyOperation *modOp = new SmfImmModifyOperation(*(*modit));
                                modOp->setDn(dn);
                                i_newStep->addModification(modOp);
                        }
                } else {
                        SmfImmModifyOperation *modOp = new SmfImmModifyOperation(*(*modit));
			modOp->setDn(i_dn);
                        i_newStep->addModification(modOp);
                }
	}

        TRACE_LEAVE();
        return true;
}

//------------------------------------------------------------------------------
// createImmSteps()
//------------------------------------------------------------------------------
bool 
SmfUpgradeProcedure::createImmSteps()
{
	TRACE_ENTER();
	SaAisErrorT rc;

	std::vector < SmfUpgradeStep * >::iterator iter;
	iter = m_procSteps.begin();
	while (iter != m_procSteps.end()) {
                if ((rc=createImmStep(*iter)) != SA_AIS_OK){
                        LOG_ER("SmfUpgradeProcedure::createImmSteps: createImmStep returns SaAisErrorT=%d", rc);
                        return false;
                }
		iter++;
	}

	TRACE_LEAVE();
        return true;
}

//------------------------------------------------------------------------------
// createImmStep()
//------------------------------------------------------------------------------
SaAisErrorT 
SmfUpgradeProcedure::createImmStep(SmfUpgradeStep * i_step)
{
	TRACE_ENTER();
	SaAisErrorT rc = SA_AIS_OK;
        std::string dnDeactUnit = "safSmfDu=smfDeactivationUnit";
        std::string dnActUnit   = "safSmfAu=smfActivationUnit";
        char str[64];
        SmfImmRTCreateOperation ico;
        SmfImmAttribute attr;

        /* Create the SaSmfStep object */
        SmfImmRTCreateOperation icoSaSmfStep;
        icoSaSmfStep.setClassName("SaSmfStep");
        icoSaSmfStep.setParentDn(getDn());
        icoSaSmfStep.setImmHandle(getProcThread()->getImmHandle());

        SmfImmAttribute attrSafSmfStep;
        attrSafSmfStep.setName("safSmfStep");
        attrSafSmfStep.setType("SA_IMM_ATTR_SASTRINGT");
        attrSafSmfStep.addValue(i_step->getRdn());
        icoSaSmfStep.addValue(attrSafSmfStep);

        SmfImmAttribute attrsaSmfStepMaxRetry;
        attrsaSmfStepMaxRetry.setName("saSmfStepMaxRetry");
        attrsaSmfStepMaxRetry.setType("SA_IMM_ATTR_SAUINT32T");
        sprintf(str,"%d",i_step->getMaxRetry());
        attrsaSmfStepMaxRetry.addValue(str);
        icoSaSmfStep.addValue(attrsaSmfStepMaxRetry);

        SmfImmAttribute attrsaSmfStepRetryCount;
        attrsaSmfStepRetryCount.setName("saSmfStepRetryCount");
        attrsaSmfStepRetryCount.setType("SA_IMM_ATTR_SAUINT32T");
        sprintf(str,"%d",i_step->getRetryCount());
        attrsaSmfStepRetryCount.addValue(str);
        icoSaSmfStep.addValue(attrsaSmfStepRetryCount);
 
        SmfImmAttribute attrsaSmfStepRestartOption;
        attrsaSmfStepRestartOption.setName("saSmfStepRestartOption");
        attrsaSmfStepRestartOption.setType("SA_IMM_ATTR_SAUINT32T");
        sprintf(str,"%d",i_step->getRestartOption());
        attrsaSmfStepRestartOption.addValue(str);
        icoSaSmfStep.addValue(attrsaSmfStepRestartOption);

        SmfImmAttribute attrsaSmfStepState;
        attrsaSmfStepState.setName("saSmfStepState");
        attrsaSmfStepState.setType("SA_IMM_ATTR_SAUINT32T");
        sprintf(str,"%d",i_step->getState());
        attrsaSmfStepState.addValue(str);
        icoSaSmfStep.addValue(attrsaSmfStepState);

        SmfImmAttribute attrsaSmfStepError;
        attrsaSmfStepError.setName("saSmfStepError");
        attrsaSmfStepError.setType("SA_IMM_ATTR_SASTRINGT");
        attrsaSmfStepError.addValue("");
        icoSaSmfStep.addValue(attrsaSmfStepError);

        if ((rc=icoSaSmfStep.execute()) != SA_AIS_OK) { //Create the object
                LOG_ER("SmfUpgradeProcedure::createImmStep: icoSaSmfStep.execute() returned %d", rc);
                TRACE_LEAVE();
                return rc;
        }

	std::list < std::string >::const_iterator iter;
	std::list < std::string >::const_iterator iterE;

	if (i_step->getDeactivationUnitList().size() > 0) {
		/* Create the SaSmfDeactivationUnit object */
		SmfImmRTCreateOperation icoSaSmfDeactivationUnit;
		icoSaSmfDeactivationUnit.setClassName("SaSmfDeactivationUnit");
		icoSaSmfDeactivationUnit.setParentDn(i_step->getRdn() + "," + getDn());
		icoSaSmfDeactivationUnit.setImmHandle(getProcThread()->getImmHandle());

		SmfImmAttribute attrsafSmfDu;
		attrsafSmfDu.setName("safSmfDu");
		attrsafSmfDu.setType("SA_IMM_ATTR_SASTRINGT");
		attrsafSmfDu.addValue("safSmfDu=smfDeactivationUnit");
		icoSaSmfDeactivationUnit.addValue(attrsafSmfDu);

		SmfImmAttribute attrsaSmfDuActedOn;
		attrsaSmfDuActedOn.setName("saSmfDuActedOn");
		attrsaSmfDuActedOn.setType("SA_IMM_ATTR_SANAMET");
		const std::list < std::string > deactList = i_step->getDeactivationUnitList();
		if(deactList.size() != 0) {
			iter = deactList.begin();
			iterE = deactList.end();
			while (iter != iterE) {
				attrsaSmfDuActedOn.addValue(*iter);
				iter++;
			}
		} else {
			/* A value must always be supplied to a cached runtime attribute */
			attrsaSmfDuActedOn.addValue("");
		}
		icoSaSmfDeactivationUnit.addValue(attrsaSmfDuActedOn);

		SmfImmAttribute attrsaSmfDuEntityToRemove;
		attrsaSmfDuEntityToRemove.setName("saSmfDuEntityToRemove");
		attrsaSmfDuEntityToRemove.setType("SA_IMM_ATTR_SANAMET");

		if (!setEntitiesToAddRemMod(i_step, &attrsaSmfDuEntityToRemove)) {
			rc = SA_AIS_ERR_CAMPAIGN_ERROR_DETECTED;
		}
		icoSaSmfDeactivationUnit.addValue(attrsaSmfDuEntityToRemove);

                if ((rc=icoSaSmfDeactivationUnit.execute()) != SA_AIS_OK) { //Create the object
                        LOG_ER("SmfUpgradeProcedure::createImmStep: icoSaSmfDeactivationUnit.execute() returned %d", rc);
                        TRACE_LEAVE();
                        return rc;
                }
	}

        /* Create the SaSmfImageNodes objects as childrens to the SaSmfDeactivationUnit instance */
        /* One SaSmfImageNodes object for each removed software bundle                           */

        std::list < SmfBundleRef >::const_iterator bundleRefiter;
	std::list < SmfBundleRef >::const_iterator bundleRefiterE;

        const std::list < SmfBundleRef > removedBundles = i_step->getSwRemoveList();  //Get the list of bundles to remove in this step
	bundleRefiter = removedBundles.begin();
	bundleRefiterE = removedBundles.end();

        //For each sw bundle, create a SaSmfImageNodes object as child to the SaSmfDeactivationUnit
	while (bundleRefiter != bundleRefiterE) {
                SmfImmRTCreateOperation icoSaSmfImageNodes;
                icoSaSmfImageNodes.setClassName("SaSmfImageNodes");
                icoSaSmfImageNodes.setParentDn(dnDeactUnit + "," + i_step->getRdn() + "," + getDn());
                icoSaSmfImageNodes.setImmHandle(getProcThread()->getImmHandle());

                SmfImmAttribute safIMageNode;
                safIMageNode.setName("safImageNode");
                safIMageNode.setType("SA_IMM_ATTR_SASTRINGT");
                //Extract the bundle name from the DN
                std::string imageNode = "safImageNode=";
                std::string bundleName = (*bundleRefiter).getBundleDn();
                std::string::size_type pos = bundleName.find("safSmfBundle=");
                pos += strlen("safSmfBundle=");
                imageNode += bundleName.substr(pos, bundleName.find(",") - pos);
                safIMageNode.addValue(imageNode);
                icoSaSmfImageNodes.addValue(safIMageNode);

                SmfImmAttribute saSmfINSwBundle;
                saSmfINSwBundle.setName("saSmfINSwBundle");
                saSmfINSwBundle.setType("SA_IMM_ATTR_SANAMET");
                saSmfINSwBundle.addValue((*bundleRefiter).getBundleDn());
                icoSaSmfImageNodes.addValue(saSmfINSwBundle);

                SmfImmAttribute saSmfINNode;
                saSmfINNode.setName("saSmfINNode");
                saSmfINNode.setType("SA_IMM_ATTR_SANAMET");
                saSmfINNode.addValue(i_step->getSwNode());
                icoSaSmfImageNodes.addValue(saSmfINNode);

                if ((rc=icoSaSmfImageNodes.execute()) != SA_AIS_OK) { //Create the object
                        LOG_ER("SmfUpgradeProcedure::createImmStep: icoSaSmfImageNodes.execute() returned %d", rc);
                        TRACE_LEAVE();
                        return rc;
                }

		bundleRefiter++;
	}

	if (i_step->getActivationUnitList().size() > 0) {
		/* Create the SaSmfActivationUnit object */
		SmfImmRTCreateOperation icoSaSmfActivationUnit;
		icoSaSmfActivationUnit.setClassName("SaSmfActivationUnit");
		icoSaSmfActivationUnit.setParentDn(i_step->getRdn() + "," + getDn());
		icoSaSmfActivationUnit.setImmHandle(getProcThread()->getImmHandle());

		SmfImmAttribute attrsafSmfAu;
		attrsafSmfAu.setName("safSmfAu");
		attrsafSmfAu.setType("SA_IMM_ATTR_SASTRINGT");
		attrsafSmfAu.addValue("safSmfAu=smfActivationUnit");
		icoSaSmfActivationUnit.addValue(attrsafSmfAu);

		SmfImmAttribute attrsaSmfAuActedOn;
		attrsaSmfAuActedOn.setName("saSmfAuActedOn");
		attrsaSmfAuActedOn.setType("SA_IMM_ATTR_SANAMET");
		const std::list < std::string > actList = i_step->getActivationUnitList();
		if(actList.size() != 0) {
			iter = actList.begin();
			iterE = actList.end();
			while (iter != iterE) {
				attrsaSmfAuActedOn.addValue(*iter);
				iter++;
			}
		} else {
			/* A value must always be supplied to a cached runtime attribute */
			attrsaSmfAuActedOn.addValue("");
		}
		icoSaSmfActivationUnit.addValue(attrsaSmfAuActedOn);

		SmfImmAttribute attrsaSmfAuEntityToAdd;
		attrsaSmfAuEntityToAdd.setName("saSmfAuEntityToAdd");
		attrsaSmfAuEntityToAdd.setType("SA_IMM_ATTR_SANAMET");

		if (!setEntitiesToAddRemMod(i_step, &attrsaSmfAuEntityToAdd)) {
			rc = SA_AIS_ERR_CAMPAIGN_ERROR_DETECTED;
		}
		icoSaSmfActivationUnit.addValue(attrsaSmfAuEntityToAdd);

                if ((rc=icoSaSmfActivationUnit.execute()) != SA_AIS_OK) { //Create the object
                        LOG_ER("SmfUpgradeProcedure::createImmStep: icoSaSmfActivationUnit.execute() returned %d", rc);
                        TRACE_LEAVE();
                        return rc;
                }
	}

        /* Create the SaSmfImageNodes objects as childrens to the SaSmfActivationUnit instance */
        /* One SaSmfImageNodes object for each added software bundle                           */

        const std::list < SmfBundleRef > addBundles = i_step->getSwAddList();  //Get the list of bundles to add in this step
	bundleRefiter = addBundles.begin();
	bundleRefiterE = addBundles.end();

        //For each sw bundle, create a SaSmfImageNodes object as child to the SaSmfActivationUnit
	while (bundleRefiter != bundleRefiterE) {
                SmfImmRTCreateOperation icoSaSmfImageNodes;
                icoSaSmfImageNodes.setClassName("SaSmfImageNodes");
                icoSaSmfImageNodes.setParentDn(dnActUnit + "," + i_step->getRdn() + "," + getDn());
                icoSaSmfImageNodes.setImmHandle(getProcThread()->getImmHandle());

                SmfImmAttribute safIMageNode;
                safIMageNode.setName("safImageNode");
                safIMageNode.setType("SA_IMM_ATTR_SASTRINGT");
                //Extract the bundle name from the DN
                std::string imageNode = "safImageNode=";
                std::string bundleName = (*bundleRefiter).getBundleDn();
                std::string::size_type pos = bundleName.find("safSmfBundle=");
                pos += strlen("safSmfBundle=");
                imageNode += bundleName.substr(pos, bundleName.find(",") - pos);
                safIMageNode.addValue(imageNode);
                icoSaSmfImageNodes.addValue(safIMageNode);

                SmfImmAttribute saSmfINSwBundle;
                saSmfINSwBundle.setName("saSmfINSwBundle");
                saSmfINSwBundle.setType("SA_IMM_ATTR_SANAMET");
                saSmfINSwBundle.addValue((*bundleRefiter).getBundleDn());
                icoSaSmfImageNodes.addValue(saSmfINSwBundle);

                SmfImmAttribute saSmfINNode;
                saSmfINNode.setName("saSmfINNode");
                saSmfINNode.setType("SA_IMM_ATTR_SANAMET");
                saSmfINNode.addValue(i_step->getSwNode());
                icoSaSmfImageNodes.addValue(saSmfINNode);

                if ((rc=icoSaSmfImageNodes.execute()) != SA_AIS_OK) { //Create the object
                        LOG_ER("SmfUpgradeProcedure::createImmStep: icoSaSmfImageNodes.execute() returned %d", rc);
                        TRACE_LEAVE();
                        return rc;
                }

		bundleRefiter++;
	}

	TRACE_LEAVE();
	return rc;
}

//------------------------------------------------------------------------------
// getImmSteps()
//------------------------------------------------------------------------------
SaAisErrorT 
SmfUpgradeProcedure::getImmSteps()
{
	SmfImmUtils immutil;
	SaImmAttrValuesT_2 **attributes;
	SaAisErrorT rc = SA_AIS_OK;
	std::list < std::string > stepList;

	TRACE_ENTER();

	if (immutil.getChildren(getDn(), stepList, SA_IMM_SUBLEVEL, "SaSmfStep") == false) {
		LOG_ER("Failed to get steps for procedure %s", getDn().c_str());
		return SA_AIS_ERR_NOT_EXIST;
	}

	/* TODO We need to sort the list of steps here since the order from IMM is not guaranteed */
	std::list < std::string >::iterator stepit;

	/* Fetch IMM data for our upgrade procedure steps */
	for (stepit = stepList.begin(); stepit != stepList.end(); ++stepit) {
		if (immutil.getObject((*stepit), &attributes) == false) {
			LOG_ER("IMM data for step %s not found", (*stepit).c_str());
			rc = SA_AIS_ERR_NOT_EXIST;
			goto done;
		}

		SmfUpgradeStep *newStep = new SmfUpgradeStep();
		if (newStep->init((const SaImmAttrValuesT_2 **)attributes) != SA_AIS_OK) {
			LOG_ER("Initialization failed for step %s", (*stepit).c_str());
			delete newStep;
			rc = SA_AIS_ERR_INIT;
			goto done;
		}
		newStep->setDn((*stepit));

		TRACE("Adding procedure step %s from IMM", newStep->getDn().c_str());
		addProcStep(newStep);

		/* The step children objects 
		   -SaSmfActivationUnit, 
		   -SaSmfDeactivationUnit 
		   -SaSmfImageNodes
		   does not need to be fetched. Once created the procedure will
		   not use them any more. These objects are created only to provide
		   information about which entities are affected by the step.
		*/
	}

 done:
	TRACE_LEAVE();
	return rc;
}

//------------------------------------------------------------------------------
// setEntitiesToAddRemMod()
//------------------------------------------------------------------------------
bool
SmfUpgradeProcedure::setEntitiesToAddRemMod(SmfUpgradeStep * i_step, SmfImmAttribute* io_smfEntityToAddRemove)
{
        TRACE_ENTER();

        //Modifications shall be visible in both attrsaSmfDuEntityToRemove and attrsaSmfDuEntityToAdd
        std::list < SmfImmOperation * > modList;
        modList = i_step->getModifications();

        SmfImmCreateOperation* createOper;
        SmfImmDeleteOperation* deleteOper;
        SmfImmModifyOperation* modifyOper;

	if (modList.size() == 0) {
		/* A value must always be supplied to a cached runtime attribute */
		io_smfEntityToAddRemove->addValue("");
		TRACE_LEAVE();
		return true;
	}

        std::list < SmfImmOperation * >::iterator it;
        for (it = modList.begin(); it != modList.end(); ++it) {
                if ((createOper = dynamic_cast<SmfImmCreateOperation*>(*it)) != 0) {
                        SmfImmUtils immUtil;
                        //Get class decription
                        SaImmAttrDefinitionT_2** attrDefinitionsOut = NULL;
                        if (!immUtil.getClassDescription(createOper->getClassName(), &attrDefinitionsOut)) {
				LOG_ER("getClassDescription FAILED for [%s]", createOper->getClassName().c_str());
				return false;
			}
			assert(attrDefinitionsOut != NULL);

                        //Look in SaImmAttrDefinitionT_2 for an attribute with the SA_IMM_ATTR_RDN flag set
                        std::string rdnAttr;
                        for (int i = 0; attrDefinitionsOut[i] != 0; i++) {
                                SaImmAttrFlagsT flags = attrDefinitionsOut[i]->attrFlags;
                                if (flags & SA_IMM_ATTR_RDN == SA_IMM_ATTR_RDN) {
                                        rdnAttr = (char*)attrDefinitionsOut[i]->attrName;
                                        break;
                                }
                        }
                        
                        immUtil.classDescriptionMemoryFree(attrDefinitionsOut);

                        //Find attribute name in SmfImmCreateOperation attributes
                        const std::list < SmfImmAttribute > values = createOper->getValues();
                        if ( values.size() != 0 ) {
                                std::list < SmfImmAttribute >::const_iterator it;
                                for (it = values.begin(); it != values.end(); ++it) {
                                        if (((SmfImmAttribute)(*it)).getName() == rdnAttr) {
                                                std::string rdn = ((SmfImmAttribute)(*it)).getValues().front(); //RDN always one value
                                                std::string str = rdn + "," + createOper->getParentDn();
                                                TRACE("TEST TEST create DN = %s", str.c_str());
                                                io_smfEntityToAddRemove->addValue(rdn + "," + createOper->getParentDn());
                                                break;
                                        }
                                }
                        } else {
                                LOG_ER("SmfUpgradeProcedure::setEntitiesToAddRemMod: no values set in create operation for instance of class %s",createOper->getClassName().c_str());
                                return false;
                        }
                } else if ((deleteOper = dynamic_cast<SmfImmDeleteOperation*>(*it)) != 0) {
                        io_smfEntityToAddRemove->addValue(deleteOper->getDn());
                } else if ((modifyOper = dynamic_cast<SmfImmModifyOperation*>(*it)) != 0) {
                        io_smfEntityToAddRemove->addValue(modifyOper->getDn());
                } else {
                        LOG_ER("SmfUpgradeProcedure::setEntitiesToAddRemMod: unknown operation");
                        return false;
                }
        }

        TRACE_LEAVE();

        return true;
}

//------------------------------------------------------------------------------
// execute()
//------------------------------------------------------------------------------
void SmfUpgradeProcedure::execute()
{
	TRACE_ENTER();
	m_state->execute(this);
	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// executeInit()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::executeInit()
{
	TRACE_ENTER();
	m_state->executeInit(this);
	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// executeStep()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::executeStep()
{
	TRACE_ENTER();
	m_state->executeStep(this);
	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// executeWrapup()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::executeWrapup()
{
	TRACE_ENTER();
	m_state->executeWrapup(this);
	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// suspend()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::suspend()
{
	TRACE_ENTER();
	m_state->suspend(this);
	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// rollback()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::rollback()
{
	TRACE_ENTER();
	m_state->rollback(this);
	TRACE_LEAVE();
}

//------------------------------------------------------------------------------
// commit()
//------------------------------------------------------------------------------
void 
SmfUpgradeProcedure::commit()
{
	TRACE_ENTER();
	m_state->commit(this);
	TRACE_LEAVE();
}
