/*      -*- OpenSAF  -*-
 *
 * (C) Copyright 2008 The OpenSAF Foundation
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. This file and program are licensed
 * under the GNU Lesser General Public License Version 2.1, February 1999.
 * The complete license can be accessed from the following location:
 * http://opensource.org/licenses/lgpl-license.php
 * See the Copying file included with the OpenSAF distribution for full
 * licensing terms.
 *
 * Author(s): Emerson Network Power
 *            Ericsson
 *
 */

/*****************************************************************************
..............................................................................

..............................................................................

  DESCRIPTION: 
  
  This module deals with the creation, accessing and deletion of the SI-SI
  dependency database.

..............................................................................

  FUNCTIONS INCLUDED in this module:

******************************************************************************
*/

/*
 * Module Inclusion Control...
 */

#include <saImmOm.h>
#include <immutil.h>
#include <logtrace.h>

#include <avd_dblog.h>
#include <avd_imm.h>
#include <avd_si_dep.h>
#include <avd_susi.h>
#include <avd_proc.h>

/* static function prototypes */
static uns32 avd_check_si_dep_sponsors(AVD_CL_CB *cb, AVD_SI *si, NCS_BOOL take_action);
static void avd_update_si_dep_state_for_spons_unassign(AVD_CL_CB *cb, AVD_SI *dep_si, AVD_SI_SI_DEP *si_dep_rec);
static uns32 avd_si_dep_si_unassigned(AVD_CL_CB *cb, AVD_SI *si);
static uns32 avd_check_si_state_enabled(AVD_CL_CB *cb, AVD_SI *si);
static uns32 avd_sg_red_si_process_assignment(AVD_CL_CB *cb, AVD_SI *si);
static uns32 avd_si_dep_state_evt(AVD_CL_CB *cb, AVD_SI *si, AVD_SI_SI_DEP_INDX *si_dep_idx);
static void avd_si_dep_spons_state_modif(AVD_CL_CB *cb, AVD_SI *si, AVD_SI *si_dep,
					 AVD_SI_DEP_SPONSOR_SI_STATE spons_state);
static uns32 avd_si_si_dep_cyclic_dep_find(AVD_CL_CB *cb, AVD_SI_SI_DEP_INDX *indx);
static void avd_si_dep_start_unassign(AVD_CL_CB *cb, AVD_EVT *evt);

/*****************************************************************************
 * Function: avd_check_si_state_enabled 
 *
 * Purpose:  This function check the active and the quiescing HA states for
 *           SI of all service units.
 *
 * Input:   cb - Pointer to AVD ctrl-block
 *          si - Pointer to AVD_SI struct
 *
 * Returns: NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE.
 *
 * NOTES:
 *
 **************************************************************************/
uns32 avd_check_si_state_enabled(AVD_CL_CB *cb, AVD_SI *si)
{
	AVD_SU_SI_REL *susi = NULL;
	uns32 rc = NCSCC_RC_FAILURE;

	susi = si->list_of_sisu;
	while (susi != AVD_SU_SI_REL_NULL) {
		if ((susi->state == SA_AMF_HA_ACTIVE) || (susi->state == SA_AMF_HA_QUIESCING)) {
			rc = NCSCC_RC_SUCCESS;
			break;
		}

		susi = susi->si_next;
	}

	if (rc != NCSCC_RC_SUCCESS)
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_SPONSOR_UNASSIGNED);

	return rc;
}

/*****************************************************************************
 * Function: avd_si_dep_spons_list_del
 *
 * Purpose:  This function deletes the spons-SI node from the spons-list of 
 *           dependent-SI.
 *
 * Input:  cb - ptr to AVD control block
 *         si_dep_rec - ptr to AVD_SI_SI_DEP struct 
 *
 * Returns:
 *
 * NOTES:
 * 
 **************************************************************************/
void avd_si_dep_spons_list_del(AVD_CL_CB *cb, AVD_SI_SI_DEP *si_dep_rec)
{
	AVD_SI *dep_si = NULL;
	AVD_SPONS_SI_NODE *spons_si_node = NULL;
	AVD_SPONS_SI_NODE *del_spons_si_node = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_si_dep_spons_list_del");

	/* Dependent SI row Status should be active, if not return error */
	if ((dep_si = avd_si_get(&si_dep_rec->indx_imm.si_name_sec))
	    == AVD_SI_NULL) {
		/* LOG the message */
		return;
	}

	/* SI doesn't depend on any other SIs */
	if (dep_si->spons_si_list == NULL) {
		/* LOG the message */
		return;
	}

	spons_si_node = dep_si->spons_si_list;

	/* Check if the sponsor is the first node of the list */
	if (m_CMP_HORDER_SANAMET(dep_si->spons_si_list->si->name, si_dep_rec->indx_imm.si_name_prim) == 0) {
		dep_si->spons_si_list = spons_si_node->next;
		free(spons_si_node);
	} else {
		while (spons_si_node->next != NULL) {
			if (m_CMP_HORDER_SANAMET(spons_si_node->next->si->name,
						 si_dep_rec->indx_imm.si_name_prim) != 0) {
				spons_si_node = spons_si_node->next;
				continue;
			}

			/* Matched spons si node found, then delete */
			del_spons_si_node = spons_si_node->next;
			spons_si_node->next = spons_si_node->next->next;
			free(del_spons_si_node);
			break;
		}
	}

	return;
}

/*****************************************************************************
 * Function: avd_si_dep_spons_list_add
 *
 * Purpose:  This function adds the spons-SI node in the spons-list of 
 *           dependent-SI.
 *
 * Input:  cb - ptr to AVD control block
 *         dep_si - ptr to AVD_SI struct (dependent-SI node).
 *         spons_si - ptr to AVD_SI struct (sponsor-SI node).
 *
 * Returns:
 *
 * NOTES: 
 * 
 **************************************************************************/
uns32 avd_si_dep_spons_list_add(AVD_CL_CB *avd_cb, AVD_SI *dep_si, AVD_SI *spons_si)
{
	AVD_SPONS_SI_NODE *spons_si_node;

	m_AVD_LOG_FUNC_ENTRY("avd_si_dep_spons_list_add");

	spons_si_node = calloc(1, sizeof(AVD_SPONS_SI_NODE));
	if (spons_si_node == NULL) {
		return NCSCC_RC_FAILURE;
	}

	spons_si_node->si = spons_si;

	spons_si_node->next = dep_si->spons_si_list;
	dep_si->spons_si_list = spons_si_node;

	return NCSCC_RC_SUCCESS;
}

/*****************************************************************************
 * Function: avd_si_dep_stop_tol_timer
 *
 * Purpose:  This function is been called when SI was unassigned, checks 
 *           whether tolerance timers are running for this SI (because
 *           of its sponsor-SI unassignment), if yes just stop them.  
 *
 * Input:  cb - ptr to AVD control block
 *         si - ptr to AVD_SI struct.
 *
 * Returns:
 *
 * NOTES: 
 * 
 **************************************************************************/
void avd_si_dep_stop_tol_timer(AVD_CL_CB *cb, AVD_SI *si)
{
	AVD_SI_SI_DEP_INDX indx;
	AVD_SI_SI_DEP *rec = NULL;
	AVD_SPONS_SI_NODE *spons_si_node = si->spons_si_list;

	m_AVD_LOG_FUNC_ENTRY("avd_si_dep_stop_tol_timer");

	memset((char *)&indx, '\0', sizeof(AVD_SI_SI_DEP_INDX));
	indx.si_name_sec.length = si->name.length;
	memcpy(indx.si_name_sec.value, si->name.value, si->name.length);

	while (spons_si_node) {
		/* Need to stop the tolerance timer */
		indx.si_name_prim.length = spons_si_node->si->name.length;
		memcpy(indx.si_name_prim.value, spons_si_node->si->name.value,
		       spons_si_node->si->name.length);

		if ((rec = avd_si_si_dep_find(cb, &indx, TRUE)) != NULL) {
			if (rec->si_dep_timer.is_active == TRUE) {
				avd_stop_tmr(cb, &rec->si_dep_timer);

				if (si->tol_timer_count > 0) {
					si->tol_timer_count--;
				}
			}
		}

		spons_si_node = spons_si_node->next;
	}

	return;
}

/*****************************************************************************
 * Function: avd_si_dep_si_unassigned
 *
 * Purpose:  This function removes the active and the quiescing HA states for
 *           SI from all service units and moves SI dependency state to
 *           SPONSOR_UNASSIGNED state.  
 *
 * Input: cb - The AVD control block
 *        si - Pointer to AVD_SI struct
 *
 * Returns: NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE.
 *
 * NOTES:
 *
 **************************************************************************/
uns32 avd_si_dep_si_unassigned(AVD_CL_CB *cb, AVD_SI *si)
{
	AVD_SU_SI_REL *susi = NULL;
	uns32 rc = NCSCC_RC_FAILURE;
	AVD_SU_SI_STATE old_fsm_state;

	if (si->sg_of_si->sg_fsm_state != AVD_SG_FSM_STABLE) {
		return rc;
	}

	susi = si->list_of_sisu;
	while (susi != AVD_SU_SI_REL_NULL) {
		if ((susi->state == SA_AMF_HA_ACTIVE) || (susi->state == SA_AMF_HA_QUIESCING)) {
			old_fsm_state = susi->fsm;
			susi->fsm = AVD_SU_SI_STATE_UNASGN;
			m_AVSV_SEND_CKPT_UPDT_ASYNC_UPDT(cb, susi, AVSV_CKPT_AVD_SI_ASS);

			rc = avd_snd_susi_msg(cb, susi->su, susi, AVSV_SUSI_ACT_DEL);
			if (NCSCC_RC_SUCCESS != rc) {
				/* LOG the erro */
				susi->fsm = old_fsm_state;
				m_AVSV_SEND_CKPT_UPDT_ASYNC_UPDT(cb, susi, AVSV_CKPT_AVD_SI_ASS);
				return rc;
			}

			/* add the su to su-oper list */
			avd_sg_su_oper_list_add(cb, susi->su, FALSE);
		}

		susi = susi->si_next;
	}

	m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_SPONSOR_UNASSIGNED);

	/* transition to sg-realign state */
	m_AVD_SET_SG_FSM(cb, si->sg_of_si, AVD_SG_FSM_SG_REALIGN);

	/* Check if this SI is a sponsor SI for some other SI, the take appropriate action */
	avd_si_dep_spons_state_modif(cb, si, NULL, AVD_SI_DEP_SPONSOR_UNASSIGNED);

	return rc;
}

/*****************************************************************************
 * Function: avd_sg_red_si_process_assignment 
 *
 * Purpose:  This function process SI assignment process on a redundancy model
 *           that was associated with the SI.
 *
 * Input: cb - The AVD control block
 *        si - Pointer to AVD_SI struct
 *
 * Returns: NCSCC_RC_SUCCESS/NCSCC_RC_INV_VAL
 *
 * NOTES:
 *
 **************************************************************************/
uns32 avd_sg_red_si_process_assignment(AVD_CL_CB *cb, AVD_SI *si)
{
	if (si->sg_of_si->sg_fsm_state != AVD_SG_FSM_STABLE) {
		return NCSCC_RC_FAILURE;
	}

	if ((si->max_num_csi == si->num_csi) &&
	    (si->saAmfSIAdminState == SA_AMF_ADMIN_UNLOCKED) && (cb->init_state == AVD_APP_STATE)) {
		switch (si->sg_of_si->sg_redundancy_model) {
		case SA_AMF_2N_REDUNDANCY_MODEL:
			if (avd_sg_2n_si_func(cb, si) != NCSCC_RC_SUCCESS) {
				return NCSCC_RC_FAILURE;
			}
			break;

		case SA_AMF_N_WAY_ACTIVE_REDUNDANCY_MODEL:
			if (avd_sg_nacvred_si_func(cb, si) != NCSCC_RC_SUCCESS) {
				return NCSCC_RC_FAILURE;
			}
			break;

		case SA_AMF_N_WAY_REDUNDANCY_MODEL:
			if (avd_sg_nway_si_func(cb, si) != NCSCC_RC_SUCCESS) {
				return NCSCC_RC_FAILURE;
			}
			break;

		case SA_AMF_NPM_REDUNDANCY_MODEL:
			if (avd_sg_npm_si_func(cb, si) != NCSCC_RC_SUCCESS) {
				return NCSCC_RC_FAILURE;
			}
			break;

		case SA_AMF_NO_REDUNDANCY_MODEL:
		default:
			if (avd_sg_nored_si_func(cb, si) != NCSCC_RC_SUCCESS) {
				return NCSCC_RC_FAILURE;
			}
			break;
		}

		if (avd_check_si_state_enabled(cb, si) == NCSCC_RC_SUCCESS) {
			m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_ASSIGNED);
			return NCSCC_RC_SUCCESS;
		}
	}

	return NCSCC_RC_FAILURE;
}

/*****************************************************************************
 * Function: avd_si_dep_delete
 *
 * Purpose:  This function deletes the SI-SI dependency records associated
 *           to a particular SI. 
 *
 * Input:  cb - ptr to AVD control block
 *         si - ptr to AVD_SI struct.
 *
 * Returns: 
 *
 * NOTES:
 * 
 **************************************************************************/
void avd_si_dep_delete(AVD_CL_CB *cb, AVD_SI *si)
{
	AVD_SI_SI_DEP *rec = NULL;
	AVD_SI_SI_DEP_INDX spons_indx;
	AVD_SI_SI_DEP_INDX dep_indx;

	m_AVD_LOG_FUNC_ENTRY("avd_si_dep_delete");

	/* Get the sponsor index */
	memset((char *)&spons_indx, '\0', sizeof(AVD_SI_SI_DEP_INDX));
	spons_indx.si_name_prim.length = si->name.length;
	memcpy(spons_indx.si_name_prim.value, si->name.value, si->name.length);

	memset((char *)&dep_indx, '\0', sizeof(AVD_SI_SI_DEP_INDX));
	dep_indx.si_name_sec.length = si->name.length;
	memcpy(dep_indx.si_name_sec.value, si->name.value, si->name.length);

	rec = avd_si_si_dep_find_next(cb, &spons_indx, TRUE);
	while (rec != NULL) {
		if (m_CMP_HORDER_SANAMET(rec->indx_imm.si_name_prim, spons_indx.si_name_prim) != 0) {
			break;
		}

		dep_indx.si_name_prim.length = spons_indx.si_name_sec.length;
		memcpy(dep_indx.si_name_prim.value, spons_indx.si_name_sec.value,
		       spons_indx.si_name_sec.length);

		if (ncs_patricia_tree_del(&cb->si_dep.spons_anchor, &rec->tree_node_imm)
		    != NCSCC_RC_SUCCESS) {
			/* LOG the message */
		}

		if ((rec = avd_si_si_dep_find(cb, &dep_indx, FALSE)) != NULL) {
			if (ncs_patricia_tree_del(&cb->si_dep.dep_anchor, &rec->tree_node)
			    != NCSCC_RC_SUCCESS) {
				/* log error */
				return;
			}
		}

		/* Since the SI Dep. is been removed, adjust the dependent SI dep state accordingly */
		si = avd_si_get(&rec->indx.si_name_prim);
		if (si == AVD_SI_NULL)
			return;

		if (rec->si_dep_timer.is_active == TRUE) {
			avd_stop_tmr(cb, &rec->si_dep_timer);

			if (si->tol_timer_count > 0)
				si->tol_timer_count--;
		}

		avd_screen_sponsor_si_state(cb, si, TRUE);

		if (rec)
			free(rec);

		rec = avd_si_si_dep_find_next(cb, &spons_indx, TRUE);
	}

	return;
}

/*****************************************************************************
 * Function: avd_si_dep_state_evt
 *
 * Purpose:  This function prepares the event to send AVD_EVT_SI_DEP_STATE
 *           event. This event is sent on expiry of tolerance timer or the 
 *           sponsor SI was unassigned and tolerance timer is "0".
 *
 * Input:  cb - ptr to AVD control block
 *         si - ptr to AVD_SI struct.
 *
 * Returns: 
 *
 * NOTES:
 * 
 **************************************************************************/
uns32 avd_si_dep_state_evt(AVD_CL_CB *cb, AVD_SI *si, AVD_SI_SI_DEP_INDX *si_dep_idx)
{
	AVD_EVT *evt = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_si_dep_state_evt");

	evt = calloc(1, sizeof(AVD_EVT));
	if (evt == AVD_EVT_NULL) {
		m_AVD_LOG_MEM_FAIL_LOC(AVD_EVT_ALLOC_FAILED);
		return NCSCC_RC_FAILURE;
	}

	/* Update evt struct, using tmr field even though this field is not
	 * relevant for this event, but it accommodates the required data.
	 */
	evt->cb_hdl = cb->cb_handle;
	evt->rcv_evt = AVD_EVT_SI_DEP_STATE;

	/* si_dep_idx is NULL for ASSIGN event non-NULL for UNASSIGN event */
	if (si_dep_idx != NULL) {
		evt->info.tmr.spons_si_name = si_dep_idx->si_name_prim;
		evt->info.tmr.dep_si_name = si_dep_idx->si_name_sec;
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_READY_TO_UNASSIGN);
	} else {		/* For ASSIGN evt, just enough to feed SI name */

		evt->info.tmr.spons_si_name.length = si->name.length;
		memcpy(evt->info.tmr.spons_si_name.value, si->name.value, si->name.length);
	}

	m_AVD_LOG_RCVD_VAL(((long)evt));
	m_AVD_LOG_EVT_INFO(AVD_RCVD_EVENT, evt->rcv_evt);

	if (m_NCS_IPC_SEND(&cb->avd_mbx, evt, NCS_IPC_PRIORITY_HIGH)
	    != NCSCC_RC_SUCCESS) {
		m_AVD_LOG_MBX_ERROR(AVSV_LOG_MBX_SEND);
		free(evt);
		return NCSCC_RC_FAILURE;
	}

	return NCSCC_RC_SUCCESS;
}

/*****************************************************************************
 * Function: avd_tmr_si_dep_tol_func
 *
 * Purpose:  On expiry of tolerance timer in SI-SI dependency context, this
 *           function initiates the process of SI unassignment due to its 
 *           sponsor moved to unassigned state.
 *
 * Input:  cb - ptr to AVD control block
 *         evt - ptr to AVD_EVT struct.
 *
 * Returns: 
 *
 * NOTES:
 * 
 **************************************************************************/
void avd_tmr_si_dep_tol_func(AVD_CL_CB *cb, AVD_EVT *evt)
{
	AVD_SI *si = NULL;
	AVD_SI *spons_si = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_tmr_si_dep_tol_func");

	if (evt->info.tmr.type != AVD_TMR_SI_DEP_TOL) {
		/* log error that a wrong timer type value */
		m_AVD_LOG_INVALID_VAL_FATAL(evt->info.tmr.type);
		return;
	}

	si = avd_si_get(&evt->info.tmr.dep_si_name);
	spons_si = avd_si_get(&evt->info.tmr.spons_si_name);

	if ((si == NULL) || (spons_si == NULL)) {
		/* Nothing to do here as SI/spons-SI itself lost their existence */
		return;
	}

	avd_log(NCSFL_SEV_NOTICE, "'%s'", si->name.value);

	/* Since the tol-timer is been expired, can decrement tol_timer_count of 
	 * the SI. 
	 */
	if (si->tol_timer_count > 0)
		si->tol_timer_count--;

	switch (si->si_dep_state) {
	case AVD_SI_NO_DEPENDENCY:
	case AVD_SI_SPONSOR_UNASSIGNED:
	case AVD_SI_ASSIGNED:
	case AVD_SI_UNASSIGNING_DUE_TO_DEP:
		/* LOG the ERROR message. Before moving to this state, need to ensure 
		 * all the tolerance timers of this SI are in stopped state.
		 */
		return;

	default:
		break;
	}

	/* Before starting unassignment process of SI, check once again whether 
	 * sponsor SIs are assigned back,if so move the SI state to ASSIGNED state 
	 */
	avd_screen_sponsor_si_state(cb, si, FALSE);
	if (si->si_dep_state == AVD_SI_ASSIGNED) {
		/* Nothing to do further */
		return;
	}

	/* If the SI is already been unassigned, nothing to proceed for 
	 * unassignment 
	 */
	if (si->list_of_sisu == AVD_SU_SI_REL_NULL) {
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_SPONSOR_UNASSIGNED);
		return;
	}

	/* Check if spons_si SI-Dep state is not in ASSIGNED state, then 
	 * only initiate the unassignment process. Because the SI can be 
	 * in this state due to some other spons-SI.
	 */
	if (avd_check_si_state_enabled(cb, spons_si) != NCSCC_RC_SUCCESS) {
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_UNASSIGNING_DUE_TO_DEP);

		if (avd_si_dep_si_unassigned(cb, si) != NCSCC_RC_SUCCESS) {
			/* Log the error */
		}
	}

	return;
}

/*****************************************************************************
 * Function: avd_check_si_dep_sponsors
 *
 * Purpose:  This function checks whether sponsor SIs of an SI are in enabled /
 *           disabled state.  
 *
 * Input:    cb - Pointer to AVD control block
 *           si - Pointer to AVD_SI struct 
 *           take_action - If TRUE, process the impacts (SI-SI dependencies)
 *                      on dependent-SIs (of sponsor-SI being enabled/disabled)
 *
 * Returns: NCSCC_RC_SUCCESS - if sponsor-SI is in enabled state 
 *          NCSCC_RC_FAILURE - if sponsor-SI is in disabled state 
 *
 * NOTES:  
 * 
 **************************************************************************/
uns32 avd_check_si_dep_sponsors(AVD_CL_CB *cb, AVD_SI *si, NCS_BOOL take_action)
{
	uns32 rc = NCSCC_RC_SUCCESS;
	AVD_SPONS_SI_NODE *spons_si_node = si->spons_si_list;

	while (spons_si_node) {
		if (avd_check_si_state_enabled(cb, spons_si_node->si) != NCSCC_RC_SUCCESS) {
			rc = NCSCC_RC_FAILURE;
			if (take_action) {
				avd_si_dep_spons_state_modif(cb, spons_si_node->si, si, AVD_SI_DEP_SPONSOR_UNASSIGNED);
			} else {
				/* Sponsors are not in enabled state */
				break;
			}
		}

		spons_si_node = spons_si_node->next;
	}

	/* All of the sponsors are in enabled state */
	if ((rc == NCSCC_RC_SUCCESS) && (take_action)) {
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_ASSIGNED);

		avd_si_dep_spons_state_modif(cb, si, NULL, AVD_SI_DEP_SPONSOR_ASSIGNED);
	}

	return rc;
}

/*****************************************************************************
 * Function: avd_sg_screen_si_si_dependencies
 *
 * Purpose:  This function screens SI-SI dependencies of the SG SIs. 
 *
 * Input: cb - The AVD control block
 *        sg - Pointer to AVD_SG struct 
 *
 * Returns: 
 *
 * NOTES:  
 * 
 **************************************************************************/
void avd_sg_screen_si_si_dependencies(AVD_CL_CB *cb, AVD_SG *sg)
{
	AVD_SI *si = NULL;

	si = sg->list_of_si;
	while (si != AVD_SI_NULL) {
		if (avd_check_si_state_enabled(cb, si) == NCSCC_RC_SUCCESS) {
			/* SI was in enabled state, so check for the SI-SI dependencies
			 * conditions and update the SI accordingly.
			 */
			avd_check_si_dep_sponsors(cb, si, TRUE);
		} else {
			avd_screen_sponsor_si_state(cb, si, TRUE);
			if ((si->si_dep_state == AVD_SI_SPONSOR_UNASSIGNED) ||
			    (si->si_dep_state == AVD_SI_NO_DEPENDENCY)) {
				/* Check if this SI is a sponsor SI for some other SI, the take appropriate action */
				avd_si_dep_spons_state_modif(cb, si, NULL, AVD_SI_DEP_SPONSOR_UNASSIGNED);
			}
		}

		si = si->sg_list_of_si_next;
	}

	return;
}

/*****************************************************************************
 * Function: avd_screen_sponsor_si_state 
 *
 * Purpose:  This function checks whether the sponsor SIs are in ASSIGNED 
 *           state. If they are in assigned state, dependent SI changes its
 *           si_dep state accordingly.
 *
 * Input: cb - The AVD control block
 *        si - Pointer to AVD_SI struct 
 *
 * Returns: 
 *
 * NOTES:  
 * 
 **************************************************************************/
void avd_screen_sponsor_si_state(AVD_CL_CB *cb, AVD_SI *si, NCS_BOOL start_assignment)
{
	m_AVD_LOG_FUNC_ENTRY("avd_screen_sponsor_si_state");

	/* Change the SI dependency state only when all of its sponsor SIs are 
	 * in assigned state.
	 */
	if (avd_check_si_dep_sponsors(cb, si, FALSE) != NCSCC_RC_SUCCESS) {
		/* Nothing to do here, just return */
		return;
	}

	switch (si->si_dep_state) {
	case AVD_SI_TOL_TIMER_RUNNING:
	case AVD_SI_READY_TO_UNASSIGN:
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_ASSIGNED);
		break;

	case AVD_SI_SPONSOR_UNASSIGNED:
		/* If SI-SI dependency cfg's are removed for this SI the update the 
		 * SI dep state with AVD_SI_NO_DEPENDENCY 
		 */
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_NO_DEPENDENCY);

	case AVD_SI_NO_DEPENDENCY:
		/* Initiate the process of ASSIGNMENT state, as all of its sponsor SIs
		 * are in ASSIGNED state, should be done only when SI is been unassigned
		 * due to SI-SI dependency (sponsor unassigned).
		 */
		if (start_assignment) {
			if (avd_sg_red_si_process_assignment(cb, si) == NCSCC_RC_SUCCESS) {
				/* Check if this SI is a sponsor SI for some other SI, the take appropriate action */
				avd_si_dep_spons_state_modif(cb, si, NULL, AVD_SI_DEP_SPONSOR_ASSIGNED);
			} else {
				/* LOG the error message */
			}
		}
		break;

	default:
		return;
	}

	return;
}

/*****************************************************************************
 * Function: avd_process_si_dep_state_evt
 *
 * Purpose:  This function starts SI unassignment process due to sponsor was  
 *           unassigned (SI-SI dependency logics).
 *
 * Input: cb - The AVD control block
 *        evt - Pointer to AVD_EVT struct 
 *
 * Returns: 
 *
 * NOTES:
 * 
 **************************************************************************/
void avd_process_si_dep_state_evt(AVD_CL_CB *cb, AVD_EVT *evt)
{
	AVD_SI *si = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_process_si_dep_state_evt");

	if (evt == NULL) {
		/* Log the message */
		return;
	}

	/* Check whether rcv_evt is AVD_EVT_SI_DEP_STATE event, if not LOG the 
	 * error message.
	 */
	if (evt->rcv_evt != AVD_EVT_SI_DEP_STATE) {
		/* Log the message */
		return;
	}

	if (evt->info.tmr.dep_si_name.length == 0) {
		/* Its an ASSIGN evt */
		si = avd_si_get(&evt->info.tmr.spons_si_name);
		if (si != NULL) {
			avd_screen_sponsor_si_state(cb, si, TRUE);
		}
	} else {		/* Process UNASSIGN evt */

		avd_si_dep_start_unassign(cb, evt);
	}

	return;
}

/*****************************************************************************
 * Function: avd_si_dep_start_unassign
 *
 * Purpose:  This function starts SI unassignment process due to sponsor was  
 *           unassigned (SI-SI dependency logics).
 *
 * Input: cb - The AVD control block
 *        evt - Pointer to AVD_EVT struct 
 *
 * Returns: 
 *
 * NOTES:
 * 
 **************************************************************************/
void avd_si_dep_start_unassign(AVD_CL_CB *cb, AVD_EVT *evt)
{
	AVD_SI *si = NULL;
	AVD_SI *spons_si = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_si_dep_start_unassign");

	si = avd_si_get(&evt->info.tmr.dep_si_name);
	spons_si = avd_si_get(&evt->info.tmr.spons_si_name);

	if ((!si) || (!spons_si)) {
		/* Log the ERROR message */
		return;
	}

	if (si->si_dep_state != AVD_SI_READY_TO_UNASSIGN) {
		/* Log the message */
		return;
	}

	/* Before starting unassignment process of SI, check again whether all its
	 * sponsors are moved back to assigned state, if so it is not required to
	 * initiate the unassignment process for the (dependent) SI.
	 */
	avd_screen_sponsor_si_state(cb, si, FALSE);
	if (si->si_dep_state == AVD_SI_ASSIGNED)
		return;

	/* If the SI is already been unassigned, nothing to proceed for 
	 * unassignment 
	 */
	if (si->list_of_sisu == AVD_SU_SI_REL_NULL) {
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_SPONSOR_UNASSIGNED);
		return;
	}

	/* Check if spons_si SI-Dep state is not in ASSIGNED state, then 
	 * only initiate the unassignment process. Because the SI can be 
	 * in this state due to some other spons-SI.
	 */
	if (avd_check_si_state_enabled(cb, spons_si) != NCSCC_RC_SUCCESS) {
		m_AVD_SET_SI_DEP_STATE(cb, si, AVD_SI_UNASSIGNING_DUE_TO_DEP);

		if (avd_si_dep_si_unassigned(cb, si) != NCSCC_RC_SUCCESS) {
			/* Log the error */
		}
	}

	return;
}

/*****************************************************************************
 * Function: avd_update_si_dep_state_for_spons_unassign 
 *
 * Purpose:  Upon sponsor SI is unassigned, this function updates the SI  
 *           dependent states either to AVD_SI_READY_TO_UNASSIGN / 
 *           AVD_SI_TOL_TIMER_RUNNING states.
 *
 * Input:  cb - ptr to AVD control block
 *         dep_si - ptr to AVD_SI struct (dependent SI).
 *         si_dep_rec - ptr to AVD_SI_SI_DEP struct 
 *
 * Returns: 
 *
 * NOTES:
 * 
 **************************************************************************/
void avd_update_si_dep_state_for_spons_unassign(AVD_CL_CB *cb, AVD_SI *dep_si, AVD_SI_SI_DEP *si_dep_rec)
{
	m_AVD_LOG_FUNC_ENTRY("avd_update_si_dep_state_for_spons_unassign");

	switch (dep_si->si_dep_state) {
	case AVD_SI_ASSIGNED:
		if (si_dep_rec->saAmfToleranceTime > 0) {
			/* Start the tolerance timer */
			m_AVD_SET_SI_DEP_STATE(cb, dep_si, AVD_SI_TOL_TIMER_RUNNING);

			/* Start the tolerance timer */
			m_AVD_SI_DEP_TOL_TMR_START(cb, si_dep_rec);

			if (dep_si->tol_timer_count) {
				/* It suppose to be "0", LOG the err msg and continue. */
			}

			dep_si->tol_timer_count = 1;
		} else {
			/* Send an event to start SI unassignment process */
			avd_si_dep_state_evt(cb, dep_si, &si_dep_rec->indx_imm);
		}
		break;

	case AVD_SI_TOL_TIMER_RUNNING:
	case AVD_SI_READY_TO_UNASSIGN:
		if (si_dep_rec->saAmfToleranceTime > 0) {
			if (si_dep_rec->si_dep_timer.is_active != TRUE) {
				/* Start the tolerance timer */
				m_AVD_SI_DEP_TOL_TMR_START(cb, si_dep_rec);

				/* SI is already in AVD_SI_TOL_TIMER_RUNNING state, so just 
				 * increment tol_timer_count indicates that >1 sponsors are 
				 * in unassigned state for this SI.
				 */
				dep_si->tol_timer_count++;
			} else {
				/* Should not happen, LOG the error message & go ahead */
			}
		} else {
			if (!si_dep_rec->unassign_event) {
				si_dep_rec->unassign_event = TRUE;

				/* Send an event to start SI unassignment process */
				avd_si_dep_state_evt(cb, dep_si, &si_dep_rec->indx_imm);
			}
		}
		break;

	case AVD_SI_NO_DEPENDENCY:
		m_AVD_SET_SI_DEP_STATE(cb, dep_si, AVD_SI_SPONSOR_UNASSIGNED);
		break;

	default:
		break;
	}

	return;
}

/*****************************************************************************
 * Function: avd_si_dep_spons_state_modif 
 *
 * Purpose:  Upon SI is assigned/unassigned and if this SI turns out to be 
 *           sponsor SI for some of the SIs (dependents),then update the states
 *           of dependent SIs accordingly (either to AVD_SI_READY_TO_UNASSIGN / 
 *           AVD_SI_TOL_TIMER_RUNNING states).
 *
 * Input:  cb - ptr to AVD control block
 *         si - ptr to AVD_SI struct (sponsor SI).
 *         si_dep - ptr to AVD_SI struct (dependent SI), NULL for all
 *                  dependent SIs of the above sponsor SI.
 *
 * Returns: 
 *
 * NOTES:
 * 
 **************************************************************************/
void avd_si_dep_spons_state_modif(AVD_CL_CB *cb, AVD_SI *si, AVD_SI *si_dep, AVD_SI_DEP_SPONSOR_SI_STATE spons_state)
{
	AVD_SI *dep_si = NULL;
	AVD_SI_SI_DEP_INDX si_indx;
	AVD_SI_SI_DEP *si_dep_rec = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_si_dep_spons_state_modif");

	memset((char *)&si_indx, '\0', sizeof(AVD_SI_SI_DEP_INDX));
	si_indx.si_name_prim.length = si->name.length;
	memcpy(si_indx.si_name_prim.value, si->name.value, si_indx.si_name_prim.length);

	/* If si_dep is NULL, means adjust the SI dep states for all depended 
	 * SIs of the sponsor SI.
	 */
	if (si_dep == NULL) {
		si_dep_rec = avd_si_si_dep_find_next(cb, &si_indx, TRUE);
		while (si_dep_rec != NULL) {
			if (m_CMP_HORDER_SANAMET(si_dep_rec->indx_imm.si_name_prim, si_indx.si_name_prim) != 0) {
				/* Seems no more node exists in spons_anchor tree with 
				 * "si_indx.si_name_prim" as primary key 
				 */
				break;
			}

			dep_si = avd_si_get(&si_dep_rec->indx_imm.si_name_sec);
			if (dep_si == NULL) {
				/* No corresponding SI node?? some thing wrong */
				si_dep_rec = avd_si_si_dep_find_next(cb, &si_dep_rec->indx_imm, TRUE);

				/* LOG the failure */
				continue;
			}

			if (spons_state == AVD_SI_DEP_SPONSOR_UNASSIGNED) {
				avd_update_si_dep_state_for_spons_unassign(cb, dep_si, si_dep_rec);
			} else {
				avd_si_dep_state_evt(cb, dep_si, NULL);
			}

			si_dep_rec = avd_si_si_dep_find_next(cb, &si_dep_rec->indx_imm, TRUE);
		}
	} else {
		/* Just ignore and return if spons_state is AVD_SI_DEP_SPONSOR_ASSIGNED */
		if (spons_state == AVD_SI_DEP_SPONSOR_ASSIGNED)
			return;

		/* Frame the index completely to the associated si_dep_rec */
		si_indx.si_name_sec.length = si_dep->name.length;
		memcpy(si_indx.si_name_sec.value, si_dep->name.value, si_dep->name.length);

		si_dep_rec = avd_si_si_dep_find(cb, &si_indx, TRUE);

		if (si_dep_rec != NULL) {
			/* se_dep_rec primary key should match with sponsor SI name */
			if (m_CMP_HORDER_SANAMET(si_dep_rec->indx_imm.si_name_prim, si_indx.si_name_prim) != 0)
				return;

			avd_update_si_dep_state_for_spons_unassign(cb, si_dep, si_dep_rec);
		}
	}

	return;
}

/*****************************************************************************
 * Function: avd_si_si_dep_struc_crt
 *
 * Purpose: This function will create and add a AVD_SI_SI_DEP structure to the
 *          trees if an element with the same key value doesn't exist in the
 *          tree.
 *
 * Input: cb - the AVD control block
 *        indx - Index for the row to be added. 
 *
 * Returns: The pointer to AVD_SI_SI_DEP structure allocated and added.
 *
 * NOTES:
 *
 **************************************************************************/
AVD_SI_SI_DEP *avd_si_si_dep_struc_crt(AVD_CL_CB *cb, AVD_SI_SI_DEP_INDX *indx)
{
	AVD_SI_SI_DEP *rec = NULL;
	uns32 si_prim_len = indx->si_name_prim.length;
	uns32 si_sec_len = indx->si_name_sec.length;

	m_AVD_LOG_FUNC_ENTRY("avd_si_si_dep_struc_crt");

	/* Allocate a new block structure for imm rec now */
	if ((rec = calloc(1, sizeof(AVD_SI_SI_DEP))) == NULL) {
		/* log an error 
		   m_AVD_LOG_MEM_FAIL(AVD_SI_SI_DEP_ALLOC_FAILED);
		 */
		return NULL;
	}

	rec->indx_imm.si_name_prim.length = indx->si_name_prim.length;
	memcpy(rec->indx_imm.si_name_prim.value, indx->si_name_prim.value, si_prim_len);

	rec->indx_imm.si_name_sec.length = indx->si_name_sec.length;
	memcpy(rec->indx_imm.si_name_sec.value, indx->si_name_sec.value, si_sec_len);

	rec->indx.si_name_prim.length = indx->si_name_sec.length;
	memcpy(rec->indx.si_name_prim.value, indx->si_name_sec.value, si_sec_len);

	rec->indx.si_name_sec.length = indx->si_name_prim.length;
	memcpy(rec->indx.si_name_sec.value, indx->si_name_prim.value, si_prim_len);

	rec->tree_node_imm.key_info = (uns8 *)&(rec->indx_imm);
	rec->tree_node_imm.bit = 0;
	rec->tree_node_imm.left = NCS_PATRICIA_NODE_NULL;
	rec->tree_node_imm.right = NCS_PATRICIA_NODE_NULL;

	rec->tree_node.key_info = (uns8 *)&(rec->indx);
	rec->tree_node.bit = 0;
	rec->tree_node.left = NCS_PATRICIA_NODE_NULL;
	rec->tree_node.right = NCS_PATRICIA_NODE_NULL;

	if (ncs_patricia_tree_add(&cb->si_dep.spons_anchor, &rec->tree_node_imm)
	    != NCSCC_RC_SUCCESS) {
		/* log an error */
		free(rec);
		return NULL;
	}

	if (ncs_patricia_tree_add(&cb->si_dep.dep_anchor, &rec->tree_node)
	    != NCSCC_RC_SUCCESS) {
		/* log an error */
		ncs_patricia_tree_del(&cb->si_dep.spons_anchor, &rec->tree_node_imm);
		free(rec);
		return NULL;
	}

	return rec;
}

/*****************************************************************************
 * Function: avd_si_si_dep_find 
 *
 * Purpose:  This function will find a AVD_SI_SI_DEP structure in the tree 
 *           with indx value as key. Indices can be provided as per the order
 *           mention in the imm or in the reverse of that. 
 *
 * Input: cb - The AVD control block
 *        indx - The key.
 *
 * Returns: The pointer to AVD_SG_SI_RANK structure found in the tree.
 *
 * NOTES: Set the isImmIdx flag value to 1 if indices are as defined by imm and 0
 *        if it is in reverse order
 * 
 **************************************************************************/
AVD_SI_SI_DEP *avd_si_si_dep_find(AVD_CL_CB *cb, AVD_SI_SI_DEP_INDX *indx, NCS_BOOL isImmIdx)
{
	AVD_SI_SI_DEP *rec = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_si_si_dep_find");

	if (isImmIdx) {
		rec = (AVD_SI_SI_DEP *)ncs_patricia_tree_get(&cb->si_dep.spons_anchor, (uns8 *)indx);
	} else {
		rec = (AVD_SI_SI_DEP *)ncs_patricia_tree_get(&cb->si_dep.dep_anchor, (uns8 *)indx);
		if (rec != NULL) {
			/* Adjust the pointer */
			rec = (AVD_SI_SI_DEP *)(((char *)rec)
						- (((char *)&(AVD_SI_SI_DEP_NULL->tree_node))
						   - ((char *)AVD_SI_SI_DEP_NULL)));
		}
	}

	return rec;
}

/*****************************************************************************
 * Function: avd_si_si_dep_find_next 
 *
 * Purpose:  This function will find next AVD_SI_SI_DEP structure in the tree
 *           with indx value as key. Indices can be provided as per the order
 *           mention in the imm or in the reverse of that. 
 *
 * Input: cb - the AVD control block
 *        indx - The key.
 *
 * Returns: The next pointer to AVD_SG_SI_RANK structure found in the tree.
 *
 * NOTES: Set the isImmIdx flag value to 1 if indices are as defined by imm and 0
 *        if it is in reverse order
 * 
 **************************************************************************/
AVD_SI_SI_DEP *avd_si_si_dep_find_next(AVD_CL_CB *cb, AVD_SI_SI_DEP_INDX *indx, NCS_BOOL isImmIdx)
{
	AVD_SI_SI_DEP *rec = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_si_si_dep_find_next");

	if (isImmIdx) {
		rec = (AVD_SI_SI_DEP *)ncs_patricia_tree_getnext(&cb->si_dep.spons_anchor, (uns8 *)indx);
	} else {
		rec = (AVD_SI_SI_DEP *)ncs_patricia_tree_getnext(&cb->si_dep.dep_anchor, (uns8 *)indx);
		if (rec != NULL) {
			/* Adjust the pointer */
			rec = (AVD_SI_SI_DEP *)(((char *)rec)
						- (((char *)&(AVD_SI_SI_DEP_NULL->tree_node))
						   - ((char *)AVD_SI_SI_DEP_NULL)));
		}
	}

	return rec;
}

/*****************************************************************************
 * Function: avd_si_si_dep_del_row 
 *
 * Purpose:  This function will delete and free AVD_SI_SI_DEP structure from 
 *           the tree. It will delete the record from both patricia trees
 *
 * Input: cb - The AVD control block
 *        si - Pointer to service instance row 
 *
 * Returns: NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE  
 *
 * NOTES:
 * 
 **************************************************************************/
uns32 avd_si_si_dep_del_row(AVD_CL_CB *cb, AVD_SI_SI_DEP *rec)
{
	AVD_SI_SI_DEP *si_dep_rec = NULL;

	m_AVD_LOG_FUNC_ENTRY("avd_si_si_dep_del_row");

	if (rec == NULL)
		return NCSCC_RC_FAILURE;

	if ((si_dep_rec = avd_si_si_dep_find(cb, &rec->indx, FALSE)) != NULL) {
		if (ncs_patricia_tree_del(&cb->si_dep.dep_anchor, &si_dep_rec->tree_node)
		    != NCSCC_RC_SUCCESS) {
			/* log error */
			return NCSCC_RC_FAILURE;
		}
	}

	si_dep_rec = NULL;

	if ((si_dep_rec = avd_si_si_dep_find(cb, &rec->indx_imm, TRUE)) != NULL) {
		if (ncs_patricia_tree_del(&cb->si_dep.spons_anchor, &si_dep_rec->tree_node_imm)
		    != NCSCC_RC_SUCCESS) {
			/* log error */
			return NCSCC_RC_FAILURE;
		}
	}

	if (si_dep_rec)
		free(si_dep_rec);

	return NCSCC_RC_SUCCESS;
}

/*****************************************************************************
 * Function: avd_si_si_dep_cyclic_dep_find 
 *
 * Purpose:  This function will use to evalute that is new record will because
 *           of cyclic dependency exist in SIs or not.
 *
 * Input: cb   - The AVD control block
 *        indx - Index of the new rec needs to be evaluated.
 *
 * Returns: NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 *
 * NOTES: Return value NCSCC_RC_SUCCESS means it found a cyclic dependency or
 *        buffer is not sufficient to process this request.
 *
 **************************************************************************/
uns32 avd_si_si_dep_cyclic_dep_find(AVD_CL_CB *cb, AVD_SI_SI_DEP_INDX *indx)
{
	AVD_SI_SI_DEP *rec = NULL;
	AVD_SI_DEP_NAME_LIST *start = NULL;
	AVD_SI_DEP_NAME_LIST *temp = NULL;
	AVD_SI_DEP_NAME_LIST *last = NULL;
	uns32 rc = NCSCC_RC_FAILURE;
	AVD_SI_SI_DEP_INDX idx;

	m_AVD_LOG_FUNC_ENTRY("avd_si_si_dep_cyclic_dep_find");

	if (m_CMP_HORDER_SANAMET(indx->si_name_prim, indx->si_name_sec) == 0) {
		/* dependent SI and Sponsor SI can not be same 
		   Cyclic dependency found return sucess
		 */
		return NCSCC_RC_SUCCESS;
	}

	if ((start = malloc(sizeof(AVD_SI_DEP_NAME_LIST))) == NULL) {
		/*Insufficient memory, record can not be added */
		return NCSCC_RC_SUCCESS;
	} else {
		start->si_name = indx->si_name_prim;
		start->next = NULL;
		last = start;
	}

	while (last) {
		memset((char *)&idx, '\0', sizeof(AVD_SI_SI_DEP_INDX));

		idx.si_name_prim.length = last->si_name.length;
		memcpy(idx.si_name_prim.value, last->si_name.value, last->si_name.length);

		rec = avd_si_si_dep_find_next(cb, &idx, FALSE);

		while ((rec != NULL) && (m_CMP_HORDER_SANAMET(rec->indx.si_name_prim, idx.si_name_prim) == 0)) {
			if (m_CMP_HORDER_SANAMET(indx->si_name_sec, rec->indx.si_name_sec) == 0) {
				/* Cyclic dependency found */
				rc = NCSCC_RC_SUCCESS;
				break;
			}

			/* Search if this SI name already exist in the list */
			temp = start;
			if (m_CMP_HORDER_SANAMET(temp->si_name, rec->indx.si_name_sec) != 0) {
				while ((temp->next != NULL) &&
				       (m_CMP_HORDER_SANAMET(temp->next->si_name, rec->indx.si_name_sec) != 0)) {
					temp = temp->next;
				}

				/* SI Name not found in the list, add it */
				if (temp->next == NULL) {
					if ((temp->next = malloc(sizeof(AVD_SI_DEP_NAME_LIST))) == NULL) {
						/* Insufficient memory space */
						rc = NCSCC_RC_SUCCESS;
						break;
					}

					temp->next->si_name = rec->indx.si_name_sec;
					temp->next->next = NULL;
				}
			}

			rec = avd_si_si_dep_find_next(cb, &rec->indx, FALSE);
		}

		if (rc == NCSCC_RC_SUCCESS) {
			break;
		} else {
			last = last->next;
		}
	}

	/* Free the allocated SI name list */
	while (start) {
		temp = start->next;
		free(start);
		start = temp;
	}

	return rc;
}

static void avd_sidep_indx_init(const SaNameT *sidep_name, AVD_SI_SI_DEP_INDX *indx)
{
	char *p;
	SaNameT tmp = *sidep_name;
	int i;

	/* The second existence of 'safSi=' should be dependent, secondary SI */

	/* find first occurence and step past it */
	p = strstr((char *)tmp.value, "safSi=") + 1;
	assert(p);

	/* find second occurence, an error if not found */
	p = strstr(p, "safSi=");
	assert(p);

	*(p - 1) = '\0';	/* null terminate at comma before SI */

	indx->si_name_sec.length = snprintf((char *)indx->si_name_sec.value, SA_MAX_NAME_LENGTH, "%s", p);

	/* Skip past the RDN tag */
	p = strchr((char *)tmp.value, '=') + 1;
	assert(p);

	/*
	 ** Example DN, need to copy to get rid of back slash escaped commas.
	 ** 'safDepend=safSi=SC2-NoRed\,safApp=OpenSAF,safSi=SC-2N,safApp=OpenSAF'
	 */

	/* Copy the RDN value which is a DN with escaped commas */
	i = 0;
	while (*p) {
		if (*p != '\\')
			indx->si_name_prim.value[i++] = *p;

		p++;
	}

	indx->si_name_prim.length = strlen((char *)indx->si_name_prim.value);
}

static AVD_SI_SI_DEP *sidep_new(SaNameT *sidep_name, const SaImmAttrValuesT_2 **attributes)
{
	int rc = -1;
	AVD_SI_SI_DEP *sidep;
	AVD_SI_SI_DEP_INDX indx;

	avd_sidep_indx_init(sidep_name, &indx);

	/* Sponsor SI need to exist */
	if (avd_si_get(&indx.si_name_prim) == NULL) {
		avd_log(NCSFL_SEV_ERROR, "avd_si_struc_find failed for '%s'", indx.si_name_prim.value);
		goto done;
	}

	if (avd_si_si_dep_cyclic_dep_find(avd_cb, &indx) == NCSCC_RC_SUCCESS) {
		/* Return value that record cannot be added due to cyclic dependency */
		avd_log(NCSFL_SEV_ERROR, "cyclic dependency for '%s'", sidep_name->value);
		goto done;
	}

	if ((sidep = avd_si_si_dep_struc_crt(avd_cb, &indx)) == NULL)
		goto done;

	if (immutil_getAttr("saAmfToleranceTime", attributes, 0, &sidep->saAmfToleranceTime) != SA_AIS_OK) {
		/* Empty, assign default value */
		sidep->saAmfToleranceTime = 0;
	}

	rc = 0;

 done:
	if (rc != 0) {
		avd_si_si_dep_struc_crt(avd_cb, &indx);
		sidep = NULL;
	}
	return sidep;
}

/**
 * Get configuration for all AMF SI dependency objects from IMM
 * and create AVD internal objects.
 * 
 * @return int
 */
SaAisErrorT avd_sidep_config_get(void)
{
	SaAisErrorT error;
	SaImmSearchHandleT searchHandle;
	SaImmSearchParametersT_2 searchParam;
	SaNameT sidep_name;
	const SaImmAttrValuesT_2 **attributes;
	const char *className = "SaAmfSIDependency";

	searchParam.searchOneAttr.attrName = "SaImmAttrClassName";
	searchParam.searchOneAttr.attrValueType = SA_IMM_ATTR_SASTRINGT;
	searchParam.searchOneAttr.attrValue = &className;

	error = immutil_saImmOmSearchInitialize_2(avd_cb->immOmHandle, NULL, SA_IMM_SUBTREE,
		SA_IMM_SEARCH_ONE_ATTR | SA_IMM_SEARCH_GET_ALL_ATTR, &searchParam,
		NULL, &searchHandle);

	if (SA_AIS_OK != error) {
		avd_log(NCSFL_SEV_ERROR, "saImmOmSearchInitialize failed: %u", error);
		goto done1;
	}

	while (immutil_saImmOmSearchNext_2(searchHandle, &sidep_name, (SaImmAttrValuesT_2 ***)&attributes) == SA_AIS_OK) {
		avd_log(NCSFL_SEV_NOTICE, "'%s'", sidep_name.value);

		if (sidep_new(&sidep_name, attributes) == NULL) {
			error = SA_AIS_ERR_FAILED_OPERATION;
			goto done2;
		}
	}

	error = SA_AIS_OK;

 done2:
	(void)immutil_saImmOmSearchFinalize(searchHandle);
 done1:

	return error;
}

static SaAisErrorT sidep_ccb_completed_cb(CcbUtilOperationData_t *opdata)
{
	SaAisErrorT rc = SA_AIS_ERR_BAD_OPERATION;
	AVD_SI_SI_DEP *sidep;

	TRACE_ENTER2("CCB ID %llu, '%s'", opdata->ccbId, opdata->objectName.value);

	switch (opdata->operationType) {
	case CCBUTIL_CREATE:
		if ((sidep = sidep_new(&opdata->objectName, opdata->param.create.attrValues)) == NULL)
			goto done;

		rc = SA_AIS_OK;
		break;
	case CCBUTIL_DELETE:
		rc = SA_AIS_OK;
		break;
	case CCBUTIL_MODIFY:
		rc = SA_AIS_OK;
		break;
	default:
		assert(0);
		break;
	}

done:
	return rc;
}

static void sidep_ccb_apply_cb(CcbUtilOperationData_t *opdata)
{
	AVD_SI_SI_DEP *sidep;
	AVD_SI_SI_DEP_INDX indx;
	const SaImmAttrModificationT_2 *attr_mod;
	int i = 0;

	TRACE_ENTER2("CCB ID %llu, '%s'", opdata->ccbId, opdata->objectName.value);

	switch (opdata->operationType) {
	case CCBUTIL_CREATE:
		break;
	case CCBUTIL_DELETE:
		avd_sidep_indx_init(&opdata->objectName, &indx);
		sidep = avd_si_si_dep_find(avd_cb, &indx, FALSE);
		avd_si_si_dep_del_row(avd_cb, sidep);
		break;
	case CCBUTIL_MODIFY:
		avd_sidep_indx_init(&opdata->objectName, &indx);
		sidep = avd_si_si_dep_find(avd_cb, &indx, FALSE);
		while ((attr_mod = opdata->param.modify.attrMods[i++]) != NULL) {
			if (!strcmp(attr_mod->modAttr.attrName, "saAmfToleranceTime")) {
				sidep->saAmfToleranceTime = *((SaTimeT *)attr_mod->modAttr.attrValues[0]);
			}
		}
		break;
	default:
		assert(0);
		break;
	}
}

void avd_sidep_constructor(void)
{
	avd_class_impl_set("SaAmfSIDependency", NULL, NULL, sidep_ccb_completed_cb, sidep_ccb_apply_cb);
}
