// $Id: HD.cc 9276 2009-02-28 06:58:00Z mfeingol $

#include "HD.hh"
#include "File.hh"
#include "FileContext.hh"
#include "FileException.hh"
#include "XMLElement.hh"
#include "CliComm.hh"
#include "MSXMotherBoard.hh"
#include "MSXException.hh"
#include "HDCommand.hh"
#include "serialize.hh"
#include <bitset>
#include <cassert>

namespace openmsx {

using std::string;
using std::vector;

static const unsigned MAX_HD = 26;
typedef std::bitset<MAX_HD> HDInUse;

HD::HD(MSXMotherBoard& motherBoard_, const XMLElement& config)
	: motherBoard(motherBoard_)
{
	MSXMotherBoard::SharedStuff& info =
		motherBoard.getSharedStuff("hdInUse");
	if (info.counter == 0) {
		assert(info.stuff == NULL);
		info.stuff = new HDInUse();
	}
	++info.counter;
	HDInUse& hdInUse = *reinterpret_cast<HDInUse*>(info.stuff);

	unsigned id = 0;
	while (hdInUse[id]) {
		++id;
		if (id == MAX_HD) {
			throw MSXException("Too many HDs");
		}
	}
	// for exception safety, set hdInUse only at the end
	name = string("hd") + char('a' + id);

	// For the initial hd image, savestate should only try exactly this
	// (resolved) filename. For user-specified hd images (commandline or
	// via hda command) savestate will try to re-resolve the filename.
	string original = config.getChildData("filename");
	string resolved = config.getFileContext().resolveCreate(original);
	filename = Filename(resolved);
	try {
		file.reset(new File(filename));
		filesize = file->getSize();
	} catch (FileException&) {
		// Image didn't exist yet, but postpone image creation:
		// we don't want to create images during 'testconfig'
		filesize = config.getChildDataAsInt("size") * 1024 * 1024;
	}
	alreadyTried = false;

	hdInUse[id] = true;
	hdCommand.reset(new HDCommand(motherBoard.getCommandController(),
	                              motherBoard.getMSXEventDistributor(),
	                              motherBoard.getScheduler(),
	                              *this));

	motherBoard.getMSXCliComm().update(CliComm::HARDWARE, name, "add");
}

HD::~HD()
{
	MSXMotherBoard::SharedStuff& info =
		motherBoard.getSharedStuff("hdInUse");
	assert(info.counter);
	assert(info.stuff);
	HDInUse& hdInUse = *reinterpret_cast<HDInUse*>(info.stuff);

	motherBoard.getMSXCliComm().update(CliComm::HARDWARE, name, "remove");
	unsigned id = name[2] - 'a';
	assert(hdInUse[id]);
	hdInUse[id] = false;

	--info.counter;
	if (info.counter == 0) {
		assert(hdInUse.none());
		delete &hdInUse;
		info.stuff = NULL;
	}
}

const string& HD::getName() const
{
	return name;
}

const Filename& HD::getImageName() const
{
	return filename;
}

void HD::openImage()
{
	if (file.get()) return;

	// image didn't exist yet, create new
	if (alreadyTried) {
		throw FileException("No HD image");
	}
	alreadyTried = true;
	try {
		file.reset(new File(filename, File::CREATE));
		file->truncate(filesize);
	} catch (FileException& e) {
		motherBoard.getMSXCliComm().printWarning(
			"Couldn't create HD image: " + e.getMessage());
		throw;
	}
}

void HD::switchImage(const Filename& name)
{
	file.reset(new File(name));
	filename = name;
	filesize = file->getSize();
	motherBoard.getMSXCliComm().update(CliComm::MEDIA, getName(),
	                                   filename.getResolved());
}

unsigned HD::getNbSectorsImpl() const
{
	const_cast<HD&>(*this).openImage();
	return filesize / SECTOR_SIZE;
}

void HD::readSectorImpl(unsigned sector, byte* buf)
{
	openImage();
	file->seek(sector * SECTOR_SIZE);
	file->read(buf, SECTOR_SIZE);
}

void HD::writeSectorImpl(unsigned sector, const byte* buf)
{
	openImage();
	file->seek(sector * SECTOR_SIZE);
	file->write(buf, SECTOR_SIZE);
}

bool HD::isWriteProtectedImpl() const
{
	const_cast<HD&>(*this).openImage();
	return file->isReadOnly();
}

SectorAccessibleDisk* HD::getSectorAccessibleDisk()
{
	return this;
}

const std::string& HD::getContainerName() const
{
	return getName();
}

bool HD::diskChanged()
{
	return false; // TODO not implemented
}

int HD::insertDisk(const std::string& filename)
{
	try {
		switchImage(Filename(filename));
		return 0;
	} catch (MSXException&) {
		return -1;
	}
}


template<typename Archive>
void HD::serialize(Archive& ar, unsigned /*version*/)
{
	Filename tmp = file.get() ? filename : Filename();
	ar.serializeNoID("filename", tmp);
	if (ar.isLoader()) {
		if (tmp.empty()) {
			// lazily open file specified in config
		} else {
			tmp.updateAfterLoadState(motherBoard.getCommandController());
			switchImage(tmp);
			assert(file.get());
		}
	}

	if (file.get()) {
		string oldChecksum;
		if (!ar.isLoader()) {
			oldChecksum = getSHA1Sum();
		}
		ar.serialize("checksum", oldChecksum);
		if (ar.isLoader()) {
			string newChecksum = getSHA1Sum();
			if (oldChecksum != newChecksum) {
				motherBoard.getMSXCliComm().printWarning(
				    "The content of the harddisk " +
				    tmp.getResolved() +
				    " has changed since the time this savestate was "
				    "created. This might result in emulation problems "
				    "or even diskcorruption. To prevent the latter, "
				    "the harddisk is now write-protected.");
				forceWriteProtect();
			}
		}
	}
}
INSTANTIATE_SERIALIZE_METHODS(HD);

} // namespace openmsx
