/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Wrapper_p.h"

#include <llvm/LLVMContext.h>
#include <llvm/Module.h>

#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Function_p.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/ModuleData_p.h>
#include <GTLCore/Type_p.h>
#include <GTLCore/TypesManager.h>
#include <GTLCore/TypesManager_p.h>
#include <GTLCore/Value.h>

#include <GTLFragment/Library.h>

#include <OpenRijn/LibrariesManager.h>

using namespace OpenRijn;

struct Wrapper::Private {
  bool loadRijnWrapperLibrary;
};

Wrapper::Wrapper(GTLCore::ModuleData* _moduleData, llvm::Module* module, bool loadRijnWrapperLibrary) : GTLFragment::Wrapper(_moduleData, module, -1), d(new Private)
{
  d->loadRijnWrapperLibrary = loadRijnWrapperLibrary;
}

void Wrapper::fillTypesManager( GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter)
{
  const GTLCore::Type* colorType = createColorType(_typesManager, _convertCenter);
  const GTLCore::Type* canvasType = createCanvasType(_typesManager, _convertCenter);
  const GTLCore::Type* brushType = createBrushType(_typesManager, _convertCenter, colorType, canvasType);
  createPaletteType(_typesManager, _convertCenter, brushType);

  if(d->loadRijnWrapperLibrary)
  {
    GTLFragment::Library* library = OpenRijn::LibrariesManager::instance()->loadLibrary( "rijnwrappers", -1);
    GTL_ASSERT( library );
    if(not library->isCompiled())
    {
      library->compile();
      if(not library->isCompiled())
      {
        GTL_ABORT("rijnwrappers.rijn compilation failed, " << library->compilationMessages().toString());
        return;
      }
    }
    moduleData()->linkWith( library->data() );
    addFunctionFromModuleToType(brushType, library, GTLCore::ScopedName("wrappers", "drawPoint"));
    addFunctionFromModuleToType(brushType, library, GTLCore::ScopedName("wrappers", "drawLine"));
    addFunctionFromModuleToType(brushType, library, GTLCore::ScopedName("wrappers", "drawLineTo"));
  }
}

const GTLCore::Type* Wrapper::createCanvasType(GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter)
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // it's llvm declaration must be changed too in        //
  // struct CanvasWrap !                                 //
  //---------------------- WARNING ----------------------//
//   llvm::LLVMContext& context = d->moduleData->llvmModule()->getContext();
  
  std::vector<GTLCore::Type::StructDataMember> painterDataMembers;
  const GTLCore::Type* type = _typesManager->d->createStructure( "canvas", painterDataMembers );
  
  return type;
}

#define NULL_P (void*)0

const GTLCore::Type* Wrapper::createBrushType(GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter, const GTLCore::Type* _colorType, const GTLCore::Type* _canvasType)
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // it's llvm declaration must be changed too in        //
  // struct BrushWrap !                                  //
  //---------------------- WARNING ----------------------//
  llvm::LLVMContext& context = moduleData()->llvmModule()->getContext();
  
  std::vector<GTLCore::Type::StructDataMember> brushDataMembers;
  brushDataMembers.push_back( GTLCore::Type::StructDataMember( "data", GTLCore::Type::Pointer ) );
  brushDataMembers.push_back( GTLCore::Type::StructDataMember( "paintColor", _colorType ) );
  const GTLCore::Type* type = _typesManager->d->createStructure( "brush", brushDataMembers );
  
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                moduleData(), llvmModule(), context, "drawPoint", "brush_wrap_draw_point", GTLCore::Type::Void,
                GTLCore::Function::Private::EFP_TYPE_AND_DEFAULT, 6,
                type, NULL_P, _canvasType, NULL_P,
                GTLCore::Type::Float32, NULL_P,
                GTLCore::Type::Float32, NULL_P,
                GTLCore::Type::Float32, new GTLCore::Value(1.0f),
                GTLCore::Type::Float32, new GTLCore::Value(0.0f) ) ) );
  
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                moduleData(), llvmModule(), context, "drawLine", "brush_wrap_draw_line", GTLCore::Type::Void,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 8,
                type, _canvasType,
                GTLCore::Type::Float32, GTLCore::Type::Float32, // x1, y1
                GTLCore::Type::Float32, GTLCore::Type::Float32, // t1, x2
                GTLCore::Type::Float32, GTLCore::Type::Float32 ) ) ); // y2, t2
  
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                moduleData(), llvmModule(), context, "drawLineTo", "brush_wrap_draw_line_to", GTLCore::Type::Void,
                GTLCore::Function::Private::EFP_TYPE_AND_DEFAULT, 6,
                type, NULL_P, _canvasType, NULL_P,
                GTLCore::Type::Float32, NULL_P,
                GTLCore::Type::Float32, NULL_P,
                GTLCore::Type::Float32, new GTLCore::Value(1.0f),
                GTLCore::Type::Float32, new GTLCore::Value(0.0f) ) ) );
  
  return type;
}

const GTLCore::Type* Wrapper::createPaletteType(GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter, const GTLCore::Type* _brushType)
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // it's llvm declaration must be changed too in        //
  // struct PaletteWrap !                                  //
  //---------------------- WARNING ----------------------//
  llvm::LLVMContext& context = moduleData()->llvmModule()->getContext();
  
  std::vector<GTLCore::Type::StructDataMember> paletteDataMembers;
  paletteDataMembers.push_back( GTLCore::Type::StructDataMember( "data", GTLCore::Type::Pointer ) );
  const GTLCore::Type* type = _typesManager->d->createStructure( "palette", paletteDataMembers );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                moduleData(), llvmModule(), context, "createCircleBrush", "palette_wrap_create_circle_brush", _brushType,
                GTLCore::Function::Private::EFP_TYPE_AND_DEFAULT, 3,
                type, NULL_P, GTLCore::Type::Float32, NULL_P, GTLCore::Type::Float32, new GTLCore::Value(0.5f) ) ) );
  return type;
}
