/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ParserBase_p.h"

#include <map>

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Type.h>

#include "CodeGenerator_p.h"
#include "ErrorMessage.h"
#include "ExpressionResult_p.h"
#include "Type.h"
#include "Type_p.h"
#include "Value.h"

#include "AST/AccessorExpression.h"
#include "AST/BinaryExpression.h"
#include "AST/ConstantCoumpoundExpression.h"
#include "AST/CoumpoundExpression.h"
#include "AST/Expression.h"
#include "AST/FunctionDeclaration.h"
#include "AST/Tree.h"
#include "AST/UnaryExpression.h"
#include "CompilerBase_p.h"
#include "ConvertCenter_p.h"
#include "LexerBase_p.h"
#include "Debug.h"
#include "Macros_p.h"
#include "Token_p.h"
#include "Function_p.h"
#include "Utils_p.h"
#include "VariablesManager_p.h"
#include "ExpressionGenerationContext_p.h"

#include "TypesManager.h"
#include "TypesManager_p.h"
#include <llvm/Module.h>
#include "ModuleData_p.h"

using namespace GTLCore;

struct ParserBase::Private {
  Token currentToken;
  CompilerBase* compiler;
  LexerBase* lexer;
  String nameSpace;
  VariablesManager variablesManager;
  const GTLCore::Type* currentReturnType;
};

ParserBase::ParserBase(CompilerBase* _compiler, LexerBase* _lexer) : d(new Private)
{
  d->compiler = _compiler;
  d->lexer = _lexer;
  d->currentReturnType = 0;
}

ParserBase::~ParserBase()
{
  delete d;
}

void ParserBase::getNextToken()
{
  d->currentToken = d->lexer->nextToken();
}

const Token& ParserBase::currentToken()
{
  return d->currentToken;
}


void ParserBase::parseConstantDeclaration()
{
  getNextToken();
  const GTLCore::Type* type = parseType();
  if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
  {
    GTLCore::String name = currentToken().string;
    getNextToken();
    // Decode the size of array (if applicable)
    std::list<int> memberArraySize = expressionsListToIntegersList( parseArraySize(true) );
    if( d->compiler) // This is needed for the Parser's test
    {
      type = d->compiler->typesManager()->getArray( type, memberArraySize.size() );
    }
    // Check that there is an initialiser
    if( isOfType( currentToken(), GTLCore::Token::EQUAL ) )
    {
      getNextToken();
      AST::Expression* expression = 0 ;
      // Check if it's a coumpound expression
      if( currentToken().type == GTLCore::Token::STARTBRACE )
      {
        GTLCore::AST::Expression* coumpoundExpression = parseCoumpoundExpression( type, true );
      /* if( coumpoundExpression->size() != *memberArraySize.begin()) // TODO: recurisvely check other sizes
        {
          GTL_DEBUG( memberArraySize.size() << " " << coumpoundExpression->size() );
          reportError("Wrong coumpound initialiser size", currentToken() );
        }*/
        expression = coumpoundExpression;
      } else {
        expression = parseExpression(true);
      }
      if( isOfType( currentToken(), GTLCore::Token::SEMI ) )
      {
        getNextToken();
        if(expression)
        {
          GTLCore::ScopedName scopedName( nameSpace(), name );
          if( tree()->containsGlobalConstant( scopedName ) )
          {
            reportError("Constant '" + scopedName.toString() + "' has allready been declared", currentToken());
          } else {
            AST::GlobalConstantDeclaration* gcd = new AST::GlobalConstantDeclaration( scopedName, type , expression, false );
            variablesManager()->declareConstant( scopedName, gcd->variable() );
            tree()->append( gcd );
          }
        }
      }
    } else {
      GTL_DEBUG("Unexpected");
      reportUnexpected( currentToken() );
      reachNextSemi();
    }
  } else {
    GTL_DEBUG("Unexpected");
    reportUnexpected( currentToken() );
    reachNextSemi();
  }
}

void ParserBase::parseStructDefinition()
{
  GTL_ASSERT( d->currentToken.type == Token::STRUCT );
  getNextToken();
  if( isOfType( d->currentToken, Token::IDENTIFIER ) )
  {
    String name = d->currentToken.string;
    getNextToken();
    if( d->compiler->typesManager()->d->isKnownType( name ) )
    { // It has allready been declared
      reportError( name + " has allready been declared", d->currentToken);
    } else {
      std::vector<Type::StructDataMember> members;
      if( isOfType( d->currentToken, Token::STARTBRACE ) )
      {
        getNextToken(); // Positionate on the first struct member
        while( true )
        { // Parse struct member
          if( isType(d->currentToken) )
          {
            const Type* smtype = parseType();
            GTL_ASSERT( smtype );
            if( isOfType( d->currentToken, Token::IDENTIFIER ) )
            {
              String smname = d->currentToken.string;
              getNextToken();
              // Check if it's an array
              std::list<int> memberArraySize = expressionsListToIntegersList( parseArraySize(true) );
              smtype = d->compiler->typesManager()->getArray( smtype, memberArraySize.size() );
              // Check if expression end by a SEMI and add the struct member
              if( isOfType( d->currentToken, Token::SEMI ) )
              {
                getNextToken();
                members.push_back( Type::StructDataMember( smname, smtype, memberArraySize ) );
              } else {
                return;
              }
            } else {
              return;
            }
          } else if( d->currentToken.type == Token::ENDBRACE )
          {
            getNextToken();
            break;
          } else {
            GTL_DEBUG("Unexpected");
            reportUnexpected( d->currentToken );
            return;
          }
        }
        if( isOfType( d->currentToken, Token::SEMI ) )
        {
          getNextToken();
          d->compiler->typesManager()->d->createStructure( name, members);
        }
      }
    }
  }
}

AST::Expression* ParserBase::parseMemberArrayConstantExpression(AST::Expression* _expression)
{
  if( d->currentToken.type == Token::DOT or d->currentToken.type == Token::STARTBOXBRACKET )
  {
    // Check we have a coumpound expression
    AST::ConstantCoumpoundExpression* coumpound = dynamic_cast<AST::ConstantCoumpoundExpression*>( _expression );
    if( not coumpound )
    {
      getNextToken();
      reportError( "Expected value of array, structure or vector type.", d->currentToken );
      return 0;
    }
    // Compute the idx
    int idx = -1;
    if( d->currentToken.type == Token::DOT )
    {
      getNextToken();
      if( d->currentToken.type == Token::SIZE )
      {
        return new AST::NumberExpression<int>( coumpound->size() );
      } else if( isOfType( d->currentToken, Token::IDENTIFIER ) ) {
        String name = d->currentToken.string;
        getNextToken();
        if( _expression->type()->dataType() == Type::STRUCTURE ) {
          idx = structMemberNameToIdx( _expression->type(), name );
        } else if( _expression->type()->dataType() == Type::VECTOR ) {
          idx = vectorMemberNameToIdx( name );
        } 
      }
    } else if( d->currentToken.type == Token::STARTBOXBRACKET )
    {
      getNextToken();
      AST::Expression* expr = parseExpression( true );
      GTL_ASSERT( expr->isConstant() );
      expr = d->compiler->convertCenter()->createConvertExpression( expr, GTLCore::Type::Integer32 );
      if( expr and expr->isConstant() )
      {
        llvm::LLVMContext context;
        GenerationContext gc( 0, &context, 0, 0 ,0 );
        ExpressionGenerationContext egc(0);
        llvm::ConstantInt* idxCST = dynamic_cast<llvm::ConstantInt*>( expr->generateValue( gc, egc).constant() );
        GTL_ASSERT( idxCST );
        idx = idxCST->getLimitedValue();
      } else {
        delete expr;
        reportError( "Expected integer constant", d->currentToken );
        return 0;
      }
      if( isOfType( d->currentToken, Token::ENDBOXBRACKET ) )
      {
        getNextToken();
      }
    } else {
      GTL_ABORT("Impossible error.");
    }
    GTL_DEBUG( idx );
    if( idx >= 0 and idx < coumpound->size() )
    {
      return parseMemberArrayConstantExpression( coumpound->expressionAt( idx ) );
    } else {
      reportError( "Invalid index.", d->currentToken );
      return 0;
    }
  } else {
    return _expression;
  }
}

AST::AccessorExpression* ParserBase::parseMemberArrayExpression(AST::AccessorExpression* _expression, bool _constantExpression)
{
  GTL_ASSERT( not _constantExpression);
  if( d->currentToken.type == Token::DOT )
  {
    getNextToken();
    if( d->currentToken.type == Token::SIZE )
    {
      getNextToken(); // eat size
      return new AST::ArraySizeAccessorExpression( _expression );
    } else if( isOfType( d->currentToken, Token::IDENTIFIER ) ) {
      String name = d->currentToken.string;
      getNextToken();
      if( d->currentToken.type == Token::STARTBRACKET )
      {
        getNextToken(); // Eat the start bracket
        const Type::StructFunctionMember* sfm = _expression->type()->d->functionMember( name );
        if( not sfm )
        {
          reportError("Unknown member: '" + name + "' for structure " + _expression->type()->structName(), d->currentToken );
        } else {
          std::list<Function*> functions;
          std::list<AST::Expression*> arguments = parseArguments( sfm->name() );
          functions.push_back(const_cast<Function*>( sfm->function() ) );
          GTLCore::Function* function = 0;
          arguments = selectFunction(arguments, functions, &function, 1);
          if( not function )
          {
            delete _expression;
            return 0;
          }
          GTL_ASSERT(function == sfm->function());
          return new AST::FunctionMemberAccessorExpression(_expression, sfm, arguments);
        }
      } else if( _expression->type()->dataType() == Type::STRUCTURE ) {
        return parseMemberArrayExpression( new AST::StructAccessorExpression( _expression ,  structMemberNameToIdx( _expression->type(), name ) ), _constantExpression);
      } else if( _expression->type()->dataType() == Type::VECTOR ) {
        return new AST::ArrayAccessorExpression( _expression, new AST::NumberExpression<int>( vectorMemberNameToIdx( name ) ) );
      } else {
        GTL_DEBUG("unexpected");
        reportUnexpected( d->currentToken );
      }
    }
  } else if( d->currentToken.type == Token::STARTBOXBRACKET )
  {
    getNextToken();
    AST::Expression* expr = parseExpression( _constantExpression );
    if( isOfType( d->currentToken, Token::ENDBOXBRACKET ) )
    {
      getNextToken();
      return parseMemberArrayExpression( new AST::ArrayAccessorExpression( _expression , expr  ), _constantExpression);
    } else {
      delete expr;
    }
  } else {
    return _expression;
  
  }
  delete _expression;
  return 0;
}

std::list< AST::Expression* > ParserBase::parseArraySize(bool _constantExpression)
{
  std::list< AST::Expression* > list_;
  while( d->currentToken.type == Token::STARTBOXBRACKET )
  {
    getNextToken();
    if( d->currentToken.type == Token::ENDBOXBRACKET )
    {
      getNextToken();
      list_.push_back( 0 );
    } else {
      AST::Expression* expr = parseExpression( _constantExpression );
      if( expr and d->currentToken.type == Token::ENDBOXBRACKET )
      {
        getNextToken();
        list_.push_back( expr );
      } else {
        delete expr;
        GTL_DEBUG("unexpected");
        reportUnexpected( d->currentToken );
        list_.push_back( 0 );
      }
    }
  }
  return list_;
}

AST::Expression* ParserBase::parseCoumpoundExpression( const Type* _type, bool _constantExpression )
{
//   GTL_ASSERT( _type );
  bool autoType = false;
  if( not _type )
  {
    _type = GTLCore::TypesManager::getArray(GTLCore::Type::Float); // The real type is guessed later
    autoType = true;
  }
  GTL_DEBUG( *_type );
  
  Token::Type endToken;
  switch( d->currentToken.type)
  {
    case Token::STARTBRACE:
      endToken = Token::ENDBRACE;
      break;
    case Token::STARTBRACKET:
      endToken = Token::ENDBRACKET;
      break;
    default:
      GTL_DEBUG("unexpected");
      reportUnexpected( d->currentToken );
      return 0;
  }
  
  getNextToken(); // Eat '{'
  std::vector< AST::Expression* > expressions_;
  unsigned int index = 0;
  while(true)
  {
    AST::Expression* expression = 0;
    if( _type->dataType() == Type::STRUCTURE and index >= _type->countStructDataMembers() )
    {
      index = _type->countStructDataMembers() - 1;
      reportError("Invalid number of constant in coumpound expression.", currentToken());
    }
    const Type* subtype = _type->d->subtypeAt( index );
    if( d->currentToken.type == Token::STARTBRACE )
    {
      GTL_DEBUG( index << " " <<  *subtype );
      if( subtype->dataType() != Type::STRUCTURE and subtype->dataType() != Type::ARRAY )
      {
        if(autoType) {
          subtype = 0;
        } else {
          GTL_DEBUG("unexpected");
          reportUnexpected( d->currentToken );
          return 0;
        }
      }
      expression = parseCoumpoundExpression( subtype, _constantExpression );
    } else {
      expression = parseExpression( _constantExpression );
    }
    if( not expression )
    {
      deleteAll( expressions_ );
      return 0;
    }
    if(autoType)
    {
      subtype = expression->type();
      _type = GTLCore::TypesManager::getArray(subtype);
      autoType = false;
    }
    expression = d->compiler->convertCenter()->createConvertExpression( expression, subtype );
    if( not expression )
    {
      reportError( "can't convert coumpound expression at " + String::number( index ) + ".", d->currentToken);
      return 0;
    }
    expressions_.push_back( expression );
    if( d->currentToken.type == Token::COMA )
    {
      getNextToken();
    } else if( d->currentToken.type == endToken )
    {
      getNextToken();
      if( _constantExpression )
      {
        return new AST::ConstantCoumpoundExpression(_type, expressions_);
      } else {
        return new AST::CoumpoundExpression(_type, expressions_);
      }
    }
    ++index;
  }
}

AST::Expression* ParserBase::parseTypeCoumpoundExpression(bool _constantExpression )
{
  const Type* type = parseType();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    AST::Expression* expression = 0;
    if( type->dataType() == Type::VECTOR or type->dataType() == Type::ARRAY or type->dataType() == Type::STRUCTURE )
    {
      expression = parseCoumpoundExpression( type, _constantExpression);
    } else {
      getNextToken();
      expression = parseExpression(_constantExpression, type);
      isOfType( currentToken(), Token::ENDBRACKET );
      getNextToken();
    }
    if( expression )
    {
      return d->compiler->convertCenter()->createConvertExpression( expression, type );
    } else {
      delete expression;
    }
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
  }
  return 0;
}

AST::Expression* ParserBase::parseExpression(bool _constantExpression, const GTLCore::Type* _type)
{
  // The first token of an expression is either a primary, or an unary expression or a parenthesis
  if( d->currentToken.type == Token::STARTBRACKET )
  {
    getNextToken();
    AST::Expression* expr = parseExpression(_constantExpression);
    if( expr and isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      getNextToken();
      if( expr and d->currentToken.isBinaryOperator() )
      {
        expr = parseFlatBinaryOperator( _constantExpression, expr );
      }
    }
    return expr;
  } else if( d->currentToken.isUnaryOperator() )
  {
    AST::Expression* expr = parseUnaryOperator(_constantExpression);
    // What's next ? An unary operator is either followed by a binary operator or a semi
    if( d->currentToken.isExpressionTerminal())
    {
      return expr;
    } else if( expr and d->currentToken.isBinaryOperator() ) {
      return parseFlatBinaryOperator( _constantExpression, expr );
    } else {
      delete expr;
      reportUnexpected(d->currentToken);
    }
  } else if( currentToken().type == Token::STARTBRACE )
  {
    AST::Expression* coumpoundExpression = parseCoumpoundExpression( _type, true );
    if( coumpoundExpression )
    {
      return new AST::GlobalDataExpression( coumpoundExpression );
    }
  } else {
    AST::Expression* expression = 0;
    if( d->currentToken.isPrimary() )
    {
      GTL_DEBUG( d->currentToken.isPrimary() << " " << d->currentToken);
      expression = parsePrimaryExpression(_constantExpression);
      if(not expression)
      {
        // An error has occured
        reportError("Parse error while parsing constant", d->currentToken );
        return 0;
      }
    } else if( isType(currentToken()) ) {
      expression = parseTypeCoumpoundExpression( _constantExpression );
    } else {
      GTL_DEBUG("unexpected");
      reportUnexpected( d->currentToken );
      return 0;
    }
    if( expression )
    { // What's next ? A binary operator or a ;
      if( d->currentToken.isExpressionTerminal() )
      {
        return expression;
      } else if( d->currentToken.isBinaryOperator() ) {
        return parseFlatBinaryOperator( _constantExpression, expression );
      }
      delete expression;
    }
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    getNextToken();
  }
  return 0;
}

#if 1
AST::Expression* ParserBase::parseFlatBinaryOperator( bool _constantExpression, AST::Expression* _lhs)
{
  GTL_ASSERT(_lhs);
  AST::Expression* current = _lhs;
  while( d->currentToken.isBinaryOperator() )
  {
    GTL_DEBUG( "Looping in parseFlatBinaryOperator");
    current = parseBinaryOperator( _constantExpression, current );
    if( not current ) // An error has occured
    {
      return 0;
    }
  }
  return current;
}

AST::Expression* ParserBase::parseBinaryOperator( bool _constantExpression, AST::Expression* _lhs)
{
  if( not _lhs) return 0;
  Token binOp = d->currentToken;
  GTL_DEBUG( "parseBinaryOperator: " << binOp )
  getNextToken();
  AST::Expression* expr = 0;
  while(true)
  {
    GTL_DEBUG( binOp << " " << expr );
    // What's next ? A primary expression or an unary operator or a parenthesis
    if( expr == 0 )
    {
      if( d->currentToken.type == Token::STARTBRACKET )
      {
        getNextToken();
        expr = parseExpression(_constantExpression);
        if( not isOfType( d->currentToken, Token::ENDBRACKET ) )
        {
          delete expr;
          return 0;
        }
        getNextToken();
      } else if( d->currentToken.isPrimary() )
      {
        expr = parsePrimaryExpression(_constantExpression);
      } else if( d->currentToken.isUnaryOperator() ) {
        expr = parseUnaryOperator(_constantExpression);
      } else if( currentToken().type == Token::STARTBRACE ) {
        AST::Expression* coumpoundExpression = parseCoumpoundExpression( _lhs->type() , true );
        if( coumpoundExpression )
        {
          expr = new AST::GlobalDataExpression( coumpoundExpression );
        }
      } else if( isType(currentToken()) ) {
        expr = parseTypeCoumpoundExpression(_constantExpression);      
      } else {
        GTL_DEBUG("unexpected");
        reportUnexpected( d->currentToken );
        return 0;
      }
    }
    // What's next, either a semi or an other binary operator
    if( d->currentToken.isExpressionTerminal() ) {
      GTL_DEBUG( binOp << " terminal" );
      return createBinaryOperator( binOp, _lhs, expr);
    } else if( d->currentToken.isBinaryOperator() ) {
      GTL_DEBUG( binOp << " vs " << d->currentToken.isBinaryOperator());
      GTL_ASSERT( binOp.binaryOperationPriority() != -1 );
      GTL_ASSERT( d->currentToken.binaryOperationPriority() != -1 );
      
      if( d->currentToken.binaryOperationPriority() == binOp.binaryOperationPriority() )
      {
        GTL_DEBUG( binOp << " collapse _lhs and expr and keep going" );
        _lhs = createBinaryOperator( binOp, _lhs, expr );
        binOp = d->currentToken;
        getNextToken(); // eat the current operator
        expr = 0;
      } else if( d->currentToken.binaryOperationPriority() > binOp.binaryOperationPriority() )
      {
        GTL_DEBUG( binOp << " keep parsing to collapse expr with what is next" );
        expr = parseBinaryOperator( _constantExpression, expr );
        if( expr == 0 ) return 0;
        if( d->currentToken.isExpressionTerminal() )
        {
          return createBinaryOperator( binOp, _lhs, expr);
        }
      } else {
        GTL_DEBUG( binOp << " collapse _lhs and expr and return" );
        return createBinaryOperator( binOp, _lhs, expr );
      }
    } else {
      GTL_DEBUG("unexpected");
      reportUnexpected( d->currentToken );
      return 0;
    }
  }
}

#endif

#if 0

AST::Expression* ParserBase::parseFlatBinaryOperator( bool _constantExpression, AST::Expression* _lhs )
{
  AST::Expression* currentExpr = _lhs;
  while( true )
  {
    Token binOp = d->currentToken;
    AST::Expression* nextExpr = currentExpr;
    bool lhsUsed = false;
    do {
      nextExpr = parseBinaryOperator( _constantExpression, nextExpr, lhsUsed );
      if( not d->currentToken.isBinaryOperator() )
      {
        if( lhsUsed )
        {
          return nextExpr;
        } else {
          return createBinaryOperator( binOp, currentExpr, nextExpr );
        }
      }
      GTL_ASSERT( binOp.binaryOperationPriority() != -1 );
      GTL_ASSERT( d->currentToken.binaryOperationPriority() != -1 );
    } while( binOp.binaryOperationPriority() < d->currentToken.binaryOperationPriority() );
    currentExpr = createBinaryOperator( binOp, currentExpr, nextExpr );
  }
}

AST::Expression* ParserBase::parseBinaryOperator( bool _constantExpression, AST::Expression* lhs, bool& lhsUsed )
{
  GTL_ASSERT(lhs);
  Token binOp = d->currentToken;
  getNextToken();
  AST::Expression* expr = 0;
  // What's next ? A primary expression or an unary operator or a parenthesis
  if( d->currentToken.type == Token::STARTBRACKET )
  {
    getNextToken();
    expr = parseExpression(_constantExpression);
    if( not isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      delete expr;
      return 0;
    }
    getNextToken();
  } else if( d->currentToken.isPrimary() )
  {
    expr = parsePrimaryExpression(_constantExpression);
  } else if( d->currentToken.isUnaryOperator() ) {
    expr = parseUnaryOperator(_constantExpression);
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
  // What's next, either a semi or an other binary operator
  if( d->currentToken.isExpressionTerminal() ) {
    lhsUsed = true;
    return createBinaryOperator( binOp, lhs, expr);
  } else if( d->currentToken.isBinaryOperator() ) {
    GTL_ASSERT( binOp.binaryOperationPriority() != -1 );
    GTL_ASSERT( d->currentToken.binaryOperationPriority() != -1 );
    // Check which operator has the priority over the other one
    if( binOp.binaryOperationPriority() < d->currentToken.binaryOperationPriority())
    { // This operator has lowest priority, which means the result of the second operator will be the right hand side for this operator
//       AST::Expression* rhs = parseBinaryOperator( _constantExpression, expr );
//       return createBinaryOperator( binOp, lhs, rhs );
      lhsUsed = false;
      return expr;
    } else {
      // This operator has more priority, which means the result of this operator will be the left hand side for the second operator
      lhsUsed = true;
      bool v = false;
      AST::Expression* nlhs = createBinaryOperator( binOp, lhs, expr );
      return parseBinaryOperator( _constantExpression, nlhs, v );
    }
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
}

#endif

AST::Expression* ParserBase::parseUnaryOperator( bool _constantExpression )
{
  Token unaryTok = d->currentToken;
  GTL_ASSERT(unaryTok.isUnaryOperator() );
  getNextToken();
  // What's next ? An unary operator is either followed by a parenthesis or an other unary operator or a primary
  AST::Expression* expr = 0;
  if( d->currentToken.type == Token::STARTBRACKET ) {
    expr = parseExpression(_constantExpression);
    if( not isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      delete expr;
      return 0;
    }
    getNextToken();
  } else if( d->currentToken.isPrimary() )
  {
    expr = parsePrimaryExpression(_constantExpression);
  } else if( d->currentToken.isUnaryOperator() ) {
    expr = parseUnaryOperator(_constantExpression);
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
  switch( unaryTok.type )
  {
    case Token::MINUS:
      return new AST::MinusUnaryExpression( expr );
    case Token::MINUSMINUS:
    {
      AST::AccessorExpression* varexpr = dynamic_cast<AST::AccessorExpression*>( expr );
      if( varexpr  )
      {
        if( expr->type() == Type::Integer32 )
        {
          return new AST::MinusMinusUnaryExpression( varexpr );
        } else {
          reportError( "'--' operator works only on integer variable", unaryTok);
        }
      } else {
        reportError( "'--' operator requires a variable", unaryTok );
      }
    }
    case Token::PLUSPLUS:
    {
      AST::AccessorExpression* varexpr = dynamic_cast<AST::AccessorExpression*>( expr );
      if( varexpr  )
      {
        if( expr->type() == Type::Integer32 )
        {
          return new AST::PlusPlusUnaryExpression( varexpr );
        } else {
          reportError( "'++' operator works only on integer variable", unaryTok);
        }
      } else {
        reportError( "'++' operator requires a variable", unaryTok );
      }
    }
    case Token::NOT:
    {
      return new AST::NotUnaryExpression( expr );
    }
    case Token::TILDE:
    {
      if( expr->type() != Type::Integer32 )
      {
        reportError( "'~' operator only work with integer" , unaryTok );
        return 0;
      }
      return new AST::TildeUnaryExpression( expr );
    }
    default:
    {
      GTL_ASSERT( true );
      return 0;
    }
  }
}

void ParserBase::parseFunction()
{
  GTL_ASSERT( d->currentToken.isFunctionType() );
  // Parse the type
  const Type* returnType = parseFunctionType();
  if( not returnType ) return;
  d->currentReturnType = returnType;
  struct SetCurrentReturnTypeToZero {
    ~SetCurrentReturnTypeToZero()
    {
      *currentReturnTypePtr = 0;
    }
    const Type** currentReturnTypePtr;
  } setCurrentReturnTypeToZero;
  setCurrentReturnTypeToZero.currentReturnTypePtr = &d->currentReturnType;
  
  // Clear the list of variables
  d->variablesManager.reset();
  
  // Next is the name of the function
  if( isOfType( d->currentToken, Token::IDENTIFIER ) )
  {
    Token nameToken = d->currentToken;
    ScopedName name( d->nameSpace, d->currentToken.string );
    startParsingFunction( name.name() );
    getNextToken();
    // Next is start bracket
    if( isOfType( d->currentToken, Token::STARTBRACKET ) )
    {
      std::vector< AST::FunctionParameter* > params;
      bool needToHaveInitialiser = false;
      getNextToken();
      while(true)
      {
        // Next is either a type or a end bracket
        if( isType(d->currentToken) or d->currentToken.type == Token::OUTPUT or d->currentToken.type == Token::INPUT or d->currentToken.type == Token::VARYING )
        {
          bool output = false;
          bool varying = false;
          if( d->currentToken.type == Token::OUTPUT )
          {
            output = true;
            getNextToken();
          } else if( d->currentToken.type == Token::VARYING )
          {
            varying = true;
            getNextToken();
          }else if( d->currentToken.type == Token::INPUT )
          {
            getNextToken();
          }
          if( isType( d->currentToken ) )
          {
            const Type* ptype = parseType();
            if( not ptype )
            {
              d->currentReturnType = 0;
              return;
            }
            if(isOfType( d->currentToken, Token::IDENTIFIER ) )
            {
              String pname = d->currentToken.string;
              getNextToken();
              AST::Expression* initialiser = 0;
              std::list<AST::Expression*> memberArraySize = parseArraySize(true);
              ptype = d->compiler->typesManager()->getArray( ptype, memberArraySize.size() );
              deleteAll( memberArraySize);
              if( d->currentToken.type == Token::EQUAL )
              {
                getNextToken();
                initialiser = d->compiler->convertCenter()->createConvertExpression( parseExpression( false, ptype ), ptype);
                if( not initialiser )
                {
                  reportError( "Can't convert initialiser expression to the parameter type, for parameter " + String::number((unsigned int)(params.size() + 1)) + ".", d->currentToken);
                  deleteAll( params );
                  d->currentReturnType = 0;
                  return;
                }
                needToHaveInitialiser = true;
              } else if( needToHaveInitialiser )
              {
                reportError( "Parameter need to have default initialiser once a parameter on the left had one.", d->currentToken );
                deleteAll( params );
                return;
              }
              if( d->currentToken.type == Token::COMA or d->currentToken.type == Token::ENDBRACKET )
              {
                if( d->currentToken.type == Token::COMA )
                {
                  getNextToken();
                }
                Value initialiserValue;
                if( initialiser)
                {
                  CodeGenerator cg( 0 );
                  llvm::LLVMContext context;
                  ModuleData* moduleData = new ModuleData(new llvm::Module("fake", context));
                  GenerationContext gc( &cg, &context, 0, 0, moduleData);
                  ExpressionGenerationContext egc(0);
                  llvm::Constant* constant = initialiser->generateValue( gc, egc ).constant();
                  if( constant )
                  { // TODO don't generate a constant, do the calculation with a function in the AST
                    initialiserValue = CodeGenerator::constantToValue(constant, ptype);
                  } else {
                    reportError("Couldn't generate value for default argument", currentToken());
                  }
                  delete moduleData;
                }
                params.push_back( new AST::FunctionParameter( Parameter(pname, ptype, output, varying, initialiserValue), initialiser ) );
              } else {
                GTL_DEBUG("Unexpected");
                reportUnexpected( d->currentToken );
                deleteAll( params );
                return;
              }
            } else {
              GTL_DEBUG("Unexpected");
              reportUnexpected( d->currentToken );
              deleteAll( params );
              return;
            }
          } else {
            GTL_DEBUG("Unexpected");
            reportUnexpected( d->currentToken );
          }
        }
        else if( isOfType( d->currentToken, Token::ENDBRACKET ) )
        {
          getNextToken();
          if( isOfType( d->currentToken, Token::STARTBRACE ) )
          {
            AST::FunctionDeclaration* fd = new AST::FunctionDeclaration( name, returnType, params );
            // TODO insert functions arguments in the context
            for( unsigned int i = 0; i < params.size(); ++i)
            {
              d->variablesManager.declareParameter( ScopedName("",params[i]->parameter().name() ), fd->parametersVariable()[i] );
            }
            d->variablesManager.startContext();
            AST::Statement* statement =appendCurrentContextGarbageCollecting(parseStatementList());
            d->variablesManager.endContext();
            if( statement )
            {
              bool v = d->compiler->declareFunction( name, fd->function() );
              if(v)
              {
                fd->setStatement( statement );
                tree()->append( fd );
              } else {
                delete fd->function();
                delete fd;
                delete statement;
                reportError( "Function '" + name.toString() + "' has allready been declared", nameToken);
              }
            }
            return;
          }
        } else {
          GTL_DEBUG("Unexpected");
          reportUnexpected( d->currentToken );
          return;
        }
      }
    }
    endParsingFunction( name.name() );
  }
}

AST::StatementsList* ParserBase::appendCurrentContextGarbageCollecting( AST::StatementsList* _statementsList )
{
  if( _statementsList and not _statementsList->isReturnStatement() )
  {
    _statementsList->appendStatement( d->variablesManager.garbageCollectCurrentContext() );
  }
  return _statementsList;
}

AST::Statement* ParserBase::appendCurrentContextGarbageCollecting( AST::Statement* _statement )
{
  if( _statement and not _statement->isReturnStatement() )
  {
    std::list< AST::Statement* > statements;
    statements.push_back( _statement );
    statements.push_back( d->variablesManager.garbageCollectCurrentContext() );
    return new AST::StatementsList( statements );
  }
  return _statement;
}

AST::StatementsList* ParserBase::parseStatementList()
{
  GTL_ASSERT( d->currentToken.type == Token::STARTBRACE );
  getNextToken();
  std::list<AST::Statement*> list;
  while(d->currentToken.type != Token::ENDBRACE)
  {
    AST::Statement* statement = parseStatement();
    if(not statement) break;
    list.push_back( statement );
  }
  getNextToken();
  return new AST::StatementsList( list );
}

AST::Statement* ParserBase::parsePrintStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::PRINT );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    std::list<AST::Expression*> expressions;
    while(true)
    {
      AST::Expression* expr = 0;
      if( d->currentToken.type == Token::STRING_CONSTANT )
      {
        expr = new AST::StringExpression( d->currentToken.string );
        getNextToken();
      } else {
        expr = parseExpression(false);
      }
      expressions.push_back(expr);
      if( d->currentToken.type == Token::ENDBRACKET )
      {
        getNextToken();
        break;
      } else if( not isOfType(d->currentToken, Token::COMA) ) {
        getNextToken();
        return 0;
      }
      getNextToken();
    }
    isOfType( d->currentToken, Token::SEMI);
    getNextToken();
    return new AST::PrintStatement( expressions );
  }
  return 0;
}

AST::Statement* ParserBase::parseStatementOrList()
{
  d->variablesManager.startContext();
  AST::Statement* statement;
  if( d->currentToken.type == Token::STARTBRACE )
  {
    statement = appendCurrentContextGarbageCollecting( parseStatementList() );
  } else {
    statement = appendCurrentContextGarbageCollecting( parseStatement() );
  }
  d->variablesManager.endContext();
  return statement;
}

AST::Statement* ParserBase::parseWhileStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::WHILE );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    AST::Expression* expression = parseExpression(false);
    if( isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      getNextToken(); // eat the ')'
      AST::Statement* statement = parseStatementOrList();
      return new AST::WhileStatement( expression, statement);
    }
  }
  return 0;
}

AST::Statement* ParserBase::parseForStatement()
{
  d->variablesManager.startContext();
  GTL_ASSERT( d->currentToken.type == Token::FOR );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    AST::Statement* initexpression = 0;
    if( d->currentToken.type == Token::SEMI )
    {
      getNextToken(); // eat the ';'
    } else {
      initexpression = parseStatement();
    }
    AST::Expression* comparexpression = parseExpression(false);
    if( isOfType( d->currentToken, Token::SEMI ) )
    {
      getNextToken(); // eat the ';'
      AST::Expression* updateexpression = 0;
      if( d->currentToken.type != Token::ENDBRACKET )
      {
        updateexpression = parseExpression(false);
      }
      if( isOfType( d->currentToken, Token::ENDBRACKET ) )
      {
        getNextToken(); // eat the ')'
        AST::Statement* statement = appendCurrentContextGarbageCollecting(parseStatementOrList());
        d->variablesManager.endContext();
        return new AST::ForStatement(initexpression, comparexpression, updateexpression, statement);
      }
      delete updateexpression;
      delete comparexpression;
    }
    delete initexpression;
  }
  d->variablesManager.endContext();
  return 0;
}

AST::Statement* ParserBase::parseIfStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::IF );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    AST::Expression* expression = parseExpression(false);
    if( isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      getNextToken(); // eat the ')'
      AST::Statement* statement = parseStatementOrList();
      if( d->currentToken.type == Token::ELSE )
      {
        getNextToken(); // eat the else
        AST::Statement* elsestatement = parseStatementOrList();
        return new AST::IfElseStatement( expression, statement, elsestatement);
      } else {
        return new AST::IfStatement( expression, statement);
      }
    }
  }
  return 0;
}

AST::Statement* ParserBase::parseReturnStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::RETURN );
  getNextToken();
  GTL_ASSERT( d->currentReturnType );
  if( d->currentReturnType->dataType() == Type::VOID  )
  {
    isOfType(d->currentToken, Token::SEMI);
    getNextToken(); // eat the coma
    return new AST::ReturnStatement( 0, d->variablesManager.garbageCollectEverything() );
  } else {
    AST::Expression* expr = parseExpression(false);
    if( expr )
    {
      expr = d->compiler->convertCenter()->createConvertExpression( expr, d->currentReturnType );
      if(not expr)
      {
        reportError("Can't conert expression to return type", d->currentToken);
      }
      if(isOfType( d->currentToken, Token::SEMI ) )
      {
        getNextToken();
        return new AST::ReturnStatement( expr, d->variablesManager.garbageCollectEverything() );
      } else {
        delete expr;
        return 0;
      }
    } else {
      return 0;
    }
  }
}

AST::Statement* ParserBase::parseExpressionStatement()
{
  AST::Statement* statement = parseExpression(false);
  if(isOfType( d->currentToken, Token::SEMI ) )
  {
    getNextToken();
    return statement;
  } else {
    delete statement;
    getNextToken();
    return 0;
  }
}

AST::Statement* ParserBase::parseVariableDeclaration()
{
  bool constant = false;
  if( d->currentToken.type == Token::CONST )
  {
    constant = true;
    getNextToken();
  }
  if( isType(d->currentToken) )
  {
    const Type* type = parseType();
    GTL_ASSERT(type);
    if( isOfType( d->currentToken, Token::IDENTIFIER ) )
    {
      ScopedName name("", d->currentToken.string);
      if( not d->variablesManager.hasVariableInCurrentContext( name ) )
      {
        getNextToken();
        std::list<AST::Expression*> initialsize = parseArraySize(false);
        type = d->compiler->typesManager()->getArray( type, initialsize.size() );
        bool initialised = false;
        AST::Expression* initialiser = 0;
        if( d->currentToken.type == Token::EQUAL )
        {
          getNextToken(); // eat the equal
          // Check if it's a coumpound expression
          initialiser = parseExpression( false, type );
          initialised = true;
          if( initialiser )
          {
            AST::Expression* previousInitialiser = initialiser;
            initialiser = d->compiler->convertCenter()->createConvertExpression( initialiser, type );
            if( not initialiser )
            {
              delete previousInitialiser;
              reportError( "Can't convert value at initialisation to variable type.", d->currentToken );
              return 0;
            }
          }
        }
        // Create the variable declaration statement
        AST::VariableDeclaration* variable = new AST::VariableDeclaration( type, initialiser, constant, initialsize );
        // Insert the variable in the context
        d->variablesManager.declareVariable( name, variable->variable());
        GTL_ASSERT( variable->variable() );
        GTL_DEBUG(initialised << " " << Token::typeToString( d->currentToken.type ) );
        if( d->currentToken.type == Token::COMA and not initialised ) {
          // it's possible to initialise a variable by calling a function taking as output the variable
          getNextToken(); // eat the coma
          variable->setFunctionIntialiser(parseExpression( false ));
          initialised = true;
        }
        if( constant and not initialised )
        {
          delete variable;
          reportError( "Unitialised constant.", d->currentToken );
          return 0;
        }
        if( isOfType(d->currentToken, Token::SEMI ) )
        {
          getNextToken(); // eat the semi
          return variable;
        } else {
          delete variable;
          GTL_DEBUG("Expecting ';'");
          getNextToken();
          return 0;
        }
      } else {
        reportError("Variable " + name.name() + " has allrady been defined.", d->currentToken);
        return 0;
      }
    }
    return 0;
  } else {
    GTL_DEBUG("Unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
}

const Type* ParserBase::parseFunctionType()
{
  if( d->currentToken.type == Token::VOID )
  {
    getNextToken();
    return Type::Void;
  } else {
    const Type* type = parseType();
    if( not type ) return 0;
    std::list< AST::Expression* > sizes = parseArraySize(true);
    deleteAll( sizes );
    return d->compiler->typesManager()->getArray( type, sizes.size() );
  }
}

const Type* ParserBase::parseType()
{
  switch( d->currentToken.type )
  {
    case Token::BOOL:
      getNextToken();
      return Type::Boolean;
    case Token::INT:
      getNextToken();
      return Type::Integer32;
    case Token::HALF:
    case Token::FLOAT:
      getNextToken();
      return Type::Float;
    case Token::UNSIGNED:
    {
      getNextToken();
      if( d->currentToken.type == Token::INT )
      {
        getNextToken();
      }
      return Type::UnsignedInteger32;
    }
    case Token::IDENTIFIER:
    {
      const Type* type = d->compiler->typesManager()->getStructure( d->currentToken.string );
      if( not type )
      {
        reportError("Unknown type : " + d->currentToken.string, d->currentToken );
      }
      getNextToken();
      return type;
    }
    default:
      reportError("Expected type before " + Token::typeToString( d->currentToken.type ), d->currentToken);
      getNextToken();
      return Type::Integer32;
  }
}

AST::Expression* ParserBase::parsePrimaryExpression(bool _constantExpression)
{
  switch( d->currentToken.type )
  {
    case Token::INTEGER_CONSTANT:
    {
      int v = d->currentToken.i;
      getNextToken();
      return new AST::NumberExpression<int>( v );
    }
    case Token::FLOAT_CONSTANT:
    {
      float v = d->currentToken.f;
      getNextToken();
      return new AST::NumberExpression<float>( v );
    }
    case Token::TTRUE:
      getNextToken();
      return new AST::NumberExpression<bool>( true );
    case Token::TFALSE:
      getNextToken();
      return new AST::NumberExpression<bool>( false );
    case Token::IDENTIFIER:
    {
      if(_constantExpression)
      { // If we are computing a constant expression, we can only use the previously declared constant variable
        for( std::list< AST::GlobalConstantDeclaration*>::const_iterator it = tree()->globalConstantDeclarations().begin();
             it != tree()->globalConstantDeclarations().end(); ++it)
        {
          if( (*it)->name().name() == d->currentToken.string )
          {
            getNextToken();
            // Put the expr in a proxy to avoid double free when the AST tree is deleted
            return new AST::ProxyExpression( parseMemberArrayConstantExpression( (*it)->initialiser() ) );
          }
        }
        AST::Expression* stdconst = d->compiler->standardConstant( d->currentToken.string );
        if( stdconst )
        {
          return stdconst;
        }
        delete stdconst;
        getNextToken();
        reportError( "Unknown constant: " + d->currentToken.string, d->currentToken );
        return 0;
      } else {
        // It can be either a call to a function or a variable access
        ScopedName name("", d->currentToken.string );
        getNextToken(); // eat the identifier
        if( d->currentToken.type == Token::COLONCOLON )
        {
          getNextToken(); // eat the ::
          name = ScopedName( name.name(), d->currentToken.string );
          getNextToken(); // eat the name
        }
        if( d->currentToken.type == Token::STARTBRACKET )
        { // It's a function call
          GTL_DEBUG( name );
          ScopedName fname( ( name.nameSpace() == "" ) ? d->nameSpace : name.nameSpace(),
                              name.name() );
          const std::list<Function*>* functionlist = d->compiler->function( fname );
          if( not functionlist )
          {
            reportError("Unknown function: " + name.toString(), d->currentToken);
            getNextToken();
            return 0;
          }
          GTL_ASSERT( not functionlist->empty() );
          getNextToken();
          // Parse arguments
          std::list<AST::Expression*> arguments = parseArguments( fname.toString() );
          GTLCore::Function* function = 0;
          arguments = selectFunction(arguments, *functionlist, &function, 0);
          if( not function )
          {
            return 0;
          }
          if( arguments.size() >= function->d->data->minimumParameters() and arguments.size() <= function->d->data->maximumParameters() )
          {
            return new AST::FunctionCallExpression( function, arguments ) ;
          } else {
            reportError("Incorrect number of paramaters to function: " + function->name().toString(), d->currentToken);
            return 0;
          }
        } else {
          VariableNG* var = d->variablesManager.getVariable( name );
          if(not var)
          {
            AST::Expression* stdconst = d->compiler->standardConstant( name.name() );
            if( stdconst )
            {
              return stdconst;
            } else {
              reportError( "Unknown variable: " + name.toString(), d->currentToken );
              delete stdconst;
              return 0;
            }
          }
          return parseMemberArrayExpression(new AST::VariableAccessorExpression( var ) , _constantExpression);
        }
      }
      break;
    }
    default:
      GTL_DEBUG("unexpected");
      reportUnexpected( d->currentToken );
  }
  return 0;
}

std::list<AST::Expression*> ParserBase::parseArguments( const String& _name )
{
  std::list<AST::Expression*> arguments;
  while(true)
  {
    if( d->currentToken.type == Token::ENDBRACKET )
    {
      break;
    } else {
      AST::Expression* expression = parseExpression( false, 0 );
      if(not expression)
      {
        return arguments;
      }
      arguments.push_back( expression);
      if( d->currentToken.type == Token::COMA )
      {
        getNextToken();
      } else if( d->currentToken.type != Token::ENDBRACKET )
      {
        GTL_DEBUG("Unexpected");
        reportUnexpected( d->currentToken );
        return std::list<AST::Expression*>();
      }
    }
  }
  GTL_ASSERT( d->currentToken.type == Token::ENDBRACKET );
  getNextToken(); // eat the end bracket
  return arguments;
}

std::list<AST::Expression*> ParserBase::selectFunction( const std::list<AST::Expression*>& _expressions, const std::list<GTLCore::Function*>& _functions, GTLCore::Function** _selectedFunction, int firstParameter)
{
  GTL_ASSERT( _functions.size() > 0 );
  GTL_ASSERT( *_selectedFunction == 0 );
  std::size_t realParameterCount = firstParameter + _expressions.size();
  if( _functions.size() == 1 )
  {
    GTLCore::Function* function = _functions.front();
    GTL_DEBUG( function->name().toString()  << " " << function->d->data->minimumParameters()  << " " << realParameterCount << " " << function->d->data->maximumParameters() );
    if( function->d->data->minimumParameters() > realParameterCount
        or realParameterCount > function->d->data->maximumParameters() )
    {
      reportError("Invalid number of arguments for function: " + function->name().toString() + ".", currentToken());
      return _expressions;
    }
    *_selectedFunction = function;
  } else {
    GTLCore::Function* bestFunction = 0;
    int bestLosslessConversionCount = _expressions.size() + 1;
    int bestLossConversionCount = _expressions.size() + 1;
    foreach( GTLCore::Function* function, _functions)
    {
      GTL_DEBUG("Test for " << *function);
      if( function->d->data->minimumParameters() <= realParameterCount
          and realParameterCount <= function->d->data->maximumParameters() )
      {
        GTL_DEBUG("Good range of arguments");
        int posArg = 0;
        int currentLossLessConversionCount = 0;
        int currentLossConversionCount = 0;
        bool canUse = true;
        foreach( AST::Expression* expression, _expressions)
        {
          if( function->parameters()[ posArg + firstParameter ].isOutput() )
          {
            if( not dynamic_cast< AST::AccessorExpression* >( expression ) )
            {
              // Not an accessor, meaning we can't use that function since the parameter is an output
              canUse = false;
              break;
            }
          }
          ConvertCenter::ConversionQuality cq = d->compiler->convertCenter()->conversionQuality( expression->type(), function->parameters()[ posArg + firstParameter ].type());
          GTL_DEBUG("Quality for " << posArg << " " << cq);
          if( cq == ConvertCenter::NOT_CONVERTIBLE )
          {
            canUse = false;
            break;
          } else {
            switch(cq)
            {
              case ConvertCenter::NOT_CONVERTIBLE:
                GTL_ABORT("Shouldn't be possible");
              case ConvertCenter::NONEEDED_CONVERSION:
                break;
              case ConvertCenter::LOSS_CONVERSION:
                ++currentLossConversionCount;
                break;
              case ConvertCenter::LOSSLESS_CONVERSION:
                ++currentLossLessConversionCount;
                break;
            }
          }
          ++posArg;
        }
        if(canUse )
        {
          if( currentLossLessConversionCount == bestLosslessConversionCount and currentLossConversionCount == bestLossConversionCount )
          {
            // Can't know which function to use
            bestFunction = 0;
          } else if( currentLossConversionCount < bestLossConversionCount or (currentLossLessConversionCount < bestLosslessConversionCount and currentLossConversionCount <= bestLossConversionCount ) ) {
            bestFunction = function;
            bestLosslessConversionCount = currentLossLessConversionCount;
            bestLossConversionCount = currentLossConversionCount;
          }
        }
      }
    }
    if( bestFunction )
    {
      GTL_DEBUG("Best function " << *bestFunction);
      *_selectedFunction = bestFunction;
    } else {
      *_selectedFunction = 0;
      reportError("Can't find which overloaded functions of '" + _functions.front()->name().toString() + "' to use.", currentToken() );
      return _expressions;
    }
  }
  // Attempt conversion
  std::list<AST::Expression*> convertedExpressions;
  GTLCore::Function* function = *_selectedFunction;
  GTL_ASSERT(function);
  foreach( AST::Expression* expression, _expressions )
  {
    int posArg = convertedExpressions.size();
    expression = d->compiler->convertCenter()->createConvertExpression( expression,
                                  function->parameters()[ posArg  + firstParameter ].type() );
    if( not expression )
    {
      *_selectedFunction = 0;
      reportError("Can't convert parameter " + String::number( posArg + 1 ), d->currentToken  );
      return convertedExpressions;
    }
    convertedExpressions.push_back( expression);
    if( function->parameters()[ posArg + firstParameter ].isOutput() )
    {
      if( not dynamic_cast< AST::AccessorExpression* >( expression ) )
      {
        *_selectedFunction = 0;
        reportError( "Parameter of function '" + function->name().toString() + "' is an output parameter and requires a variable as argument", d->currentToken );
      }
    }

  }
  GTL_ASSERT( convertedExpressions.size() == _expressions.size() );
  return convertedExpressions;
}

void ParserBase::reachNextSemi()
{
  while( d->currentToken.type != Token::SEMI and d->currentToken.type != Token::END_OF_FILE )
  {
    getNextToken();
  }
  getNextToken();
}

// Utilities

bool ParserBase::isOfType( const Token& token, Token::Type type )
{
  if( token.type == type )
  {
    return true;
  } else {
    reportError("Expected " + Token::typeToString(type) + " before " + Token::typeToString(token.type)  + ".", token);
    return false;
  }
}

void ParserBase::checkNextTokenIsSemi()
{
  getNextToken();
  if( d->currentToken.type != Token::SEMI )
  {
    reportError("Expected ';' before " + Token::typeToString(d->currentToken.type) + ".", d->currentToken );
  }
}

void ParserBase::reportError( const String& errMsg, const Token& token )
{
  if( d->compiler)
  {
    d->compiler->appendError( ErrorMessage( errMsg, token.line, "" ) );
  } else {
    GTL_DEBUG( errMsg);
  }
}

void ParserBase::reportUnexpected( const Token& token )
{
  reportError("Unexpected: " + Token::typeToString( token.type ), token );
  getNextToken();
}

AST::Expression* ParserBase::createBinaryOperator( const Token& token, AST::Expression* lhs, AST::Expression* rhs )
{
  if( not lhs or not rhs )
  {
    delete lhs;
    delete rhs;
    return 0;
  }
  if( token.type == Token::EQUAL or token.type == Token::PLUSEQUAL or token.type == Token::MINUSEQUAL or token.type == Token::MULTIPLYEQUAL or token.type == Token::DIVIDEEQUAL)
  {
    rhs = d->compiler->convertCenter()->createConvertExpression( rhs, lhs->type() );
    if( not rhs )
    {
      delete lhs;
      delete rhs;
      reportError("Can't convert right hand side value to the left hand side type.", token);
      return 0;
    }
    AST::AccessorExpression* ve = dynamic_cast<AST::AccessorExpression*>( lhs );
    GTL_ASSERT( rhs->type() == ve->type() );
    if( ve )
    {
      if( token.type == Token::PLUSEQUAL )
      {
        rhs = new AST::AdditionBinaryExpression( new AST::ProxyExpression(ve), rhs );
      } else if(token.type == Token::MINUSEQUAL) {
        rhs = new AST::SubstractionBinaryExpression( new AST::ProxyExpression(ve), rhs );
      } else if(token.type == Token::MULTIPLYEQUAL) {
        rhs = new AST::MultiplicationBinaryExpression( new AST::ProxyExpression(ve), rhs );
      } else if(token.type == Token::DIVIDEEQUAL) {
        rhs = new AST::DivisionBinaryExpression( new AST::ProxyExpression(ve), rhs );
      }
      if( ve->isConstant() )
      {
        delete lhs;
        delete rhs;
        reportError("Left hand side of an assignement expression can't be a constant.", token);
        return 0;
      } else {
        return new AST::AssignementBinaryExpression( ve, rhs );
      }
    }
    delete lhs;
    delete rhs;
    reportError("Left hand side of an assignement expression must be a variable.", token);
    return 0;
    
  } else {
    if( d->compiler )
    {
      std::pair<AST::Expression*, AST::Expression*>  ce = d->compiler->convertCenter()->createConvertExpressions( lhs, rhs );
      if( not ce.first or not ce.second )
      {
        GTL_DEBUG( "Convert from (" << *lhs->type() << ", " << *rhs->type() << " ) failed.");
        reportError( "Can do a binary operation only on two numbers", token );
        if(ce.first)
        {
          delete ce.first;
        } else {
          delete lhs;
        }
        if(ce.second)
        {
          delete ce.second;
        } else {
          delete rhs;
        }
        return 0;
      }
      GTL_DEBUG( "Convert from (" << *lhs->type() << ", " << *rhs->type() << " ) to ( " << *ce.first->type() << ", " << *ce.second->type() << " )" );
      lhs = ce.first;
      rhs = ce.second;
    }
    GTL_ASSERT( lhs->type() == rhs->type() );
    switch( token.type )
    {
      case Token::OR:
        return new AST::OrBinaryExpression( lhs, rhs );
      case Token::AND:
        return new AST::AndBinaryExpression( lhs, rhs );
      case Token::BITOR:
        return new AST::BitOrBinaryExpression( lhs, rhs );
      case Token::BITXOR:
        return new AST::BitXorBinaryExpression( lhs, rhs );
      case Token::BITAND:
        return new AST::BitAndBinaryExpression( lhs, rhs );
      case Token::EQUALEQUAL:
        return new AST::EqualEqualBinaryExpression( lhs, rhs );
      case Token::DIFFERENT:
        return new AST::DifferentBinaryExpression( lhs, rhs );
      case Token::INFERIOREQUAL:
        return new AST::InferiorEqualBinaryExpression( lhs, rhs );
      case Token::INFERIOR:
        return new AST::InferiorBinaryExpression( lhs, rhs );
      case Token::SUPPERIOREQUAL:
        return new AST::SupperiorEqualBinaryExpression( lhs, rhs );
      case Token::SUPPERIOR:
        return new AST::SupperiorBinaryExpression( lhs, rhs );
      case Token::RIGHTSHIFT:
        return new AST::RightShiftBinaryExpression( lhs, rhs );
      case Token::LEFTSHIFT:
        return new AST::LeftShiftBinaryExpression( lhs, rhs );
      case Token::PLUS:
        return new AST::AdditionBinaryExpression( lhs, rhs );
      case Token::MINUS:
        return new AST::SubstractionBinaryExpression( lhs, rhs );
      case Token::MULTIPLY:
        return new AST::MultiplicationBinaryExpression( lhs, rhs );
      case Token::DIVIDE:
        return new AST::DivisionBinaryExpression( lhs, rhs );
      case Token::MODULO:
      {
        if( lhs->type() != rhs->type() or rhs->type()  != Type::Integer32 )
        {
          reportError( "'%' operator only work with integer" , token );
          delete lhs;
          delete rhs;
          return 0;
        }
        return new AST::ModuloBinaryExpression( lhs, rhs );
      }
      default:
      {
        delete lhs;
        delete rhs;
        GTL_ABORT("Unknown operator: " << Token::typeToString( token.type ) );
        return 0;
      }
    }
  }
}

bool ParserBase::isType( const Token& token )
{
  return token.isNativeType() or ( token.type == Token::IDENTIFIER and d->compiler->typesManager()->d->isKnownType( token.string ) );
}

std::list<int> ParserBase::expressionsListToIntegersList( const std::list< AST::Expression* >& list )
{
  std::list<int> integersList;
  CodeGenerator cg( 0 );
  llvm::LLVMContext context;
  GenerationContext gc( &cg, &context, 0, 0, 0);
  for( std::list< AST::Expression* >::const_iterator it = list.begin();
       it != list.end(); ++it)
  {
    if( *it )
    {
      if( (*it)->isConstant() )
      {
        ExpressionGenerationContext egc(0);
        llvm::ConstantInt* v = dynamic_cast<llvm::ConstantInt* >( (*it)->generateValue( gc, egc ).constant() );
        GTL_ASSERT( v );
        integersList.push_back( v->getValue().getLimitedValue() );
      } else {
        reportError( "Expected constant expression.", d->currentToken );
      }
    } else {
      integersList.push_back( -1 );
    }
    delete *it;
  }
  return integersList;
}

void ParserBase::setNameSpace( const String& _nameSpace )
{
  d->nameSpace = _nameSpace;
  d->variablesManager.setNameSpace( _nameSpace );
}

const String& ParserBase::nameSpace() const
{
  return d->nameSpace;
}

TypesManager* ParserBase::typesManager()
{
  return d->compiler->typesManager();
}

VariablesManager* ParserBase::variablesManager()
{
  return &d->variablesManager;
}

int ParserBase::vectorMemberNameToIdx( const GTLCore::String& _name )
{
  GTL_DEBUG(_name);
  if( _name == "x" or _name == "r" ) {
    return 0;
  } else if( _name == "y" or _name == "g"  ) {
    return 1;
  } else if( _name == "z" or _name == "b"  ) {
    return 2;
  } else if( _name == "w" or _name == "a"  ) {
    return 3;
  }
  GTL_DEBUG("unexpected");
  reportUnexpected( d->currentToken );
  return -1;
}

int ParserBase::structMemberNameToIdx( const GTLCore::Type* _type, const GTLCore::String& _name )
{
  int index = _type->d->memberToIndex( _name );
  if( index == -1 )
  {
    reportError("Unknown member: '" + _name + "' for structure " + _type->structName(), d->currentToken );
  }
  return index;
}

void ParserBase::startParsingFunction( const GTLCore::String& _name )
{
}

void ParserBase::endParsingFunction( const GTLCore::String& _name )
{
}
