#include "jdxfilter.h"
#include "jdxnumbers.h"


class Gauss : public JDXfunctionPlugIn {

  JDXdouble filterwidth;

 public:
  Gauss() : JDXfunctionPlugIn("Gauss")  {

    filterwidth=0.36169; filterwidth.set_minmaxval(0.1,1.0);

    append_member(filterwidth,"FilterWidth");
  }

 private:

  float calculate_filter (float rel_kradius) const {
    float factor = (-1) * log (0.5) * secureInv( (filterwidth * filterwidth)) ;
    if(rel_kradius<0.0) rel_kradius=0.0;
    return exp(-(rel_kradius*rel_kradius)*factor);
  }

  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new Gauss;}
};


///////////////////////////////////////////////////////////



class NoFilter : public JDXfunctionPlugIn {

 public:

  NoFilter() : JDXfunctionPlugIn("NoFilter") {}

  float calculate_filter (float rel_kradius) const {
    if(rel_kradius>1.0) return 0.0;
    return 1.0;
  }

  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new NoFilter;}
};

///////////////////////////////////////////////////////////

class Triangle : public JDXfunctionPlugIn {

 public:

  Triangle() : JDXfunctionPlugIn("Triangle") { }

  float calculate_filter (float rel_kradius) const {
    if(rel_kradius<0.0) rel_kradius=0.0;
    if(rel_kradius>1.0) rel_kradius=1.0;
    return 1.0-rel_kradius;
  }

  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new Triangle;}
};

///////////////////////////////////////////////////////////

class Hann : public JDXfunctionPlugIn {

 public:

  Hann() : JDXfunctionPlugIn("Hann") { }

  float calculate_filter (float rel_kradius) const {
    if(rel_kradius<0.0) rel_kradius=0.0;
    if(rel_kradius>1.0) rel_kradius=1.0;
    return 0.5*(1.0+cos(rel_kradius*PII));
  }

  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new Hann;}
};


///////////////////////////////////////////////////////////

class Hamming : public JDXfunctionPlugIn {

 public:

  Hamming() : JDXfunctionPlugIn("Hamming") { }

  float calculate_filter (float rel_kradius) const {
    if(rel_kradius<0.0) rel_kradius=0.0;
    if(rel_kradius>1.0) rel_kradius=1.0;
	 return (0.53836+0.46164*cos(rel_kradius*PII));
  }

  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new Hamming;}
};

///////////////////////////////////////////////////////////

class CosSq : public JDXfunctionPlugIn {

 public:

  CosSq() : JDXfunctionPlugIn("CosSq") { }

 private:
  float calculate_filter (float rel_kradius) const {
    if(rel_kradius<0.0) rel_kradius=0.0;
    if(rel_kradius>1.0) rel_kradius=1.0;

    double value=pow(cos(rel_kradius*PII/2.0),2.0);
    return (value);
  }

  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new CosSq;}
};

///////////////////////////////////////////////////////////

class Blackman : public JDXfunctionPlugIn {

 public:

  Blackman() : JDXfunctionPlugIn("Blackman") { }

  float calculate_filter (float rel_kradius) const {
    if(rel_kradius<0.0) rel_kradius=0.0;
    if(rel_kradius>1.0) rel_kradius=1.0;
    return 0.42+0.5*cos(rel_kradius*PII)+0.08*cos(2.0*rel_kradius*PII);
  }

  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new Blackman;}
};

///////////////////////////////////////////////////////////

class BlackmanNuttall : public JDXfunctionPlugIn {

 public:

  BlackmanNuttall() : JDXfunctionPlugIn("BlackmanNuttall") {  }


 private:

  float calculate_filter (float rel_kradius) const {

    if(rel_kradius<0.0) rel_kradius=0.0;
    if(rel_kradius>1.0) rel_kradius=1.0;

    float step = PII*0.5*rel_kradius;
    float a0 = 0.3635819;
    float a1 = 0.4891775;
    float a2 = 0.1365995;
    float a3 = 0.0106411;
    double value=a0+a1*cos(2.0*step)+a2*cos(4.0*step)+a3*cos(6.0*step);
    return (value);
  }
  // implementing virtual function of JDXfunctionPlugIn
  JDXfunctionPlugIn* clone() const {return new BlackmanNuttall;}
};


///////////////////////////////////////////////////////////


/*
class Fermi : public JDXfunctionTemplate<Fermi> {

 public:

  Fermi() : JDXfunctionTemplate<Fermi>("Fermi") { }

  float calculate (float knorm, float kernel) const {
    float dummy=(knorm*kernel*2.0-8.0);
    return 1.0/(exp(dummy)+1.0);
  }
};
*/

///////////////////////////////////////////////////////////


void JDXfilter::init_static() {

  (new Gauss)->register_function(filterFunc, funcMode(0));
  (new NoFilter)->register_function(filterFunc, funcMode(0));
  (new Triangle)->register_function(filterFunc, funcMode(0));
  (new Hann)->register_function(filterFunc, funcMode(0));
  (new Hamming)->register_function(filterFunc, funcMode(0));
  (new CosSq)->register_function(filterFunc, funcMode(0));
  (new Blackman)->register_function(filterFunc, funcMode(0));
  (new BlackmanNuttall)->register_function(filterFunc, funcMode(0));
//  (new Fermi)->register_function(filterFunc, funcMode(0));

}

void JDXfilter::destroy_static() {
  // pulgins will be deleted by JDXfunction
}

EMPTY_TEMPL_LIST bool StaticHandler<JDXfilter>::staticdone=false;

