## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_mesh(@var{x},@var{y},@var{z},@var{c},[@var{prop},@var{val}])
##
## Displays a mesh plot defined by @var{x},@var{y},@var{z} matrices,
## where @var{x} and @var{y} are typically formed by meshgrid.
## @var{c} is an optional matrix to give a color scale.
##
## Optional @var{prop},@var{val} arguments can be used to change
## properties of the plot.  Currently, valid properties are
## ";BallRadius;LineRadius;Fancy;".
## 
## Example: Triaxial Teardrop
## @example
## [u,v] = meshgrid(0.01:pi/20:pi,0:pi/20:2*pi);
## x = ( 1 - cos(u) ) .* cos( u + 2*pi/3 ) .* cos( v + 2*pi/3 ) / 2; 
## y = ( 1 - cos(u) ) .* cos( u + 2*pi/3 ) .* cos( v - 2*pi/3 ) / 2; 
## z = cos( u - 2*pi/3 ); 
## vtk_mesh(x,y,z,'Fancy',1);
## @end example
## 
## @end deftypefn
## @seealso{vtk_trimesh, vtk_surf}

## Author: Dragan Tubic

function vtk_mesh( varargin )
  
  valid_props = ";BallRadius;LineRadius;Fancy;";
  
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, varargin{:});
  
  if ( no_numerical_params < 1 | no_numerical_params > 4 )
    error("Syntax is vtk_mesh( x, (y), (z), (c) )");
  end
  
  if ( no_numerical_params == 1 )
    z = nth (varargin,1);
    [nr nc] = size(z);
    [x y] = meshgrid(1:nr,1:nc);
    c = z;
  elseif ( no_numerical_params == 2 )
    z = nth (varargin,1);
    [nr nc] = size(z);
    [x y] = meshgrid(1:nr,1:nc);
    c = nth (varargin,1);
  elseif ( no_numerical_params == 3 )
    x = nth (varargin,1);
    y = nth (varargin,2);
    z = nth (varargin,3);
    c = z;
  elseif ( no_numerical_params == 4 )
    x = nth (varargin,1);
    y = nth (varargin,2);
    z = nth (varargin,3);
    c = nth (varargin,4);
  end
  
  [nr nc] = size(x);
  
  if ( nr < 2 | nc < 2 )
    error("input has to be at least 2xN or Nx2 matrix");
  end	
  
  m=nc; n=nr;
  i=0:m-2;
  j=0:n-2;
  [i,j]=meshgrid(i,j);
  i=i(:); j=j(:);
  ##t=[i*n+j,i*n+(j+1),(i+1)*n+(j+1) ; (i+1)*n+(j+1),(i+1)*n+j,i*n+j];  %1; one way to construct the triangles
  t=[i*n+j,(i+1)*n+j,(i+1)*n+(j+1) ; (i+1)*n+(j+1),i*n+(j+1),i*n+j];  %2; another way, seems a bit better
  if ( first_prop_index > 0 )
    f=vtk_trimesh(t+1,x(:),y(:),z(:),c(:),varargin{first_prop_index:length(varargin)});
  else
    f=vtk_trimesh(t+1,x(:),y(:),z(:),c(:));
  end
  
endfunction
