#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright(C) 2007-2008 INL
Written by Romain Bignon <romain AT inl.fr>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

$Id: table.py 12192 2008-01-14 16:48:27Z romain $
"""

from nevow import rend, loaders, tags#, static
from nevow.i18n import _, render as i18nrender

import os
from tools import args2url, Args, i2state
import time
from sessions import getSession
from frags import BaseFragment
from copy import copy

mypath = os.path.dirname(__file__) + os.sep

class TableFragment(BaseFragment):
    """ This is a rend.Fragment of a webpage, with is own XML template. """

    docFactory = loaders.xmlfile(mypath + 'xml/table.xml', ignoreDocType=True)

    render_i18n = i18nrender()

    def __init__(self, name, function, title, urlbase, args, icon='', switch=False, frag=None):
        """ Init.
            @param function [string] this is the name of the table (TODO: rename?)
            @param title [string] title of table.
            @param icon [string] if specified, a picture will be showed before title name. (file in img/)
            @param multitable [bool] if True, we will use multitable arguments formats (Tablename_arg)
        """
        BaseFragment.__init__(self, function, urlbase, args, frag)
        self.title = title
        self.name = name
        self.icon = icon
        self.switch = switch
        self.total_time = time.time()
        self.fraglink = 'frag' + '/' + self.name + '/'

    #################
    #    RENDER     #
    #################

    def render_gettable(self, context, data):
        """ First function call after data_table. """

        if hasattr(data, 'table'):
            self.table = data
        elif isinstance(data, list) and hasattr(data[0], 'table'):
            self.table = data[0]
            data = data[0]
        elif isinstance(data, str):
            print 'Error: %s' % data
            self.error = data

        return ''

    def render_table(self, context, data):
        """ Used by XML template after call data_table (defined in BaseFragment). """

        if not hasattr(self, "table"):
            if hasattr(self, "error"):
                return self.error
            else:
                return _('Unable to find this table')

        if not self.table.table:
            return tags.div(_class='noinfo')[tags.p[_('No data')]]

        # This function create patterns.
        return BaseFragment.render_sequence(self, context, self.table)

    def render_icon(self, context, data):

        """ Table's icon """
        if self.icon:
            return context.tag.clear()[tags.img(src=("img/" + self.icon), alt="icon")]
        else:
            return ''

    def render_title(self, context, data):
        """ Table's title. """

        if not hasattr(self, 'table') or not hasattr(self.table, 'table'):
            return _('Error')

        if not self.title:

            # If there isn't title setted, we build it by args.
            self.title = str()

            myargs = self.args

            # Get label
            self.title = Args(myargs).labels(context, self.function)

        if not self.title:
            self.title = self.function

        return context.tag.clear()[self.title]

    def render_toolbar(self, context, data):

        if not hasattr(self, 'table'):
            return ''

        session = getSession(context)

        render = []

        render += [tags.div(_class='toolbar_btn')[tags.a(href='csv/%s' % self.name + args2url(self.args, tiny=False))[tags.img(src='img/csv.png')]]]

        if self.table.args.has_key('limit'):
            limit = self.table.args['limit']

            limits = [5,10,20,30]

            options = []

            if not limit in limits:
                options += [tags.option(value=limit, selected='')[limit]]
            for n in limits:
                if n == limit:
                    options += [tags.option(value=n, selected='')[n]]
                else:
                    options += [tags.option(value=n)[n]]

            render += [tags.div(_class='toolbar_btn')[tags.select(_class='chlimit')[options]]]

        render += [tags.div(_class='toolbar_btn')[
                        tags.a(_class='load', href=self.fraglink + args2url(self.args))[tags.img(src='img/reload.png')]
                    ]]

        try:
            # To switch to other views (graphes), there must be at least two columns,
            # and second column must be an integer.
            int(self.table.table[0][1])
            self.switch = True
        except Exception:
            self.switch = False

        if self.switch:
            # If self.switch == True, I can switch on render, from
            # table to histogram or pie.

            # First we get actual render label.
            myargs = self.args

            my_render = 'table'
            if myargs.has_key('~render'):
                my_render = myargs['~render']

            renders = ['table', 'histo', 'pie']
            renders.remove(my_render)

            # Then we show a link on an icon to other renders
            render += [tags.div(_class='toolbar_btn')[tags.a(_class='load', href=self.fraglink + args2url(self.args, **{'~render': render}))
                            [tags.img(alt=render, src='img/%s.png' % render)]]
                        for render in renders]

        if self.args.has_key('~showstate'):
            args = self.args

            # If state is opened established, we show an other menu.
            if (args.has_key('state') and (isinstance(args['state'], int) or isinstance(args['state'], str) and args['state'].isdigit())
                and int(args['state']) in [1,2,4]):
                render += [
                        tags.table[
                            tags.tr[
                                tags.td[
                                    tags.a(_class='load', href=self.fraglink + args2url(args, state=-1))[_('All')]
                                ]
                            ],
                            tags.tr(**{'class': 'open'})[
                                tags.td[
                                    tags.a(_class='load', href=self.fraglink + args2url(args, state=1))[_('Opened')]
                                ]
                            ],
                            tags.tr(**{'class': 'established'})[
                                tags.td[
                                    tags.a(_class='load', href=self.fraglink + args2url(args, state=2))[_('Established')]
                                ]
                            ]
                        ]
                    ]
            else:
                render += [
                        tags.table[
                            tags.tr[
                                tags.td[
                                    tags.a(_class='load', href=self.fraglink + args2url(args, state=-1))[_('All')]
                                ]
                            ],
                            tags.tr(**{'class': 'drop'})[
                                tags.td[
                                    tags.a(_class='load', href=self.fraglink + args2url(args, state=0))[_('Dropped')]
                                ]
                            ],
                            tags.tr(**{'class': 'open'})[
                                tags.td[
                                    tags.a(_class='load', href=self.fraglink + args2url(args, state=4))[_('Active connection tracking')]
                                ]
                            ],
                            tags.tr(**{'class': 'close'})[
                                tags.td[
                                    tags.a(_class='load', href=self.fraglink + args2url(args, state=3))[_('Closed')]
                                ]
                            ]
                        ]
                    ]

        return context.tag.clear()[render]

    def render_buttons(self, context, data):

        if not hasattr(self, 'table'):
            return ''

        session = getSession(context)

        render = []

        try:
            page = session.getIndexPage()
            image = None
            for side in page.frags:
                if self.name in side:
                    image = tags.td(_class='buttons')[tags.a(_class='cmd', href='cmd/session/remfrag/?page=%s&frag=%s' % (page.name, self.name))[
                                    tags.img(src='img/remove_g.png', alt='remove')
                                ]
			    ]

            if not image:
                image =  tags.td(_class='buttons')[tags.a(_class='cmd', href='cmd/session/addfrag/' + args2url(self.args, page=page.name, frag=self.name))[
                                tags.img(src='img/bookmark_g.png', alt='bookmark')
                            ]
			 ]

            if image:
                render += [image]

        except:
            pass

        return context.tag.clear()[render]

    def _get_head(self, head):
        """
             @param head [string]] column name
        """

        args = self.args
        table = self.table

        new_args = {}
        new_args['sortby'] = head
        if table.args['sortby'] == head and table.args['sort'] != 'ASC':
            new_args['sort'] = 'ASC'
        else:
            new_args['sort'] = 'DESC'

        content = []
        try:
            content += [Args().arg_types[head].label]
        except:
            content += [head]

        if table.args['sortby'] == head:
            content += [tags.img(alt=_('Sort'),
                                 width=20,
                                 height=15,
                                 valign="bottom",
                                 src= (table.args['sort'] == 'ASC' and 'img/up.gif' or 'img/down.gif')
                                )]

        return tags.a(_class='load', href=self.fraglink + args2url(args, **new_args))[content]

    def render_head(self, context, data):
        """ This is the column titles. """

        if not hasattr(self, "table"):
            return ''

        table = self.table

        # TODO: I want to *CLEAN* this code!
        return context.tag.clear()[[tags.td[self._get_head(head)] for head in table.columns]]

    def _get_cell_link(self, column, label, value):
        """
            @param column [string] column name
            @param label [string] label (shown in column)
            @param value [string] value (used for link)
        """

        table = self.table

        link = self.frag.getLink(column)

        if not link:
            return None

        args = copy(link.args)
        for key, arg in args.items():
            if arg == link.CellValue:
                args[key] = value
            elif arg == link.CellLabel:
                args[key] = label

        lastargs = Args(self.args).filters()

        args = args2url(lastargs, **args)
        return link.page + args

    def _render_cell(self, case, data, a):
        """ Function which returns the content to put in cell. """

        table = self.table

        if isinstance(data[case], tuple):
            label = data[case][0]
            value = data[case][1]
        else:
            label = data[case]
            value = data[case]

        content = a.get_data(table.columns[case], label)

        if self.frag.getLink(table.columns[case]) is not None:
            return tags.a(href=self._get_cell_link(table.columns[case], label, value))[content]
        else:
            return content

    def render_entry(self, context, data):
        """ Print an entry.
            @param data [list] this is the list of args on this line
        """

        if not hasattr(self, "table"):
            return ''

        table = self.table

        args = self.args

        a = Args(self.table.args)

        tr_params = {}
        if isinstance(data[0], tuple):
            pk_id = data[0][1]
        else:
            pk_id = data[0]

        if hasattr(table, 'states') and table.states.has_key(pk_id):
            tr_params['class'] = i2state(table.states[pk_id])

        # TODO: I want to *CLEAN* this code!
        return loaders.stan(
            tags.tr(**tr_params)[
                [
                    (case < len(data) and (data[case] is not None)
                        and tags.td[self._render_cell(case, data, a)]
                    or tags.td[''])
                 for case in xrange(len(table.columns))]
            ])

    def render_prevnext(self, ctx, data):
        """ Print previous and next links at the bottom of table. """

        if not hasattr(self, "table"):
            return ''

        if not self.table.args.has_key('limit') or self.table.args['limit'] < 0:
            return ctx.tag.clear()[tags.td(), tags.td(), tags.td()]

        args = self.args

        start = self.table.args['start']
        end = start + self.table.args['limit'] - 1

        before = start - self.table.args['limit']
        after = start + self.table.args['limit']

        render = []

        if start == 0:
            render += [tags.td(width='33%', align="left")]
        else:
            if before < 0:
                before = 0
            render += [tags.td(width='33%', align="left")[
                        tags.a(_class='load',
                               href=self.fraglink + args2url(args, **{'start': before}))[_('prev')]
                      ]]

        if not hasattr(self, "table"):
            return ''

        render += [tags.td(width='33%', _class="range")["[%d->%d]" % (start, end)]]

        if self.table.args['limit'] > len(self.table.table):
            render += [tags.td(width='33%', align="right")]
        else:
            render += [tags.td(width='33%', align="right")[
                        tags.a(_class='load',
                               href=self.fraglink + args2url(args, **{'start': after }))[_('next')]
                      ]]

        render += [tags.comment['Time: %1.3fs' % (time.time() - self.total_time)]]

        return ctx.tag.clear()[render]
