// -*- indent-tabs-mode: nil -*-

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <algorithm>

#include <arc/StringConv.h>
#include <arc/ArcConfig.h>
#include <arc/client/Broker.h>
#include <arc/client/ExecutionTarget.h>
#include <arc/client/Job.h>
#include <arc/credential/Credential.h>
#include <arc/loader/FinderLoader.h>

namespace Arc {

  Logger Broker::logger(Logger::getRootLogger(), "Broker");
  Logger BrokerPlugin::logger(Logger::getRootLogger(), "BrokerPlugin");
  Logger ExecutionTargetSet::logger(Logger::getRootLogger(), "ExecutionTargetSet");
  
  BrokerPluginLoader Broker::l;

  BrokerPluginLoader::BrokerPluginLoader() : Loader(BaseConfig().MakeConfig(Config()).Parent()) {}

  BrokerPluginLoader::~BrokerPluginLoader() {
    for (std::list<BrokerPlugin*>::iterator it = plugins.begin();
         it != plugins.end(); ++it) {
      delete *it;
    }
  }
  
  BrokerPlugin* BrokerPluginLoader::load(const UserConfig& uc, const JobDescription* j, const std::string& name, bool keep_ownership) {
    BrokerPluginArgument arg(uc);
    if (name.empty()) {
      BrokerPlugin* p = new BrokerPlugin(&arg);
      if (!p) {
        return NULL;
      }
      if (j) {
        p->set(*j);
      }
      
      if (keep_ownership) {
        plugins.push_back(p);
      }
      return p;
    }

    if(!factory_->load(FinderLoader::GetLibrariesList(), "HED:BrokerPlugin", name)) {
      logger.msg(ERROR, "Broker plugin \"%s\" not found.", name);
      return NULL;
    }

    BrokerPlugin *p = factory_->GetInstance<BrokerPlugin>("HED:BrokerPlugin", name, &arg, false);

    if (!p) {
      logger.msg(ERROR, "Broker %s could not be created", name);
      return NULL;
    }

    if (j) {
      p->set(*j);
    }
    if (keep_ownership) {
      plugins.push_back(p);
    }
    logger.msg(INFO, "Loaded Broker %s", name);
    return p;
  }

  Broker::Broker(const UserConfig& uc, const JobDescription& j, const std::string& name) : uc(uc), j(&j), p(l.load(uc, j, name, false)) {
    Credential credential(uc);
    proxyDN = credential.GetDN();
    proxyIssuerCA = credential.GetCAName();
  }

  Broker::Broker(const UserConfig& uc, const std::string& name) : uc(uc), j(NULL), p(l.load(uc, name, false)) {
    Credential credential(uc);
    proxyDN = credential.GetDN();
    proxyIssuerCA = credential.GetCAName();
  }

  bool Broker::match(const ExecutionTarget& t) const {
    logger.msg(VERBOSE, "Performing matchmaking against target (%s).", t.ComputingEndpoint->URLString);

    if ( !(t.ComputingEndpoint->TrustedCA.empty()) && (find(t.ComputingEndpoint->TrustedCA.begin(), t.ComputingEndpoint->TrustedCA.end(), proxyIssuerCA)
            == t.ComputingEndpoint->TrustedCA.end()) ){
        logger.msg(VERBOSE, "The CA issuer (%s) of the credentials (%s) is not trusted by the target (%s).", proxyIssuerCA, proxyDN, t.ComputingEndpoint->URLString);
        return false;
    }

    std::map<std::string, std::string>::const_iterator itAtt;
    if ((itAtt = j->OtherAttributes.find("nordugrid:broker;reject_queue")) != j->OtherAttributes.end()) {
      if (t.ComputingShare->Name.empty()) {
        logger.msg(VERBOSE, "ComputingShareName of ExecutionTarget (%s) is not defined", t.ComputingEndpoint->URLString);
        return false;
      }

      if (t.ComputingShare->Name == itAtt->second) {
        logger.msg(VERBOSE, "ComputingShare (%s) explicitly rejected", itAtt->second);
        return false;
      }
    }

    if (!j->Resources.QueueName.empty()) {
      if (t.ComputingShare->Name.empty()) {
        logger.msg(VERBOSE, "ComputingShareName of ExecutionTarget (%s) is not defined", t.ComputingEndpoint->URLString);
        return false;
      }

      if (t.ComputingShare->Name != j->Resources.QueueName) {
        logger.msg(VERBOSE, "ComputingShare (%s) does not match selected queue (%s)", t.ComputingShare->Name, j->Resources.QueueName);
        return false;
      }
    }

    if ((int)j->Application.ProcessingStartTime.GetTime() != -1) {
      if ((int)t.ComputingEndpoint->DowntimeStarts.GetTime() != -1 && (int)t.ComputingEndpoint->DowntimeEnds.GetTime() != -1) {
        if (t.ComputingEndpoint->DowntimeStarts <= j->Application.ProcessingStartTime && j->Application.ProcessingStartTime <= t.ComputingEndpoint->DowntimeEnds) {
          logger.msg(VERBOSE, "ProcessingStartTime (%s) specified in job description is inside the targets downtime period [ %s - %s ].", (std::string)j->Application.ProcessingStartTime, (std::string)t.ComputingEndpoint->DowntimeStarts, (std::string)t.ComputingEndpoint->DowntimeEnds);
          return false;
        }
      }
      else
        logger.msg(WARNING, "The downtime of the target (%s) is not published. Keeping target.", t.ComputingEndpoint->URLString);
    }

    if (!t.ComputingEndpoint->HealthState.empty()) {

      if (t.ComputingEndpoint->HealthState != "ok") { // Enumeration for healthstate: ok, critical, other, unknown, warning
        logger.msg(VERBOSE, "HealthState of ExecutionTarget (%s) is not OK (%s)", t.ComputingEndpoint->URLString, t.ComputingEndpoint->HealthState);
        return false;
      }
    }
    else {
      logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, HealthState is not defined", t.ComputingEndpoint->URLString);
      return false;
    }

    if (!j->Resources.CEType.empty()) {
      if (!t.ComputingEndpoint->Implementation().empty()) {
        if (!j->Resources.CEType.isSatisfied(t.ComputingEndpoint->Implementation)) {
          logger.msg(VERBOSE, "Matchmaking, Computing endpoint requirement not satisfied. ExecutionTarget: %s", (std::string)t.ComputingEndpoint->Implementation);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, ImplementationName is not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    {
      typedef std::pair<std::string, int> EtTimePair;
      EtTimePair etTime[] = {EtTimePair("MaxWallTime", (int)t.ComputingShare->MaxWallTime.GetPeriod()),
                             EtTimePair("MinWallTime", (int)t.ComputingShare->MinWallTime.GetPeriod()),
                             EtTimePair("MaxCPUTime", (int)t.ComputingShare->MaxCPUTime.GetPeriod()),
                             EtTimePair("MinCPUTime", (int)t.ComputingShare->MinCPUTime.GetPeriod())};

      typedef std::pair<std::string, const ScalableTime<int>*> JobTimePair;
      JobTimePair jobTime[] = {JobTimePair("TotalWallTime", &j->Resources.TotalWallTime),
                               JobTimePair("TotalCPUTime", &j->Resources.TotalCPUTime)};

      int i = 0;
      for (; i < 4; i++) {
        JobTimePair *jTime = &jobTime[i/2];
        if (((i%2 == 0) && (jTime->second->range.max != -1)) ||
            ((i%2 == 1) && (jTime->second->range.min != -1))) {
          if (etTime[i].second != -1) {
            if (jTime->second->benchmark.first.empty()) { // No benchmark defined, do not scale.
              if (((i%2 == 0) && (jTime->second->range.max > etTime[i].second)) ||
                  ((i%2 == 1) && (jTime->second->range.min < etTime[i].second))) {
                logger.msg(VERBOSE,
                           "Matchmaking, %s (%d) is %s than %s (%d) published by the ExecutionTarget.",
                           jTime->first,
                           (i%2 == 0 ? jTime->second->range.max
                                     : jTime->second->range.min),
                           (i%2 == 0 ? "greater" : "less"),
                           etTime[i].first,
                           etTime[i].second);
                return false;
              }
            }
            else { // Benchmark defined => scale using benchmark.
              double targetBenchmark = -1.;
              for (std::map<std::string, double>::const_iterator itTBench = t.Benchmarks->begin();
                   itTBench != t.Benchmarks->end(); itTBench++) {
                if (lower(jTime->second->benchmark.first) == lower(itTBench->first)) {
                  targetBenchmark = itTBench->second;
                  break;
                }
              }

              // Make it possible to scale according to clock rate.
              if (targetBenchmark <= 0. && lower(jTime->second->benchmark.first) == "clock rate") {
                targetBenchmark = (t.ExecutionEnvironment->CPUClockSpeed > 0. ? (double)t.ExecutionEnvironment->CPUClockSpeed : 1000.);
              }

              if (targetBenchmark > 0.) {
                if (((i%2 == 0) && (jTime->second->scaleMax(targetBenchmark) > etTime[i].second)) ||
                    ((i%2 == 1) && (jTime->second->scaleMin(targetBenchmark) < etTime[i].second))) {
                  logger.msg(VERBOSE,
                             "Matchmaking, The %s scaled %s (%d) is %s than the %s (%d) published by the ExecutionTarget.",
                             jTime->second->benchmark.first,
                             jTime->first,
                             (i%2 == 0 ? jTime->second->scaleMax(targetBenchmark)
                                       : jTime->second->scaleMin(targetBenchmark)),
                             (i%2 == 0 ? "greater" : "less"),
                             etTime[i].first,
                             etTime[i].second);
                  return false;
                }
              }
              else {
                logger.msg(VERBOSE, "Matchmaking, Benchmark %s is not published by the ExecutionTarget.", jTime->second->benchmark.first);
                return false;
              }
            }
          }
          // Do not drop target if it does not publish attribute.
        }
      }
    }

    if (j->Resources.IndividualPhysicalMemory != -1) {
      if (t.ExecutionEnvironment->MainMemorySize != -1) {     // Example: 678
        if (t.ExecutionEnvironment->MainMemorySize < j->Resources.IndividualPhysicalMemory) {
          logger.msg(VERBOSE, "Matchmaking, MainMemorySize problem, ExecutionTarget: %d (MainMemorySize), JobDescription: %d (IndividualPhysicalMemory)", t.ExecutionEnvironment->MainMemorySize, j->Resources.IndividualPhysicalMemory.max);
          return false;
        }
      }
      else if (t.ComputingShare->MaxMainMemory != -1) {     // Example: 678
        if (t.ComputingShare->MaxMainMemory < j->Resources.IndividualPhysicalMemory) {
          logger.msg(VERBOSE, "Matchmaking, MaxMainMemory problem, ExecutionTarget: %d (MaxMainMemory), JobDescription: %d (IndividualPhysicalMemory)", t.ComputingShare->MaxMainMemory, j->Resources.IndividualPhysicalMemory.max);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget: %s, MaxMainMemory and MainMemorySize are not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (j->Resources.IndividualVirtualMemory != -1) {
      if (t.ComputingShare->MaxVirtualMemory != -1) {     // Example: 678
        if (t.ComputingShare->MaxVirtualMemory < j->Resources.IndividualVirtualMemory) {
          logger.msg(VERBOSE, "Matchmaking, MaxVirtualMemory problem, ExecutionTarget: %d (MaxVirtualMemory), JobDescription: %d (IndividualVirtualMemory)", t.ComputingShare->MaxVirtualMemory, j->Resources.IndividualVirtualMemory.max);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget: %s, MaxVirtualMemory is not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (!j->Resources.Platform.empty()) {
      if (!t.ExecutionEnvironment->Platform.empty()) {    // Example: i386
        if (t.ExecutionEnvironment->Platform != j->Resources.Platform) {
          logger.msg(VERBOSE, "Matchmaking, Platform problem, ExecutionTarget: %s (Platform) JobDescription: %s (Platform)", t.ExecutionEnvironment->Platform, j->Resources.Platform);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, Platform is not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (!j->Resources.OperatingSystem.empty()) {
      if (!t.ExecutionEnvironment->OperatingSystem.empty()) {
        if (!j->Resources.OperatingSystem.isSatisfied(t.ExecutionEnvironment->OperatingSystem)) {
          logger.msg(VERBOSE, "Matchmaking, ExecutionTarget: %s, OperatingSystem requirements not satisfied", t.ComputingEndpoint->URLString);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s,  OperatingSystem is not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (!j->Resources.RunTimeEnvironment.empty()) {
      if (!t.ApplicationEnvironments->empty()) {
        if (!j->Resources.RunTimeEnvironment.isSatisfied(*t.ApplicationEnvironments)) {
          logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, RunTimeEnvironment requirements not satisfied", t.ComputingEndpoint->URLString);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget: %s, ApplicationEnvironments not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (!j->Resources.NetworkInfo.empty())
      if (!t.ComputingManager->NetworkInfo.empty()) {    // Example: infiniband
        if (std::find(t.ComputingManager->NetworkInfo.begin(), t.ComputingManager->NetworkInfo.end(),
                      j->Resources.NetworkInfo) == t.ComputingManager->NetworkInfo.end()) {
          logger.msg(VERBOSE, "Matchmaking, NetworkInfo demand not fulfilled, ExecutionTarget do not support %s, specified in the JobDescription.", j->Resources.NetworkInfo);
          return false;
        }
        else {
          logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, NetworkInfo is not defined", t.ComputingEndpoint->URLString);
          return false;
        }
      }

    if (j->Resources.DiskSpaceRequirement.SessionDiskSpace > -1) {
      if (t.ComputingShare->MaxDiskSpace > -1) {     // Example: 5656
        if (t.ComputingShare->MaxDiskSpace*1024 < j->Resources.DiskSpaceRequirement.SessionDiskSpace) {
          logger.msg(VERBOSE, "Matchmaking, MaxDiskSpace problem, ExecutionTarget: %d MB (MaxDiskSpace); JobDescription: %d MB (SessionDiskSpace)", t.ComputingShare->MaxDiskSpace*1024, j->Resources.DiskSpaceRequirement.SessionDiskSpace);
          return false;
        }
      }

      if (t.ComputingManager->WorkingAreaFree > -1) {     // Example: 5656
        if (t.ComputingManager->WorkingAreaFree*1024 < j->Resources.DiskSpaceRequirement.SessionDiskSpace) {
          logger.msg(VERBOSE, "Matchmaking, WorkingAreaFree problem, ExecutionTarget: %d MB (WorkingAreaFree); JobDescription: %d MB (SessionDiskSpace)", t.ComputingManager->WorkingAreaFree*1024, j->Resources.DiskSpaceRequirement.SessionDiskSpace);
          return false;
        }
      }

      if (t.ComputingShare->MaxDiskSpace <= -1 && t.ComputingManager->WorkingAreaFree <= -1) {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, MaxDiskSpace and WorkingAreaFree are not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (j->Resources.DiskSpaceRequirement.DiskSpace.max > -1 && j->Resources.DiskSpaceRequirement.CacheDiskSpace > -1) {
      if (t.ComputingShare->MaxDiskSpace > -1) {     // Example: 5656
        if (t.ComputingShare->MaxDiskSpace*1024 < j->Resources.DiskSpaceRequirement.DiskSpace.max - j->Resources.DiskSpaceRequirement.CacheDiskSpace) {
          logger.msg(VERBOSE, "Matchmaking, MaxDiskSpace*1024 >= DiskSpace - CacheDiskSpace problem, ExecutionTarget: %d MB (MaxDiskSpace); JobDescription: %d MB (DiskSpace), %d MB (CacheDiskSpace)", t.ComputingShare->MaxDiskSpace*1024, j->Resources.DiskSpaceRequirement.DiskSpace.max, j->Resources.DiskSpaceRequirement.CacheDiskSpace);
          return false;
        }
      }

      if (t.ComputingManager->WorkingAreaFree > -1) {     // Example: 5656
        if (t.ComputingManager->WorkingAreaFree*1024 < j->Resources.DiskSpaceRequirement.DiskSpace.max - j->Resources.DiskSpaceRequirement.CacheDiskSpace) {
          logger.msg(VERBOSE, "Matchmaking, WorkingAreaFree*1024 >= DiskSpace - CacheDiskSpace problem, ExecutionTarget: %d MB (MaxDiskSpace); JobDescription: %d MB (DiskSpace), %d MB (CacheDiskSpace)", t.ComputingManager->WorkingAreaFree*1024, j->Resources.DiskSpaceRequirement.DiskSpace.max, j->Resources.DiskSpaceRequirement.CacheDiskSpace);
          return false;
        }
      }

      if (t.ComputingShare->MaxDiskSpace <= -1 && t.ComputingManager->WorkingAreaFree <= -1) {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, MaxDiskSpace and WorkingAreaFree are not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (j->Resources.DiskSpaceRequirement.DiskSpace.max > -1) {
      if (t.ComputingShare->MaxDiskSpace > -1) {     // Example: 5656
        if (t.ComputingShare->MaxDiskSpace*1024 < j->Resources.DiskSpaceRequirement.DiskSpace.max) {
          logger.msg(VERBOSE, "Matchmaking, MaxDiskSpace problem, ExecutionTarget: %d MB (MaxDiskSpace); JobDescription: %d MB (DiskSpace)", t.ComputingShare->MaxDiskSpace*1024, j->Resources.DiskSpaceRequirement.DiskSpace.max);
          return false;
        }
      }

      if (t.ComputingManager->WorkingAreaFree > -1) {     // Example: 5656
        if (t.ComputingManager->WorkingAreaFree*1024 < j->Resources.DiskSpaceRequirement.DiskSpace.max) {
          logger.msg(VERBOSE, "Matchmaking, WorkingAreaFree problem, ExecutionTarget: %d MB (WorkingAreaFree); JobDescription: %d MB (DiskSpace)", t.ComputingManager->WorkingAreaFree*1024, j->Resources.DiskSpaceRequirement.DiskSpace.max);
          return false;
        }
      }

      if (t.ComputingManager->WorkingAreaFree <= -1 && t.ComputingShare->MaxDiskSpace <= -1) {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, MaxDiskSpace and WorkingAreaFree are not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (j->Resources.DiskSpaceRequirement.CacheDiskSpace > -1) {
      if (t.ComputingManager->CacheTotal > -1) {     // Example: 5656
        if (t.ComputingManager->CacheTotal*1024 < j->Resources.DiskSpaceRequirement.CacheDiskSpace) {
          logger.msg(VERBOSE, "Matchmaking, CacheTotal problem, ExecutionTarget: %d MB (CacheTotal); JobDescription: %d MB (CacheDiskSpace)", t.ComputingManager->CacheTotal*1024, j->Resources.DiskSpaceRequirement.CacheDiskSpace);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, CacheTotal is not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if (j->Resources.SlotRequirement.NumberOfSlots != -1) {
      if (t.ComputingManager->TotalSlots != -1) {     // Example: 5656
        if (t.ComputingManager->TotalSlots < j->Resources.SlotRequirement.NumberOfSlots) {
          logger.msg(VERBOSE, "Matchmaking, TotalSlots problem, ExecutionTarget: %d (TotalSlots) JobDescription: %d (NumberOfProcesses)", t.ComputingManager->TotalSlots, j->Resources.SlotRequirement.NumberOfSlots);
          return false;
        }
      }
      if (t.ComputingShare->MaxSlotsPerJob != -1) {     // Example: 5656
        if (t.ComputingShare->MaxSlotsPerJob < j->Resources.SlotRequirement.NumberOfSlots) {
          logger.msg(VERBOSE, "Matchmaking, MaxSlotsPerJob problem, ExecutionTarget: %d (MaxSlotsPerJob) JobDescription: %d (NumberOfProcesses)", t.ComputingManager->TotalSlots, j->Resources.SlotRequirement.NumberOfSlots);
          return false;
        }
      }

      if (t.ComputingManager->TotalSlots == -1 && t.ComputingShare->MaxSlotsPerJob == -1) {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, TotalSlots and MaxSlotsPerJob are not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if ((int)j->Resources.SessionLifeTime.GetPeriod() != -1) {
      if ((int)t.ComputingManager->WorkingAreaLifeTime.GetPeriod() != -1) {     // Example: 123
        if (t.ComputingManager->WorkingAreaLifeTime < j->Resources.SessionLifeTime) {
          logger.msg(VERBOSE, "Matchmaking, WorkingAreaLifeTime problem, ExecutionTarget: %s (WorkingAreaLifeTime) JobDescription: %s (SessionLifeTime)", (std::string)t.ComputingManager->WorkingAreaLifeTime, (std::string)j->Resources.SessionLifeTime);
          return false;
        }
      }
      else {
        logger.msg(VERBOSE, "Matchmaking, ExecutionTarget:  %s, WorkingAreaLifeTime is not defined", t.ComputingEndpoint->URLString);
        return false;
      }
    }

    if ((j->Resources.NodeAccess == NAT_INBOUND ||
         j->Resources.NodeAccess == NAT_INOUTBOUND) &&
        !t.ExecutionEnvironment->ConnectivityIn) {     // Example: false (boolean)
      logger.msg(VERBOSE, "Matchmaking, ConnectivityIn problem, ExecutionTarget: %s (ConnectivityIn) JobDescription: %s (InBound)", (j->Resources.NodeAccess == NAT_INBOUND ? "INBOUND" : "INOUTBOUND"), (t.ExecutionEnvironment->ConnectivityIn ? "true" : "false"));
      return false;
    }

    if ((j->Resources.NodeAccess == NAT_OUTBOUND ||
         j->Resources.NodeAccess == NAT_INOUTBOUND) &&
        !t.ExecutionEnvironment->ConnectivityOut) {     // Example: false (boolean)
      logger.msg(VERBOSE, "Matchmaking, ConnectivityOut problem, ExecutionTarget: %s (ConnectivityOut) JobDescription: %s (OutBound)", (j->Resources.NodeAccess == NAT_OUTBOUND ? "OUTBOUND" : "INOUTBOUND"), (t.ExecutionEnvironment->ConnectivityIn ? "true" : "false"));
      return false;
    }

    bool plugin_match = p?p->match(t):true;
    if (plugin_match) {
      logger.msg(VERBOSE, "Matchmaking, ExecutionTarget: %s matches job description", t.ComputingEndpoint->URLString);
    }
    return plugin_match;
  }

  void ExecutionTargetSet::addEntities(const std::list<ComputingServiceType>& csList) {
    for (std::list<ComputingServiceType>::const_iterator it = csList.begin(); it != csList.end(); ++it) {
      addEntity(*it);
    }
  }
} // namespace Arc
