# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import re
import urllib
import urlparse
import random

if 'DISPLAY' in os.environ:
    import gtk
    import cairo

import ninix.seriko
import ninix.menu
import ninix.pix


class Surface:

    # keyval/name mapping
    if 'DISPLAY' in os.environ:
        from ninix.keymap import keymap_old, keymap_new
    else:
        keymap_old = {}
        keymap_new = {}

    def __init__(self, callback, debug=0):
        self.debug = debug
        self.window = []
        self.__menu = None
        self.__scale = 100 # %
        self.desc = None
        self.__use_pna = False
        self.__alpha_channel = 1.0
        self.callback = callback

    def set_debug(self, debug):
        self.debug = debug
        for surface_window in self.window:
            surface_window.set_debug(self.debug)

    def set_use_pna(self, flag):
        if self.__use_pna == bool(flag):
            return
        self.__use_pna = bool(flag)
        for surface_window in self.window:
            surface_window.set_use_pna(self.__use_pna)

    def set_alpha_channel(self, alpha):
        if not 0.1 <= alpha <= 1.0 or alpha is None:
            alpha = 1.0
        if self.__alpha_channel == alpha:
            return
        self.__alpha_channel = alpha
        for surface_window in self.window:
            surface_window.set_alpha_channel(self.__alpha_channel)

    def get_scale(self):
        return self.__scale

    def set_scale(self, scale):
        if self.__scale == scale:
            return
        self.__scale = scale # %
        for surface_window in self.window:
            surface_window.set_scale(self.__scale)
        self.callback['notify_observer']('set scale') ## FIXME

    def set_animation_quality(self, quality):
        for surface_window in self.window:
            surface_window.set_animation_quality(quality)

    def set_seriko_inactive(self, flag):
        for surface_window in self.window:
            surface_window.set_seriko_inactive(flag)

    def finalize(self):
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []
        ##self.__menu.finalize()
        self.__menu = None

    def create_gtk_window(self, title, skip_taskbar):
        window = ninix.pix.TransparentWindow()
        window.set_focus_on_map(False)
        window.set_title(title)
        window.set_decorated(False)
        window.set_resizable(False)
        if skip_taskbar:
            window.set_skip_taskbar_hint(True)
        window.connect('delete_event', self.delete)
        window.connect('key_press_event', self.key_press)
        window.connect('window_state_event', self.window_state)
        window.set_events(gtk.gdk.KEY_PRESS_MASK)
        window.realize()
        return window

    def identify_window(self, win):
        for surface_window in self.window:
            if win == surface_window.window.window:
                return True
        return False

    def window_stayontop(self, flag):
        for surface_window in self.window:
            gtk_window = surface_window.window
            gtk_window.set_keep_above(flag)
                
    def window_iconify(self, flag):
        gtk_window = self.window[0].window
        if flag:
            if not gtk_window.window.get_state() & \
               gtk.gdk.WINDOW_STATE_ICONIFIED:
                gtk_window.iconify()
        else:
            if gtk_window.window.get_state() & gtk.gdk.WINDOW_STATE_ICONIFIED:
                gtk_window.deiconify()

    def window_state(self, window, event):
        if not self.callback['is_running']():
            return
        if event.new_window_state & gtk.gdk.WINDOW_STATE_ICONIFIED:
            if window == self.window[0].window:
                self.callback['notify_iconified']()
                self.callback['notify_observer']('iconified')
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and \
                   not gtk_window.window.get_state() & \
                   gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.iconify()
        else:
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and \
                   gtk_window.window.get_state() & \
                   gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.deiconify()
            if window == self.window[0].window:
                self.callback['notify_deiconified']()
        return

    def delete(self, window, event):
        return True

    def key_press(self, window, event):
        name = self.keymap_old.get(event.keyval, event.string)
        keycode = self.keymap_new.get(event.keyval, event.string)
        if name or keycode:
            self.callback['notify_event']('OnKeyPress', name, keycode)
        return True

    def window_stick(self, action):
        stick = self.__menu.get_stick()
        for window in self.window:
            if stick:
                window.window.stick()
            else:
                window.window.unstick()

    def open_popup_menu(self, button, side):
        self.__menu.popup(button, side)

    re_surface_id = re.compile('^surface([0-9]+)$')

    def get_seriko(self, surface):
        seriko = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define animation patterns
            seriko[key] = ninix.seriko.get_actors(config)
        return seriko

    def new(self, desc, alias, surface, name, prefix, tooltips):
        self.desc = desc
        self.__tooltips = tooltips
        self.name = name
        self.prefix = prefix
        if alias is None:
            alias0 = alias1 = None
        else:
            alias0 = alias.get('sakura.surface.alias')
            alias1 = alias.get('kero.surface.alias')
        # load surface
        pixbufs = {}
        elements = {}
        for basename, (path, config) in surface.iteritems():
            if path is None:
                continue
            if not os.path.exists(path):
                name, suffix = os.path.splitext(path)
                dgp_path = ''.join((name, '.dgp'))
                if not os.path.exists(dgp_path):
                    ddp_path = ''.join((name, '.ddp'))
                    if not os.path.exists(ddp_path):
                        print '%s: file not found (ignored)' % path
                        continue
                    else:
                        path = ddp_path
                else:
                    path = dgp_path
            elements[basename] = [path]
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            pixbufs[key] = elements[basename]
        # compose surface elements
        composite_pixbuf = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            if 'element0' in config:
                if self.debug & 8192:
                    print 'surface', key
                composite_pixbuf[key] = self.compose_elements(elements, config)
        pixbufs.update(composite_pixbuf)
        # check if necessary surfaces have been loaded
        for key in ['0', '10']:
            if key not in pixbufs:
                raise SystemExit, 'cannot load surface #%s (abort)\n' % key
        self.__pixbufs = pixbufs
        # arrange surface configurations
        region = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define collision areas
            buf = []
            for n in range(256):
                # "redo" syntax
                rect = config.get(''.join(('collision', str(n))))
                if rect is None:
                    continue
                values = rect.split(',')
                if len(values) != 5:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in values[:4]]
                except ValueError:
                    continue
                buf.append((values[4].strip(), x1, y1, x2, y2))
            for part in ['head', 'face', 'bust']:
                # "inverse" syntax
                rect = config.get(''.join(('collision.', part)))
                if not rect:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in rect.split(',')]
                except ValueError:
                    pass
                buf.append((part.capitalize(), x1, y1, x2, y2))
            region[key] = buf
        self.__region = region
        # MAYUNA
        mayuna = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define animation patterns
            mayuna[key] = ninix.seriko.get_mayuna(config)
        bind = {}
        for side in ['sakura', 'kero']:
            bind[side] = {}
            for index in range(128):
                name = self.desc.get(
                    '%s.bindgroup%d.name' % (side, index), None)
                default = self.desc.get(
                    '%s.bindgroup%d.default' % (side, index), 0)
                if name is not None:
                    bind[side][index] = [name, default]
        self.mayuna = {}
        for side in ['sakura', 'kero']:
            self.mayuna[side] = []
            for index in range(128):
                key = self.desc.get('%s.menuitem%d' % (side, index), None)
                if key == '-':
                    self.mayuna[side].append([key, None, 0])
                else:
                    try:
                        key = int(key)
                    except:
                        pass
                    else:
                        if key in bind[side]:
                            name = bind[side][key][0].split(',')
                            self.mayuna[side].append([key, name[1],
                                                      bind[side][key][1]])
        # create surface windows
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []
        self.__surface = surface
        self.add_window(0, '0', alias0, mayuna, bind['sakura'])
        self.add_window(1, '10', alias1, mayuna, bind['kero'])
        menu_callback = {
            'stick_window': self.window_stick,
            'toggle_bind': self.toggle_bind,
            }
        for key in ['network_update', 'vanish_sakura', 'edit_preferences',
                    'open_ghost_manager', 'show_usage', 'about',
                    'close_sakura', 'close_all', 'getstring',
                    'notify_site_selection', 'get_ghost_list', 'select_sakura',
                    'start_sakura', 'get_shell_list', 'get_balloon_list',
                    'select_shell', 'select_balloon',
                    'get_current_balloon_directory', 'get_current_item',
                    'get_plugin_list', 'select_plugin',
                    'get_nekodorif_list', 'select_nekodorif',
                    'get_kinoko_list', 'select_kinoko']:
            menu_callback[key] = self.callback[key]
        self.__menu = ninix.menu.Menu(menu_callback)
        top_dir = self.prefix
        name = self.desc.get('menu.background.bitmap.filename',
                             'menu_background.png')
        name = name.replace('\\', '/')
        path = os.path.join(top_dir, name)
        if not os.path.exists(path):
            top_dir = os.path.join(self.callback['get_prefix'](),
                                   'ghost', 'master')
            path = os.path.join(top_dir, 'menu_background.png')
            if os.path.exists(path):
                path_background = path
            else:
                path_background = None
            path = os.path.join(top_dir, 'menu_sidebar.png')
            if os.path.exists(path):
                path_sidebar = path
            else:
                path_sidebar = None
            path = os.path.join(top_dir, 'menu_foreground.png')
            if os.path.exists(path):
                path_foreground = path
            else:
                path_foreground = None
        else:
            path_background = path
            name = self.desc.get('menu.sidebar.bitmap.filename',
                                 'menu_sidebar.png')
            name = name.replace('\\', '/')
            path = os.path.join(top_dir, name)
            if os.path.exists(path):
                path_sidebar = path
            else:
                path_sidebar = None
            name = self.desc.get('menu.foreground.bitmap.filename',
                                 'menu_foreground.png')
            name = name.replace('\\', '/')
            path = os.path.join(top_dir, name)
            if os.path.exists(path):
                path_foreground = path
            else:
                path_foreground = None
        if path_background:
            self.__menu.set_pixmap(
                path_background, path_sidebar, path_foreground)
        fontcolor_r = self.desc.get_with_type('menu.background.font.color.r', int, 0)
        fontcolor_g = self.desc.get_with_type('menu.background.font.color.g', int, 0)
        fontcolor_b = self.desc.get_with_type('menu.background.font.color.b', int, 0)
        background = (fontcolor_r, fontcolor_g, fontcolor_b)
        fontcolor_r = self.desc.get_with_type('menu.foreground.font.color.r', int, 0)
        fontcolor_g = self.desc.get_with_type('menu.foreground.font.color.g', int, 0)
        fontcolor_b = self.desc.get_with_type('menu.foreground.font.color.b', int, 0)
        foreground = (fontcolor_r, fontcolor_g, fontcolor_b)
        self.__menu.set_fontcolor(background, foreground)
        self.__menu.create_mayuna_menu(self.get_mayuna_menu())

    def add_window(self, side, default, alias=None, mayuna={}, bind={}):
        assert len(self.window) == side
        if side == 0:
            name = 'sakura'
            title = self.callback['get_selfname']() or \
                ''.join(('surface.', name))
        elif side == 1:
            name = 'kero'
            title = self.callback['get_keroname']() or \
                ''.join(('surface.', name))
        else:
            name = 'char%d' % side
            title = ''.join(('surface.', name))
        skip_taskbar = bool(side >= 1)
        gtk_window = self.create_gtk_window(title, skip_taskbar)
        seriko = self.get_seriko(self.__surface)
        tooltips = {}
        if name in self.__tooltips:
            tooltips = self.__tooltips[name]
        surface_window_callback = {
            'reset_balloon_position': self.reset_balloon_position
            }
        for key in ['enqueue_event', 'notify_observer', 'get_preference',
                    'set_balloon_direction', 'reset_idle_time',
                    'notify_surface_click', 'notify_event', 'busy',
                    'notify_surface_mouse_motion']:
            surface_window_callback[key] = self.callback[key]
        surface_window = SurfaceWindow(
            gtk_window, side, surface_window_callback, self.desc, alias,
            self.__surface, tooltips, self.__pixbufs, seriko, self.__region,
            mayuna, bind, default, self.debug)
        self.window.append(surface_window)
        # SurfaceWindow default settings should be same as current Surface ones
        surface_window.set_scale(self.__scale)
        surface_window.set_use_pna(self.__use_pna)
        surface_window.set_alpha_channel(self.__alpha_channel)
        # Seriko default settings
        quality = self.callback['get_preference']('animation_quality')
        surface_window.set_animation_quality(quality)
        flag = self.callback['get_preference']('seriko_inactive')
        surface_window.set_seriko_inactive(flag)
        #surface_window.set_surface(default) # Don't do this.

    def get_mayuna_menu(self):
        for side, index in [('sakura', 0), ('kero', 1)]:
            for menu in self.mayuna[side]:
                if menu[0] != '-':
                    menu[2] = self.window[index].bind[menu[0]][1]
        return self.mayuna

    def compose_elements(self, elements, config):
        error = None
        for n in range(256):
            key = ''.join(('element', str(n)))
            if key not in config:
                break
            spec = [value.strip() for value in config[key].split(',')]
            try:
                method, filename, x, y = spec
                x = int(x)
                y = int(y)
            except ValueError:
                error = 'invalid element spec for %s: %s' % (key, config[key])
                break
            basename, suffix = os.path.splitext(filename)
            if suffix.lower() not in ['.png', '.dgp', '.ddp']:
                error = 'unsupported file format for %s: %s' % (key, filename)
                break
            basename = basename.lower()
            if basename not in elements:
                error = '%s file not found: %s' % (key, filename)
                break
            pixbuf = elements[basename][0]
            if n == 0: # base surface
                pixbuf_list = [pixbuf]
            elif method == 'overlay':
                pixbuf_list.append((pixbuf, x, y))
            elif method == 'overlayfast': # XXX
                pixbuf_list.append((pixbuf, x, y))
            elif method == 'base':
                pixbuf_list.append((pixbuf, x, y))
            else:
                error = 'unknown method for %s: %s' % (key, method)
                break
            if self.debug & 8192:
                print '%s: %s %s, x=%d, y=%d' % (key, method, filename, x, y)
        if error is not None:
            print error
            pixbuf_list = []
        return pixbuf_list

    def get_window(self, side):
        if len(self.window) > side:
            return self.window[side].window # FIXME
        else: 
            return None

    def reset_surface(self, side):
        if len(self.window) > side:
            self.window[side].reset_surface()

    def set_surface_default(self, side):
        if side is None:
            for side in range(len(self.window)):
                self.window[side].set_surface_default()
        elif 0 <= side < len(self.window):
            self.window[side].set_surface_default()

    def set_surface(self, side, surface_id):
        if len(self.window) > side:
            self.window[side].set_surface(surface_id)

    def get_surface(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface()
        else:
            return 0

    def get_surface_size(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_size()
        else:
            return 0, 0

    def get_surface_offset(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_offset()
        else:
            return 0, 0

    def get_touched_region(self, side, x, y):
        if len(self.window) > side:
            return self.window[side].get_touched_region(x, y)
        else:
            return ''

    def get_center(self, side):
        if len(self.window) > side:
                return self.window[side].get_center()
        else:
            return None, None

    def get_kinoko_center(self, side):
        if len(self.window) > side:
                return self.window[side].get_kinoko_center()
        else:
            return None, None

    def get_direction(self, side):
        if len(self.window) > side:
            return self.window[side].get_direction()
        else:
            return 0

    def set_direction(self, side, direction):
        if len(self.window) > side:
            self.window[side].set_direction(direction)

    def reset_balloon_position(self):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        for side in range(len(self.window)):
            x, y = self.get_position(side)
            sox, soy = self.window[side].get_surface_offset()
            w, h = self.get_surface_size(side)
            bw, bh = self.callback['get_balloon_size'](side)
            ox, oy = self.get_balloon_offset(side)
            direction = self.get_direction(side)
            align = self.get_alignment(side)
            if direction == 0:
                bx = max(x + sox - bw + ox, left)
            else:
                bx =  min(x + sox + w - ox, left + scrn_w - bw)
            if align == 0:
                by = min(y + soy + oy, top + scrn_h - bh)
            elif align == 1:
                by = max(y + soy + oy, top)
            else:
                if y + soy < top + scrn_h / 2:
                    by = max(y + soy + oy, top)
                else:
                    by = min(y + soy + oy, top + scrn_h - bh)
            self.callback['set_balloon_direction'](side, direction)
            self.callback['set_balloon_position'](side, bx, by)

    def reset_position(self):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        for side in range(len(self.window)):
            align = self.get_alignment(side)
            if side == 0: # sakura
                w, h = self.get_surface_size(side)
                x = left + scrn_w - w
                if align == 1: # top
                    y = top
                else:
                    y = top + scrn_h - h
                bw, bh = self.callback['get_balloon_size'](side)
                ox, oy = self.get_balloon_offset(side)
                direction = 0 # left
                bx = max(x - bw + ox, left)
            else:
                b0w, b0h = self.callback['get_balloon_size'](side - 1)
                b1w, b1h = self.callback['get_balloon_size'](side)
                o0x, o0y = self.get_balloon_offset(side - 1)
                o1x, o1y = self.get_balloon_offset(side)
                w, h = self.get_surface_size(side)
                offset = max(0, b1w - (b0w - o0x))
                if (s0x + o0x - b0w) - offset - w + o1x < left:
                    x = left
                else:
                    x = (s0x + o0x - b0w) - offset - w + o1x
                if align == 1: # top
                    y = top
                else:
                    y = top + scrn_h - h
                direction = 1 # right
                bx =  min(x + w - o1x, left + scrn_w - b1w)
            if align == 0:
                by = min(y + oy, top + scrn_h - bh)
            elif align == 1:
                by = max(y + oy, top)
            else:
                if y < top + scrn_h / 2:
                    by = max(y + oy, top)
                else:
                    by = min(y + oy, top + scrn_h - bh)
            self.set_position(side, x, y)
            self.set_direction(side, direction)
            self.callback['set_balloon_position'](side, bx, by)
            s0x, s0y, s0w, s0h = x, y, w, h # for next loop

    def set_position(self, side, x, y):
        if len(self.window) > side:
            self.window[side].set_position(x, y)

    def get_position(self, side):
        if len(self.window) > side:
            return self.window[side].get_position()
        else:
            return 0, 0

    def set_alignment_current(self):
        for side in range(len(self.window)):
            self.window[side].set_alignment_current()

    def set_alignment(self, side, align):
        if len(self.window) > side:
            self.window[side].set_alignment(align)

    def get_alignment(self, side):
        if len(self.window) > side:
            return self.window[side].get_alignment()
        else:
            return 0

    def reset_alignment(self):
        if self.desc.get('seriko.alignmenttodesktop') == 'free':
            align = 2
        else:
            align = 0
        for side in range(len(self.window)):
            self.set_alignment(side, align)

    def is_shown(self, side):
        if len(self.window) > side:
            return self.window[side].is_shown()
        else:
            return False

    def show(self, side):
        if len(self.window) > side:
            self.window[side].show()
            self.callback['notify_observer']('show') ## FIXME
            self.callback['notify_observer']('raise', (side)) # XXX ## FIXME

    def hide_all(self):
        for side in range(len(self.window)):
            self.window[side].hide()
        self.callback['notify_observer']('hide') ## FIXME

    def hide(self, side):
        if len(self.window) > side:
            self.window[side].hide()
        self.callback['notify_observer']('hide') ## FIXME

    def raise_all(self):
        for side in range(len(self.window)):
            self.window[side].raise_()
            self.callback['notify_observer']('raise', (side)) ## FIXME

    def raise_(self, side):
        if len(self.window) > side:
            self.window[side].raise_()
            self.callback['notify_observer']('raise', (side)) ## FIXME

    def lower_all(self):
        for side in range(len(self.window)):
            self.window[side].lower()
        self.callback['notify_observer']('lower') ## FIXME

    def lower(self, side):
        if len(self.window) > side:
            self.window[side].lower()
        self.callback['notify_observer']('lower') ## FIXME

    def invoke(self, side, actor_id):
        if len(self.window) > side:
            self.window[side].invoke(actor_id)

    def invoke_yen_e(self, side, surface_id):
        if len(self.window) > side:
            self.window[side].invoke_yen_e(surface_id)

    def invoke_talk(self, side, surface_id, count):
        if len(self.window) > side:
            return self.window[side].invoke_talk(surface_id, count)
        else:
            return 0

    def set_icon(self, path):
        pixbuf = None
        if path is not None:
            try:
                pixbuf = ninix.pix.create_pixbuf_from_file(path, is_pnr=False)
            except:
                pixbuf = None
        for window in self.window:
            window.window.set_icon(pixbuf)

    def get_mikire(self):
        if not self.is_shown(0):
            return 0
        ##return 1 if self.is_truncated(0) else 0
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        x0, y0 = self.get_position(0)
        s0w, s0h = self.get_surface_size(0)
        if x0 + s0w / 3 < left or x0 + s0w * 2 / 3 > left + scrn_w or \
           y0 + s0h / 3 < top or y0 + s0h * 2 / 3 > top + scrn_h:
            return 1
        else:
            return 0

    def get_kasanari(self):
        if not self.is_shown(0) or not self.is_shown(1):
            return 0
        x0, y0 = self.get_position(0)
        s0w, s0h = self.get_surface_size(0)
        x1, y1 = self.get_position(1)
        s1w, s1h = self.get_surface_size(1)
        if (x0 < x1 + s1w / 2 < x0 + s0w and y0 < y1 + s1h / 2 < y0 + s0h) or \
           (x1 < x0 + s0w / 2 < x1 + s1w and y1 < y0 + s0h / 2 < y1 + s1h):
            return 1
        else:
            return 0

    def get_name(self):
        return self.name

    def get_username(self):
        return None if self.desc is None else self.desc.get('user.defaultname')

    def get_selfname(self):
        return None if self.desc is None else self.desc.get('sakura.name')

    def get_selfname2(self):
        return None if self.desc is None else self.desc.get('sakura.name2')

    def get_keroname(self):
        return None if self.desc is None else self.desc.get('kero.name')

    def get_friendname(self):
        return None if self.desc is None else self.desc.get('sakura.friend.name')

    def get_balloon_offset(self, side):
        if len(self.window) > side:
            return self.window[side].get_balloon_offset()
        return 0, 0

    def toggle_bind(self, event, args):
        side, bind_id = args
        self.window[side].toggle_bind(bind_id)

    def get_collision_area(self, side, part):
        if len(self.window) > side:
            return self.window[side].get_collision_area(part)
        return None


class SurfaceWindow:

    # DnD data types
    dnd_targets = [
        ('text/plain', 0, 0),
        ]

    def __init__(self, window, side, callback, desc, alias, surface, tooltips,
                 pixbuf, seriko, region, mayuna, bind, default_id, debug):
        self.window = window
        self.side = side
        self.callback = callback
        self.desc = desc
        self.alias = alias
        self.tooltips = tooltips
        self.align = 0
        self.__scale = 100 # %
        self.__use_pna = False
        self.__alpha_channel = 1.0
        self.__current_part = ''
        if self.alias is not None:
            default_id = self.alias.get(default_id, [default_id])[0]
        self.surface = surface
        self.surface_id = default_id
        self.pixbuf = pixbuf
        self.current_surface_pixbuf = None # XXX
        self.seriko = ninix.seriko.Controler(seriko)
        self.region = region
        self.mayuna = mayuna
        self.bind = bind
        self.default_id = default_id
        self.debug = debug
        self.__shown = False
        self.truncated = False
        self.truncate = (0, 0)
        self.window_offset = (0, 0)
        self.set_position(0, 0)
        self.set_direction(0)
        self.dragged = False
        self.x_root = None
        self.y_root = None
        self.window.connect('leave_notify_event', self.window_leave_notify) # XXX
        self.window.connect('enter_notify_event', self.window_enter_notify) # XXX
        # create drawing area
        self.darea = gtk.DrawingArea()
        self.darea.show()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                              gtk.gdk.BUTTON_PRESS_MASK|
                              gtk.gdk.BUTTON_RELEASE_MASK|
                              gtk.gdk.POINTER_MOTION_MASK|
                              gtk.gdk.POINTER_MOTION_HINT_MASK|
                              gtk.gdk.SCROLL_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.darea.connect('button_press_event', self.button_press)
        self.darea.connect('button_release_event', self.button_release)
        self.darea.connect('motion_notify_event', self.motion_notify)
        self.darea.connect('drag_data_received', self.drag_data_received)
        self.darea.connect('scroll_event', self.scroll)
        self.darea.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.dnd_targets,
                                 gtk.gdk.ACTION_COPY)
        self.fixed = gtk.Fixed()
        self.fixed.put(self.darea, 0, 0)
        self.fixed.show()
        self.window.add(self.fixed)
        self.window.realize()
        self.window.window.set_back_pixmap(None, False)
        self.set_surface(None)

    def set_debug(self, debug):
        self.debug = debug

    def set_use_pna(self, flag):
        self.__use_pna = bool(flag)
        self.reset_surface()

    def set_alpha_channel(self, value):
        self.__alpha_channel = value
        self.reset_surface()

    def get_scale(self):
        return self.__scale

    def set_scale(self, scale):
        self.__scale = scale # %
        self.reset_surface()

    def set_animation_quality(self, quality):
        self.seriko.set_animation_quality(quality, self)

    def set_seriko_inactive(self, flag):
        self.seriko.set_inactive(flag, self)

    def drag_data_received(self, widget, context, x, y, data, info, time):
        if self.debug & 8192:
            print 'Content-type:', data.type
            print 'Content-length:', data.get_length()
            print repr(data.data)
        if str(data.type) == 'text/plain':
            filelist = []
            for line in data.data.split('\r\n'):
                scheme, host, path, params, query, fragment = \
                        urlparse.urlparse(line)
                pathname = urllib.url2pathname(path)
                if scheme == 'file' and os.path.exists(pathname):
                    filelist.append(pathname)
            if filelist:
                self.callback['enqueue_event'](
                    'OnFileDrop2', chr(1).join(filelist), self.side)
        return True

    def append_actor(self, frame, actor):
        self.seriko.append_actor(frame, actor)

    def invoke(self, actor_id, update=0):
        self.seriko.invoke(self, actor_id, update)

    def invoke_yen_e(self, surface_id):
        self.seriko.invoke_yen_e(self, surface_id)

    def invoke_talk(self, surface_id, count):
        return self.seriko.invoke_talk(self, surface_id, count)

    def reset_surface(self):
        surface_id = self.get_surface()
        self.set_surface(surface_id)

    def set_surface_default(self):
        self.set_surface(self.default_id)

    def set_surface(self, surface_id):
        if self.alias is not None and surface_id in self.alias:
            aliases = self.alias.get(surface_id)
            if aliases:
                surface_id = random.choice(aliases)
        if surface_id == '-2':
            self.seriko.terminate(self)
        if surface_id in ['-1', '-2']:
            pass
        elif surface_id not in self.pixbuf:
            self.surface_id = self.default_id
        else:
            self.surface_id = surface_id
        self.seriko.reset(self, surface_id)
        # define collision areas
        self.collisions = self.region[self.surface_id]
        # update window offset
        x, y = self.position # XXX: without window_offset
        w, h = self.get_surface_size(self.surface_id)
        dw, dh = self.get_surface_size(self.default_id) # default surface size
        xoffset = (dw - w) / 2
        if self.get_alignment() == 0:
            yoffset = dh - h
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            y = top + scrn_h - dh
        elif self.get_alignment() == 1:
            yoffset = 0
        else:
            yoffset = (dh - h) / 2
        self.window_offset = (xoffset, yoffset)
        # resize window
        self.darea.set_size_request(w, h)
        self.window.queue_resize()
        self.seriko.start(self)
        # relocate window
        if not self.dragged: # XXX
            self.set_position(x, y)
        if self.side < 2:
            self.callback['notify_observer']('set surface')

    def iter_mayuna(self, surface_width, surface_height, mayuna, done):
        for surface, interval, method, args in mayuna.patterns:
            if method in ['bind', 'add']:
                if surface in self.pixbuf:
                    x, y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    # overlay surface pixbuf
                    if x + w > surface_width:
                        w = surface_width - x
                    if y + h > surface_height:
                        h = surface_height - y
                    if x < 0:
                        dest_x = 0
                        w += x
                    else:
                        dest_x = x
                    if y < 0:
                        dest_y = 0
                        h += y
                    else:
                        dest_y = y
                    yield method, pixbuf, dest_x, dest_y, w, h, x, y
            elif method == 'reduce':
                if surface in self.pixbuf:
                    dest_x, dest_y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    x = y = 0 # XXX
                    yield method, pixbuf, dest_x, dest_y, w, h, x, y
            elif method == 'insert':
                index = args[0]
                for actor in self.mayuna[self.surface_id]:
                    actor_id = actor.get_id()
                    if actor_id == index:
                        if actor_id in self.bind and self.bind[actor_id][1] and \
                           actor_id not in done:
                            done.append(actor_id)
                            for result in self.iter_mayuna(surface_width, surface_height, actor, done):
                                yield result
                        else:
                            break
            else:
                raise RuntimeError, 'should not reach here'

    def create_pixbuf_from_file(self, pixbuf_id):
        assert pixbuf_id in self.pixbuf
        try:
            pixbuf = ninix.pix.create_pixbuf_from_file(
                self.pixbuf[pixbuf_id][0], use_pna=self.__use_pna)
        except:
            if self.debug & 4:
                print 'cannot load surface #%s' % pixbuf_id
            return ninix.pix.create_blank_pixbuf(100, 100)
        for element, x, y in self.pixbuf[pixbuf_id][1:]:
            try:
                overlay = ninix.pix.create_pixbuf_from_file(
                    element, use_pna=self.__use_pna)
            except:
                continue
            w = overlay.get_width()
            h = overlay.get_height()
            sw = pixbuf.get_width()
            sh = pixbuf.get_height()
            if x + w > sw:
                w = sw - x
            if y + h > sh:
                h = sh - y
            if x < 0:
                dest_x = 0
                w += x
            else:
                dest_x = x
            if y < 0:
                dest_y = 0
                h += y
            else:
                dest_y = y
            overlay.composite(pixbuf, dest_x, dest_y,
                              w, h, x, y, 1.0, 1.0,
                              gtk.gdk.INTERP_BILINEAR, 255)
        return pixbuf

    def get_pixbuf(self, pixbuf_id):
        if pixbuf_id not in self.pixbuf:
            if self.debug & 4:
                print 'cannot load pixbuf #%s' % pixbuf_id
            return ninix.pix.create_blank_pixbuf(100, 100)
        return self.create_pixbuf_from_file(pixbuf_id)

    def draw_region(self):
        cr = self.darea.window.cairo_create()
        cr.save()
        scale = self.get_scale()
        for part, x1, y1, x2, y2 in self.collisions:
            x1 = x1 * scale / 100
            x2 = x2 * scale / 100
            y1 = y1 * scale / 100
            y2 = y2 * scale / 100
            cr.set_operator(cairo.OPERATOR_ATOP)
            cr.set_source_rgba(0.2, 0.0, 0.0, 0.4) # XXX
            cr.rectangle(x1, y1, x2 - x1, y2 - y1)
            cr.fill_preserve()
            cr.set_operator(cairo.OPERATOR_SOURCE)
            cr.set_source_rgba(0.4, 0.0, 0.0, 0.8) # XXX
            cr.stroke()
        cr.restore()
        del cr

    def create_surface_pixbuf(self, surface_id=None):
        if surface_id is None:
            surface_id = self.surface_id
        if surface_id in self.mayuna and self.mayuna[surface_id]:
            surface_pixbuf = self.get_pixbuf(surface_id)
            done = []
            for actor in self.mayuna[surface_id]:
                actor_id = actor.get_id()
                if actor_id in self.bind and self.bind[actor_id][1] and \
                   actor_id not in done:
                    done.append(actor_id)
                    #surface_pixbuf = self.compose_surface(
                    #    surface_pixbuf, actor, done)
                    surface_width = surface_pixbuf.get_width()
                    surface_height = surface_pixbuf.get_height()
                    for method, pixbuf, dest_x, dest_y, w, h, x, y in self.iter_mayuna(surface_width, surface_height, actor, done):
                        if method in ['bind', 'add']:
                            pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                                             w, h, x, y, 1.0, 1.0,
                                             gtk.gdk.INTERP_BILINEAR, 255)
                        elif method == 'reduce':
                            if pixbuf.get_has_alpha():
                                dest_w = surface_width
                                dest_h = surface_height
                                surface_array = surface_pixbuf.get_pixels_array()
                                array = pixbuf.get_pixels_array()
                                for i in range(h):
                                    for j in range(w):
                                        if array[i][j][3] == 0: # alpha
                                            x = j + dest_x
                                            y = i + dest_y
                                            if 0 <= x < dest_w and 0 <= y < dest_h:
                                                surface_array[y][x][3] = 0 # alpha
                        else:
                            raise RuntimeError, 'should not reach here'
        else:
            surface_pixbuf = self.get_pixbuf(surface_id)
        return surface_pixbuf

    def update_frame_buffer(self):
        if self.seriko.is_active():
            surface_pixbuf = self.create_surface_pixbuf(self.seriko.base_id)
            assert surface_pixbuf is not None
            # draw overlays
            for pixbuf_id, x, y in self.seriko.iter_overlays():
                try:
                    pixbuf = self.get_pixbuf(pixbuf_id)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                except:
                    continue
                # overlay surface pixbuf
                sw = surface_pixbuf.get_width()
                sh = surface_pixbuf.get_height()
                if x + w > sw:
                    w = sw - x
                if y + h > sh:
                    h = sh - y
                if x < 0:
                    dest_x = 0
                    w += x
                else:
                    dest_x = x
                if y < 0:
                    dest_y = 0
                    h += y
                else:
                    dest_y = y
                pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                                 w, h, x, y, 1.0, 1.0,
                                 gtk.gdk.INTERP_BILINEAR, 255)
        else:
            surface_pixbuf = self.create_surface_pixbuf()
        w = surface_pixbuf.get_width()
        h = surface_pixbuf.get_height()
        scale = self.get_scale()
        w = max(8, w * scale / 100)
        h = max(8, h * scale / 100)
        surface_pixbuf = surface_pixbuf.scale_simple(
            w, h, gtk.gdk.INTERP_BILINEAR)
        mask_pixmap = gtk.gdk.Pixmap(None, w, h, 1)
        surface_pixbuf.render_threshold_alpha(
            mask_pixmap, 0, 0, 0, 0, w, h, 1)
        x, y = self.truncate
        if self.window.is_composited():
            self.window.input_shape_combine_mask(mask_pixmap, x, y)
        else:
            self.window.shape_combine_mask(mask_pixmap, x, y)
        self.current_mask = mask_pixmap
        self.current_surface_pixbuf = surface_pixbuf
        self.darea.queue_draw()

    def redraw(self, darea, event):
        if self.current_surface_pixbuf is None: # XXX
            return
        cr = darea.window.cairo_create()
        cr.save()
        cr.set_operator(cairo.OPERATOR_CLEAR)
        cr.paint()
        cr.restore()
        cr.set_source_pixbuf(self.current_surface_pixbuf, 0, 0)
        cr.paint_with_alpha(self.__alpha_channel)
        del cr
        if self.callback['get_preference']('check_collision'):
            self.draw_region()

    def remove_overlay(self, actor):
        self.seriko.remove_overlay(actor)

    def add_overlay(self, actor, pixbuf_id, x, y):
        self.seriko.add_overlay(self, actor, pixbuf_id, x, y)

    def __move(self, xoffset=0, yoffset=0):
        x, y = self.get_position()
        w, h = self.get_surface_size()
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        new_x = min(max(x + xoffset, -w + 1), left + scrn_w - 1)
        new_y = min(max(y + yoffset, -h + 1), top + scrn_h - 1)
        new_w = w
        new_h = h
        offset_x = 0
        offset_y = 0
        truncate = False
        if new_x < left:
            offset_x = new_x - left
            new_x = left
            new_w += offset_x
            truncate = True
        if new_x + w > left + scrn_w:
            new_w -= new_x + w - (left + scrn_w)
            truncate = True
        if new_y < top:
            offset_y = new_y - top
            new_y = top
            new_h += offset_y
            truncate = True
        if new_y + h > top + scrn_h:
            new_h -= new_y + h - (top + scrn_h)
            truncate = True
        if not truncate:
            if self.truncated:
                self.reset_truncate()
        else:
            self.fixed.move(self.darea, offset_x, offset_y)
            self.window.set_size_request(new_w, new_h)
            if self.window.is_composited():
                self.window.input_shape_combine_mask(
                    self.current_mask, offset_x, offset_y)
            else:
                self.window.shape_combine_mask(
                    self.current_mask, offset_x, offset_y)
            self.truncated = True
            self.truncate = (offset_x, offset_y)
        if not self.dragged: # XXX
            ### XXX ###
            while gtk.events_pending():
                gtk.main_iteration(True)
            gtk.gdk.flush()
            ### XXX ###
        self.window.move(new_x, new_y)
        self.darea.queue_draw()

    def reset_truncate(self):
        self.fixed.move(self.darea, 0, 0)
        self.window.set_size_request(-1, -1)
        if self.window.is_composited():
            self.window.input_shape_combine_mask(self.current_mask, 0, 0)
        else:
            self.window.shape_combine_mask(self.current_mask, 0, 0)
        self.truncate = (0, 0)
        self.truncated = False

    def move_surface(self, xoffset, yoffset):
        self.__move(xoffset, yoffset)
        if self.side < 2:
            args = (self.side, xoffset, yoffset)
            self.callback['notify_observer']('move surface', args) # animation

    def get_balloon_offset(self):
        path, config = self.surface[''.join(('surface', self.surface_id))]
        side = self.side
        if side == 0:
            name = 'sakura'
            x = config.get_with_type('%s.balloon.offsetx' % name, int)
            y = config.get_with_type('%s.balloon.offsety' % name, int)
        elif side == 1:
            name = 'kero'
            x = config.get_with_type('%s.balloon.offsetx' % name, int)
            y = config.get_with_type('%s.balloon.offsety' % name, int)
        else:
            name = 'char%d' % side
            x, y = None, None # XXX
        if x is None:
            x = self.desc.get_with_type('%s.balloon.offsetx' % name, int, 0)
        if y is None:
            y = self.desc.get_with_type('%s.balloon.offsety' % name, int, 0)
        scale = self.get_scale()
        x = x * scale / 100
        y = y * scale / 100
        return x, y

    def get_collision_area(self, part):
        for p, x1, y1, x2, y2 in self.collisions: ## FIXME
            if p == part:
                scale = self.get_scale()
                x1 = x1 * scale / 100
                x2 = x2 * scale / 100
                y1 = y1 * scale / 100
                y2 = y2 * scale / 100
                return x1, y1, x2, y2
        return None

    def get_surface(self):
        return self.surface_id

    def get_surface_size(self, surface_id=None):
        if surface_id is None:
            surface_id = self.surface_id
        if surface_id not in self.pixbuf:
            w, h = 100, 100 # XXX
        else:
            w, h = ninix.pix.get_png_size(self.pixbuf[surface_id][0])
        scale = self.get_scale()
        w = max(8, int(w * scale / 100))
        h = max(8, int(h * scale / 100))
        return w, h

    def get_surface_offset(self):
        return self.window_offset

    def get_touched_region(self, x, y):
        for part, x1, y1, x2, y2 in self.collisions:
            if x1 <= x <= x2 and y1 <= y <= y2:
                ##print part, 'touched'
                return part
        return ''

    def __get_with_scaling(self, name, conv):
        basename = ''.join(('surface', self.surface_id))
        path, config = self.surface[basename]
        value = config.get_with_type(name, conv)
        if value is not None:
            scale = self.get_scale()
            value = conv(value * scale / 100)
        return value

    def get_center(self):
        centerx = self.__get_with_scaling('point.centerx', int)
        centery = self.__get_with_scaling('point.centery', int)
        return centerx, centery

    def get_kinoko_center(self):
        centerx = self.__get_with_scaling('point.kinoko.centerx', int)
        centery = self.__get_with_scaling('point.kinoko.centery', int)
        return centerx, centery

    def get_direction(self):
        return self.direction

    def set_direction(self, direction):
        self.direction = direction # 0: left, 1: right
        self.callback['set_balloon_direction'](self.side, direction)

    def set_position(self, x, y):
        self.position = (x, y)
        if self.__shown:
            self.__move()
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if x > left + scrn_w / 2:
            direction = 0
        else:
            direction = 1
        self.set_direction(direction)
        self.callback['notify_observer']('set position')

    def get_position(self):
        return map(lambda x, y: x + y, self.position, self.window_offset)

    def set_alignment_current(self):
        self.set_alignment(self.get_alignment())

    def set_alignment(self, align):
        if align in [0, 1, 2]:
            self.align = align
        if self.dragged:
            # XXX: position will be reset after button release event
            return
        if align == 0:
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            sw, sh = self.get_surface_size(self.default_id)
            sx, sy = self.position # XXX: without window_offset
            sy = top + scrn_h - sh
            self.set_position(sx, sy)
        elif align == 1:
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            sx, sy = self.position # XXX: without window_offset
            sy = top
            self.set_position(sx, sy)
        else: # free
            pass

    def get_alignment(self):
        return self.align

    def destroy(self):
        self.seriko.destroy()
        self.fixed.remove(self.darea)
        self.darea.destroy()
        self.window.remove(self.fixed)
        self.fixed.destroy()
        self.window.destroy()

    def is_shown(self):
        return self.__shown

    def show(self):
        if self.__shown:
            return
        self.__shown = True
        self.__move() # XXX: call before showing the window
        self.darea.show()
        self.window.show()

    def hide(self):
        if self.__shown:
            ##self.window.hide()
            self.darea.hide()
            self.__shown = False

    def raise_(self):
        self.window.window.raise_()

    def lower(self):
        self.window.window.lower()

    def button_press(self, window, event):
        self.callback['reset_idle_time']()
        x = int(event.x)
        y = int(event.y)
        scale = self.get_scale()
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        self.x_root = event.x_root
        self.y_root = event.y_root
        click = 1 if event.type == gtk.gdk.BUTTON_PRESS else 2
        # automagical raise
        self.callback['notify_observer']('raise', (self.side))
        self.callback['notify_surface_click'](
            event.button, click, self.side, x, y)
        return True

    def button_release(self, window, event):
        if self.dragged:
            self.dragged = False
            self.set_alignment_current()
            self.callback['reset_balloon_position']()
        self.x_root = None
        self.y_root = None
        return True

    def motion_notify(self, darea, event):
        if event.is_hint:
            x, y, state = self.darea.window.get_pointer()
        else:
            x, y, state = event.x, event.y, event.state
        scale = self.get_scale()
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        part = self.get_touched_region(x, y)
        if part != self.__current_part:
            if part == '':
                self.window.set_tooltip_text(None)
                self.darea.window.set_cursor(None)
                self.callback['notify_event'](
                    'OnMouseLeave', x, y, '', self.side, self.__current_part)
            else:
                if part in self.tooltips:
                    tooltip = self.tooltips[part]
                    self.window.set_tooltip_text(tooltip)
                else:
                    self.window.set_tooltip_text(None)
                cursor = gtk.gdk.Cursor(gtk.gdk.HAND1)
                self.darea.window.set_cursor(cursor)
                self.callback['notify_event'](
                    'OnMouseEnter', x, y, '', self.side, part)
        self.__current_part = part
        if not self.callback['busy']():
            if state & gtk.gdk.BUTTON1_MASK:
                if self.x_root is not None and \
                   self.y_root is not None:
                    self.dragged = True
                    x_delta = int(event.x_root - self.x_root)
                    y_delta = int(event.y_root - self.y_root)
                    x, y = self.position # XXX: without window_offset
                    self.set_position(x + x_delta, y + y_delta)
                    self.x_root = event.x_root
                    self.y_root = event.y_root
            elif state & gtk.gdk.BUTTON2_MASK or \
               state & gtk.gdk.BUTTON3_MASK:
                pass
            else:
                self.callback['notify_surface_mouse_motion'](
                    self.side, x, y, part)
        return True

    def scroll(self, darea, event):
        x = int(event.x)
        y = int(event.y)
        scale = self.get_scale()
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        if event.direction == gtk.gdk.SCROLL_UP:
            count = 1
        elif event.direction == gtk.gdk.SCROLL_DOWN:
            count = -1
        else:
            count = 0
        if count != 0:
            part = self.get_touched_region(x, y)
            self.callback['notify_event'](
                'OnMouseWheel', x, y, count, self.side, part)
        return True

    def toggle_bind(self, bind_id):
        if bind_id in self.bind:
            current = self.bind[bind_id][1]
            self.bind[bind_id][1] = not current
            self.reset_surface()

    def window_enter_notify(self, window, event):
        x, y, state = event.x, event.y, event.state
        scale = self.get_scale()
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        self.callback['notify_event'](
            'OnMouseEnterAll', x, y, '', self.side, '')

    def window_leave_notify(self, window, event):
        x, y, state = event.x, event.y, event.state
        scale = self.get_scale()
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        if self.__current_part != '': # XXX
            self.callback['notify_event'](
                'OnMouseLeave', x, y, '', self.side, self.__current_part)
            self.__current_part = ''
        self.callback['notify_event'](
            'OnMouseLeaveAll', x, y, '', self.side, '')
        return True


def test():
    pass

if __name__ == '__main__':
    test()
