# -*- coding: utf-8 -*-
#
#  Copyright (C) 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import re
import sys
import random

import gtk
import gobject

def print_error(message):
    sys.stderr.write(''.join((message, '\n')))


class Controler:

    def __init__(self, seriko):
        self.seriko = seriko
        self.exclusive_actor = None
        self.base_id = None
        self.timeout_id = None
        self.idle_id = None
        self.reset_overlays()
        self.queue = []
        self.frame = {'step': 0.033, # [sec]
                      'next': 0}
        self.active = []
        self.__move = None
        self.__dirty = 1
        self.__inactive = False

    def is_active(self):
        if self.__inactive:
            return 0
        else:
            return 1

    def set_base_id(self, window, surface_id):
        if surface_id == -2:
            self.terminate(window)
            self.base_id = window.surface_id
        else:
            self.base_id = surface_id
        self.__dirty = 1

    def move_surface(self, xoffset, yoffset):
        self.__move = (xoffset, yoffset)

    def append_actor(self, frame, actor):
        self.active.append((frame, actor))

    def set_animation_quality(self, quality, window):
        if self.timeout_id is not None:
            gobject.source_remove(self.timeout_id)
        if quality is None:
            quality = 1.0
        step = 0.033 / quality ## FIXME
        self.frame['step'] = step
        if self.timeout_id is not None:
            self.timeout_id = gobject.timeout_add(int(step * 1000.0), self.update, window)

    def set_inactive(self, flag, window):
        self.__inactive = True if flag else False
        self.reset(window, window.surface_id)

    def update_frame(self, window):
        if self.terminate_flag:
            return False
        if len(window.frame_buffer) > 1.0 / self.frame['step']: # XXX
            return True
        self.frame['next'] += self.frame['step']
        frame, actor = self.get_actor_next(window)
        last_actor = actor
        while actor is not None:
            actor.update(window, frame)
            last_actor = actor
            frame, actor = self.get_actor_next(window)
        self.update_surface_frame_buffer(window)
        if last_actor is not None and last_actor.exclusive and \
           last_actor.terminate_flag and self.exclusive_actor is None: # XXX
            self.invoke_restart(window)
        self.idle_id = gobject.timeout_add(10, self.update_frame, window) # XXX
        return False

    def get_actor_next(self, window):
        if self.active:
            self.active.sort()
            if self.active[0][0] <= self.frame['next']:
                return self.active.pop(0)
        return None, None

    def update_surface_frame_buffer(self, window):
        window.update_frame_buffer(
            self, self.frame['next'], self.__move, self.__dirty)
        self.__dirty = 0

    def update(self, window):
        if self.terminate_flag:
            return False
        if not self.__inactive:
            window.update()
        self.timeout_id = gobject.timeout_add(int(self.frame['step'] * 1000), self.update, window)
        return False

    def lock_exclusive(self, window, actor):
        assert self.exclusive_actor is None
        self.terminate(window)
        self.exclusive_actor = actor
        actor.set_post_proc(self.unlock_exclusive, (window, actor))
        self.update_frame(window) # XXX
        self.__dirty = 1

    def unlock_exclusive(self, window, actor):
        assert self.exclusive_actor == actor
        self.exclusive_actor = None

    def reset_overlays(self):
        self.overlays = {}
        self.__dirty = 1

    def remove_overlay(self, actor):
        try:
            del self.overlays[actor]
        except KeyError:
            pass
        self.__dirty = 1

    def add_overlay(self, window, actor, pixbuf_id, x, y):
        if pixbuf_id == '-2':
            self.terminate(window)
        if pixbuf_id in ['-1', '-2']:
            self.remove_overlay(actor)
            return
        self.overlays[actor] = (pixbuf_id, x, y)
        self.__dirty = 1

    def invoke_actor(self, window, actor):
        if self.exclusive_actor is not None:
            interval = actor.get_interval()
            if interval.startswith('talk') or interval == 'yen-e':
                return
            self.queue.append(actor)
            return
        if actor.exclusive:
            self.lock_exclusive(window, actor)
        actor.invoke(window, self.frame['next'])

    def invoke(self, window, actor_id, update=0):
        if self.base_id not in self.seriko:
            return
        for actor in self.seriko[self.base_id]:
            if actor_id == actor.get_id():
                self.invoke_actor(window, actor)
                break

    def invoke_yen_e(self, window, surface_id):
        if surface_id not in self.seriko:
            return
        for actor in self.seriko[surface_id]:
            if actor.get_interval() == 'yen-e':
                self.invoke_actor(window, actor)
                break

    def invoke_talk(self, window, surface_id, count):
        if surface_id not in self.seriko:
            return 0
        interval_count = None
        for actor in self.seriko[surface_id]:
            interval = actor.get_interval()
            if interval.startswith('talk'):
                interval_count = int(interval[5]) # XXX
                break
        if interval_count is not None and count >= interval_count:
            self.invoke_actor(window, actor)
            return 1
        else:
            return 0

    def invoke_runonce(self, window):
        if self.base_id not in self.seriko:
            return
        for actor in self.seriko[self.base_id]:
            if actor.get_interval() == 'runonce':
                self.invoke_actor(window, actor)

    def invoke_always(self, window):
        if self.base_id not in self.seriko:
            return
        for actor in self.seriko[self.base_id]:
            interval = actor.get_interval()
            if interval in ['always', 'sometimes', 'rarely'] or \
               interval.startswith('random'):
                self.invoke_actor(window, actor)

    def invoke_restart(self, window):
        if self.base_id not in self.seriko:
            return
        for actor in self.seriko[self.base_id]:
            if actor in self.queue:
                self.queue.remove(actor)
                self.invoke_actor(window, actor)

    def invoke_kinoko(self, window): # XXX
        if self.base_id not in self.seriko:
            return
        for actor in self.seriko[self.base_id]:
            if actor.get_interval() in ['always', 'runonce',
                                        'sometimes', 'rarely',]:
                self.invoke_actor(window, actor)

    def reset(self, window, surface_id):
        self.queue = []
        self.terminate(window)
        self.frame['next'] = 0
        self.set_base_id(window, window.surface_id)
        if surface_id in self.seriko:
            self.base_id = surface_id
        else:
            self.base_id = window.surface_id
        if self.base_id in self.seriko:
            surface_ids = []
            for actor in self.seriko[self.base_id]:
                for surface_id in actor.get_surface_ids():
                    if surface_id not in surface_ids:
                        surface_ids.append(surface_id)
            if surface_ids:
                window.prefetch(surface_ids)
        self.update_frame(window)

    def start(self, window):
        self.invoke_runonce(window)
        self.invoke_always(window)
        self.timeout_id = gobject.timeout_add(int(self.frame['step'] * 1000), self.update, window)

    def terminate(self, window):
        self.terminate_flag = 1
        if self.timeout_id is not None:
            gobject.source_remove(self.timeout_id)
            self.timeout_id = None
        if self.idle_id is not None:
            gobject.source_remove(self.idle_id)
            self.idle_id = None
        if self.base_id in self.seriko:
            for actor in self.seriko[self.base_id]:
                actor.terminate()
        self.reset_overlays()
        self.active = []
        self.__move = None
        self.__dirty = 1
        window.clear_frame_buffer()
        self.terminate_flag = 0

    def iter_overlays(self):
        actors = self.overlays.keys()
        actors[:] = [(actor.get_id(), actor) for actor in actors]
        actors.sort()
        actors[:] = [actor for actor_id, actor in actors]
        for actor in actors:
            pixbuf_id, x, y = self.overlays[actor]
            ##print 'actor=%d, id=%s, x=%d, y=%d' % (actor.get_id(), pixbuf_id, x, y)
            yield pixbuf_id, x, y

class Actor:

    def __init__(self, actor_id, interval):
        self.id = actor_id
        self.interval = interval
        self.patterns = []
        self.last_method = None
        self.exclusive = 0
        self.post_proc = None
        self.terminate_flag = 1

    def set_post_proc(self, proc, args):
        assert self.post_proc is None
        self.post_proc = (proc, args)

    def set_exclusive(self):
        self.exclusive = 1

    def get_id(self):
        return self.id

    def get_interval(self):
        return self.interval

    def get_patterns(self):
        return self.patterns

    def add_pattern(self, surface, interval, method, args):
        self.patterns.append((surface, interval, method, args))

    def invoke(self, base_frame):
        self.terminate_flag = 0

    def update(self, window, base_frame):
        if self.terminate_flag:
            return False
        pass

    def terminate(self):
        self.terminate_flag = 1
        if self.post_proc is not None:
            proc, args = self.post_proc
            self.post_proc = None
            proc(*args)

    def get_surface_ids(self):
        surface_ids = []
        for surface, interval, method, args in self.patterns:
            if method == 'base':
                surface_ids.append(surface)
        return surface_ids

    def show_pattern(self, window, surface, method, args):
        if self.last_method in ['overlay', 'overlayfast']:
            window.remove_overlay(self)
        if method == 'move':
            window.seriko.move_surface(args[0], args[1]) ## FIXME
        elif method in ['overlay', 'overlayfast']:
            window.add_overlay(self, surface, args[0], args[1])
        elif method == 'base':
            window.seriko.set_base_id(window, surface) ## FIXME
        elif method == 'start':
            window.invoke(args[0], update=1)
        elif method == 'alternativestart':
            window.invoke(random.choice(args), update=1)
        else:
            raise RuntimeError, 'should not reach here'
        self.last_method = method

class ActiveActor(Actor): # always

    def __init__(self, actor_id, interval):
        Actor.__init__(self, actor_id, interval)
        self.wait = 0
        self.pattern = 0

    def invoke(self, window, base_frame):
        self.terminate()
        self.terminate_flag = 0
        self.pattern = 0
        self.update(window, base_frame)

    def update(self, window, base_frame):
        if self.terminate_flag:
            return False
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.pattern += 1
        if self.pattern == len(self.patterns):
            self.pattern = 0
        self.show_pattern(window, surface, method, args)
        window.append_actor(base_frame + interval / 1000.0, self)
        return False

class RandomActor(Actor): # sometimes, rarely, randome

    def __init__(self, actor_id, interval, wait_min, wait_max):
        Actor.__init__(self, actor_id, interval)
        self.wait_min = wait_min
        self.wait_max = wait_max
        self.reset()

    def reset(self):
        self.wait = random.randint(self.wait_min, self.wait_max)
        self.pattern = 0

    def invoke(self, window, base_frame):
        self.terminate()
        self.terminate_flag = 0
        self.reset()
        window.append_actor(base_frame + self.wait / 1000.0, self)

    def update(self, window, base_frame):
        if self.terminate_flag:
            return False
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.pattern += 1
        if self.pattern < len(self.patterns):
            self.wait = interval
        else:
            self.reset()
        self.show_pattern(window, surface, method, args)
        window.append_actor(base_frame + self.wait / 1000.0, self)
        return False

class OneTimeActor(Actor): # runone

    def __init__(self, actor_id, interval):
        Actor.__init__(self, actor_id, interval)
        self.wait = -1
        self.pattern = 0

    def invoke(self, window, base_frame):
        self.terminate()
        self.terminate_flag = 0
        self.wait = 0
        self.pattern = 0
        self.update(window, base_frame)

    def update(self, window, base_frame):
        if self.terminate_flag:
            return False
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.pattern += 1
        if self.pattern < len(self.patterns):
            self.wait = interval
        else:
            self.wait = -1 # done
            self.terminate()
        self.show_pattern(window, surface, method, args)
        if self.wait >= 0:
            window.append_actor(base_frame + self.wait / 1000.0, self)
        return False

class PassiveActor(Actor): # never, yen-e, talk

    def __init__(self, actor_id, interval):
        Actor.__init__(self, actor_id, interval)
        self.wait = -1

    def invoke(self, window, base_frame):
        self.terminate()
        self.terminate_flag = 0
        self.wait = 0
        self.pattern = 0
        self.update(window, base_frame)

    def update(self, window, base_frame):
        if self.terminate_flag:
            return False
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.pattern += 1
        if self.pattern < len(self.patterns):
            self.wait = interval
        else:
            self.wait = -1 # done
            self.terminate()
        self.show_pattern(window, surface, method, args)
        if self.wait >= 0:
            window.append_actor(base_frame + self.wait / 1000.0, self)
        return False

class Mayuna(Actor):

    def set_exclusive(self):
        pass

    def show_pattern(self, window, surface, method, args):
        pass


re_seriko_interval = re.compile('^([0-9]+)interval$')
re_seriko_interval_value = re.compile('^(sometimes|rarely|random,[0-9]+|always|runonce|yesn-e|talk,[0-9]+|never)$')
re_seriko_pattern = re.compile(r'^([0-9]+|-[12])\s*,\s*([+-]?[0-9]+)\s*,\s*(overlay|overlayfast|base|move|start|alternativestart|)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\[[0-9]+(\.[0-9]+)*\])?$')

re_seriko2_interval = re.compile('^animation([0-9]+)\.interval$')
re_seriko2_interval_value = re.compile('^(sometimes|rarely|random,[0-9]+|always|runonce|yesn-e|talk,[0-9]+|never)$')
re_seriko2_pattern = re.compile(r'^(overlay|overlayfast|base|move|start|alternativestart|)\s*,\s*([0-9]+|-[12])\s*,\s*([+-]?[0-9]+)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\([0-9]+(\.[0-9]+)*\))?$')

def get_actors(config):
    version = None
    buf = []
    for key, value in config.iteritems():
        if version == 1:
            match = re_seriko_interval.match(key)
        elif version == 2:
            match = re_seriko2_interval.match(key)
        else:
            match1 = re_seriko_interval.match(key)
            match2 = re_seriko2_interval.match(key)
            if match1:
                version = 1
                match = match1
            elif match2:
                version = 2
                match = match2
            else:
                continue
        if not match:
            continue
        if version == 1 and not re_seriko_interval_value.match(value):
            continue
        if version == 2 and not re_seriko2_interval_value.match(value):
            continue
        buf.append((int(match.group(1)), value))
    actors = []
    for actor_id, interval in buf:
        if interval == 'always':
            actor = ActiveActor(actor_id, interval)
        elif interval == 'sometimes':
            actor = RandomActor(actor_id, interval, 0, 10000) # 0 to 10 seconds
        elif interval == 'rarely':
            actor = RandomActor(actor_id, interval, 20000, 60000)
        elif interval.startswith('random'):
            actor = RandomActor(actor_id, interval, 0, 1000 * int(interval[7]))
        elif interval == 'runonce':
            actor = OneTimeActor(actor_id, interval)
        elif interval == 'yen-e':
            actor = PassiveActor(actor_id, interval)
        elif interval.startswith('talk'):
            actor = PassiveActor(actor_id, interval)
        elif interval == 'never':
            actor = PassiveActor(actor_id, interval)
        if version == 1:
            key = ''.join((str(actor_id), 'option'))
        else:
            key = ''.join(('animation', str(actor_id), '.option'))
        if key in config and config[key] == 'exclusive':
            actor.set_exclusive()
        try:
            for n in range(128): # up to 128 patterns (0 - 127)
                if version == 1:
                    key = ''.join((str(actor_id), 'pattern', str(n)))
                else:
                    key = ''.join(('animation', str(actor_id), '.pattern', str(n)))
                if key not in config:
                    key = ''.join((str(actor_id), 'patturn', str(n))) # only for version 1
                    if key not in config:
                        continue # XXX
                pattern = config[key]
                if version == 1:
                    match = re_seriko_pattern.match(pattern)
                else:
                    match = re_seriko2_pattern.match(pattern)
                if not match:
                    raise ValueError, 'unsupported pattern: %s' % pattern
                if version == 1:
                    surface = str(int(match.group(1)))
                    interval = abs(int(match.group(2))) * 10
                    method = match.group(3)
                else:
                    method = match.group(1)
                    surface = str(int(match.group(2)))
                    interval = abs(int(match.group(3)))
                if method == '':
                    method = 'base'
                if method == 'start':
                    group = match.group(4)
                    if group is None:
                        raise ValueError, 'syntax error: %s' % pattern
                    args = [int(group)]
                elif method == 'alternativestart':
                    args = match.group(6)
                    if args is None:
                        raise ValueError, 'syntax error: %s' % pattern
                    args = [int(s) for s in args[1:-1].split('.')]
                else:
                    if surface in ['-1', '-2']:
                        x = 0
                        y = 0
                    else:
                        x = int(match.group(4) or 0)
                        y = int(match.group(5) or 0)
                    args = [x, y]
                actor.add_pattern(surface, interval, method, args)
        except ValueError, error:
            print_error(''.join(('seriko.py: ', str(error))))
            continue
        if not actor.get_patterns():
            print_error(
                'seriko.py: animation group #%d has no pattern (ignored)' % actor_id)
            continue
        actors.append(actor)
    actors[:] = [(actor.get_id(), actor) for actor in actors]
    actors.sort()
    actors[:] = [actor for actor_id, actor in actors]
    return actors

re_mayuna_interval = re.compile('^([0-9]+)interval$')
re_mayuna_interval_value = re.compile('^(bind)$')
re_mayuna_pattern = re.compile(r'^([0-9]+|-[12])\s*,\s*([0-9]+)\s*,\s*(bind|add|reduce|insert)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\[[0-9]+(\.[0-9]+)*\])?$')

re_mayuna2_interval = re.compile('^animation([0-9]+)\.interval$')
re_mayuna2_interval_value = re.compile('^(bind)$')
re_mayuna2_pattern = re.compile(r'^(bind|add|reduce|insert)\s*,\s*([0-9]+|-[12])\s*,\s*([0-9]+)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\([0-9]+(\.[0-9]+)*\))?$')

def get_mayuna(config):
    version = None
    buf = []
    for key, value in config.iteritems():
        if version == 1:
            match = re_mayuna_interval.match(key)
        elif version == 2:
            match = re_mayuna2_interval.match(key)
        else:
            match1 = re_mayuna_interval.match(key)
            match2 = re_mayuna2_interval.match(key)
            if match1:
                version = 1
                match = match1
            elif match2:
                version = 2
                match = match2
            else:
                continue
        if not match:
            continue
        if version == 1 and not re_mayuna_interval_value.match(value):
            continue
        if version == 2 and not re_mayuna2_interval_value.match(value):
            continue
        buf.append((int(match.group(1)), value))
    mayuna = []
    for mayuna_id, interval in buf:
        ##assert interval == 'bind'
        actor = Mayuna(mayuna_id, interval)
        try:
            for n in range(128): # up to 128 patterns (0 - 127)
                if version == 1:
                    key = ''.join((str(mayuna_id), 'pattern', str(n)))
                else:
                    key = ''.join(('animation', str(mayuna_id), '.pattern', str(n)))
                if key not in config:
                    key = ''.join((str(mayuna_id), 'patturn', str(n))) # only for version 1
                    if key not in config:
                        continue # XXX
                pattern = config[key]
                if version == 1:
                    match = re_mayuna_pattern.match(pattern)
                else:
                    match = re_mayuna2_pattern.match(pattern)
                if not match:
                    raise ValueError, 'unsupported pattern: %s' % pattern
                if version == 1:
                    surface = str(int(match.group(1)))
                    interval = abs(int(match.group(2))) * 10
                    method = match.group(3)
                else:
                    method = match.group(1)
                    surface = str(int(match.group(2)))
                    interval = abs(int(match.group(3)))
                if method not in ['bind', 'add', 'reduce', 'insert']:
                    continue
                else:
                    if surface in ['-1', '-2']:
                        x = 0
                        y = 0
                    else:
                        x = int(match.group(4) or 0)
                        y = int(match.group(5) or 0)
                    args = [x, y]
                actor.add_pattern(surface, interval, method, args)
        except ValueError, error:
            print_error(''.join(('seriko.py: ', str(error))))
            continue
        if not actor.get_patterns():
            print_error(
                'seriko.py: animation group #%d has no pattern (ignored)' % mayuna_id)
            continue
        mayuna.append(actor)
    mayuna[:] = [(actor.get_id(), actor) for actor in mayuna]
    mayuna.sort()
    mayuna[:] = [actor for actor_id, actor in mayuna]
    return mayuna

# find ~/.ninix -name 'surface*a.txt' | xargs python seriko.py
def test():
    import ninix.config
    if len(sys.argv) == 1:
        print 'Usage:', sys.argv[0], '[surface??a.txt ...]'
    for filename in sys.argv[1:]:
        print 'Reading', filename, '...'
        for actor in get_actors(ninix.config.open(filename)):
            print '#%d' % actor.get_id(),
            print actor.__class__.__name__,
            print '(%s)' % actor.get_interval()
            print 'number of patterns =', len(actor.get_patterns())
            for pattern in actor.get_patterns():
                print 'surface=%s, interval=%d, method=%s, args=%s' % pattern
        for actor in get_mayuna(ninix.config.open(filename)):
            print '#%d' % actor.get_id(),
            print actor.__class__.__name__,
            print '(%s)' % actor.get_interval()
            print 'number of patterns =', len(actor.get_patterns())
            for pattern in actor.get_patterns():
                print 'surface=%s, interval=%d, method=%s, args=%s' % pattern

if __name__ == '__main__':
    test()
