# -*- coding: utf-8 -*-
#
#  niseshiori.py - a "偽栞" compatible Shiori module for ninix
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import __builtin__
import random
import time
import sys
import os
import re
import StringIO


REVISION = '$Revision: 1.28 $'

# tree node types
ADD_EXPR     = 1
MUL_EXPR     = 2
UNARY_EXPR   = 3
PRIMARY_EXPR = 4

def list_dict(top_dir):
    buf = []
    try:
        filelist = os.listdir(top_dir)
    except OSError:
        filelist = []
    re_dict_filename = re.compile(r'^ai.*\.(dtx|txt)$')
    for filename in filelist:
        if re_dict_filename.match(filename):
            buf.append(os.path.join(top_dir, filename))
    return buf


class NiseShiori:

    def __init__(self, debug=0):
        self.debug = debug
        self.dict = {}
        self.type_chains = {}
        self.word_chains = {}
        self.keywords = {}
        self.responses = {}
        self.greetings = {}
        self.events = {}
        self.resources = {}
        self.variables = {}
        self.dbpath = None
        self.expr_parser = ExprParser()
        self.username = ''
        self.ai_talk_interval = 180
        self.ai_talk_count = 0
        self.surf0 = 0
        self.surf1 = 10
        self.event = None
        self.reference = None
        self.jump_entry = None
        self.motion_area = None
        self.motion_count = 0
        self.mikire = 0
        self.kasanari = 0
        self.otherghost = []
        self.to = ''
        self.sender = ''

    def load(self, top_dir):
        # read dictionaries
        dictlist = list_dict(top_dir)
        if not dictlist:
            return 1
        for path in dictlist:
            self.read_dict(path)
        # read variables
        self.dbpath = os.path.join(top_dir, 'niseshiori.db')
        self.load_database(self.dbpath)
        return 0

    def load_database(self, path):
        try:
            f = __builtin__.open(path)
        except IOError:
            return
        dic = {}
        for line in f:
            if line.startswith('# ns_st: '):
                try:
                    self.ai_talk_interval = int(line[9:])
                except ValueError:
                    pass
                continue
            elif line.startswith('# ns_tn: '):
                self.username = line[9:].strip()
                continue
            try:
                name, value = [s.strip() for s in line.split('=')]
            except ValueError:
                print 'niseshiori.py: malformed database (ignored)'
                return
            dic[name] = value
        self.variables = dic

    def save_database(self):
        if self.dbpath is None:
            return
        try:
            f = __builtin__.open(self.dbpath, 'w')
        except IOError:
            print 'niseshiori.py: cannot write database (ignored)'
            return
        f.write('# ns_st: %d\n' % self.ai_talk_interval)
        f.write('# ns_tn: %s\n' % self.username)
        for name, value in self.variables.iteritems():
            if not name.startswith('_'):
                f.write('%s=%s\n' % (name, str(value)))
        f.close()

    def finalize(self):
        pass

    re_type = re.compile(r'\\(m[szlchtep?]|[dk])')
    re_user = re.compile(r'\\u[a-z]')
    re_category = re.compile(r'\\(m[szlchtep]|[dk])?\[([^\]]+)\]')

    def read_dict(self, path):
        # read dict file and decrypt if necessary
        f = __builtin__.open(path)
        if path.endswith('.dtx'):
            buf = self.decrypt(f.read())
        else:
            buf = f.readlines()
        f.close()
        # omit empty lines and comments and merge continuous lines
        definitions = []
        decode = lambda line: unicode(line, 'Shift_JIS', 'replace')
        in_comment = 0
        i, j = 0, len(buf)
        while i < j:
            line = buf[i].strip()
            i += 1
            if not line:
                continue
            elif i == 1 and line.startswith('#Charset:'):
                charset = line[9:].strip()
                if charset == 'UTF-8':
                    decode = lambda line: unicode(line, 'utf-8')
                elif charset in ['EUC-JP', 'EUC-KR']:
                    decode = lambda line: unicode(line, charset)
                continue
            elif line.startswith('/*'):
                in_comment = 1
                continue
            elif line.startswith('*/'):
                in_comment = 0
                continue
            elif in_comment or line.startswith('#') or line.startswith('//'):
                continue
            lines = [line]
            while i < j and buf[i] and \
                  (buf[i].startswith(' ') or buf[i].startswith('\t')):
                lines.append(buf[i].strip())
                i += 1
            definitions.append(''.join(lines))
        # parse each line
        for line in definitions:
            line = decode(line)
            # special case: words in a category
            match = self.re_category.match(line)
            if match:
                line = line[match.end():].strip()
                if not line or not line.startswith(','):
                    self.syntax_error(path, line)
                    continue
                words = [s.strip() for s in self.split(line) if s.strip()]
                cattype, catlist = match.groups()
                for cat in [s.strip() for s in self.split(catlist)]:
                    if cattype is None:
                        keylist = [(None, cat)]
                    else:
                        keylist = [(None, cat), (cattype, cat)]
                    for key in keylist:
                        value = self.dict.get(key, [])
                        value.extend(words)
                        self.dict[key] = value
                if cattype is not None:
                    key = ''.join(('\\', cattype))
                    value = self.dict.get(key, [])
                    value.extend(words)
                    self.dict[key] = value
                continue
            # other cases
            try:
                command, argv = [s.strip() for s in self.split(line, 1)]
            except ValueError:
                self.syntax_error(path, line)
                continue
            if command == r'\ch':
                argv = [s.strip() for s in self.split(argv)]
                if len(argv) == 5:
                    t1, w1, t2, w2, c = argv
                elif len(argv) == 4:
                    t1, w1, t2, w2 = argv
                    c = None
                elif len(argv) == 3:
                    t1, w1, c = argv
                    t2 = w2 = None
                else:
                    self.syntax_error(path, line)
                    continue
                if not self.re_type.match(t1) and not self.re_user.match(t1):
                    self.syntax_error(path, line)
                    continue
                if c is not None:
                    ch_list = self.type_chains.get(t1, [])
                    ch_list.append((c, w1))
                    self.type_chains[t1] = ch_list
                if t2 is None:
                    continue
                if not self.re_type.match(t2) and not self.re_user.match(t2):
                    self.syntax_error(path, line)
                    continue
                if c is not None:
                    ch_list = self.type_chains.get(t2, [])
                    ch_list.append((c, w2))
                    self.type_chains[t2] = ch_list
                m1 = ''.join(('%', t1[1:]))
                m2 = ''.join(('%', t2[1:]))
                key = (m1, w1)
                dic = self.word_chains.get(key, {})
                ch_list = dic.get(m2, [])
                if (c, w2) not in ch_list:
                    ch_list.append((c, w2))
                    dic[m2] = ch_list
                    self.word_chains[key] = dic
                key = (m2, w2)
                dic = self.word_chains.get(key, {})
                ch_list = dic.get(m1, [])
                if (c, w1) not in ch_list:
                    ch_list.append((c, w1))
                    dic[m1] = ch_list
                    self.word_chains[key] = dic
                ch_list = self.dict.get(t1, [])
                if w1 not in ch_list:
                    ch_list.append(w1)
                    self.dict[t1] = ch_list
                ch_list = self.dict.get(t2, [])
                if w2 not in ch_list:
                    ch_list.append(w2)
                    self.dict[t2] = ch_list
            elif self.re_type.match(command) or self.re_user.match(command):
                words = [s.strip() for s in self.split(argv) if s.strip()]
                value = self.dict.get(command, [])
                value.extend(words)
                self.dict[command] = value
            elif command in [r'\dms', r'\e']:
                value = self.dict.get(command, [])
                value.append(argv)
                self.dict[command] = value
            elif command == r'\ft':
                argv = [s.strip() for s in self.split(argv, 2)]
                if len(argv) != 3:
                    self.syntax_error(path, line)
                    continue
                w, t, s = argv
                if not self.re_type.match(t):
                    self.syntax_error(path, line)
                    continue
                self.keywords[(w, t)] = s
            elif command == r'\re':
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    cond = self.parse_condition(argv[0])
                    re_list = self.responses.get(cond, [])
                    re_list.append(argv[1])
                    self.responses[cond] = re_list
            elif command == r'\hl':
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    hl_list = self.greetings.get(argv[0], [])
                    hl_list.append(argv[1])
                    self.greetings[argv[0]] = hl_list
            elif command == r'\ev':
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    cond = self.parse_condition(argv[0])
                    ev_list = self.events.get(cond, [])
                    ev_list.append(argv[1])
                    self.events[cond] = ev_list
            elif command == r'\id':
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    if argv[0] in ['sakura.recommendsites',
                                   'kero.recommendsites',
                                   'sakura.portalsites']:
                        id_list = self.resources.get(argv[0], '')
	                if id_list:
                            id_list = ''.join((id_list, '\2'))
	                id_list = ''.join((id_list, argv[1].replace(' ', '\1')))
                        self.resources[argv[0]] = id_list
                    else:
                        self.resources[argv[0]] = argv[1]
            elif command == r'\tc':
                pass
            else:
                self.syntax_error(path, line)

    def split(self, line, maxcount=None):
        buf = []
        count = 0
        end = pos = 0
        while maxcount is None or count < maxcount:
            pos = line.find(',', pos)
            if pos < 0:
                break
            elif pos > 0 and line[pos - 1] == '\\':
                pos += 1
            else:
                buf.append(line[end:pos])
                count += 1
                end = pos = pos + 1
        buf.append(line[end:])
        return [s.replace('\\,', ',') for s in  buf]

    def syntax_error(self, path, line):
        if self.debug & 4:
            print 'niseshiori.py: syntax error in', os.path.basename(path)
            print line

    re_comp_op = re.compile('<[>=]?|>=?|=')
    COND_COMPARISON = 1
    COND_STRING     = 2

    def parse_condition(self, condition):
        buf = []
        for expr in [s.strip() for s in condition.split('&')]:
            match = self.re_comp_op.search(expr)
            if match:
                buf.append((self.COND_COMPARISON, (
                    expr[:match.start()].strip(),
                    match.group(),
                    expr[match.end():].strip())))
            else:
                buf.append((self.COND_STRING, expr))
        return tuple(buf)

    def decrypt(self, data):
        buf = []
        a = 0x61
        i = 0
        j = len(data)
        line = []
        while i < j:
            if data[i] == '@':
                i += 1
                buf.append(''.join(line))
                line = []
                continue
            y = ord(data[i])
            i += 1
            x = ord(data[i])
            i += 1
            x -= a
            a += 9
            y -= a
            a += 2
            if a > 0xdd:
                a = 0x61
            line.append(chr((x & 0x03) | ((y & 0x03) << 2) | \
                            ((y & 0x0c) << 2) | ((x & 0x0c) << 4)))
        return buf

    # SHIORI/1.0 API
    def getaistringrandom(self):
        result = self.get_event_response('OnNSRandomTalk') or self.get(r'\e')
        return result.encode('utf-8', 'ignore')

    def getaistringfromtargetword(self, word):
        word = unicode(word, 'utf-8', 'ignore')
        return self.get(r'\e').encode('utf-8', 'ignore') # XXX

    def getdms(self):
        return self.get(r'\dms').encode('utf-8', 'ignore')

    def getword(self, word_type):
        return self.get(''.join(('\\', word_type))).encode('utf-8', 'ignore')

    # SHIORI/2.4 API
    def teach(self, word):
        return None

    # SHIORI/2.5 API
    def getstring(self, name):
        name = unicode(name, 'utf-8', 'ignore')
        result = self.resources.get(name)
        if result is not None:
            result = result.encode('utf-8', 'ignore')
        return result

    # SHIORI/2.2 API
    def get_event_response(self, event,
                           ref0=None, ref1=None, ref2=None, ref3=None,
                           ref4=None, ref5=None, ref6=None, ref7=None):
        def proc(ref):
            if ref is not None and \
               not isinstance(ref, int) and not isinstance(ref, float):
                ref = unicode(ref, 'utf-8', 'ignore')
            return ref
        ref = [proc(ref) for ref in [ref0, ref1, ref2, ref3, ref4,
                                     ref5, ref6, ref7]]
        if event in ['OnSecondChange', 'OnMinuteChange']:
            if ref[1]:
                self.mikire += 1
                script = self.get_event_response('OnNSMikireHappen')
                if script is not None:
                    return script
            elif self.mikire > 0:
                self.mikire = 0
                return self.get_event_response('OnNSMikireSolve')
            if ref[2]:
                self.kasanari += 1
                script = self.get_event_response('OnNSKasanariHappen')
                if script is not None:
                    return script
            elif self.kasanari > 0:
                self.kasanari = 0
                return self.get_event_response('OnNSKasanariHappen')
            if event == 'OnSecondChange' and self.ai_talk_interval > 0:
                self.ai_talk_count += 1
                if self.ai_talk_count == self.ai_talk_interval:
                    self.ai_talk_count = 0
                    if self.otherghost and \
                       random.randint(0, 10) == 0:
                        target = []
                        for name, s0, s1 in self.otherghost:
                            if name in self.greetings:
                                target.append(name)
                        if target:
                            self.to = random.choice(target)
                        if self.to:
                            self.current_time = time.localtime(time.time())
                            s = random.choice(self.greetings[self.to])
                            if s:
                                s = self.replace_meta(s)
                                while 1:
                                    match = self.re_ns_tag.search(s)
                                    if not match:
                                        break
                                    value = self.eval_ns_tag(match.group())
                                    s = ''.join((s[:match.start()],
                                                 str(value),
                                                 s[match.end():]))
                            return s.encode('utf-8', 'ignore')
                    return self.getaistringrandom()
            return None
        elif event == 'OnMouseMove':
            if self.motion_area != (ref[3], ref[4]):
                self.motion_area = (ref[3], ref[4])
                self.motion_count = 0
            else:
                self.motion_count += 5 # sensitivity
        elif event == 'OnSurfaceChange':
            self.surf0 = ref[0]
            self.surf1 = ref[1]
        elif event == 'OnUserInput' and ref[0] == 'ninix.niseshiori.username':
            self.username = ref[1]
            self.save_database()
            return '\e'
        elif event == 'OnCommunicate':
            self.event = ''
            self.reference = (ref[1], )
            if ref[0] == 'user':
                self.sender = 'User'
            else:
                self.sender = ref[0]
            candidate = []
            for cond in self.responses:
                if self.eval_condition(cond):
                    candidate.append(cond)
            script = None
            self.to = ref[0]
            if candidate:
                cond = random.choice(candidate)
                script = random.choice(self.responses[cond])
            if not script and 'nohit' in self.responses:
                script = random.choice(self.responses['nohit'])
            if not script:
                self.to = ''
            else:
                script = self.replace_meta(script)
                while 1:
                    match = self.re_ns_tag.search(script)
                    if not match:
                        break
                    value = self.eval_ns_tag(match.group())
                    script = ''.join((script[:match.start()],
                                      str(value),
                                      script[match.end():]))
            self.sender = ''
            return script.encode('utf-8', 'ignore')
        key = 'action'
        self.dict[key] = []
        self.event = event
        self.reference = tuple(ref)
        self.current_time = time.localtime(time.time())
        for condition, actions in self.events.iteritems():
            if self.eval_condition(condition):
                self.dict[key].extend(actions)
        script = self.get(key, default=None)
        if script is not None:
            self.ai_talk_count = 0
        self.event = None
        self.reference = None
        if script is not None:
            script = script.encode('utf-8', 'ignore')
        return script

    def otherghostname(self, ghost_list):
        self.otherghost = []
        for ghost in ghost_list:
            name, s0, s1 = ghost.split(chr(1))
            self.otherghost.append([unicode(name, 'utf-8', 'ignore'), s0, s1])
        return ''

    def communicate_to(self):
        to = self.to
        self.to = ''
        return to.encode('utf-8', 'ignore')

    def eval_condition(self, condition):
        for cond_type, expr in condition:
            if not self.eval_conditional_expression(cond_type, expr):
                return 0
        return 1

    def eval_conditional_expression(self, cond_type, expr):
        if cond_type == NiseShiori.COND_COMPARISON:
            value1 = self.expand_meta(expr[0])
            value2 = expr[2]
            try:
                op1 = int(value1)
                op2 = int(value2)
            except ValueError:
                op1 = str(value1)
                op2 = str(value2)
            if expr[1] == '>=':
                return op1 >= op2
            elif expr[1] == '<=':
                return op1 <= op2
            elif expr[1] == '>':
                return op1 > op2
            elif expr[1] == '<':
                return op1 < op2
            elif expr[1] == '=':
                return op1 == op2
            elif expr[1] == '<>':
                return op1 != op2
        elif cond_type == NiseShiori.COND_STRING:
            if expr in self.event:
                return 1
            for ref in self.reference:
                if ref is not None and expr in str(ref):
                    return 1
            return 0
        else:
            return 0

    re_ns_tag = re.compile(r'\\(ns_(st(\[[0-9]+\])?|cr|hl|rf\[[^\]]+\]|ce|tc\[[^\]]+\]|tn(\[[^\]]+\])?|jp\[[^\]]+\]|rt)|set\[[^\]]+\])')

    def get(self, key, default=''):
        self.current_time = time.localtime(time.time())
        s = self.expand(key, '', default)
        if s:
            while 1:
                match = self.re_ns_tag.search(s)
                if not match:
                    break
                value = self.eval_ns_tag(match.group())
                s = ''.join((s[:match.start()], str(value), s[match.end():]))
        return s

    def eval_ns_tag(self, tag):
        value = ''
        if tag == r'\ns_cr':
            self.ai_talk_count = 0
        elif tag == r'\ns_ce':
            self.to = ''
        elif tag == r'\ns_hl':
            if self.otherghost:
                self.to = random.choice(self.otherghost)[0]
        elif tag.startswith(r'\ns_st[') and tag.endswith(']'):
            try:
                num = int(tag[7:-1])
            except ValueError:
                pass
            else:
                if num == 0:
                    self.ai_talk_interval = 0
                elif num == 1:
                    self.ai_talk_interval = 420
                elif num == 2:
                    self.ai_talk_interval = 180
                elif num == 3:
                    self.ai_talk_interval = 60
                else:
                    self.ai_talk_interval = min(max(num, 4), 999)
                self.save_database()
                self.ai_talk_count = 0
        elif tag.startswith(r'\ns_jp[') and tag.endswith(']'):
            name = tag[7:-1]
            self.jump_entry = name
            value = self.get_event_response('OnNSJumpEntry') or ''
            self.jump_entry = None
        elif tag.startswith(r'\set[') and tag.endswith(']'):
            statement = tag[5:-1]
            if '=' not in statement:
                if self.debug & 1024:
                    print 'niseshiori.py: syntax error'
                    print tag
            else:
                name, expr = [x.strip() for x in statement.split('=', 1)]
                value = self.eval_expression(expr)
                if value is not None:
                    self.variables[name] = value
                    if not name.startswith('_'):
                        self.save_database()
                    if self.debug & 2048:
                        print 'set %s = "%s"' % (name, value)
                elif self.debug & 1024:
                    print 'niseshiori.py: syntax error'
                    print tag
        elif tag == r'\ns_rt':
            value = r'\a'
        elif tag == r'\ns_tn':
            value = r'\![open,inputbox,ninix.niseshiori.username,-1]'
        elif tag.startswith(r'\ns_tn[') and tag.endswith(']'):
            self.username = tag[7:-1]
            self.save_database()
        return value

    re_meta = re.compile(r'%((rand([1-9]|\[[0-9]+\])|selfname2?|sakuraname|keroname|username|friendname|year|month|day|hour|minute|second|week|ghostname|sender|ref[0-7]|surf[01]|word|ns_st|get\[[^\]]+\]|jpentry|plathome|platform|move|mikire|kasanari|ver)|(dms|(m[szlchtep?]|[dk])(\[[^\]]*\])?|\[[^\]]*\]|u[a-z])([2-9]?))')

    def replace_meta(self, s):
        pos = 0
        buf = []
        variables = {}
        variable_chains = []
        while 1:
            match = self.re_meta.search(s, pos)
            if not match:
                buf.append(s[pos:])
                break
            buf.append(s[pos:match.start()])
            meta = match.group()
            if match.group(4) is not None: # %ms, %dms, %ua, etc.
                if meta not in variables:
                    chained_meta = ''.join(('%', match.group(4)))
                    for chains in variable_chains:
                        if chained_meta in chains:
                            candidates_A, candidates_B = chains[chained_meta]
                            if candidates_A:
                                word = random.choice(candidates_A)
                                candidates_A.remove(word)
                            else:
                                word = random.choice(candidates_B)
                                candidates_B.remove(word)
                            if not candidates_A and not candidates_B:
                                del chains[chained_meta]
                            if self.debug & 16:
                                print 'chained: %s => %s' % (meta, word)
                            break
                    else:
                        if match.group(4) == 'm?':
                            word = self.expand(
                                ''.join(('\\',
                                         random.choice(['ms', 'mz', 'ml',
                                                        'mc', 'mh', 'mt',
                                                        'me', 'mp']))), s)
                        else:
                            word = self.expand(
                                ''.join(('\\', match.group(4))), s)
                    chains = self.find_chains((chained_meta, word), s)
                    if self.debug & 16:
                        prefix = 'chain:'
                        for k, (candidates_A, candidates_B) in chains.iteritems():
                            for w in candidates_A:
                                print prefix, '%s, %s' % (k, w)
                                prefix = '      '
                            for w in candidates_B:
                                print prefix, '%s, %s' % (k, w)
                                prefix = '      '
                    variables[meta] = word
                    variable_chains.append(chains)
                buf.append(variables[meta])
            else:
                buf.append(str(self.expand_meta(meta)))
            pos = match.end()
        t = ''.join(buf)
        return t

    def expand(self, key, context, default=''):
        choices = []
        for keyword, word in self.type_chains.get(key, []):
            if keyword in context:
                if self.debug & 16:
                    print 'chain keyword:', keyword
                choices.append(word)
        if not choices:
            match = self.re_category.match(key)
            if match:
                key = match.groups()
            choices = self.dict.get(key)
        if not choices:
            if self.debug & 8:
                if isinstance(key, tuple):
                    key = '(%s, %s)' % key
                sys.stderr.write('%s not found\n' % key)
            return default
        s = random.choice(choices)
        t = self.replace_meta(s)
        if self.debug & 16:
            if isinstance(key, tuple):
                key = '\\%s[%s]' % (key[0] or '', key[1])
            print key.encode('UTF-8', 'ignore'), '=>', \
                  s.encode('UTF-8', 'ignore')
            print ' ' * len(key),
            print '=>', t.encode('UTF-8', 'ignore')
        return t

    def find_chains(self, key, context):
        chains = {}
        dic = self.word_chains.get(key, {})
        for chained_meta in dic:
            candidates_A = []
            candidates_B = []
            for keyword, chained_word in dic[chained_meta]:
                if keyword and keyword in context:
                    candidates_A.append(chained_word)
                else:
                    candidates_B.append(chained_word)
            chains[chained_meta] = (candidates_A, candidates_B)
        return chains

    WEEKDAY_NAMES = ['mon', 'tue', 'wed', 'thu', 'fri', 'sat', 'sun']

    def expand_meta(self, name):
        if name in ['%selfname', '%selfname2', '%keroname', '%friendname']:
            result = name
        elif name == '%sakuraname':
            result = '%selfname'
        elif name == '%username':
            result = self.username or '%username'
        elif name.startswith('%get[') and name.endswith(']'):
            value = self.variables.get(name[5:-1], '?')
            try:
                result = int(value)
            except ValueError:
                result = value
        elif name.startswith('%rand[') and name.endswith(']'):
            n = int(name[6:-1])
            result = random.randint(1, n)
        elif name.startswith('%rand') and len(name) == 6:
            n = int(name[5])
            result = random.randint(10**(n - 1), 10**n - 1)
        elif name.startswith('%ref') and len(name) == 5 and \
             name[4] in '01234567':
            if self.reference is None:
                result = ''
            else:
                n = int(name[4])
                if self.reference[n] is None:
                    result = ''
                else:
                    result = self.reference[n]
        elif name == '%jpentry':
            if self.jump_entry is None:
                result = ''
            else:
                result = self.jump_entry
        elif name == '%year':
            result = str(self.current_time[0])
        elif name == '%month':
            result = str(self.current_time[1])
        elif name == '%day':
            result = str(self.current_time[2])
        elif name == '%hour':
            result = str(self.current_time[3])
        elif name == '%minute':
            result = str(self.current_time[4])
        elif name == '%second':
            result = str(self.current_time[5])
        elif name == '%week':
            result = self.WEEKDAY_NAMES[self.current_time[6]]
        elif name == '%ns_st':
            result = self.ai_talk_interval
        elif name == '%surf0':
            result = str(self.surf0)
        elif name == '%surf1':
            result = str(self.surf1)
        elif name in ['%plathome', '%platform']:
            result = 'ninix'
        elif name == '%move':
            result = str(self.motion_count)
        elif name == '%mikire':
            result = str(self.mikire)
        elif name == '%kasanari':
            result = str(self.kasanari)
        elif name == '%ver':
            if REVISION[1:11] == 'Revision: ':
                result = '偽栞 for ninix (rev.%s)' % REVISION[11:-2]
            else:
                result = '偽栞 for ninix'
        elif name == '%sender':
            if self.sender:
                result = self.sender
            else:
                result = ''
        elif name == '%ghost':
            if self.to:
                result = self.to
            elif self.otherghost:
                result = random.choice(self.otherghost)[0]
            else:
                result = ''
        else:
            result = ''.join(('\\', name))
        return result

    def eval_expression(self, expr):
        tree = self.expr_parser.parse(expr)
        if tree is None:
            return None
        return self.interp_expr(tree)

    def __interp_add_expr(self, tree):
        value = self.interp_expr(tree[0])
        for i in range(1, len(tree), 2):
            operand = self.interp_expr(tree[i + 1])
            try:
                if tree[i] == '+':
                    value = int(value) + int(operand)
                elif tree[i] == '-':
                    value = int(value) - int(operand)
            except ValueError:
                value = ''.join((str(value), tree[i], str(operand)))
        return value

    def __interp_mul_expr(self, tree):
        value = self.interp_expr(tree[0])
        for i in range(1, len(tree), 2):
            operand = self.interp_expr(tree[i + 1])
            try:
                if tree[i] == '*':
                    value = int(value) * int(operand)
                elif tree[i] == '/':
                    value = int(value) / int(operand)
                elif tree[i] == '\\':
                    value = int(value) % int(operand)
            except (ValueError, ZeroDivisionError):
                value = ''.join((str(value), tree[i], str(operand)))
        return value

    def __interp_unary_expr(self, tree):
        operand = self.interp_expr(tree[1])
        try:
            if tree[0] == '+':
                return int(operand)
            elif tree[0] == '-':
                return - int(operand)
        except ValueError:
            return ''.join((tree[0], str(operand)))

    def __interp_primary_expr(self, tree):
        if self.is_number(tree[0]):
            return int(tree[0])
        elif tree[0].startswith('%'):
            return self.expand_meta(tree[0])
        try:
            return self.variables[tree[0]]
        except KeyError:
            return '?'

    __expr = {
        ADD_EXPR: __interp_add_expr,
        MUL_EXPR: __interp_mul_expr,
        UNARY_EXPR: __interp_unary_expr,
        PRIMARY_EXPR: __interp_primary_expr,
        }

    def interp_expr(self, tree):
        key = tree[0]
        if key in self.__expr:
            return self.__expr[key](self, tree[1:])
        else:
            raise RuntimeError, 'should not reach here'

    def is_number(self, s):
        return s and filter(lambda c: c in '0123456789', s) == s

class ExprError(ValueError):
    pass

class ExprParser:

    def __init__(self):
        self.debug = 0
        # bit 0 = trace get_expr()
        # bit 1 = trace all 'get_' functions

    def show_progress(self, func, buf):
        if buf is None:
            print '%s() -> syntax error' % func
        else:
            print '%s() -> %s' % (func, buf)

    re_token = re.compile(r'[()*/\+-]|\d+|\s+')

    def tokenize(self, data):
        buf = []
        end = 0
        while 1:
            match = NiseShiori.re_meta.match(data, end)
            if match:
                buf.append(match.group())
                end = match.end()
                continue
            match = self.re_token.match(data, end)
            if match:
                if match.group().strip():
                    buf.append(match.group())
                end = match.end()
                continue
            match = self.re_token.search(data, end)
            if match:
                buf.append(data[end:match.start()])
                if match.group().strip():
                    buf.append(match.group())
                end = match.end()
            else:
                if end < len(data):
                    buf.append(data[end:])
                break
        return buf

    def parse(self, data):
        self.tokens = self.tokenize(data)
        try:
            return self.get_expr()
        except ExprError:
            return None # syntax error

    # internal
    def done(self):
        return not self.tokens

    def pop(self):
        try:
            return self.tokens.pop(0)
        except IndexError:
            raise ExprError

    def look_ahead(self, index=0):
        try:
            return self.tokens[index]
        except IndexError:
            raise ExprError

    def match(self, s):
        if self.pop() != s:
            raise ExprError

    def get_expr(self):
        buf = self.get_add_expr()
        if not self.done():
            raise ExprError
        if self.debug & 3:
            self.show_progress('get_expr', buf)
        return buf

    def get_add_expr(self):
        buf = [ADD_EXPR]
        while 1:
            buf.append(self.get_mul_expr())
            if self.done() or self.look_ahead() not in ['+', '-']:
                break
            buf.append(self.pop()) # operator
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_add_expr', buf)
        return buf

    def get_mul_expr(self):
        buf = [MUL_EXPR]
        while 1:
            buf.append(self.get_unary_expr())
            if self.done() or self.look_ahead() not in ['*', '/', '\\']:
                break
            buf.append(self.pop()) # operator
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_mul_expr', buf)
        return buf

    def get_unary_expr(self):
        if self.look_ahead() in ['+', '-']:
            buf = [UNARY_EXPR, self.pop(), self.get_unary_expr()]
        else:
            buf = self.get_primary_expr()
        if self.debug & 2:
            self.show_progress('get_unary_expr', buf)
        return buf

    def get_primary_expr(self):
        if self.look_ahead() == '(':
            self.match('(')
            buf = self.get_add_expr()
            self.match(')')
        else:
            buf = [PRIMARY_EXPR, self.pop()]
        if self.debug & 2:
            self.show_progress('get_primary_expr', buf)
        return buf

# <<< EXPRESSION SYNTAX >>>
# expr         := add-expr
# add-expr     := mul-expr (add-op mul-expr)*
# add-op       := '+' | '-'
# mul-expr     := unary-expr (mul-op unary-expr)*
# mul-op       := '*' | '/' | '\'
# unary-expr   := unary-op unary-expr | primary-expr
# unary-op     := '+' | '-'
# primary-expr := identifier | constant | '(' add-expr ')'

class Shiori(NiseShiori):

    def __init__(self, dll_name, debug=0):
        NiseShiori.__init__(self, debug)
        self.dll_name = dll_name

    def load(self, top_dir):
        NiseShiori.load(self, top_dir)
        return 1

    def unload(self):
        NiseShiori.finalize(self)

    def find(self, top_dir, dll_name):
        result = 0
        if list_dict(top_dir):
            result = 100
        return result

    def show_description(self):
        sys.stdout.write(
            'Shiori: NiseShiori compatible module for ninix\n'
            '        Copyright (C) 2001, 2002 by Tamito KAJIYAMA\n'
            '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko\n'
            '        Copyright (C) 2002-2009 by Shyouzou Sugitani\n'
            '        Copyright (C) 2003 by Shun-ichi TAHARA\n')

    def request(self, req_string):
        header = StringIO.StringIO(req_string)
        req_header = {}
        line = header.readline()
        if line:
            line = line.strip()
            req_list = line.split()
            if len(req_list) >= 2:
                command = req_list[0].strip()
                protocol = req_list[1].strip()
            for line in header:
                line = line.strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = [x.strip() for x in line.split(':', 1)]
                try:
                    value = int(value)
                except:
                    value = str(value)
                req_header[key] = value
        result = ''
        to = None
        if 'ID' in req_header:
            if req_header['ID'] == 'charset':
                result = 'UTF-8'
            elif req_header['ID'] == 'dms':
                result = self.getdms()
            elif req_header['ID'] == 'OnAITalk':
                result = self.getaistringrandom()
            elif req_header['ID'] in ['\\ms', '\\mz', '\\ml', '\\mc', '\\mh',
                                      '\\mt', '\\me', '\\mp']:
                result = self.getword(req_header['ID'][1:])
            elif req_header['ID'] == '\\m?':
                result = self.getword(random.choice(['ms', 'mz', 'ml', 'mc',
                                                     'mh', 'mt', 'me', 'mp']))
            elif req_header['ID'] == 'otherghostname':
                otherghost = []
                for n in range(128):
                    if ''.join(('Reference', str(n))) in req_header:
                        otherghost.append(
                            req_header[''.join(('Reference', str(n)))])
                result = self.otherghostname(otherghost)
            elif req_header['ID'] == 'OnTeach':
                if 'Reference0' in req_header:
                    self.teach(req_header['Reference0'])
            else:
                result = self.getstring(req_header['ID'])
                if result is None:
                    ref = []
                    for n in range(8):
                        if ''.join(('Reference', str(n))) in req_header:
                            ref.append(
                                req_header[''.join(('Reference', str(n)))])
                        else:
                            ref.append(None)
                    ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7 = ref
                    result = self.get_event_response(
                        req_header['ID'],
                        ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7)
            if result is None:
                result = ''
            to = self.communicate_to()
        result = 'SHIORI/3.0 200 OK\r\n' \
                 'Sender: Niseshiori\r\n' \
                 'Value: %s\r\n' % result
        if to is not None:
            result = ''.join((result, 'Reference0: %s\r\n' % to))
        result = ''.join((result, '\r\n'))
        return result


def open(shiori_dir, debug=0):
    ns = NiseShiori(debug)
    ns.load(shiori_dir)
    return ns

def test():
    if len(sys.argv) == 2:
        top_dir = sys.argv[1]
    else:
        top_dir = os.curdir
    if not list_dict(top_dir):
        print 'no dictionary'
        return
    ns = open(top_dir, 4 + 8 + 16)
    dump_dict(ns)
    while 1:
        print ns.getaistringrandom()
        try:
            raw_input()
        except:
            break

def dump_dict(ns):
    print 'DICT'
    for k, v in ns.dict.iteritems():
        if isinstance(k, tuple):
            k = '(%s, %s)' % k
        print k.encode('UTF-8', 'ignore')
        for e in v: print '\t', e.encode('UTF-8', 'ignore')
    print '*' * 50
    print 'CHAINS'
    for t, chain_list in ns.type_chains.iteritems():
        print t.encode('UTF-8', 'ignore')
        for c, w in chain_list:
            print ('-> %s, %s' % (c, w)).encode('UTF_8', 'ignore')
    for key, dic in ns.word_chains.iteritems():
        print ('(%s, %s)' % key).encode('UTF-8', 'ignore')
        for t, chain_list in dic.iteritems():
            prefix = '-> %s' % t.encode('UTF-8', 'ignore')
            for c, w in chain_list:
                print prefix, ('-> %s, %s' % (c, w)).encode('UTF-8', 'ignore')
                prefix = ''.join(('   ', ' ' * len(t)))
    print '*' * 50
    print 'KEYWORDS'
    for (t, w), s in ns.keywords.iteritems():
        print t.encode('UTF-8', 'ignore'), w.encode('UTF-8', 'ignore')
        print '->', s.encode('UTF-8', 'ignore')
    print '*' * 50
    print 'RESPONSES'
    for k, v in ns.responses.iteritems():
        print_condition(k)
        print '->', v
    print '*' * 50
    print 'GREETINGS'
    for k, v in ns.greetings.iteritems():
        print k.encode('UTF-8', 'ignore')
        print '->', v
    print '*' * 50
    print 'EVENTS'
    for k, v in ns.events.iteritems():
        print_condition(k)
        for e in v:
            print '->', e.encode('UTF-8', 'ignore')
    
def print_condition(condition):
    prefix = 'Condition'
    for cond_type, expr in condition:
        if cond_type == NiseShiori.COND_COMPARISON:
            print prefix, ("'%s' '%s' '%s'" % expr).encode('UTF-8', 'ignore')
        elif cond_type == NiseShiori.COND_STRING:
            print prefix, "'%s'" % expr.encode('UTF-8', 'ignore')
        prefix = '      and'

if __name__ == '__main__':
    test()
