/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// X509_ACL_Validator.h : interface for the X509_ACL_Validator class.
//
//////////////////////////////////////////////////////////////////////

#ifndef X509_ACL_VALIDATOR_H
#define X509_ACL_VALIDATOR_H

#include <openssl/x509.h>
#include <ASN1/Asn1Conf.h>
#include <Objects.h>
#include <ReadersWriter.h>
#include <PKI_CERT.h>
#include <PKI_CRL.h>


/*!
	This function provides users validation functionnalities
*/
class X509_ACL_Validator  
{
public:
	/*! \brief This is the constructor.
	 */
	X509_ACL_Validator();

	/*! \brief This is the destructor.
	 */
	virtual ~X509_ACL_Validator();

	/*! \brief This function sets the PKI's internal CA's CRLs.
	 *  \param crl [IN] The PKI's internal CA's CRLs.
	 *  \return true on success, false on failure.
	 */
	bool SetCRL(const InternalPkiCrl & crl);

	/*! \brief This function sets the PKI's internal CA's certificates.
	 *  \param ca [IN] The PKI's internal CA's certificates.
	 *  \return true on success, false on failure.
	 */
	bool SetCA(const InternalPkiCa & ca);

	/*! \brief This function sets the ACLs to check users against.
	 *  \param acl [IN] The ACLs.
	 *  \return true on success, false on failure.
	 */
	bool SetACL(const X509Acl & acl);

	/*! \brief This function sets the groups to check users against.
	 *  \param groups [IN] The groups.
	 *  \return true on success, false on failure.
	 */
	bool SetGroups(const mVector<UsersGroup> & groups);

	/*! \brief This function verifies that a user can perform an action.
	 *  \param usercert [IN] The user certificate.
	 *  \param command [IN] The command.
	 *  \param CheckCert [IN] Should the certificate be verified.
	 *  \return true on success, false on failure.
	 */
	bool CanUserPerform(const PKI_CERT & usercert, int command, bool CheckCert=true) const;

	/*! \brief This function verifies that a certificate is valid.
	 *  \param usercert [IN] The user certificate.
	 *  \return true on success, false on failure.
	 */
	int ValidateCert(const PKI_CERT & usercert) const;

	/*! \brief This function checks if a user if a PKI admin.
	 *  \param usercert [IN] The user certificate.
	 *  \return true when user is a PKI admin, false when user is not a PKI admin.
	 */
	bool IsPkiAdministrator(const PKI_CERT & usercert) const;

	/*! \brief This function checks if a user if a PKI admin.
	 *  \param serial [IN] The user certificate's serial.
	 *  \return true when user is a PKI admin, false when user is not a PKI admin.
	 */
	bool IsPkiAdministrator(long serial) const;	

	/*! \brief This function checks if a user if a PKI admin.
	 *  \param cert [IN] The user certificate.
	 *  \param AdminSerials [IN] The list of PKI Admins.
	 *  \return true when user is a PKI admin, false when user is not a PKI admin.
	 */
	static bool Static_IsPkiAdministrator(const X509 *cert, const mVector<PkiAdminEntry> & AdminSerials);

	/*! \brief This function checks if a user if a PKI admin.
	 *  \param serial [IN] The user certificate's serial.
	 *  \param AdminSerials [IN] The list of PKI Admins.
	 *  \return true when user is a PKI admin, false when user is not a PKI admin.
	 */
	static bool Static_IsPkiAdministrator(long serial, const mVector<PkiAdminEntry> & AdminSerials);

	/*! \brief This function returns the ACLs of a user.
	 *  \param resAcl [OUT] The ACLs.
	 *  \param serial [IN] The user certificate's serial.
	 *  \return true on success, false on failure.
	 */
	bool GetUserRights(ASN1_BIT_STRING * resAcl, unsigned long serial) const;
private:
	X509Acl m_acl;
	mVector<UsersGroup> m_groups;
	InternalPkiCrl m_crls;
	InternalPkiCa m_certs;

	ReadersWriter AccessLock;
	const AclEntry & FindAcl(int type, unsigned long serial) const;
};

#endif
