/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



// Entity_SERVER.h: interface for the Entity_SERVER class.
//
//////////////////////////////////////////////////////////////////////

#ifndef ENTITY_SERVER_H
#define ENTITY_SERVER_H

class OcspServers;


#include <newpki_threads.h>
#include <CriticalSection.h>
#include <HashTable/HashTable_String.h>

#include "Entity.h"
#include "ASN1/Asn1Entity.h"
#include <mString.h>
#include <openssl/ocsp.h>

#include <map>
using namespace std;


/*!
	This class represents a Server Entity
*/
class Entity_SERVER : public Entity
{
public:
	void PrintInfo(FILE * out);
	Entity_SERVER(ENTITY_CONSTRUCTOR_PARAMETERS);
	virtual ~Entity_SERVER();

	bool Load();
	bool Create(const EntityCreationDatas & Params, AdminResponseBody & response);
	bool ParseAdminCommand(AdminResponseBody & response, const PKI_CERT & ClientCert, const AdminRequest & AdminRequest);
	bool Upgrade(const char * Version);
	bool Init(const EntitySignatureResp & init_datas);

	/*! \brief This function is called when a new request is received, the server entity will send it to the destination entity.
	 *  \param response [OUT] The response.
	 *  \param ClientIp [IN] The end user IP.
	 *  \param ClientCert [IN] The end user certificate.
	 *  \param AdminReq [IN] The request.
	 *  \return true on success, false on failure.
	 */
	bool OnADMIN_REQUEST(AdminResponse & response, const char * ClientIp, const PKI_CERT & ClientCert, const AdminRequest & AdminReq);

	/*! \brief This function is called when a new OCSP request arrives, it will send it to the right Entity_PUBLICATION.
	 *  \param Ip [IN] The IP of the requester.
	 *  \param request [IN] The OCSP request.
	 *  \param response [OUT] The OCSP response.
	 *  \param EntityName [IN] The name of the Entity_PUBLICATION that should process it.
	 *  \param ShouldStopServer [OUT] Set to true when the Entity_PUBLICATION is not available anymore.
	 *  \return true on success, false on failure.
	 */
	bool OnNewOCSP(const char * Ip, const OCSP_REQUEST * request, OCSP_RESPONSE ** response, const mString & EntityName, bool & ShouldStopServer);

	/*! \brief This function is called to initialize all the OCSP responders.
	 *  \param OcspServersHandler [IN] The OCSP servers provider.
	 */
	void SetOcspServersHandler(OcspServers * OcspServersHandler);

	/*! \brief This function is called when the connection to a user is closed.
	 */
	void OnConnectionClosed();

private:
	bool PrepareConfToWrite();
	#define DEFAULT_ADMIN "root"
	#define DEFAULT_ADMIN_ID 1

	#define SELECT_VERSION		"SELECT version FROM version;"
	#define UPDATE_VERSION		"UPDATE version SET version='"NEWPKI_VERSION"';"
	#define INITIAL_DB			"newpki_server"
	#define SELECT_TABLES		"SHOW TABLES;"

	#define INITIAL_CREATE_1			"create table entities_list (entity_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, entity_name TEXT NOT NULL, entity_type INT UNSIGNED NOT NULL);"
	#define INITIAL_CREATE_2			"create table users (user_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, username CHAR(49) NOT NULL UNIQUE, password TEXT NOT NULL, activated INT NOT NULL, flags INT NOT NULL);"
	#define INITIAL_CREATE_3			"INSERT INTO users (username, password, activated, flags) VALUES ('"DEFAULT_ADMIN"','8a88e6bd80673eab335ee79a35428b26f3c98ab97217c7cb352ba3be', 1, 0);"
	//Version table
	#define INITIAL_CREATE_4			"create table version (version TEXT NOT NULL);"
	#define INITIAL_CREATE_5			"INSERT INTO version (version) VALUES ('"NEWPKI_VERSION"');"

	#define INITIAL_GET_ENTITIES		"SELECT * FROM entities_list;" 
	#define INITIAL_INSERT_ENTITY		"INSERT INTO entities_list (entity_name, entity_type) VALUES ('%s', '%d');"
	#define INITIAL_REMOVE_ENTITY		"DELETE FROM entities_list WHERE entity_name='%s';" 

	#define SQL_SELECT_USERS				"SELECT * FROM users;"
	#define SQL_INSERT_USER					"INSERT INTO users(username, password, activated, flags) VALUES ('%s', '%s', %d, %ld);"
	#define SQL_UPDATE_USER					"UPDATE users SET username='%s', activated=%d, flags=%ld WHERE user_id=%ld;"
	#define SQL_UPDATE_USER_PWD				"UPDATE users SET password='%s' WHERE user_id=%ld;"
	#define SQL_SELECT_USER					"SELECT * FROM users WHERE username = '%s';"

	class ENTITY_INFO
	{
	public:
		ENTITY_INFO()
		{
			Type = 0;
			entity = NULL;
		}
		ENTITY_INFO(const ENTITY_INFO & other)
		{
			*this = other;
		}
		~ENTITY_INFO()
		{
		}
		bool operator=(const ENTITY_INFO & other)
		{
			Type = other.Type;
			entity = other.entity;
			return true;
		}
		int Type;
		Entity * entity;
	};

	CriticalSection GlobalUsersLock;
	map<NEWPKI_THREAD_ID, mString> GlobalUsers;

	bool ParseAdminRequest(AdminResponseBody & response, const PKI_CERT & ClientCert, const AdminRequest & AdminReq, mString & EntityName);
	bool EnumEntities(COMMAND_PARAMETERS);
	bool DeleteEntity(COMMAND_PARAMETERS);
	bool CreateEntity(COMMAND_PARAMETERS);
	bool EnumUsers(COMMAND_PARAMETERS);
	bool CreateUser(COMMAND_PARAMETERS);
	bool UpdateUser(COMMAND_PARAMETERS);
	bool ChangeUserPassword(COMMAND_PARAMETERS);
	bool ChangePassword(COMMAND_PARAMETERS);
	bool EntityLoad(COMMAND_PARAMETERS);
	bool EntityUnload(COMMAND_PARAMETERS);


	bool Private_EnumUsers(AdminResponseBody & response, const UserHandle & hUser);
	bool Private_CreateUser(AdminResponseBody & response, const UserHandle & hUser, const mString & Username, unsigned long Flags, int Activated);
	bool Private_UpdateUser(AdminResponseBody & response, const UserHandle & hUser, unsigned long user_id, const mString & Username, unsigned long Flags, int Activated);
	bool Private_ChangeUserPassword(AdminResponseBody & response, const UserHandle & hUser, unsigned long user_id, const mString & NewPassword);


	OcspServers * m_OcspServersHandler;
	static bool Private_ParseAdminCommand(bool ExecuteCmd, Entity * me_this, vector<LOG_MESSAGE_TYPE> &  mLogsType, AdminResponseBody & response, PKI_CERT * ClientCert, AdminRequest & AdminRequest, UserHandle * Param);

	bool InitEntity(COMMAND_PARAMETERS);
	Entity* GetEntity();


	Entity * Entity_new(const mString & Name, int Type);
	bool ParseNewConf();

	bool LoginUser(UserHandle & hUser, int & UserType);
	void LogoutUser(const UserHandle & hUser);

	/*! \brief This function signs an response using the entity's certificate.
	 *  \param response [IN/OUT] The response.
	 *  \return true on success, false on failure.
	 */
	bool EntitySignResponse(AdminResponse & response);


	bool CreateSSL_CERT();
	void CleanEntities();


	ReadersWriter m_EntitiesLock;
	map<mString, ENTITY_INFO> m_EntitiesList;

	NewpkiThread m_ThreadDeleteEntity;
	static void ThreadDeleteEntity(const NewpkiThread * Thread, void *param);
	vector<Entity*> m_DeleteEntitiesList;
	CriticalSection m_DeleteEntitiesLock;
	
	NewpkiThread m_ThreadUnloadEntity;
	static void ThreadUnloadEntity(const NewpkiThread * Thread, void *param);
	vector<Entity*> m_UnloadEntitiesList;
	CriticalSection m_UnloadEntitiesLock;

	Entity * LoadEntity(const mString & Name, int Type);

	mString currentVersion;

	DECLARE_COMMAND_PARSER(Entity_SERVER);
};

#endif
