/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// mString.cpp: implementation of the mString class.
//
//////////////////////////////////////////////////////////////////////

#include "mString.h"
#include <PEM_DER.h>
#include <PKI_ERR.h>
#include <iconv.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

mString mString ::EmptyInstance;

mString::mString()
{
	Reset();
	*this = "";
}


mString::mString( const mString & other )
{
	Reset();
	*this = other;
}

mString::mString( const string & other )
{
	Reset();
	*this = other;
}

mString::mString( const char * other )
{
	Reset();
	*this = other;
}

mString::mString( const unsigned char * other )
{
	Reset();
	*this = other;
}

mString::mString( const ASN1_STRING * other )
{
	Reset();
	*this = other;
}

mString::mString( const int other )
{
	Reset();
	*this = other;
}

mString::mString( const unsigned int other )
{
	Reset();
	*this = other;
}

mString::mString( const long other )
{
	Reset();
	*this = other;
}

mString::mString( const unsigned long other )
{
	Reset();
	*this = other;
}


mString::~mString()
{
	Reset();
}

const string & mString::c_string() const
{
	return buffer;
}

string & mString::c_string()
{
	return buffer;
}



void mString::reserve(size_t t)
{
	buffer.reserve(t);
}

int mString::find(const char *str, int pos) const
{
	return buffer.find(str, pos);
}

void mString::replace(int pos, int num, const char *datas)
{
	buffer.replace(pos, num, datas);
}

unsigned int mString::capacity() const
{
	return buffer.capacity();
}

unsigned int mString::size() const
{
	return buffer.size();
}

const char * mString::c_str() const
{
	return buffer.c_str();
}

int mString::c_int() const
{
	if(!buffer.size()) return 0;
	int m_int;
	sscanf(buffer.c_str(), "%d", &m_int);
	return m_int;
}

unsigned int mString::c_uint() const
{
	if(!buffer.size()) return 0;
	unsigned int m_uint;
	sscanf(buffer.c_str(), "%d", &m_uint);
	return m_uint;
}

long mString::c_lng() const
{
	if(!buffer.size()) return 0;
	long m_lng;
	sscanf(buffer.c_str(), "%ld", &m_lng);
	return m_lng;
}

unsigned long mString::c_ulng() const
{
	if(!buffer.size()) return 0;
	unsigned long m_ulng;
	sscanf(buffer.c_str(), "%ld", &m_ulng);
	return m_ulng;
}

bool mString::c_ASN1_INTEGER(ASN1_INTEGER ** dest) const
{
	if(!*dest)
	{
		if(! (*dest = ASN1_INTEGER_new()) )
		{
			return false;
		}
	}

	if(ASN1_INTEGER_set(*dest, c_lng()) <= 0)
	{
		return false;
	}
	return true;
}

bool mString::c_ASN1_UTF8STRING(ASN1_UTF8STRING ** dest) const
{
	if(!*dest)
	{
		if(! (*dest = ASN1_UTF8STRING_new()) )
		{
			return false;
		}
	}

	if(ASN1_STRING_set(*dest, buffer.c_str(), buffer.size()) <= 0)
	{
		return false;
	}
	return true;
}

int mString::compare(const char *other) const
{
	return buffer.compare(other);
}

int mString::compare(unsigned char *other) const
{
	return buffer.compare((char*)other);
}

int mString::compare(mString & other) const
{
	return buffer.compare(other.buffer);
}

int mString::compare(string & other) const
{
	return buffer.compare(other);
}


int mString::compare(int from, int max, char *other) const
{
	return buffer.compare(from, max, other);
}

int mString::compare(int from, int max, unsigned char *other) const
{
	return buffer.compare(from, max, (char*)other);
}

int mString::compare(int from, int max, mString & other) const
{
	return buffer.compare(from, max, other.buffer);
}

int mString::compare(int from, int max, string & other) const
{
	return buffer.compare(from, max, other);
}


void mString::resize(unsigned int size)
{
	buffer.reserve(size+1);
	buffer.resize(size, ' ');
}

char & mString::operator[](unsigned int pos)
{
	return buffer[pos];
}

const char mString::operator[](unsigned int pos) const 
{
	return buffer[pos];
}

mString & mString::operator=( const mString & other )
{
	Reset();
	buffer = other.buffer;
	return *this;
}

mString & mString::operator=( const string & other )
{
	Reset();
	buffer = other;
	return *this;
}

mString & mString::operator=( const char * other )
{
	Reset();
	if(other)
		buffer = other;
	else
		buffer = "";
	return *this;
}

mString & mString::operator=( const unsigned char * other )
{
	Reset();
	if(other)
		buffer = (const char*)other;
	else
		buffer = "";
	return *this;
}

mString & mString::operator=( const ASN1_STRING * other)
{
	Reset();
	if(other)
	{
		switch(other->type)
		{
			case V_ASN1_INTEGER:
				sprintf("%ld", ASN1_INTEGER_get((ASN1_STRING *)other));
				break;
			case V_ASN1_UTF8STRING:
				buffer = (char*)ASN1_STRING_data((ASN1_STRING *)other);
				break;
		}
	}
	else
		buffer = "";

	return *this;
}

mString & mString::operator=( const int other )
{
	Reset();
	sprintf("%d", other);
	return *this;
}

mString & mString::operator=( const unsigned int other )
{
	Reset();
	sprintf("%d", other);
	return *this;
}

mString & mString::operator=( const long other )
{
	Reset();
	sprintf("%ld", other);
	return *this;
}

mString & mString::operator=( const unsigned long other )
{
	Reset();
	sprintf("%ld", other);
	return *this;
}

mString & mString::operator+=( const mString & other )
{
	buffer += other.buffer;
	return *this;
}

mString & mString::operator+=( const string & other )
{
	buffer += other;
	return *this;
}

mString & mString::operator+=( const char * other )
{
	if(other)
		buffer += other;
	return *this;
}

mString & mString::operator+=( const unsigned char * other )
{
	if(other)
		buffer += (const char*)other;
	return *this;
}

mString & mString::operator+=( const ASN1_STRING * other)
{
	if(other)
	{
		switch(other->type)
		{
			case V_ASN1_INTEGER:
				::sprintf(buffer_lng, "%ld", ASN1_INTEGER_get((ASN1_STRING *)other));
				buffer += buffer_lng;
				break;
			case V_ASN1_UTF8STRING:
				buffer += (char*)ASN1_STRING_data((ASN1_STRING *)other);
				break;
		}
	}
	return *this;
}

mString & mString::operator+=( const int other )
{
	::sprintf(buffer_lng, "%d", other);
	buffer += buffer_lng;
	return *this;
}

mString & mString::operator+=( const unsigned int other )
{
	::sprintf(buffer_lng, "%d", other);
	buffer += buffer_lng;
	return *this;
}

mString & mString::operator+=( const long other )
{
	::sprintf(buffer_lng, "%ld", other);
	buffer += buffer_lng;
	return *this;
}

mString & mString::operator+=( const unsigned long other )
{
	::sprintf(buffer_lng, "%ld", other);
	buffer += buffer_lng;
	return *this;
}


bool mString::operator==( const mString & other ) const
{
	return (buffer.compare(other.buffer) == 0);
}

bool mString::operator==( const string & other ) const
{
	return (buffer.compare(other) == 0);
}

bool mString::operator==( const char * other ) const
{
	if(!other)
		return false;

	return (buffer.compare(other) == 0);
}

bool mString::operator==( const unsigned char * other ) const
{
	if(!other)
		return false;

	return (buffer.compare((const char*)other) == 0);
}

bool mString::operator==( const ASN1_STRING * other) const
{
	if(!other)
		return false;

	switch(other->type)
	{
		case V_ASN1_INTEGER:
			return (c_lng() == ASN1_INTEGER_get((ASN1_STRING *)other));
			break;
		case V_ASN1_UTF8STRING:
			return (buffer.compare((char*)ASN1_STRING_data((ASN1_STRING *)other)) == 0);
			break;
		default:
			return false;
	}
}

bool mString::operator==( const int other ) const
{
	return (c_int() == other);
}

bool mString::operator==( const unsigned int other ) const
{
	return (c_uint() == other);
}

bool mString::operator==( const long other ) const
{
	return (c_lng() == other);
}

bool mString::operator==( const unsigned long other ) const
{
	return (c_ulng() == other);
}


bool mString::operator!=( const mString & other ) const
{
	return (buffer.compare(other.buffer) != 0);
}

bool mString::operator!=( const string & other ) const
{
	return (buffer.compare(other) != 0);
}

bool mString::operator!=( const char * other ) const
{
	if(!other)
		return false;

	return (buffer.compare(other) != 0);
}

bool mString::operator!=( const unsigned char * other ) const
{
	if(!other)
		return false;

	return (buffer.compare((const char*)other) != 0);
}

bool mString::operator!=( const int other ) const
{
	return (c_int() != other);
}

bool mString::operator!=( const unsigned int other ) const
{
	return (c_uint() != other);
}

bool mString::operator!=( const long other ) const
{
	return (c_lng() != other);
}

bool mString::operator!=( const unsigned long other ) const
{
	return (c_ulng() != other);
}

inline void mString::Reset()
{
	buffer.empty();
}

int mString::vsprintf(const char *format, va_list args)
{
    char ch;
    long value;
    double fvalue;
    char *strvalue;
    int min;
    int max;
    int state;
    int flags;
    int cflags;
    size_t currlen;

    state = DP_S_DEFAULT;
    flags = currlen = cflags = min = 0;
    max = -1;
    ch = *format++;

	buffer = "";

    while (state != DP_S_DONE) {
        if (ch == '\0')
            state = DP_S_DONE;

        switch (state) {
        case DP_S_DEFAULT:
            if (ch == '%')
                state = DP_S_FLAGS;
            else
                buffer += ch;
            ch = *format++;
            break;
        case DP_S_FLAGS:
            switch (ch) {
            case '-':
                flags |= DP_F_MINUS;
                ch = *format++;
                break;
            case '+':
                flags |= DP_F_PLUS;
                ch = *format++;
                break;
            case ' ':
                flags |= DP_F_SPACE;
                ch = *format++;
                break;
            case '#':
                flags |= DP_F_NUM;
                ch = *format++;
                break;
            case '0':
                flags |= DP_F_ZERO;
                ch = *format++;
                break;
            default:
                state = DP_S_MIN;
                break;
            }
            break;
        case DP_S_MIN:
            if (isdigit((unsigned char)ch)) {
                min = 10 * min + char_to_int(ch);
                ch = *format++;
            } else if (ch == '*') {
                min = va_arg(args, int);
                ch = *format++;
                state = DP_S_DOT;
            } else
                state = DP_S_DOT;
            break;
        case DP_S_DOT:
            if (ch == '.') {
                state = DP_S_MAX;
                ch = *format++;
            } else
                state = DP_S_MOD;
            break;
        case DP_S_MAX:
            if (isdigit((unsigned char)ch)) {
                if (max < 0)
                    max = 0;
                max = 10 * max + char_to_int(ch);
                ch = *format++;
            } else if (ch == '*') {
                max = va_arg(args, int);
                ch = *format++;
                state = DP_S_MOD;
            } else
                state = DP_S_MOD;
            break;
        case DP_S_MOD:
            switch (ch) {
            case 'h':
                cflags = DP_C_SHORT;
                ch = *format++;
                break;
            case 'l':
                if (*format == 'l') {
                    cflags = DP_C_LLONG;
                    format++;
                } else
                    cflags = DP_C_LONG;
                ch = *format++;
                break;
            case 'q':
                cflags = DP_C_LLONG;
                ch = *format++;
                break;
            case 'L':
                cflags = DP_C_LDOUBLE;
                ch = *format++;
                break;
            default:
                break;
            }
            state = DP_S_CONV;
            break;
        case DP_S_CONV:
            switch (ch) {
            case 'd':
            case 'i':
                switch (cflags) {
                case DP_C_SHORT:
                    value = (short int)va_arg(args, int);
                    break;
                case DP_C_LONG:
                    value = va_arg(args, long int);
                    break;
                case DP_C_LLONG:
                    value = va_arg(args, long);
                    break;
                default:
                    value = va_arg(args, int);
                    break;
                }
                fmtint(value, 10, min, max, flags);
                break;
            case 'X':
                flags |= DP_F_UP;
                /* FALLTHROUGH */
            case 'x':
            case 'o':
            case 'u':
                flags |= DP_F_UNSIGNED;
                switch (cflags) {
                case DP_C_SHORT:
                    value = (unsigned short int)va_arg(args, unsigned int);
                    break;
                case DP_C_LONG:
                    value = (long) va_arg(args,
                        unsigned long int);
                    break;
                case DP_C_LLONG:
                    value = va_arg(args, unsigned long);
                    break;
                default:
                    value = (long) va_arg(args,
                        unsigned int);
                    break;
                }
                fmtint(value, ch == 'o' ? 8 : (ch == 'u' ? 10 : 16), min, max, flags);
                break;
            case 'f':
                if (cflags == DP_C_LDOUBLE)
                    fvalue = va_arg(args, double);
                else
                    fvalue = va_arg(args, double);
                fmtfp(fvalue, min, max, flags);
                break;
            case 'E':
                flags |= DP_F_UP;
            case 'e':
                if (cflags == DP_C_LDOUBLE)
                    fvalue = va_arg(args, double);
                else
                    fvalue = va_arg(args, double);
                break;
            case 'G':
                flags |= DP_F_UP;
            case 'g':
                if (cflags == DP_C_LDOUBLE)
                    fvalue = va_arg(args, double);
                else
                    fvalue = va_arg(args, double);
                break;
            case 'c':
                buffer += (char)va_arg(args, int);
                break;
            case 's':
                strvalue = va_arg(args, char *);
                if (max < 0)
				{
					max = INT_MAX;
				}
                fmtstr(strvalue, flags, min, max);
                break;
            case 'p':
                value = (long)va_arg(args, void *);
                fmtint(value, 16, min, max, flags|DP_F_NUM);
                break;
            case 'n': /* XXX */
                if (cflags == DP_C_SHORT) {
                    short int *num;
                    num = va_arg(args, short int *);
                    *num = currlen;
                } else if (cflags == DP_C_LONG) { /* XXX */
                    long int *num;
                    num = va_arg(args, long int *);
                    *num = (long int) currlen;
                } else if (cflags == DP_C_LLONG) { /* XXX */
                    long *num;
                    num = va_arg(args, long *);
                    *num = (long) currlen;
                } else {
                    int    *num;
                    num = va_arg(args, int *);
                    *num = currlen;
                }
                break;
            case '%':
                buffer += ch;
                break;
            case 'w':
                /* not supported yet, treat as next char */
                ch = *format++;
                break;
            default:
                /* unknown, skip */
                break;
            }
            ch = *format++;
            state = DP_S_DEFAULT;
            flags = cflags = min = 0;
            max = -1;
            break;
        case DP_S_DONE:
            break;
        default:
            break;
        }
    }

	return buffer.size();
}

int mString::sprintf(const char *format, ...)
{	
	va_list args;
	int ret;
	va_start(args, format);
	ret = vsprintf(format, args);
	va_end(args);
	return ret;
}


void mString::fmtstr(const char *value, int flags, int min, int max)
{
    int padlen, strln;
    int cnt = 0;
	string padding;

    if (value == 0)
        value = "<NULL>";
    for (strln = 0; value[strln]; ++strln)
        ;

	if(max < min)
		min = max;

    padlen = min - strln;
    if (padlen < 0)
        padlen = 0;
    if (flags & DP_F_MINUS)
        padlen = -padlen;


	if(padlen > 0)
	{
		cnt = padlen>max?max:padlen;
		padding.resize(cnt, ' ');
		buffer += padding;
	}

	if( (strln + cnt) > max)
	{
		strln = max - cnt;
	}
	if(strln)
	{
		buffer.append(value, strln);
		cnt += strln;
	}

	if(padlen < 0)
	{
		padlen = -padlen;
		if( (padlen + cnt) > max)
		{
			padlen = max - cnt;
		}
		if(padlen)
		{
			padding.resize(padlen, ' ');
			buffer += padding;
		}
	}
}

void mString::fmtint(long value, int base, int min, int max, int flags)
{
    char signvalue = 0;
    char *prefix = "";
    unsigned long uvalue;
    char convert[((sizeof(value)*8+2)/3+1)+3];
    int place = 0;
    int spadlen = 0;
    int zpadlen = 0;
    int caps = 0;

    if (max < 0)
        max = 0;
    uvalue = value;
    if (!(flags & DP_F_UNSIGNED)) {
        if (value < 0) {
            signvalue = '-';
            uvalue = -value;
        } else if (flags & DP_F_PLUS)
            signvalue = '+';
        else if (flags & DP_F_SPACE)
            signvalue = ' ';
    }
    if (flags & DP_F_NUM) {
	if (base == 8) prefix = "0";
	if (base == 16) prefix = "0x";
    }
    if (flags & DP_F_UP)
        caps = 1;
    do {
        convert[place++] =
            (caps ? "0123456789ABCDEF" : "0123456789abcdef")
            [uvalue % (unsigned) base];
        uvalue = (uvalue / (unsigned) base);
    } while (uvalue && (place < (int)sizeof convert));
    if (place == sizeof convert)
        place--;
    convert[place] = 0;

    zpadlen = max - place;
    spadlen = min - OSSL_MAX(max, place) - (signvalue ? 1 : 0) - strlen(prefix);
    if (zpadlen < 0)
        zpadlen = 0;
    if (spadlen < 0)
        spadlen = 0;
    if (flags & DP_F_ZERO) {
        zpadlen = OSSL_MAX(zpadlen, spadlen);
        spadlen = 0;
    }
    if (flags & DP_F_MINUS)
        spadlen = -spadlen;

    /* spaces */
    while (spadlen > 0) {
        buffer += ' ';
        --spadlen;
    }

    /* sign */
    if (signvalue)
        buffer += signvalue;

    /* prefix */
    while (*prefix) {
	buffer += *prefix;
	prefix++;
    }

    /* zeros */
    if (zpadlen > 0) {
        while (zpadlen > 0) {
            buffer += '0';
            --zpadlen;
        }
    }
    /* digits */
    while (place > 0)
        buffer += convert[--place];

    /* left justified spaces */
    while (spadlen < 0) {
        buffer += ' ';
        ++spadlen;
    }
    return;
}

double mString::abs_val(double value)
{
    double result = value;
    if (value < 0)
        result = -value;
    return result;
}

double mString::pow10(int exp)
{
    double result = 1;
    while (exp) {
        result *= 10;
        exp--;
    }
    return result;
}

long mString::roundv(double value)
{
    long intpart;
    intpart = (long) value;
    value = value - intpart;
    if (value >= 0.5)
        intpart++;
    return intpart;
}

void mString::fmtfp(double fvalue, int min, int max, int flags)
{
    char signvalue = 0;
    double ufvalue;
    char iconvert[20];
    char fconvert[20];
    int iplace = 0;
    int fplace = 0;
    int padlen = 0;
    int zpadlen = 0;
    int caps = 0;
    long intpart;
    long fracpart;

    if (max < 0)
        max = 6;
    ufvalue = abs_val(fvalue);
    if (fvalue < 0)
        signvalue = '-';
    else if (flags & DP_F_PLUS)
        signvalue = '+';
    else if (flags & DP_F_SPACE)
        signvalue = ' ';

    intpart = (long)ufvalue;

    /* sorry, we only support 9 digits past the decimal because of our
       conversion method */
    if (max > 9)
        max = 9;

    /* we "cheat" by converting the fractional part to integer by
       multiplying by a factor of 10 */
    fracpart = roundv((pow10(max)) * (ufvalue - intpart));

    if (fracpart >= pow10(max)) {
        intpart++;
        fracpart -= (long)pow10(max);
    }

    /* convert integer part */
    do {
        iconvert[iplace++] =
            (caps ? "0123456789ABCDEF"
              : "0123456789abcdef")[intpart % 10];
        intpart = (intpart / 10);
    } while (intpart && (iplace < (int)sizeof(iplace)));
    if (iplace == sizeof(iplace))
        iplace--;
    iconvert[iplace] = 0;

    /* convert fractional part */
    do {
        fconvert[fplace++] =
            (caps ? "0123456789ABCDEF"
              : "0123456789abcdef")[fracpart % 10];
        fracpart = (fracpart / 10);
    } while (fplace < max);
    if (fplace == sizeof fplace)
        fplace--;
    fconvert[fplace] = 0;

    /* -1 for decimal point, another -1 if we are printing a sign */
    padlen = min - iplace - max - 1 - ((signvalue) ? 1 : 0);
    zpadlen = max - fplace;
    if (zpadlen < 0)
        zpadlen = 0;
    if (padlen < 0)
        padlen = 0;
    if (flags & DP_F_MINUS)
        padlen = -padlen;

    if ((flags & DP_F_ZERO) && (padlen > 0)) {
        if (signvalue) {
            buffer += signvalue;
            --padlen;
            signvalue = 0;
        }
        while (padlen > 0) {
            buffer += '0';
            --padlen;
        }
    }
    while (padlen > 0) {
        buffer += ' ';
        --padlen;
    }
    if (signvalue)
        buffer += signvalue;

    while (iplace > 0)
        buffer += iconvert[--iplace];

    /*
     * Decimal point. This should probably use locale to find the correct
     * char to print out.
     */
    if (max > 0 || (flags & DP_F_NUM)) {
        buffer += '.';

        while (fplace > 0)
            buffer += fconvert[--fplace];
    }
    while (zpadlen > 0) {
        buffer += '0';
        --zpadlen;
    }

    while (padlen < 0) {
        buffer += ' ';
        ++padlen;
    }
}

bool mString::FromDER(const unsigned char *indatas, const int indataslen)
{
	PEM_DER pem_cvrt;
	char * outdatas;
	int outdataslen;
	

	//We test the params
	if(!indatas || !indataslen)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}


	outdatas = NULL;
	if(!pem_cvrt.Der2Pem((const char*)indatas, indataslen, &outdatas, &outdataslen))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	buffer.assign(outdatas, outdataslen);
	free(outdatas);
	return true;
}

bool mString::ToDER(mBuffer & outdatas) const
{
	PEM_DER pem_cvrt;
	unsigned char * t_outdatas;
	int t_outdataslen;

	if(!pem_cvrt.Pem2Der(c_str(), size(), (char **)&t_outdatas, &t_outdataslen))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	outdatas.Assign(t_outdatas, t_outdataslen);
	return true;
}

bool mString::ToDER(unsigned char **outdatas, int * outdataslen) const
{	
	PEM_DER pem_cvrt;

	//We test the params
	if(!outdatas || !outdataslen)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!pem_cvrt.Pem2Der(c_str(), size(), (char **)outdatas, outdataslen))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool mString::left_compare(const char *cmp) const
{
	return (buffer.compare(0, strlen(cmp), cmp) == 0);
}

mString mString::Right(int count) const
{
	if(count <= 0 || (size_t)count > size())
		return "";

	return buffer.substr(size()-count, count);
}

mString mString::Left(int count) const
{
	if(count <= 0)
		return "";
	if((size_t)count > size())
		count = (int)size();
	
	return buffer.substr(0, count);
}

mString mString::Mid(int start, int count) const
{
	if(count <= 0)
		return "";
	if((size_t)count > size())
		count = (int)size();
	
	return buffer.substr(start, count);
}

void mString::TrimLeft(const char *filters)
{
	int filters_len = strlen(filters);
	int i, j;
	int len = size();

	for(i=0; i<len; i++)
	{
		for(j=0; j<filters_len; j++)
		{
			if(buffer[i] == filters[j])
			{
				break;
			}			
		}
		if(j == filters_len)
		{
			buffer = buffer.substr(i, len - i);
			return;
		}
	}
	buffer = "";
}

void mString::TrimRight(const char *filters)
{
	int filters_len = strlen(filters);
	int i, j;
	int len = size();
	char tt;

	for(i=len-1; i >= 0; i--)
	{
		for(j=0; j<filters_len; j++)
		{
			tt = buffer[i];
			if(buffer[i] == filters[j])
			{
				break;
			}			
		}
		if(j == filters_len)
		{
			buffer = buffer.substr(0, i+1);
			return;
		}
	}
	buffer = "";
}

void mString::Split(const char *sep, vector<mString> & results) const
{
	int pos = 0;
	int lastpos = 0;
	results.clear();
	mString newEntry;
	int seplen = strlen(sep);

	do
	{
		pos = find(sep, pos); 
		if(pos != -1)
		{
			newEntry = buffer.substr(lastpos, pos - lastpos);
			results.push_back(newEntry);
			pos += seplen;
			lastpos = pos;
		}
	}
	while(pos != -1);

	newEntry = buffer.substr(lastpos, size() - lastpos);
	results.push_back(newEntry);
}

bool mString::operator<(const mString& rhs) const 
{
	return (buffer.compare(rhs.buffer) < 0);
}

void mString::Encode(const char * from, const char * to, const mString & src, mString & dst)
{
	char * inbuf;
	char * outbuf;
	size_t inbytesleft, outbytesleft;
	iconv_t   UTF8ToLoc = (iconv_t) -1;

	dst.resize(src.size()*2);

	UTF8ToLoc = iconv_open(to, from);
	inbuf = (char*)src.c_str();
	inbytesleft = src.size(); 
	outbuf = &dst[0];
	outbytesleft = dst.size();

	if (iconv(UTF8ToLoc, (char**)&inbuf, &inbytesleft, 
				&outbuf, &outbytesleft) == (size_t)-1) 
	{
		iconv_close(UTF8ToLoc);
		dst = src;
		return;
	}
	iconv_close(UTF8ToLoc);

	dst[dst.size() - outbytesleft] = '\0';
}





mBuffer::mBuffer()
{
	m_buffer = NULL;
	m_bufferlen = 0;
}

mBuffer::mBuffer(const mBuffer & other)
{
	m_buffer = NULL;
	m_bufferlen = 0;
	*this = other;
}

mBuffer::~mBuffer()
{
	Clear();
}

bool mBuffer::Copy(const unsigned char * buffer, size_t bufferlen)
{
	Clear();
	m_buffer = (unsigned char *)malloc(bufferlen);
	if(!m_buffer)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	memcpy(m_buffer, buffer, bufferlen);
	m_bufferlen = bufferlen;
	return true;
}

void mBuffer::Assign(const unsigned char * buffer, size_t bufferlen)
{
	Clear();
	m_bufferlen = bufferlen;
	m_buffer = (unsigned char *)buffer;
}

const unsigned char * mBuffer::get_Buffer() const
{
	return m_buffer;
}

unsigned char * mBuffer::get_Buffer()
{
	return m_buffer;
}

size_t mBuffer::get_BufferLen() const
{
	return m_bufferlen;
}

void mBuffer::Clear()
{
	if(m_buffer)
	{
		free(m_buffer);
		m_buffer = NULL;
		m_bufferlen = 0;
	}
}

bool mBuffer::operator=(const mBuffer & other)
{
	Clear();
	if(other.m_buffer && other.m_bufferlen)
	{
		if(!Copy(other.m_buffer, other.m_bufferlen))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}

mBuffer::operator int() const
{
	return (isOK()?1:0);
}

bool mBuffer::isOK() const
{
	return (m_buffer && m_bufferlen?true:false);
}

void mString::replaceall(const mString & from, const mString & to)
{
	int pos;
	size_t to_len = to.size();
	size_t from_len = from.size();

	pos = 0;
	do
	{
		pos = buffer.find(from.buffer, pos); 
		if(pos != -1)
		{
			buffer.replace(pos, from_len, to.buffer);
			pos+=to_len;
		}
	}
	while(pos != -1);
}
