/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_RSA.cpp: implementation of the PKI_RSA class.
//
//////////////////////////////////////////////////////////////////////

#include "PKI_RSA.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


PKI_RSA::PKI_RSA(const PKI_RSA & other)
{
	Reset();
	*this = other;
}

PKI_RSA::PKI_RSA()
{
	Reset();
}

PKI_RSA::PKI_RSA(const EVP_PKEY *RSAkey)
{
	Reset();
	if(!SetKey(RSAkey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKI_RSA::PKI_RSA(const mString & pem_key, ENGINE * engine)
{
	Reset();
	if(!SetKey(pem_key, engine))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKI_RSA::PKI_RSA(int KeyBitsLen, ENGINE * engine)
{
	Reset();
	if(!GenerateKey(KeyBitsLen, engine))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKI_RSA::~PKI_RSA()
{
	Clear();
}


const EVP_PKEY * PKI_RSA::GetRsaKey() const
{
	return pkey;
}

const mString & PKI_RSA::GetRsaKeyPem() const
{
	return PemKey;
}

bool PKI_RSA::SetKey(const RSA *RSAkey)
{
	if(!RSAkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	Clear();
	pkey = EVP_PKEY_new();
	if(!pkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!EVP_PKEY_set1_RSA(pkey, (RSA*)RSAkey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!EVP_PKEYToString())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(X509_PUBKEY_set(&pubKey, pkey) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	KeyType = PKI_RSA_KEY_TYPE_NORMAL;
	return true;
}

bool PKI_RSA::SetKey(const EVP_PKEY *RSAkey)
{
	if(!RSAkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	Clear();
	RSA * rsakey = EVP_PKEY_get1_RSA((EVP_PKEY*)RSAkey);
	if(!rsakey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!SetKey(rsakey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		RSA_free(rsakey);
		return false;
	}
	RSA_free(rsakey);
	return true;
}


bool PKI_RSA::SetKey(const mString &pem_key, ENGINE *engine)
{
	Clear();
	mString tmpStr;

	e = engine;


	if(pem_key.find(FILE_HEAD, 0) == 0)
	{
		KeyType = PKI_RSA_KEY_TYPE_FILE;
		InitialString = pem_key;

		tmpStr = pem_key;
		tmpStr.replace(0, strlen(FILE_HEAD), "");
		if(!LoadKeyFromFile(tmpStr))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!EVP_PKEYToString())
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else if(pem_key.find(ENGINE_HEAD, 0) == 0)
	{
		KeyType = PKI_RSA_KEY_TYPE_ENGINE;
		InitialString = pem_key;

		tmpStr = pem_key;
		tmpStr.replace(0, strlen(ENGINE_HEAD), "");
		if(!LoadKeyFromEngine(tmpStr))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else
	{
		KeyType = PKI_RSA_KEY_TYPE_NORMAL;

		if(!StringToEVP_PKEY(pem_key))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!EVP_PKEYToString())
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(X509_PUBKEY_set(&pubKey, pkey) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool PKI_RSA::GenerateKey(int KeyBitsLen, ENGINE *engine)
{
	Clear();
	e = engine;
	RSA * rsaKey;

	if(!KeyBitsLen)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if ((pkey=EVP_PKEY_new()) == NULL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}

	if(!(rsaKey = RSA_generate_key(KeyBitsLen,0x10001,req_cb, NULL)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GENERATE_BIKEY);
		return false;
	}
	if (!EVP_PKEY_set1_RSA(pkey, rsaKey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GENERATE_BIKEY);
		RSA_free(rsaKey);
		return false;
	}
	RSA_free(rsaKey);
	if(!EVP_PKEYToString())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(X509_PUBKEY_set(&pubKey, pkey) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	KeyType = PKI_RSA_KEY_TYPE_NORMAL;	
	return true;
}

bool PKI_RSA::operator=( const PKI_RSA & other)
{
	Clear();
	//We totally reload the key (engine/file)
	//Or copy it from PEM
	switch(other.KeyType)
	{
		case PKI_RSA_KEY_TYPE_NORMAL:
			return SetKey(other.PemKey, other.e);
			break;
		case PKI_RSA_KEY_TYPE_ENGINE:
		case PKI_RSA_KEY_TYPE_FILE:
			return SetKey(other.InitialString, other.e);
			break;
		default:
			return false;
	}
	return true;
}


void PKI_RSA::Reset()
{
	e = NULL;
	pkey=NULL;
	pubKey = NULL;
}

void PKI_RSA::Clear()
{
	InitialString = "";
	PemKey = "";
	if(pkey) EVP_PKEY_free(pkey);
	if(pubKey) X509_PUBKEY_free(pubKey);
	Reset();
}

bool PKI_RSA::EVP_PKEYToString()
{
	int dsize=0;
	unsigned char *p,*data=NULL;

	if ((dsize=i2d_PrivateKey(pkey,NULL)) < 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		return false;
	}

	// dzise + 8 bytes are needed
	data=(unsigned char *)malloc(dsize+20);
	if (!data)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	p=data;
	if((dsize=i2d_PrivateKey(pkey,&p))<0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		return false;
	}
	if(!PemKey.FromDER(data, dsize))
	{
		free(data);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	free(data);
	return true;
}

bool PKI_RSA::StringToEVP_PKEY(const mString &PrivateKey)
{
	int dsize=0;
	unsigned char *data=NULL;
	unsigned char *tmp_data=NULL;

	if(!PrivateKey.ToDER(&data, &dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	tmp_data=data;
	if(!d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &data, dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		free(tmp_data);
		return false;
	}
	free(tmp_data);


	// On verifie la cohrence de la cl
	RSA * rsaKey;
	rsaKey = EVP_PKEY_get1_RSA(pkey);
	if(!rsaKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		return false;
	}

	if(RSA_check_key(rsaKey) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		RSA_free(rsaKey);
		return false;
	}
	RSA_free(rsaKey);


	return true;
}

int PKI_RSA::pem_password_cb(char *buf, int size, int rwflag, void *userdata)
{
		return 0;
}

void PKI_RSA::req_cb(int p, int n, void *arg)
{
	char c;

	switch(p)
	{
		case 0:
			c='.';
			break;

		case 1:
			c='+';
			break;

		case 2:
			c='*';
			break;

		case 3:
			c='\n';
			break;

		default:
			c='*';
			break;
	}

	//printf("%c", c);
}

bool PKI_RSA::LoadKeyFromFile(const mString &KeyFile)
{
	BIO * in;

	in=BIO_new(BIO_s_file());

	if(!in)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}

	if(BIO_read_filename(in, KeyFile.c_str()) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_LOAD_KEY);
		BIO_free(in);
		return false;
	}

	pkey = PEM_read_bio_PrivateKey(in,NULL,pem_password_cb,NULL);
	if (!pkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_LOAD_KEY);
		BIO_free(in);
		return false;
	}

	BIO_free(in);

	// On verifie la cohrence de la cl
	RSA * rsaKey;
	rsaKey = EVP_PKEY_get1_RSA(pkey);
	if(!rsaKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		EVP_PKEY_free(pkey);
		return false;
	}

	if(RSA_check_key(rsaKey) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		EVP_PKEY_free(pkey);
		RSA_free(rsaKey);
		return false;
	}
	RSA_free(rsaKey);

	return true;
}

bool PKI_RSA::LoadKeyFromEngine(const mString & KeyFile)
{
	if(!e)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_NO_ENGINE);
		return false;
	}

	pkey = ENGINE_load_private_key(e, KeyFile.c_str(), NULL, NULL);
	if(!pkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_LOAD_KEY);
		return false;
	}
/*
	// On verifie la cohrence de la cl
	RSA * rsaKey;
	rsaKey = EVP_PKEY_get1_RSA(pkey);
	if(!rsaKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		return false;
	}

	if(RSA_check_key(rsaKey) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_KEY_EVP_PKEY);
		RSA_free(rsaKey);
		return false;
	}

	RSA_free(rsaKey);
*/
	return true;
}

PKI_RSA::operator int() const
{
	if(pkey)
		return 1;
	else
		return 0;
}

long PKI_RSA::GetRsaSize() const
{
	if(!pkey) return 0;
	
	return RSA_size(pkey->pkey.rsa);
}

RSA * PKI_RSA::GetRSA(bool dup)
{
	if(dup)
	{
		CRYPTO_add(&pkey->pkey.rsa->references, 1, CRYPTO_LOCK_RSA);
	}
	return pkey->pkey.rsa;
}

const RSA * PKI_RSA::GetRSA(bool dup) const
{
	if(dup)
	{
		CRYPTO_add(&pkey->pkey.rsa->references, 1, CRYPTO_LOCK_RSA);
	}
	return pkey->pkey.rsa;
}

const X509_PUBKEY * PKI_RSA::GetPublicKey() const
{
	return pubKey;
}
