/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_CERT.h: interface for the CERT class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_CERT_H
#define PKI_CERT_H


#include <stdio.h>
#include <string.h>
#include <assert.h>


#include <PKI_ERR.h>
#include <HashTable/HashTable_Dn.h>
#include <HashTable/HashTable_String.h>
#include <PEM_DER.h>
#include <PKI_RSA.h>
#include <PKI_CSR.h>
#include <PKI_EXT.h>

#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <ExceptionNewPKI.h>

/*!
  This class is the representation of a X509 certificate
*/
class PKI_CERT
{
public:
	/*! \brief This is the constructor.
	 */
	PKI_CERT();

	/*! \brief This is the constructor.
	 *  \param other [IN] The other certificate to copy from.
	 */
	PKI_CERT(const PKI_CERT & other);

	/*! \brief This is the constructor.
	 *  \param PEM_CER [IN] A pointer to a PEM encoded certificate.
	 *  \param PrivateKey [IN] A pointer to a private key.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PKI_CERT(const char * PEM_CER, const PKI_RSA & PrivateKey);

	/*! \brief This is the constructor.
	 *  \param certificat [IN] A pointer to an OpenSSL X509 certificate.
	 *  \param PrivateKey [IN] A pointer to a private key.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PKI_CERT(const X509 * certificat, const PKI_RSA & PrivateKey);

	/*! \brief This is the destructor.
	 */
	virtual ~PKI_CERT();
	
	/*! \brief This function returns the version of the certificate.
	 *  \return The version of the certificate.
	 */
	unsigned long GetVersion() const;
	
	/*! \brief This function returns the extensions of the certificate.
	 *  \return The extensions of the certificate.
	 */
	const HashTable_String & GetExtensions() const;

	/*! \brief This function returns the serial of the certificate.
	 *  \return The serial of the certificate.
	 */
	unsigned long GetSerial() const;

	/*! \brief This function returns the signature algorithm of the certificate.
	 *  \return The signature algorithm of the certificate.
	 */
	const char * GetSignatureAlg() const;

	/*! \brief This function returns the DN of the issuer of the certificate.
	 *  \return The DN of the issuer of the certificate.
	 */
	const HashTable_Dn & GetIssuerDN() const;

	/*! \brief This function returns the length of the key of the certificate.
	 *  \return The length of the key of the certificate.
	 */
	unsigned long GetKeyLen() const;

	/*! \brief This function returns the thumprint algorithm of the certificate.
	 *  \return The thumprint algorithm of the certificate.
	 */
	const char * GetThumprintAlgo() const;

	/*! \brief This function returns the thumprint of the certificate.
	 *  \return The thumprint of the certificate.
	 */
	const char * GetThumprint() const;

	/*! \brief This function returns the public key of the certificate.
	 *  \return The public key of the certificate.
	 */
	const X509_PUBKEY * GetX509_PUBKEY() const;

	/*! \brief This function returns the the DN in string of the certificate.
	 *  \return The the DN in string of the certificate.
	 */
	const char * GetStringName() const;

	/*! \brief This function sets the certificate.
	 *  \param certificate [IN] A pointer to an OpenSSL X509 certificate.
	 *  \return true on success, false on failure.
	 */
	bool SetCert(const X509 *certificate);

	/*! \brief This function loads the certificate from a file.
	 *  \param File [IN] The filename to load the certificate from.
	 *  \return true on success, false on failure.
	 */
	bool LoadFromFile(const mString & File);

	/*! \brief This function sets the certificate.
	 *  \param certificate [IN] A pointer to an OpenSSL X509 certificate.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const X509 *certificate);

	/*! \brief This function copies the internal certificate.
	 *  \param Cert [OUT] The certificate.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(X509 ** Cert) const;

	/*! \brief This function sets the certificate.
	 *  \param PEM_CER [IN] A pointer to a PEM encoded certificate.
	 *  \return true on success, false on failure.
	 */
	bool SetCert(const char * PEM_CER);

	/*! \brief This function returns the expiration (valid to) date of the certificate.
	 *  \return The expiration (valid to) date of the certificate.
	 */
	time_t GetEndDate() const;

	/*! \brief This function returns the begining (valid from) date of the certificate.
	 *  \return The begining (valid from) date of the certificate.
	 */
	time_t GetStartDate() const;

	/*! \brief This function returns the PEM encoded certificate.
	 *  \return The PEM encoded certificate.
	 */
	const mString & GetCertPEM() const;

	/*! \brief This function returns the public key in the OpenSSL EVP_PKEY format.
	 *  \return The public key in the OpenSSL EVP_PKEY format.
	 */
	const EVP_PKEY * GetPublicKey() const;

	/*! \brief This function returns the private key of the certificate.
	 *  \return The private key of the certificate.
	 */
	const PKI_RSA & GetPrivateKey() const;

	/*! \brief This function sets the certificate private key.
	 *  \param PrivateKey [IN] The private key.
	 *  \param CheckKey [IN] Whether to check that the key and the certificate match.
	 *  \return true on success, false on failure.
	 */
	bool SetPrivateKey(const PKI_RSA & PrivateKey, bool CheckKey=true);

	/*! \brief This function returns the certificate in the OpenSSL X509 format.
	 *  \param Duplicate [IN] Should the X509 be duplicated, then freed by calling function.
	 *  \return The certificate in the OpenSSL X509 format.
	 */
	X509 * GetX509(bool Duplicate = false) const;

	/*! \brief This function returns the Dn of the certificate.
	 *  \return The Dn of the certificate.
	 */
	const HashTable_Dn & GetCertDN() const;

	/*! \brief This function clears the internal datas.
	 */
	void Clear();

	/*! \brief This function preloads all the datas.
	 *  \return true on success, false on failure.
	 */
	bool LoadAll();

	/*! \brief This function creates a selfsigned certificate.
	 *  \param ReqCSR [IN] The certificate request.
	 *  \param Exts [IN] The available extensions for the certificate.
	 *  \param Days [IN] The number of days for the certificate validity.
	 *  \param Serial [IN] The serial for the certificate.
	 *  \return true on success, false on failure.
	 */
	bool CreateSelfSigned(const PKI_CSR & ReqCSR, const HashTable_String * Exts, int Days, long Serial);

	/*! \brief This function signs a CSR.
	 *  \param ResultCert [OUT] The resulting certificate.
	 *  \param ReqCSR [IN] The certificate request.
	 *  \param Exts [IN] The available extensions for the certificate.
	 *  \param Days [IN] The number of days for the certificate validity.
	 *  \param Serial [IN] The serial for the certificate.
	 *  \param md [IN] The hash algorithm to use for the signature.
	 *  \param keepCsrExts [IN] Whether to keep the CSR's extensions.
	 *  \param csrExtsOverwrite [IN] Whether to overwrite the CSR's extensions.
	 *  \param check_sig [IN] Whether to check the CSR signature.
	 *  \return true on success, false on failure.
	 */
	bool SignCSR(PKI_CERT & ResultCert, const PKI_CSR & ReqCSR, const HashTable_String * Exts, int Days, long Serial, const char * md, bool keepCsrExts, bool csrExtsOverwrite, bool check_sig = true) const;


	/*! \brief This operator copies one PKI_CERT into another.
	 *  \param other [IN] The other certificate to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_CERT &other );

	/*! \brief This operator compares the certificate to another.
	 *  \param other [IN] The other certificate to compare to.
	 *  \return true when equal, false when different.
	 */
	bool operator==( const PKI_CERT &other ) const;

	/*! \brief This operator compares the certificate to another.
	 *  \param other [IN] The other certificate to compare to.
	 *  \return true when equal, false when different.
	 */
	bool operator==( const X509 * other ) const;

	/*! \brief This operator compares the certificate to another.
	 *  \param other [IN] The other certificate to compare to.
	 *  \return true when different, false when equal.
	 */
	bool operator!=( const X509 * other ) const;

	/*! \brief This operator compares the certificate to another.
	 *  \param pubkey [IN] The other certificate to compare to.
	 *  \return true when equal, false when different.
	 */
	bool operator==( const X509_PUBKEY * pubkey ) const;

	/*! \brief This function return 1 when certificate is operable and 0 when not.
	 *  \return return 1 when certificate is operable and 0 when not.
	 */
	operator int() const;

	/*! \brief This member is an empty instance of this class.
	 */
	static PKI_CERT EmptyInstance;
private:
	X509 * cert;
	HashTable_Dn leDN;
	HashTable_Dn IssuerDN;
	PKI_EXT InternalExts;
	mString pem_cert;
	EVP_PKEY * public_key;
	PKI_RSA rsakey;
	mString m_Thumprint;
	mString m_Name;

	void Reset();
	void LoadThumbprint();
	bool LoadDatas();
	void Init();
	bool StringToX509(const char * pem);
	bool X509ToString(const X509 * certificate);
	bool LoadDN();
	bool LoadIssuerDN();
};

#endif // PKI_CERT_H

