/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_HashTable.h: interface for the PKI_HashTable class.
//
//////////////////////////////////////////////////////////////////////

#ifndef HASHTABLE_H
#define HASHTABLE_H



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <CriticalSection.h>
#include <NewPKI.h>

#define HASHTABLE_NOT_FOUND -1

/*!
  This class is the top level hashtable
 */
class PKI_HashTable
{
public:
	/*! \brief This is the constructor.
	 */
	PKI_HashTable();

	/*! \brief This is the destructor.
	 */
	virtual ~PKI_HashTable();
	
	
	/*! \brief This function allows to enter the PKI_HashTable Critical Section.
	 */
	void Lock() const;

	/*! \brief This function allows to leave the PKI_HashTable Critical Section.
	 */
	void Unlock() const;

	/*! \brief This function search for a Name.
	 *  \param Name [IN] The name to search for.
	 *  \param LastPos [IN] The last position returned by SeekEntryName or -1 when starting.
	 *  \return The position for the name.
	 */
	long SeekEntryName(const char * Name, long LastPos) const;

	/*! \brief This function returns one Name.
	 *  \param Pos [IN] The position of the name to get, starting from 0 to (EntriesCount-1).
	 *  \return The name.
	 */
	const char * GetName(long Pos) const;
	
	/*! \brief This function empties the hashtable.
	 */
	void Clear();

	/*! \brief This function returns the length (in byte) of the datas inside the hashtable.
	 *  \return The length.
	 */
	long DatasLen() const;
	
	/*! \brief This function returns the number of entries inside the hashtable.
	 *  \return The number of entries.
	 */
	long EntriesCount() const;
	
	/*! \brief This function deletes an entry from the hashtable.
	 *  \param Name [IN] The name of the entry to delete.
	 *  \return true on success, false on failure.
	 */
	bool Delete(const char * Name);

	/*! \brief This function deletes an entry from the hashtable.
	 *  \param Pos [IN] The position of the entry to delete.
	 *  \return true on success, false on failure.
	 */
	bool Delete(long Pos);

	/*! \brief This operator copies one PKI_HashTable into another.
	 *  \param other [IN] The other PKI_HashTable to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_HashTable &other );

	/*! \brief This function tell the hashtable that we can have duplicate names.
	 */
	void AllowDuplicateNames();
protected:
	CriticalSection HashLock;
	/*! \brief This structure represents an internal item.
	 */
	struct HASH_LISTE
	{
		char * Name;			//!< The entry name
		void * Value;			//!< The entry datas
		long size;				//!< The entry size pointed by "Value"
		HASH_LISTE * psuiv;		//!< The next entry in the list
	};
	
	const void * m_GetPTR(long Pos) const;
	bool m_Add(const char * Name, const void * Value, long size);

	bool m_Modify(const char *Name, const void *Value, long size);
	bool m_Modify(long Pos, const void *Value, long size);

	const void * m_Get(const char * Name) const;
	
	HASH_LISTE * hashliste;
	long ctr; 
	long datas_len;
	bool AllowDup;
};

#endif

