/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef HASHCORRELATION_ASN1_H
#define HASHCORRELATION_ASN1_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <ASN1/Asn1Helper.h>

/*!
  This structure holds the hash
*/
typedef struct st_INTERNAL_HASH
{
	ASN1_OCTET_STRING * hash; //!< The hash
}INTERNAL_HASH;
DECLARE_ASN1_FUNCTIONS(INTERNAL_HASH);

/*!
  This class is the representation of INTERNAL_HASH
*/
class InternalHash : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalHash();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalHash(const InternalHash & other);

	/*! \brief This is the destructor.
	 */
	~InternalHash();


	/*! \brief This function converts a INTERNAL_HASH to a InternalHash.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_HASH * Datas);

	/*! \brief This function converts a InternalHash to a INTERNAL_HASH.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_HASH ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_HASH.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_OCTET_STRING::hash.
	 *  \param c_hash [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_hash(const Asn1OctetString & c_hash);

	/*! \brief This function returns ASN1_OCTET_STRING::hash.
	 *  \return The value of ASN1_OCTET_STRING::hash.
	 */
	const Asn1OctetString & get_hash() const;

	/*! \brief This function returns ASN1_OCTET_STRING::hash.
	 *  \return The value of ASN1_OCTET_STRING::hash.
	 */
	Asn1OctetString & get_hash();


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a InternalHash to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalHash & other);
	/*! \brief This member is an empty instance of InternalHash.
	 */
	static InternalHash EmptyInstance;

private:
	Asn1OctetString m_hash;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef HASHCORRELATION_ASN1_H
