/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1USER_H
#define ASN1USER_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/evp.h>
#include <openssl/pkcs12.h>
#include <mString.h>
#include <PKI_CERT.h>
#include <PKI_P7B.h>
#include <PKI_PKCS12.h>
#include "Asn1Helper.h"



#define USER_TYPE_SERVER		1		//!< The user that just logged in is a Server user
#define USER_TYPE_PKI			2		//!< The user that just logged in is a PKI user
#define USER_TYPE_CA			3		//!< The user that just logged in is a CA user
#define USER_TYPE_RA			4		//!< The user that just logged in is a RA user
#define USER_TYPE_REPOSITORY	5		//!< The user that just logged in is a Repository user
#define USER_TYPE_KEYSTORE		6		//!< The user that just logged in is a KeyStore user
#define USER_TYPE_PUBLICATION	7		//!< The user that just logged in is a Publication user
#define USER_TYPE_EE			8		//!< The user that just logged in is an EE user




/*!
	This structure represents a software key for a PKI user certificate generation
*/
typedef struct st_PKI_USER_KEY_SOFT
{
	ASN1_INTEGER * keylen;		//!< The key length
	ASN1_UTF8STRING * password;	//!< The password for the PKCS#12
}PKI_USER_KEY_SOFT;
DECLARE_ASN1_ITEM(PKI_USER_KEY_SOFT)


#define PKI_USER_KEY_TYPE_SOFTKEY 0		//!< The key should be generated
#define PKI_USER_KEY_TYPE_PUBKEY 1		//!< The key was generated by client

/*!
	This structure represents the information needed for a PKI user certificate generation
*/
typedef struct st_PKI_USER_KEY
{
	int type;
	union
	{
		PKI_USER_KEY_SOFT * softkey;		//!< The key length
		X509_PUBKEY * pubkey;		//!< The public key
	}d;
}PKI_USER_KEY;
DECLARE_ASN1_ITEM(PKI_USER_KEY)


/*!
	This structure represents a PKI user creation request.
*/
typedef struct st_CREATE_PKI_USER_REQUEST
{
	ASN1_INTEGER * pkiadmin;	//!< Is the new user a PKI admin ?
	ASN1_UTF8STRING * cn;		//!< The name of the user
	ASN1_UTF8STRING * email;	//!< The email address of the user
	PKI_USER_KEY * ukey;		//!< The key
}CREATE_PKI_USER_REQUEST;
DECLARE_ASN1_ITEM(CREATE_PKI_USER_REQUEST)

/*!
	This structure represents a PKI user creation response.
*/
typedef struct st_CREATE_PKI_USER_RESPONSE
{
	int type;
	union
	{
		PKCS12 * p12;			//!< A PKCS#12
		PKCS7 * p7b;			//!< A PKCS#7
	}d;
}CREATE_PKI_USER_RESPONSE;
DECLARE_ASN1_ITEM(CREATE_PKI_USER_RESPONSE)



/*!
	This enumeration defines the possible flags for a user
*/
typedef enum USER_FLAGS
{
	USER_FLAGS_EXTERNAL_AUTH		= 1		//!< The authentication should be processed by an external module
};


/*!
	This structure defines a Server user personal info
*/
typedef struct st_USER_INFO
{
	ASN1_UTF8STRING * Name;			//!< The user name
	ASN1_INTEGER * Activated;		//!< The user activation status
	ASN1_INTEGER * Flags;			//!< The user flags
}USER_INFO;
DECLARE_ASN1_ITEM(USER_INFO)

/*!
	This structure defines a Server user
*/
typedef struct st_USER_ENTRY
{
	ASN1_INTEGER * user_id;			//!< The user id
	USER_INFO * user_info;			//!< The user information
}USER_ENTRY;
DECLARE_ASN1_ITEM(USER_ENTRY)

/*!
	This structure defines the certificate for a PKI user
*/
typedef struct st_USER_ENTRY_CERT
{
	X509 * user_cert;				//!< Its certificate
	ASN1_INTEGER * admin;			//!< Is it a PKI Admin
}USER_ENTRY_CERT;
DECLARE_ASN1_ITEM(USER_ENTRY_CERT)


/*!
	This structure defines a password change request for a server user
*/
typedef struct st_CHANGE_USER_PASSWD
{
	ASN1_INTEGER * user_id;			//!< The user id
	ASN1_UTF8STRING * Password;		//!< The new password
}CHANGE_USER_PASSWD;
DECLARE_ASN1_ITEM(CHANGE_USER_PASSWD)


/*!
	This structure defines a password change request by a server user
*/
typedef struct st_CHANGE_PASSWD
{
	ASN1_UTF8STRING * Password;		//!< The new password
}CHANGE_PASSWD;
DECLARE_ASN1_ITEM(CHANGE_PASSWD)

/*!
  This class is the representation of USER_INFO
*/
class UserInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	UserInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	UserInfo(const UserInfo & other);

	/*! \brief This is the destructor.
	 */
	~UserInfo();


	/*! \brief This function converts a USER_INFO to a UserInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const USER_INFO * Datas);

	/*! \brief This function converts a UserInfo to a USER_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(USER_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of USER_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::Activated.
	 *  \param c_activated [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_activated(unsigned long c_activated);

	/*! \brief This function returns ASN1_INTEGER::Activated.
	 *  \return The value of ASN1_INTEGER::Activated.
	 */
	unsigned long get_activated() const;


	/*! \brief This function sets ASN1_INTEGER::Flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(unsigned long c_flags);

	/*! \brief This function returns ASN1_INTEGER::Flags.
	 *  \return The value of ASN1_INTEGER::Flags.
	 */
	unsigned long get_flags() const;


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function copies a UserInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const UserInfo & other);
	/*! \brief This member is an empty instance of UserInfo.
	 */
	static UserInfo EmptyInstance;

private:
	unsigned long m_activated;
	unsigned long m_flags;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of USER_ENTRY_CERT
*/
class UserEntryCert : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	UserEntryCert();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	UserEntryCert(const UserEntryCert & other);

	/*! \brief This is the destructor.
	 */
	~UserEntryCert();


	/*! \brief This function converts a USER_ENTRY_CERT to a UserEntryCert.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const USER_ENTRY_CERT * Datas);

	/*! \brief This function converts a UserEntryCert to a USER_ENTRY_CERT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(USER_ENTRY_CERT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of USER_ENTRY_CERT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::admin.
	 *  \param c_admin [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_admin(unsigned long c_admin);

	/*! \brief This function returns ASN1_INTEGER::admin.
	 *  \return The value of ASN1_INTEGER::admin.
	 */
	unsigned long get_admin() const;


	/*! \brief This function sets X509::user_cert.
	 *  \param c_userCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_userCert(const PKI_CERT & c_userCert);

	/*! \brief This function returns X509::user_cert.
	 *  \return The value of X509::user_cert.
	 */
	const PKI_CERT & get_userCert() const;

	/*! \brief This function returns X509::user_cert.
	 *  \return The value of X509::user_cert.
	 */
	PKI_CERT & get_userCert();


	/*! \brief This function copies a UserEntryCert to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const UserEntryCert & other);
	/*! \brief This member is an empty instance of UserEntryCert.
	 */
	static UserEntryCert EmptyInstance;

private:
	unsigned long m_admin;
	PKI_CERT m_userCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of USER_ENTRY
*/
class UserEntry : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	UserEntry();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	UserEntry(const UserEntry & other);

	/*! \brief This is the destructor.
	 */
	~UserEntry();


	/*! \brief This function converts a USER_ENTRY to a UserEntry.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const USER_ENTRY * Datas);

	/*! \brief This function converts a UserEntry to a USER_ENTRY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(USER_ENTRY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of USER_ENTRY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::user_id.
	 *  \param c_userId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_userId(unsigned long c_userId);

	/*! \brief This function returns ASN1_INTEGER::user_id.
	 *  \return The value of ASN1_INTEGER::user_id.
	 */
	unsigned long get_userId() const;


	/*! \brief This function sets USER_INFO::user_info.
	 *  \param c_userInfo [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_userInfo(const UserInfo & c_userInfo);

	/*! \brief This function returns USER_INFO::user_info.
	 *  \return The value of USER_INFO::user_info.
	 */
	const UserInfo & get_userInfo() const;

	/*! \brief This function returns USER_INFO::user_info.
	 *  \return The value of USER_INFO::user_info.
	 */
	UserInfo & get_userInfo();


	/*! \brief This function copies a UserEntry to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const UserEntry & other);
	/*! \brief This member is an empty instance of UserEntry.
	 */
	static UserEntry EmptyInstance;

private:
	unsigned long m_userId;
	UserInfo m_userInfo;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CHANGE_USER_PASSWD
*/
class ChangeUserPasswd : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ChangeUserPasswd();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ChangeUserPasswd(const ChangeUserPasswd & other);

	/*! \brief This is the destructor.
	 */
	~ChangeUserPasswd();


	/*! \brief This function converts a CHANGE_USER_PASSWD to a ChangeUserPasswd.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CHANGE_USER_PASSWD * Datas);

	/*! \brief This function converts a ChangeUserPasswd to a CHANGE_USER_PASSWD.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CHANGE_USER_PASSWD ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CHANGE_USER_PASSWD.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::Password.
	 *  \return The value of ASN1_UTF8STRING::Password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::Password.
	 *  \return The value of ASN1_UTF8STRING::Password.
	 */
	mString & get_password();


	/*! \brief This function sets ASN1_INTEGER::user_id.
	 *  \param c_userId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_userId(unsigned long c_userId);

	/*! \brief This function returns ASN1_INTEGER::user_id.
	 *  \return The value of ASN1_INTEGER::user_id.
	 */
	unsigned long get_userId() const;


	/*! \brief This function copies a ChangeUserPasswd to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ChangeUserPasswd & other);
	/*! \brief This member is an empty instance of ChangeUserPasswd.
	 */
	static ChangeUserPasswd EmptyInstance;

private:
	mString m_password;
	unsigned long m_userId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CHANGE_PASSWD
*/
class ChangePasswd : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ChangePasswd();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ChangePasswd(const ChangePasswd & other);

	/*! \brief This is the destructor.
	 */
	~ChangePasswd();


	/*! \brief This function converts a CHANGE_PASSWD to a ChangePasswd.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CHANGE_PASSWD * Datas);

	/*! \brief This function converts a ChangePasswd to a CHANGE_PASSWD.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CHANGE_PASSWD ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CHANGE_PASSWD.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::Password.
	 *  \return The value of ASN1_UTF8STRING::Password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::Password.
	 *  \return The value of ASN1_UTF8STRING::Password.
	 */
	mString & get_password();


	/*! \brief This function copies a ChangePasswd to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ChangePasswd & other);
	/*! \brief This member is an empty instance of ChangePasswd.
	 */
	static ChangePasswd EmptyInstance;

private:
	mString m_password;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PKI_USER_KEY_SOFT
*/
class PkiUserKeySoft : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PkiUserKeySoft();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PkiUserKeySoft(const PkiUserKeySoft & other);

	/*! \brief This is the destructor.
	 */
	~PkiUserKeySoft();


	/*! \brief This function converts a PKI_USER_KEY_SOFT to a PkiUserKeySoft.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKI_USER_KEY_SOFT * Datas);

	/*! \brief This function converts a PkiUserKeySoft to a PKI_USER_KEY_SOFT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKI_USER_KEY_SOFT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PKI_USER_KEY_SOFT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::keylen.
	 *  \param c_keylen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keylen(unsigned long c_keylen);

	/*! \brief This function returns ASN1_INTEGER::keylen.
	 *  \return The value of ASN1_INTEGER::keylen.
	 */
	unsigned long get_keylen() const;


	/*! \brief This function sets ASN1_UTF8STRING::password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	mString & get_password();


	/*! \brief This function copies a PkiUserKeySoft to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PkiUserKeySoft & other);
	/*! \brief This member is an empty instance of PkiUserKeySoft.
	 */
	static PkiUserKeySoft EmptyInstance;

private:
	unsigned long m_keylen;
	mString m_password;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PKI_USER_KEY
*/
class PkiUserKey : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PkiUserKey();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PkiUserKey(const PkiUserKey & other);

	/*! \brief This is the destructor.
	 */
	~PkiUserKey();


	/*! \brief This function converts a PKI_USER_KEY to a PkiUserKey.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKI_USER_KEY * Datas);

	/*! \brief This function converts a PkiUserKey to a PKI_USER_KEY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKI_USER_KEY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PKI_USER_KEY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets X509_PUBKEY::d.pubkey.
	 *  \param c_pubkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubkey(const X509_PUBKEY * c_pubkey);

	/*! \brief This function returns X509_PUBKEY::d.pubkey.
	 *  \return The value of X509_PUBKEY::d.pubkey.
	 */
	const X509_PUBKEY * get_pubkey() const;

	/*! \brief This function returns X509_PUBKEY::d.pubkey.
	 *  \return The value of X509_PUBKEY::d.pubkey.
	 */
	X509_PUBKEY * get_pubkey();


	/*! \brief This function sets PKI_USER_KEY_SOFT::d.softkey.
	 *  \param c_softkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_softkey(const PkiUserKeySoft & c_softkey);

	/*! \brief This function returns PKI_USER_KEY_SOFT::d.softkey.
	 *  \return The value of PKI_USER_KEY_SOFT::d.softkey.
	 */
	const PkiUserKeySoft & get_softkey() const;

	/*! \brief This function returns PKI_USER_KEY_SOFT::d.softkey.
	 *  \return The value of PKI_USER_KEY_SOFT::d.softkey.
	 */
	PkiUserKeySoft & get_softkey();


	/*! \brief This function copies a PkiUserKey to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PkiUserKey & other);
	/*! \brief This member is an empty instance of PkiUserKey.
	 */
	static PkiUserKey EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	X509_PUBKEY * m_pubkey;
	PkiUserKeySoft * m_softkey;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CREATE_PKI_USER_REQUEST
*/
class CreatePkiUserRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CreatePkiUserRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CreatePkiUserRequest(const CreatePkiUserRequest & other);

	/*! \brief This is the destructor.
	 */
	~CreatePkiUserRequest();


	/*! \brief This function converts a CREATE_PKI_USER_REQUEST to a CreatePkiUserRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CREATE_PKI_USER_REQUEST * Datas);

	/*! \brief This function converts a CreatePkiUserRequest to a CREATE_PKI_USER_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CREATE_PKI_USER_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CREATE_PKI_USER_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::cn.
	 *  \param c_cn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cn(const mString & c_cn);

	/*! \brief This function returns ASN1_UTF8STRING::cn.
	 *  \return The value of ASN1_UTF8STRING::cn.
	 */
	const mString & get_cn() const;

	/*! \brief This function returns ASN1_UTF8STRING::cn.
	 *  \return The value of ASN1_UTF8STRING::cn.
	 */
	mString & get_cn();


	/*! \brief This function sets ASN1_UTF8STRING::email.
	 *  \param c_email [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_email(const mString & c_email);

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	const mString & get_email() const;

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	mString & get_email();


	/*! \brief This function sets ASN1_INTEGER::pkiadmin.
	 *  \param c_pkiadmin [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pkiadmin(unsigned long c_pkiadmin);

	/*! \brief This function returns ASN1_INTEGER::pkiadmin.
	 *  \return The value of ASN1_INTEGER::pkiadmin.
	 */
	unsigned long get_pkiadmin() const;


	/*! \brief This function sets PKI_USER_KEY::ukey.
	 *  \param c_ukey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ukey(const PkiUserKey & c_ukey);

	/*! \brief This function returns PKI_USER_KEY::ukey.
	 *  \return The value of PKI_USER_KEY::ukey.
	 */
	const PkiUserKey & get_ukey() const;

	/*! \brief This function returns PKI_USER_KEY::ukey.
	 *  \return The value of PKI_USER_KEY::ukey.
	 */
	PkiUserKey & get_ukey();


	/*! \brief This function copies a CreatePkiUserRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CreatePkiUserRequest & other);
	/*! \brief This member is an empty instance of CreatePkiUserRequest.
	 */
	static CreatePkiUserRequest EmptyInstance;

private:
	mString m_cn;
	mString m_email;
	unsigned long m_pkiadmin;
	PkiUserKey m_ukey;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CREATE_PKI_USER_RESPONSE
*/
class CreatePkiUserResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CreatePkiUserResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CreatePkiUserResponse(const CreatePkiUserResponse & other);

	/*! \brief This is the destructor.
	 */
	~CreatePkiUserResponse();


	/*! \brief This function converts a CREATE_PKI_USER_RESPONSE to a CreatePkiUserResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CREATE_PKI_USER_RESPONSE * Datas);

	/*! \brief This function converts a CreatePkiUserResponse to a CREATE_PKI_USER_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CREATE_PKI_USER_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CREATE_PKI_USER_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets PKCS12::d.p12.
	 *  \param c_p12 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p12(const PKI_PKCS12 & c_p12);

	/*! \brief This function returns PKCS12::d.p12.
	 *  \return The value of PKCS12::d.p12.
	 */
	const PKI_PKCS12 & get_p12() const;

	/*! \brief This function returns PKCS12::d.p12.
	 *  \return The value of PKCS12::d.p12.
	 */
	PKI_PKCS12 & get_p12();


	/*! \brief This function sets PKCS7::d.p7b.
	 *  \param c_p7b [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p7b(const PKI_P7B & c_p7b);

	/*! \brief This function returns PKCS7::d.p7b.
	 *  \return The value of PKCS7::d.p7b.
	 */
	const PKI_P7B & get_p7b() const;

	/*! \brief This function returns PKCS7::d.p7b.
	 *  \return The value of PKCS7::d.p7b.
	 */
	PKI_P7B & get_p7b();


	/*! \brief This function copies a CreatePkiUserResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CreatePkiUserResponse & other);
	/*! \brief This member is an empty instance of CreatePkiUserResponse.
	 */
	static CreatePkiUserResponse EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	PKI_PKCS12 * m_p12;
	PKI_P7B * m_p7b;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef ASN1USER_H
