/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgBashImport.cpp: implementation of the DlgBashImport class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgBashImport.h"
#include "dlgs_wdr.h"
#include "clintl.h"
#include <PKI_CSR.h>
#include "DlgRequestCert.h"
#include "DlgProgress.h"
#include <stdio.h>
#include <sys/stat.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


BEGIN_EVENT_TABLE(DlgBashImportListCtrl, wxListCtrl)
    EVT_LIST_ITEM_SELECTED(IDC_LISTCTRL, DlgBashImportListCtrl::OnItemClick)
    EVT_MOUSE_EVENTS(DlgBashImportListCtrl::OnMouseEvent)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(DlgBashImport, wxDialog)
    EVT_BUTTON(IDC_CANCEL, DlgBashImport::OnCancel)
    EVT_BUTTON(IDC_OPEN_XML, DlgBashImport::OnImportXML)
    EVT_BUTTON(IDC_IMPORT, DlgBashImport::OnImport)
    EVT_BUTTON(IDC_SAVE_DTD, DlgBashImport::OnSaveDtd)
    EVT_BUTTON(IDC_SAVE_XML, DlgBashImport::OnSaveState)
END_EVENT_TABLE()

void DlgBashImportListCtrl::OnItemClick(wxListEvent& event)
{
	DlgBashImport * wParent = (DlgBashImport *)GetParent();
	wParent->OnItemClick(event);
}

void DlgBashImportListCtrl::OnMouseEvent(wxMouseEvent & event)
{
	DlgBashImport * wParent = (DlgBashImport *)GetParent();
	wParent->OnMouseEvent(event);
}



extern "C" int xmlDoValidityCheckingDefaultValue;

#define PROFILES_HEADER \
"<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n"

#define PROFILES_DTD \
"<!ELEMENT PROFILES		(PROFILE*, IMPORTED_PROFILE*)>\n" \
"<!ELEMENT PROFILE		(DN, REQUESTS?)>\n" \
"    <!ATTLIST PROFILE\n" \
"	owner_group	CDATA	#IMPLIED\n" \
"	ldap_uid	CDATA	#IMPLIED\n" \
">\n" \
"<!ELEMENT IMPORTED_PROFILE		(DN, REQUESTS?)>\n" \
"    <!ATTLIST IMPORTED_PROFILE\n" \
"	id	CDATA			#REQUIRED\n" \
">\n" \
"<!ELEMENT DN			(DN_ENTRY*)>\n" \
"<!ELEMENT DN_ENTRY		(NAME, VALUE)>\n" \
"<!ELEMENT NAME			(#PCDATA)>\n" \
"<!ELEMENT VALUE			(#PCDATA)>\n" \
"<!ELEMENT REQUESTS		(REQUEST*)>\n" \
"<!ELEMENT REQUEST		(REQ_P10|REQ_SC|REQ_P12)>\n" \
"    <!ATTLIST REQUEST\n" \
"	ca			CDATA	#REQUIRED\n" \
"	send_cert	(0|1)	#REQUIRED\n" \
"	validity	CDATA	#IMPLIED\n" \
">	<!-- validity: If not provided the default value for the RA is used -->\n" \
"<!ELEMENT REQ_P10		(#PCDATA)>	<!-- The content is the PEM encoded PKCS#10 -->\n" \
"<!ELEMENT REQ_SC		(#PCDATA)>	<!-- The content is the CSP name -->\n" \
"    <!ATTLIST REQ_SC\n" \
"	keylen		CDATA	#REQUIRED\n" \
">\n" \
"<!ELEMENT REQ_P12		(#PCDATA)>	<!-- The content is the password, will be ignored if gen_pwd=1 -->\n" \
"    <!ATTLIST REQ_P12\n" \
"	keylen		CDATA	#REQUIRED\n" \
"	send_pwd	(0|1)	#IMPLIED\n" \
"	gen_pwd		(0|1)	#IMPLIED\n" \
">"

static char * strDTD =  PROFILES_HEADER \
"\n" \
PROFILES_DTD;


DlgBashImport::DlgBashImport(wxWindow * wParent, const EntityConf & cEntityConf, PkiClient * ClientPki):
		wxDialog(wParent, (wxWindowID)-1, _("Batch Profiles Importation"), wxDefaultPosition, wxDefaultSize, wxSYSTEM_MENU | wxMINIMIZE_BOX | wxCAPTION),
		m_EntityConf(cEntityConf)
{
	xmlParserInputBufferPtr xMem;

	xmlInitParser();
	xmlDoValidityCheckingDefaultValue = 0;
	xmlSetGenericErrorFunc(this, xmlErrorCB);
	xDtd = NULL;

	
	xMem = xmlParserInputBufferCreateMem(strDTD, strlen(strDTD), XML_CHAR_ENCODING_NONE);
	if(xMem)
	{
		xDtd = xmlIOParseDTD(NULL, xMem, XML_CHAR_ENCODING_NONE);
		if(xDtd)
		{
			xDtd->ExternalID = NULL;
			xDtd->SystemID = NULL;
			xDtd->name = (xmlChar*)strdup("PROFILES");
		}
	}





	m_imageListSmall = NULL;
	m_ClientPki = ClientPki;

	DlgBashImport_SetWindow(this);
	((wxTextCtrl *)FindWindow(IDC_INFO))->SetBackgroundColour(GetBackgroundColour());


	m_listCtrl = ((wxListCtrl *)FindWindow(IDC_LIST_PROFILES));
	m_imageListSmall = new wxImageList(16, 16, TRUE);

	wxIcon ico;

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_EMPTY_PROFILE));
	EmptyProfileIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_PLUS_PROFILE));
	PlusProfileIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_MINUS_PROFILE));
	MinusProfileIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_ACT_MID));
	ActMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_ACT_END));
	ActEndConnectorIconId = m_imageListSmall->Add( ico );

	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);

	m_listCtrl->InsertColumn(0, _("DN"), wxLIST_FORMAT_LEFT, 385);
	m_listCtrl->InsertColumn(1, _("ID"), wxLIST_FORMAT_LEFT, 60);


	CenterOnScreen();
	ShowModal();
}

DlgBashImport::~DlgBashImport()
{
	Clear();
	if(m_imageListSmall)
		delete m_imageListSmall;
	xmlCleanupParser();
	if(xDtd)
		xmlFreeDtd(xDtd);
}

void DlgBashImport::OnItemClick(wxListEvent& event)
{
	XmlRequest * request;
	request = GetSelectedCert();
	if(request)
	{
		DlgRequestCert Dlg(this, NULL, m_EntityConf, request);
	}
}

void DlgBashImport::OnCancel(wxCommandEvent& event)
{
	Close(TRUE);
}

void DlgBashImport::OnSaveDtd(wxCommandEvent& event)
{
	FILE * fp;
	wxFileDialog dialog(this, _("Save DTD"), "", "", _("DTD File (*.dtd)|*.dtd"), wxSAVE | wxOVERWRITE_PROMPT);
	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}

	fp = fopen(dialog.GetPath(), "wb");
	if(!fp)
	{
		HandleError(strerror(errno), this);
		return;
	}
	fwrite(strDTD, 1, strlen(strDTD), fp);
	fclose(fp);
}

void DlgBashImport::OnSaveState(wxCommandEvent& event)
{
	size_t i;
	FILE * fp;
	xmlDocPtr xDoc;
	DlgProgress dlg(this);
	float val;
	size_t profilesCount = m_profiles.size();

	wxFileDialog dialog(this, _("Save State"), "", "", _("XML File (*.xml)|*.xml"), wxSAVE | wxOVERWRITE_PROMPT);
	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	
	}
	fp = fopen(dialog.GetPath(), "wb");
	if(!fp)
	{
		HandleError(strerror(errno), this);
		return;
	}
	fwrite(PROFILES_HEADER, 1, strlen(PROFILES_HEADER), fp);
	fprintf(fp, "<!DOCTYPE PROFILES [\n");
	fwrite(PROFILES_DTD, 1, strlen(PROFILES_DTD), fp);
	fprintf(fp, "\n]>\n");
	fprintf(fp, "<PROFILES>\n");
	xDoc = xmlNewDoc((const xmlChar*)"1.0");
	if(!xDoc)
	{
		fclose(fp);
		return;
	}
	xDoc->encoding = (xmlChar*)strdup("iso-8859-1");

	

	dlg.SetMax(100);
	dlg.SetTitle(_("Saving Profiles"));
	dlg.wShow();

	for(i=0; i<profilesCount; i++)
	{
		val = (float)i/(float)profilesCount;
		dlg.SetValue((int)(val*100.0));
		wxYieldIfNeeded();

		if(!m_profiles[i]->Dump(fp, xDoc))
		{
			fclose(fp);
			xmlFreeDoc(xDoc);
			dlg.wHide();
			return;
		}
	}
	dlg.wHide();

	fprintf(fp, "\n</PROFILES>\n");
	xmlFreeDoc(xDoc);
	fclose(fp);
}

void DlgBashImport::OnImportXML(wxCommandEvent& event)
{
	wxFileDialog dialog(this, _("Open XML"), "", "", _("XML File (*.xml)|*.xml"), wxOPEN);
	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}
	LoadXML(dialog.GetPath().c_str());
}

void DlgBashImport::LoadXML(const mString & file)
{
	xmlNodePtr currChild;
	xmlNodePtr currProfile;
	xmlDocPtr doc = NULL;
    xmlValidCtxt cvp;
    cvp.userData = (void *)this;
	cvp.error    = xmlErrorCB;
	cvp.warning  = xmlWarningCB;
	XmlProfile * newProfile;
	mString strError;
	DlgProgress dlg(this);
	unsigned long profilesCount = 0;
	unsigned long profilesNum = 0;
	FILE * fp;
	xmlParserCtxtPtr ctxt;
	char buffer[8192];
	size_t len;
	struct stat lstat;
    xmlDtdPtr oldExt, oldInt;
	float val;
	mString title;

	m_Errors = "";
	m_Warnings = "";

	Clear();

	if(stat(file.c_str(), &lstat) == -1)
	{
		HandleError(strerror(errno), this);
		return;
	}

	dlg.SetMax(100);
	dlg.SetTitle(_("Parsing XML"));
	dlg.wShow();
	fp = fopen(file.c_str(), "rb");
	if(!fp)
	{
		HandleError(strerror(errno), this);
		return;
	}
	ctxt = xmlCreatePushParserCtxt(NULL, NULL, NULL, NULL, file.c_str());
    if(!ctxt)
	{
		HandleError(_("Failed to create parser context"), this);
		fclose(fp);
    }
	xmlCtxtUseOptions(ctxt, XML_PARSE_NOBLANKS);

	while(!feof(fp))
	{
		len = fread(buffer, 1, sizeof(buffer), fp);
		if(len)
		{
			xmlParseChunk(ctxt, buffer, len, 0);
		}
		val = (float)ftell(fp)/(float)lstat.st_size;
		dlg.SetValue((int)(val*100.0));
		wxYieldIfNeeded();
		if(dlg.IsCanceled())
		{
			fclose(fp);
			if(ctxt->myDoc)
			{
				xmlFreeDoc(ctxt->myDoc);
			}
			xmlFreeParserCtxt(ctxt);
			dlg.wHide();
			return;
		}
	}
	xmlParseChunk(ctxt, buffer, 0, 1);
	fclose(fp);
	doc = ctxt->myDoc;
	xmlFreeParserCtxt(ctxt);
	dlg.wHide();
	if(m_Errors.size())
	{
		xmlFreeDoc(doc);
		HandleError(m_Errors.c_str(), this);
		return;
	}


	// Special tweak to make sure the validation of a DTD is done
	// inside our progress bar display
    oldExt = doc->extSubset;
    oldInt = doc->intSubset;
    doc->extSubset = xDtd;
    doc->intSubset = NULL;
    if (!xmlValidateRoot(&cvp, doc))
	{
		doc->extSubset = oldExt;
		doc->intSubset = oldInt;
		xmlFreeDoc(doc);
		HandleError(m_Errors.c_str(), this);
		return;
    }
    if (doc->ids != NULL)
	{
          xmlFreeIDTable((xmlHashTable*)doc->ids);
          doc->ids = NULL;
    }
    if (doc->refs != NULL)
	{
          xmlFreeRefTable((xmlHashTable*)doc->refs);
          doc->refs = NULL;
    }

	for(currChild=xmlDocGetRootElement(doc); currChild; currChild=currChild->next)
	{
		if(xmlStrcmp(currChild->name, (xmlChar*)"PROFILES") == 0)
		{
			// Validate the syntax of PROFILES, but not its children
			currProfile=currChild->children;
			currChild->children = NULL;
			if(!xmlValidateElement(&cvp, doc, currChild))
			{
				currChild->children = currProfile;
				doc->extSubset = oldExt;
				doc->intSubset = oldInt;
				xmlFreeDoc(doc);
				HandleError(m_Errors.c_str(), this);
				return;
			}
			currChild->children = currProfile;

			for(currProfile=currChild->children; currProfile; currProfile=currProfile->next)
			{
				profilesNum++;
			}

			dlg.SetMax(100);
			dlg.wShow();
			if(dlg.IsCanceled())
			{
				Clear();
				break;
			}
			wxYieldIfNeeded();

			profilesCount=0;
			currProfile=currChild->children;

			while(currProfile)
			{
				profilesCount++;
				title.sprintf(_("[%ld/%ld]"), profilesCount, profilesNum);
				dlg.SetTitle(title.c_str());
				val = (float)profilesCount/(float)profilesNum;
				dlg.SetValue((int)(val*100.0));
				if(dlg.IsCanceled())
				{
					Clear();
					break;
				}
				wxYieldIfNeeded();

				if(!xmlValidateElement(&cvp, doc, currProfile))
				{
					doc->extSubset = oldExt;
					doc->intSubset = oldInt;
					xmlFreeDoc(doc);
					dlg.wHide();
					HandleError(m_Errors.c_str(), this);
					return;
				}

				if(xmlStrcmp(currProfile->name, (xmlChar*)"PROFILE") == 0 ||
					xmlStrcmp(currProfile->name, (xmlChar*)"IMPORTED_PROFILE") == 0)
				{
					newProfile = new XmlProfile(m_EntityConf);
					if(!newProfile)
					{
						doc->extSubset = oldExt;
						doc->intSubset = oldInt;
						xmlFreeDoc(doc);
						dlg.wHide();
						HandleError(NEWPKIerrGetStr(ERROR_MALLOC), this);
						return;
					}
					if(!newProfile->Load(currProfile))
					{
						doc->extSubset = oldExt;
						doc->intSubset = oldInt;
						xmlFreeDoc(doc);
						dlg.wHide();
						ERR_to_mstring(strError, true);
						HandleError(strError.c_str(), this);
						return;
					}
					m_profiles.push_back(newProfile);
				}
				currChild->children = currProfile->next;
				xmlFreeNode(currProfile);
				currProfile=currChild->children;
			}
			dlg.wHide();
		}
	}
	if(!xmlValidateDocumentFinal(&cvp, doc))
	{
		doc->extSubset = oldExt;
		doc->intSubset = oldInt;
		xmlFreeDoc(doc);
		HandleError(m_Errors.c_str(), this);
		return;
	}
	if(m_Warnings.size())
	{
		HandleError(m_Warnings.c_str(), this);
	}
	doc->extSubset = oldExt;
	doc->intSubset = oldInt;
	xmlFreeDoc(doc);

	ReloadList();
}

void DlgBashImport::ReloadList()
{
	size_t i;
	mString id;
	long index;
	int IconId;
	bool has_errors = false;
	wxColour greycol(125, 125, 125);
	DlgProgress dlg(this);
	float val;
	size_t profilesCount = m_profiles.size();

	dlg.SetMax(100);
	dlg.SetTitle(_("Loading List"));
	dlg.wShow();
	m_listCtrl->DeleteAllItems();
	for(i=0; i<profilesCount; i++)
	{
		val = (float)i/(float)profilesCount;
		dlg.SetValue((int)(val*50.0));
		wxYieldIfNeeded();

		if(!m_profiles[i]->IsError())
			continue;
		has_errors = true;

		if(m_profiles[i]->get_Requests().size())
			IconId = PlusProfileIconId;
		else
			IconId = EmptyProfileIconId;

		index = m_listCtrl->InsertItem(m_listCtrl->GetItemCount(), m_profiles[i]->GetName(), IconId);
		m_listCtrl->SetItemData(index, (long)m_profiles[i]);
		m_listCtrl->SetItemTextColour(index, *wxRED);

		m_profiles[i]->set_ItemId(index);
	}

	for(i=0; i<profilesCount; i++)
	{
		val = (float)i/(float)profilesCount;
		dlg.SetValue((int)(val*50.0)+50);
		wxYieldIfNeeded();

		if(m_profiles[i]->IsError())
			continue;

		if(m_profiles[i]->get_Requests().size())
			IconId = PlusProfileIconId;
		else
			IconId = EmptyProfileIconId;

		index = m_listCtrl->InsertItem(m_listCtrl->GetItemCount(), m_profiles[i]->GetName(), IconId);
		m_listCtrl->SetItemData(index, (long)m_profiles[i]);
		if(m_profiles[i]->IsImported())
		{
			id.sprintf("%ld", m_profiles[i]->get_Id());
			m_listCtrl->SetItem(index, 1, id.c_str());
			m_listCtrl->SetItemTextColour(index, greycol);
		}

		m_profiles[i]->set_ItemId(index);
	}
	dlg.wHide();
	((wxButton *)FindWindow(IDC_IMPORT))->Enable(!has_errors);
	((wxTextCtrl *)FindWindow(IDC_INFO))->SetValue("");
}

void DlgBashImport::OnImport(wxCommandEvent& event)
{
	XmlProfile * profile;
	XmlRequest * request;
	long SelectedItem = -1;
	DlgProgress dlg(this);
	size_t i;
	int j;
	mString title;
	unsigned long profile_id;
	unsigned long req_id;
	bool has_errors = false;
	vector<XmlProfile*> lprofiles;
	mString id;

	// This will close all the nodes
	dlg.SetMax(m_profiles.size());
	dlg.wShow();
	for(i=0; i<m_profiles.size(); i++)
	{
		// Import profile
		profile = m_profiles[i];
		if(!profile)
			continue;

		m_listCtrl->SetItemState(profile->get_ItemId(), wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
		m_listCtrl->EnsureVisible(profile->get_ItemId());

		dlg.SetValue(i+1);
		title.sprintf("[%d/%d]", i+1, m_profiles.size());
		dlg.SetTitle(title.c_str());
		if(dlg.IsCanceled())
		{
			has_errors = true;
			break;
		}
		wxYieldIfNeeded();

		ERR_clear_error();
		if(!profile->IsImported())
		{
			if(!m_ClientPki->ImportProfile(profile->get_Profile(), profile_id))
			{
				profile->set_Error(m_ClientPki->GetError());
				has_errors = true;
				m_listCtrl->SetItemTextColour(profile->get_ItemId(), *wxRED);
				continue;
			}
			else
			{
				profile->set_Imported(true);
				profile->set_Id(profile_id);
				id.sprintf("%ld", profile_id);
				m_listCtrl->SetItem(profile->get_ItemId(), 1, id.c_str());
			}
		}

		DrawOpenProfile(profile->get_ItemId(), profile);
		// Import cert requests
		for(j=0; j<profile->get_Requests().size(); j++)
		{
			request = profile->get_Requests()[j];
			if(!request || request->IsImported())
				continue;

			m_listCtrl->SetItemState(profile->get_ItemId()+j+1, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
			m_listCtrl->EnsureVisible(profile->get_ItemId()+j+1);

			request->get_Request().set_profileId(profile->get_Id());

			ERR_clear_error();
			if(!m_ClientPki->RequestCertificate(request->get_Request(), req_id))
			{
				request->set_Error(m_ClientPki->GetError());
				m_listCtrl->SetItemTextColour(profile->get_ItemId(), *wxRED);
				m_listCtrl->SetItemTextColour(profile->get_ItemId()+j+1, *wxRED);
				has_errors = true;
			}
			else
			{
				request->set_Imported(true);
			}
			wxYieldIfNeeded();
			if(dlg.IsCanceled())
			{
				has_errors = true;
				break;
			}
		}
		DrawCloseProfile(profile->get_ItemId(), profile);
		wxYieldIfNeeded();
		if(dlg.IsCanceled())
		{
			has_errors = true;
			break;
		}
	}
	dlg.wHide();

	if(has_errors)
	{
		for(i=0; i<m_profiles.size(); i++)
		{
			profile = m_profiles[i];
			if(!profile)
				continue;

			if(profile->IsImported())
			{
				for(j=0; j<profile->get_Requests().size(); j++)
				{
					request = profile->get_Requests()[j];
					if(!request)
						continue;
					if(request->IsImported())
					{
						profile->get_Requests().erase(profile->get_Requests().begin() + j);
						j--;
						delete request;
					}
				}
				if(profile->get_Requests().size())
				{
					lprofiles.push_back(profile);
				}
				else
				{
					delete profile;
				}
			}
			else
			{
				lprofiles.push_back(profile);
			}
		}
		m_profiles = lprofiles;

		DisplayMessage(this, _("There were error(s) importing data, the XML state will be saved so that you can modify it."), wxOK | wxICON_ERROR);

		ReloadList();
		((wxButton *)FindWindow(IDC_IMPORT))->Enable(FALSE);
		OnSaveState(event);
	}
	else
	{
		DisplayMessage(this, _("Data imported successfuly"), wxOK);
		Close(TRUE);
	}
}

void DlgBashImport::xmlErrorCB(void *ctx, const char *msg, ...)
{
	DlgBashImport * me_this = (DlgBashImport*)ctx;
	mString strTemp;
	va_list args;
	va_start(args, msg);
	strTemp.vsprintf(msg, args);
	va_end(args);
	me_this->m_Errors += strTemp;
}

void DlgBashImport::xmlWarningCB(void *ctx, const char *msg, ...)
{
	DlgBashImport * me_this = (DlgBashImport*)ctx;
	mString strTemp;
	va_list args;
	va_start(args, msg);
	strTemp.vsprintf(msg, args);
	va_end(args);
	me_this->m_Warnings += strTemp;
}


void DlgBashImport::Clear()
{
	size_t i;
	for(i=0; i<m_profiles.size(); i++)
	{
		if(m_profiles[i])
			delete m_profiles[i];
	}
	m_profiles.clear();
	m_listCtrl->DeleteAllItems();
}


void DlgBashImport::OnMouseEvent(wxMouseEvent &event)
{
	int flags;
	long item = m_listCtrl->HitTest(event.GetPosition(), flags);
	if (item > -1)
	{
		if(event.LeftDown())
		{
			MouseDown(item);
		}
		else if(event.Moving())
		{
			MouseMove(item);
		}
	}
	event.Skip();
}

void DlgBashImport::MouseDown(long item)
{
	wxListItem info;
	XmlProfile * Profile;
	XmlRequest * request;
	
	info.m_mask = wxLIST_MASK_IMAGE;
	info.m_itemId = item;
	if (!m_listCtrl->GetItem(info))
	{
		return;
	}
	
	if( (Profile = GetSelectedProfile(item)) )
	{
		if(Profile->get_Requests().size())
		{
			if(info.m_image == PlusProfileIconId)
			{
				DrawOpenProfile(item, Profile);
			}
			else if(info.m_image == MinusProfileIconId)
			{
				DrawCloseProfile(item, Profile);
			}
		}
	}
	else if ( (request = GetSelectedCert()) )
	{
		DlgRequestCert Dlg(this, NULL, m_EntityConf, request);
	}
}

void DlgBashImport::MouseMove(long item)
{
	wxListItem info;
	XmlProfile * Profile;
	XmlRequest * Request;
	size_t i;
	mString strInfo;
	
	if(lastItem == item)
		return;

	lastItem = item;

	if( (Profile = GetSelectedProfile(item)) )
	{
		if(Profile->get_Profile())
		{
			if(Profile->get_Profile().get_eeValidation())
			{
				strInfo += "Should be validated by End User thru the EE\n";
			}
			if(Profile->get_Profile().get_ldapUid().size())
			{
				strInfo += _("LDAP UID: ");
				strInfo += Profile->get_Profile().get_ldapUid();
				strInfo += "\n";
			}

			if(Profile->get_Profile().get_ownerGroupSerial())
			{
				strInfo += _("Owner group: ");
				// Search the owner group
				for(i=0; i<m_EntityConf.get_groups().size(); i++)
				{
					if(m_EntityConf.get_groups()[i].get_serial() == Profile->get_Profile().get_ownerGroupSerial())
					{
						strInfo += m_EntityConf.get_groups()[i].get_name();
						break;
					}
				}
				if(i == m_EntityConf.get_groups().size())
				{
					strInfo += _("Unknown");
				}
				strInfo += "\n";
			}
		}

		if(Profile->IsError())
		{
			strInfo += _("\nError with profile:\n");
			strInfo += Profile->get_Error();
		}
		strInfo += "\n";
	}
	else if( (Request = GetSelectedCert(item)) )
	{
		if(Request->get_Request())
		{
			strInfo += _("Type: ");
			switch(Request->get_Request().get_type())
			{
				case NEWPKI_PROFILE_CERT_TYPE_PKCS10:
					strInfo += _("Simple Certificate");
					break;

				case NEWPKI_PROFILE_CERT_TYPE_PKCS12:
					strInfo += _("PKCS#12");
					break;

				case NEWPKI_PROFILE_CERT_TYPE_SC:
					strInfo += _("Smart Card");
					break;
			}
			
			strInfo += _("\nCA: ");
			strInfo += Request->get_Request().get_caName();
			strInfo += "\n";
		}

		if(Request->IsError())
		{
			strInfo += _("\nError with request:\n");
			strInfo += Request->get_Error();
		}
		strInfo += "\n";
	}
	else
	{
		strInfo = "";
	}
	
	((wxTextCtrl *)FindWindow(IDC_INFO))->SetValue(strInfo.c_str());
}


void DlgBashImport::DrawOpenProfile(long item, XmlProfile * Profile)
{
	size_t i, ctr;
	int IconId;
	mString Id;


	m_listCtrl->SetItemImage(item, MinusProfileIconId, MinusProfileIconId);

	for(i=0, ctr=0; i<Profile->get_Requests().size() ; i++)
	{
		if(!Profile->get_Requests()[i]->IsError())
			continue;
		
		ctr++;
		item++;

		if(ctr == Profile->get_Requests().size())
			IconId = ActEndConnectorIconId;
		else
			IconId = ActMidConnectorIconId;


		m_listCtrl->InsertItem(item, Profile->get_Requests()[i]->get_CaName().c_str(), IconId);
		m_listCtrl->SetItemData(item, (long)Profile->get_Requests()[i]);
		m_listCtrl->SetItemTextColour(item, *wxRED);
	}

	for(i=0; i<Profile->get_Requests().size() ; i++)
	{
		if(Profile->get_Requests()[i]->IsError())
			continue;

		ctr++;
		item++;
		
		if(ctr == Profile->get_Requests().size())
			IconId = ActEndConnectorIconId;
		else
			IconId = ActMidConnectorIconId;


		m_listCtrl->InsertItem(item, Profile->get_Requests()[i]->get_CaName().c_str(), IconId);
		m_listCtrl->SetItemData(item, (long)Profile->get_Requests()[i]);
		Id = ctr;
		m_listCtrl->SetItem(item, 1, Id.c_str());
	}
}

void DlgBashImport::DrawCloseProfile(long item, XmlProfile *Profile)
{
	int i;
	i = Profile->get_Requests().size();
	while(i)
	{
		m_listCtrl->DeleteItem(item + 1);
		i--;
	}
	m_listCtrl->SetItemImage(item, PlusProfileIconId, PlusProfileIconId);
}


XmlProfile * DlgBashImport::GetSelectedProfile(long Item)
{
	wxListItem info;

	if(Item == -1)
	{
		Item = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
		if(Item == -1)
			return NULL;
	}
	
	info.m_mask = wxLIST_MASK_IMAGE ;
	info.m_itemId = Item;
	if(!m_listCtrl->GetItem(info))
		return NULL;

	if(!IS_BASH_PROFILE_ICON(info.m_image))
		return NULL;
	
	return (XmlProfile *)m_listCtrl->GetItemData(Item);
}

XmlRequest * DlgBashImport::GetSelectedCert(long Item)
{
	wxListItem info;

	if(Item == -1)
	{
		Item = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
		if(Item == -1)
			return NULL;
	}
	
	info.m_mask = wxLIST_MASK_IMAGE;
	info.m_itemId = Item;
	if(!m_listCtrl->GetItem(info))
		return NULL;

	if(!IS_BASH_CERT_ICON(info.m_image))
		return NULL;
	
	return (XmlRequest *)m_listCtrl->GetItemData(Item);
}

