#ifndef FILE_MESHACCESS
#define FILE_MESHACCESS

/*********************************************************************/
/* File:   meshaccess.hpp                                            */
/* Author: Joachim Schoeberl                                         */
/* Date:   25. Mar. 2000                                             */
/*********************************************************************/

/** 
   Access to mesh geometry.
   This class provides topology, as element to vertex,
   element to face etc. Internally, NGSolve calls the Netgen
   mesh handler.
*/

class MeshAccess
{
  /// @name Finite Elements for mesh transformation
//@{
  ///
  FE_Segm1 segm1;
  ///
  FE_Segm2 segm2;
  ///
  FE_Trig1 trig1;
  ///
  FE_Trig2 trig2;
  ///
  FE_Quad1 quad1;
  ///
  FE_Quad2aniso quad2;
  ///
  FE_Tet1 tet1;
  ///
  FE_Tet2 tet2;
  ///
  FE_Prism1 prism1;
  //
  FE_Prism2aniso prism2;
  ///
  FE_Pyramid1 pyramid1;
  ///
  FE_Hex1 hex1;
  
  FE_Tet0 tet0;
  FE_Prism0 prism0;
  FE_Pyramid0 pyramid0;
  FE_Hex0 hex0;
  FE_Trig0 trig0;
  FE_Quad0 quad0;

//@}
public:
  ///
  MeshAccess ();
  ///
  virtual ~MeshAccess ();


  ///
  int GetDimension() const
  { return Ng_GetDimension(); }
  ///
  int GetNP() const
  { return Ng_GetNP(); }
  ///
  int GetNV() const
  { return Ng_GetNV(); }
  ///
  int GetNE() const
  { return Ng_GetNE(); }
  ///
  int GetNSE() const
  { return Ng_GetNSE(); }

  ///
  int GetNEdges() const
  { return Ng_GetNEdges(); }

  ///
  int GetNFaces() const
  { return Ng_GetNFaces(); }


  int GetNDomains () const;
  int GetNBoundaries () const;


  ///
  void GetPoint (int pi, double * p) const
  { Ng_GetPoint (pi+1, p); }

  template <int D>
  void GetPoint (int pi, Vec<D> & p) const
  { Ng_GetPoint (pi+1, &p(0)); }

  ///
  ELEMENT_TYPE GetElType (int elnr) const;
  ///
  int GetElIndex (int elnr) const
  { return Ng_GetElementIndex (elnr+1)-1; }

  string GetElMaterial (int elnr) const
  { return Ng_GetElementMaterial (elnr+1); }

  ///
  ELEMENT_TYPE GetSElType (int elnr) const;
  ///
  int GetSElIndex (const int elnr) const
  { return Ng_GetSurfaceElementIndex (elnr+1)-1; }

  void GetSElNeighbouringDomains(const int elnr, int & in, int & out) const
  { 
    Ng_GetSurfaceElementNeighbouringDomains(elnr+1,in,out);
    //in--; out--;
  }

  ///
  void GetElPNums (int elnr, ARRAY<int> & pnums) const;
  ///
  void GetSElPNums (int selnr, ARRAY<int> & pnums) const;

  void GetElVertices (int elnr, ARRAY<int> & vnums) const;
  ///
  void GetSElVertices (int selnr, ARRAY<int> & vnums) const;
  ///
  void GetElEdges (int elnr, ARRAY<int> & ednums) const;
  ///
  void GetElEdges (int elnr, ARRAY<int> & ednums, ARRAY<int> & orient) const;
  ///
  void GetSElEdges (int selnr, ARRAY<int> & ednums) const;
  ///
  void GetSElEdges (int selnr, ARRAY<int> & ednums, ARRAY<int> & orient) const;
  ///
  void GetElFaces (int elnr, ARRAY<int> & fnums) const;
  ///
  void GetElFaces (int elnr, ARRAY<int> & fnums, ARRAY<int> & orient) const;
  ///
  int GetSElFace (int selnr) const;
  ///
  void GetSElFace (int selnr, int & fnum, int & orient) const;
  ///
  void GetFacePNums (int fnr, ARRAY<int> & pnums) const;
  ///
  void GetEdgePNums (int enr, int & pn1, int & pn2) const;
  ///
  void GetFaceEdges (int fnr, ARRAY<int> & edges) const;
  ///
  // void GetVertexElements (int vnr, ARRAY<int> & elnrs) const;
  ///
  int GetElOrder (int enr) const
  { return Ng_GetElementOrder (enr+1); }

  ///
  double ElementVolume (int elnr) const;
  ///
  double SurfaceElementVolume (int selnr) const;





  /// multigrid:
  int GetNLevels() const;
  ///
  void GetParentNodes (int pi, int * parents) const
  { 
    Ng_GetParentNodes (pi+1, parents);
    parents[0]--; parents[1]--; 
  }
  ///
  int GetParentElement (int ei) const
  { return Ng_GetParentElement (ei+1)-1; }
  ///
  int GetParentSElement (int ei) const
  { return Ng_GetParentSElement (ei+1)-1; }
  
  /// anisotropic clusters:
  int GetClusterRepVertex (int pi) const
    { return Ng_GetClusterRepVertex (pi+1)-1; }
  ///
  int GetClusterRepEdge (int pi) const
    { return Ng_GetClusterRepEdge (pi+1)-1; }
  ///
  int GetClusterRepFace (int pi) const
    { return Ng_GetClusterRepFace (pi+1)-1; }
  ///
  int GetClusterRepElement (int pi) const
    { return Ng_GetClusterRepElement (pi+1)-1; }


  template <int A, int B, int C, int D>
  void GetSurfaceElementTransformation (int sei, const Vec<A> & xi,
                                        Vec<B> & x, Mat<C,D> & dxdxi)
  {
    double xl[2];
    double xg[3];
    double dxgdxl[6];
    int i, j;
    for (i=0; i<A; i++) xl[i] = xi(i);
    Ng_GetSurfaceElementTransformation (sei, &xl[0], &xg[0], &dxgdxl[0]);
    for (i=0; i<B; i++) x(i) = xg[i];
    for (i=0; i<D; i++)
      for (j=0; j<C; j++) dxdxi(i,j) = dxgdxl[i*C+j];
  }


  ///
  void GetElementTransformation (int elnr, ElementTransformation & eltrans, 
				 LocalHeap & lh) const; 
  ///
  void GetSurfaceElementTransformation (int elnr, ElementTransformation & eltrans,
					LocalHeap & lh) const;

  
  int FindElementOfPoint (FlatVector<double> point,
			  IntegrationPoint & ip, 
			  bool build_searchtree,
			  int index = -1) const;

  
  void GetPeriodicVertices (ARRAY<ngstd::INT<2> > & pairs) const;
  int GetNPairsPeriodicVertices () const;  

  void GetPeriodicEdges (ARRAY<ngstd::INT<2> > & pairs) const;
  int GetNPairsPeriodicEdges () const;  


  void PushStatus (const char * str) const
  { Ng_PushStatus (str); }

  void PopStatus () const
  { Ng_PopStatus (); }

  void SetThreadPercentage (double percent) const
  { Ng_SetThreadPercentage (percent); }
  
  ///// Added by Roman Stainko ....
  void GetVertexElements( int vnr, ARRAY<int>& elems) const;

  void GetVertexSurfaceElements( int vnr, ARRAY<int>& elems) const;

};



#endif
