/* nexp_log.c -- logging routines and other things common to both Network
		 Expect program and library.

   Copyright (C) 2007 Eloy Paris
   Copyright (C) 1987-2002 Don Libes <libes@cme.nist.gov>,
	National Institute of Standards and Technology <http://www.nist.gov/>

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

struct nexp_log {
    Tcl_Channel diagChannel;
    Tcl_DString diagFilename;
    int diagToStderr;

    Tcl_Channel logChannel;
    Tcl_DString logFilename;	/* if no name, then it came from -open or
				   -leaveopen */
    int logAppend;
    int logLeaveOpen;
    int logAll;	    /* if TRUE, write log of all interactions
		     * despite value of logUser - i.e., even if
		     * user is not seeing it (via stdout)
		     */
    int logUser;    /* TRUE if user sees interactions on stdout */
} nexplog;

/*
 * create a reasonably large buffer for the bulk of the output routines
 */
static char bigbuf[2000];

/* write 8-bit bytes */
static void
nexpDiagWriteBytes(char *str, int len)
{
    struct nexp_log *logptr = &nexplog;

    if (!logptr->diagChannel)
	return;

    Tcl_Write(logptr->diagChannel, str, len);
}

/* write UTF chars */
static void
nexpDiagWriteChars(char *str, int len)
{
    struct nexp_log *logptr = &nexplog;

    if (!logptr->diagChannel)
	return;

    Tcl_WriteChars(logptr->diagChannel, str, len);
}

/* send to log if open */
/* send to stderr */
/* use this function for error conditions */
/*VARARGS*/
void
nexpErrorLog TCL_VARARGS_DEF(char *, arg1)
{
    char *fmt;
    va_list args;
    struct nexp_log *logptr = &nexplog;

    fmt = TCL_VARARGS_START(char *, arg1, args);
    (void) vsnprintf(bigbuf, sizeof(bigbuf), fmt, args);

    nexpDiagWriteChars(bigbuf, -1);
    fprintf(stderr, "%s", bigbuf);
    if (logptr->logChannel)
	Tcl_WriteChars(logptr->logChannel, bigbuf, -1);

    va_end(args);
}

/* send diagnostics to Diag, Log, and stderr */
/* use this function for recording unusual things in the log */
/*VARARGS*/
void
nexpDiagLog TCL_VARARGS_DEF(char *, arg1)
{
    char *fmt;
    va_list args;
    struct nexp_log *logptr = &nexplog;

    if (!logptr->diagToStderr && logptr->diagChannel == NULL)
	return;

    fmt = TCL_VARARGS_START(char *, arg1, args);

    (void) vsnprintf(bigbuf, sizeof(bigbuf), fmt, args);

    nexpDiagWriteBytes(bigbuf, -1);
    if (logptr->diagToStderr) {
	fprintf(stderr, "%s", bigbuf);
	if (logptr->logChannel)
	    Tcl_WriteChars(logptr->logChannel, bigbuf, -1);
    }

    va_end(args);
}

void
nexpDiagInit()
{
    struct nexp_log *logptr = &nexplog;

    Tcl_DStringInit(&logptr->diagFilename);
    logptr->diagChannel = NULL;
    logptr->diagToStderr = 0;
}

void
nexpLogInit()
{
    struct nexp_log *logptr = &nexplog;

    Tcl_DStringInit(&logptr->logFilename);
    logptr->logChannel = NULL;
    logptr->logAll = 0;
    logptr->logUser = 1;
}
