/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <regex.h>

#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_raw_protocol.h>
#include <netdude/nd.h>

#include "nd_tcp.h"
#include "nd_tcb.h"
#include "nd_tcp_prefs.h"
#include "nd_tcp_callbacks.h"

#define ND_TCP_TCB_KEY "tcp_tcb_key"


static ND_ProtoField tcp_fields[] = {
  { ND_VAL_FIELD, N_("Src. port (%u)"),    N_("Source port number"),          16, nd_tcp_sport_cb   },
  { ND_VAL_FIELD, N_("Dst. port (%u)"),    N_("Destination port number"),     16, nd_tcp_dport_cb   },
  { ND_VAL_FIELD, N_("Seq. number (%lu)"), N_("Sequence number"),             32, nd_tcp_seq_cb     },
  { ND_VAL_FIELD, N_("Ack number (%lu)"),  N_("Acknowledgement number"),      32, nd_tcp_ack_cb     },
  { ND_VAL_FIELD, N_("Data offset (%u)"),  N_("Data offset (header length)"), 4,  nd_tcp_off_cb     },
  { ND_VAL_FIELD, N_("Unused (%u)"),       N_("Unused bits"),                 6,  nd_tcp_unused_cb  },
  { ND_FLG_FIELD, N_("U"),                 N_("Urgent flag"),                 1,  nd_tcp_u_cb       },
  { ND_FLG_FIELD, N_("A"),                 N_("ACK flag"),                    1,  nd_tcp_a_cb       },
  { ND_FLG_FIELD, N_("P"),                 N_("Push flag"),                   1,  nd_tcp_p_cb       },
  { ND_FLG_FIELD, N_("R"),                 N_("Reset flag"),                  1,  nd_tcp_r_cb       },
  { ND_FLG_FIELD, N_("S"),                 N_("SYN flag"),                    1,  nd_tcp_s_cb       },
  { ND_FLG_FIELD, N_("F"),                 N_("FIN flag"),                    1,  nd_tcp_f_cb       },
  { ND_VAL_FIELD, N_("Win (%u)"),          N_("TCP window size"),             16, nd_tcp_win_cb     },
  { ND_VAL_FIELD, N_("Checksum (0x%.4x)"), N_("Checksum"),                    16, nd_tcp_cksum_cb   },
  { ND_VAL_FIELD, N_("Urgent (%u)"),       N_("Urgent pointer"),              16, nd_tcp_urp_cb     },
  { 0, NULL, NULL, 0, NULL }
};


/* Field definitions for TCP options: */

static ND_ProtoField tcp_opt_fields[] = {
  { ND_VAL_FIELD, N_("Opt. (%s)"),        N_("IP option type"),             8,  nd_proto_8bit_cb   },
  { ND_VAL_FIELD, N_("Len. (%u)"),        N_("IP option length"),           8,  nd_proto_8bit_cb   },
  { ND_VAL_FIELD, "%u",                   N_("Maximum segment size"),       16, nd_proto_16bit_cb  },
  { ND_VAL_FIELD, "%u",                   N_("Window scaling shift count"), 8,  nd_proto_8bit_cb   },
  { ND_VAL_FIELD, N_("(%i bytes)"),       N_("Unknown option data"),        8,  NULL                     },
  { ND_VAL_FIELD, N_("%lu"),              N_("Timestamp value"),            32, nd_proto_32bit_cb  },
  { ND_VAL_FIELD, "%lu",                  N_("SAck left edge"),             32, nd_proto_32bit_cb  },
  { ND_VAL_FIELD, "%lu",                  N_("SAck right edge"),            32, nd_proto_32bit_cb  },
  { 0, NULL, NULL, 0, NULL }    
};


ND_MenuData tcp_menu_data[] = {
  { N_("Fix Checksums"), N_("Fixes the TCP checksums of the current selection"), 0, nd_tcp_cksum_fix_cb },
  { NULL, NULL, 0, NULL}
};


static ND_Protocol *tcp;

static regex_t      regex_seq;
static regex_t      regex_ack;

static ND_TCPStateMode tcp_state_mode;

static gboolean
tcp_header_complete(const ND_Packet *packet, guchar *data)
{
  return (data + 20 <= nd_packet_get_end(packet));
}



/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("TCP Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for the TCP protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return ("0.1.0");
}


ND_Protocol *
init(void)
{
  tcp = nd_proto_new("TCP", ND_PROTO_LAYER_TRANS, IPPROTO_TCP);

  tcp->create_gui          = nd_tcp_create_gui;
  tcp->set_gui             = nd_tcp_set_gui;
  tcp->init_packet         = nd_tcp_init_packet;
  tcp->header_complete     = nd_tcp_header_complete;
  tcp->fix_packet          = nd_tcp_fix_packet;
  tcp->init_state          = nd_tcp_init_state;
  tcp->update_state        = nd_tcp_update_state;
  tcp->update_tcpdump_line = nd_tcp_update_tcpdump_line;
  tcp->free_state          = nd_tcp_free_state;

  /* TCP is stateful, so we need some of our functions to be called
     at the right moments: */
  tcp->is_stateful = TRUE;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  tcp->fields = tcp_fields;

  tcp->header_width = 32;

  /* We provide a little menu that allows us to fix checksums. */
  tcp->menu = nd_gui_create_menu(tcp_menu_data);
	     
  /* Hook TCP options into preferences dialog! */
  nd_tcp_init_prefs();

  /* We need regexes to update seq/ack numbers dynamically when
   * the user so desires. Compile them now.
   */
  if (regcomp(&regex_seq, ".* ([0-9]+):([0-9]+).*", REG_EXTENDED) < 0)
    D(("seq regex error\n"));
  
  if (regcomp(&regex_ack, ".* ack ([0-9]+) .*", REG_EXTENDED) < 0)
    D(("ack regex error\n"));
  

  return tcp;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_tcp_create_gui(ND_Trace *trace, ND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_tcp_set_gui(const ND_Packet *packet, ND_ProtoInfo *pinf)
{
  struct ip *iphdr;
  struct tcphdr *tcphdr;
  
  if (pinf->inst.nesting != 0)
    D(("WARNING -- TCP plugin doesn't support nested TCP\n"));

  if (!nd_tcp_get_first(packet, &iphdr, &tcphdr))
    return;
  
  nd_tcp_set_gui_sport(pinf, tcphdr);
  nd_tcp_set_gui_dport(pinf, tcphdr);
  nd_tcp_set_gui_seq(pinf, tcphdr);
  nd_tcp_set_gui_ack(pinf, tcphdr);
  nd_tcp_set_gui_off(pinf, tcphdr);
  nd_tcp_set_gui_unused(pinf, tcphdr);
  nd_tcp_set_gui_flags(pinf, tcphdr);
  nd_tcp_set_gui_win(pinf, tcphdr);
  nd_tcp_set_gui_cksum(pinf, iphdr, tcphdr, packet);
  nd_tcp_set_gui_urg(pinf, tcphdr);
  nd_tcp_set_gui_options(pinf, tcphdr, packet);
}


void       
nd_tcp_init_packet(ND_Packet *packet, guchar *data, guchar *data_end)
{
  struct tcphdr *tcphdr;
  ND_Protocol *payload_proto;

  /* Cast the data pointer into your protocol's header */
  tcphdr = (struct tcphdr *) data;

  if (!tcp_header_complete(packet, data))
    {
      nd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }

  /* Adjust the data_end pointer here, if your protocol has a trailer */

  nd_packet_add_proto_data(packet, tcp, data, data_end);  
  
  /* Check the appriopriate header field value to demultiplex
     packet initialization up to the next correct protocol: */

  payload_proto = nd_proto_registry_find(ND_PROTO_LAYER_APP, ntohs(tcphdr->th_dport));
  payload_proto->init_packet(packet, data + (4 * tcphdr->th_off), data_end);
}


gboolean
nd_tcp_header_complete(const ND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = nd_packet_get_data(packet, tcp, nesting);

  return tcp_header_complete(packet, data);
}


gboolean
nd_tcp_fix_packet(ND_Packet *packet, int index)
{
  struct tcphdr *tcphdr;
  guint16        correct_sum;
  
  if (!packet)
    return FALSE;
  
  if (!nd_tcp_csum_correct(packet, &correct_sum))
    {
      tcphdr = (struct tcphdr*) nd_packet_get_data(packet, tcp, 0);
      tcphdr->th_sum = correct_sum;
      nd_packet_modified_at_index(packet, index);
      return TRUE;
    }

  return FALSE;
}


void       
nd_tcp_init_state(ND_Trace *trace)
{
  ND_TCB *tcb;

  if (!trace)
    return;

  if (tcp_state_mode == ND_TCP_STATE_NONE ||
      tcp_state_mode == ND_TCP_STATE_ONCE)
    return;

  D(("Initializing TCP state management for %s\n", trace->filename));

  tcb = nd_tcb_new();
  nd_trace_set_data(trace, ND_TCP_TCB_KEY, tcb);
}


void       
nd_tcp_update_state(const ND_Packet *packet, int index)
{
  ND_TCB *tcb;

  if (tcp_state_mode == ND_TCP_STATE_NONE ||
      tcp_state_mode == ND_TCP_STATE_ONCE)
    return;

  tcb = nd_trace_get_data(packet->trace, ND_TCP_TCB_KEY);
  D_ASSERT_PTR(tcb);
  nd_tcb_update(tcb, packet, index);
}


void       
nd_tcp_update_tcpdump_line(const ND_Packet *packet, char *line)
{
  ND_Protocol *ip;
  regmatch_t   regex_match[3];
  ND_TCB      *tcb;
  ND_TCBConn  *tcbc;
  gboolean     rel = FALSE, is_reverse;
  guint32      ack_no, seq_start, seq_end;
  char         line2[MAXPATHLEN];
  struct       ip *iphdr;
  struct       tcphdr *tcphdr;

  if (! (ip = nd_tcp_get_ip()))
    return;

  if (tcp_state_mode == ND_TCP_STATE_NONE ||
      tcp_state_mode == ND_TCP_STATE_ONCE)
    return;

  tcb = nd_trace_get_data(packet->trace, ND_TCP_TCB_KEY);
  D_ASSERT_PTR(tcb);
  tcbc = nd_tcb_lookup(tcb, packet, &is_reverse);

  if (!tcbc)
    return;

  if (!nd_tcp_get_first(packet, &iphdr, &tcphdr))
    return;

  if (regexec(&regex_seq, line, 3, regex_match, 0) == 0)
    {
      line[regex_match[1].rm_so] = '\0';
      rel = nd_tcb_conn_get_rel_seq(tcbc, iphdr, tcphdr, &seq_start, &seq_end);
      g_snprintf(line2, MAXPATHLEN, "%s%u:%u%s", line, seq_start, seq_end,
		 line + regex_match[2].rm_eo);
            
      memcpy(line, line2, MAXPATHLEN);
    }	  
  
  /* It's possible that regex_seq does not match -- pure ACK: */

  if (regexec(&regex_ack, line, 2, regex_match, 0) == 0)
    {
      line[regex_match[1].rm_so] = '\0';
      nd_tcb_conn_get_rel_ack(tcbc, iphdr, tcphdr, rel, &ack_no);
      g_snprintf(line2, MAXPATHLEN, "%s%u%s", line, ack_no,
		 line + regex_match[1].rm_eo);

      memcpy(line, line2, MAXPATHLEN);
    }
}


void       
nd_tcp_free_state(ND_Trace *trace)
{
  ND_TCB *tcb;

  if (!trace)
    return;

  tcb = nd_trace_remove_data(trace, ND_TCP_TCB_KEY);

  if (tcb)
    nd_tcb_free(tcb);
}


ND_TCPStateMode
nd_tcp_set_state_mode(ND_TCPStateMode mode)
{
  ND_TCPStateMode old_mode;

  old_mode = tcp_state_mode;

  D(("New TCP state mode: %i\n", mode));
  tcp_state_mode = mode;

  return old_mode;
}


ND_TCPStateMode
nd_tcp_get_state_mode(void)
{
  return tcp_state_mode;
}


guint16
nd_tcp_checksum(const ND_Packet *packet)
{
  ND_Protocol    *ip;
  struct ip*      iphdr;
  struct tcphdr  *tcphdr;
  guint           tcp_len, len;
  guint16         old, result = 0;
  guint32         sum, addl_pseudo;

  if (!packet)
    return 0;

  if (! (ip = nd_tcp_get_ip()))
    return 0;

  if (!nd_tcp_get_first(packet, &iphdr, &tcphdr))
    return 0;

  old = tcphdr->th_sum;
  tcphdr->th_sum = 0;
  
  /* len is payload length, tcp_len is length of entire tcp packet: */
  tcp_len = (ntohs(iphdr->ip_len) - (iphdr->ip_hl << 2));
  len = tcp_len - tcphdr->th_off*4;
      
  if (len % 2 == 1) /* Add in pad byte. */
    sum = htons(((const u_char*) tcphdr)[tcp_len - 1] << 8);
  else
    sum = 0;
      
  sum = nd_misc_ones_complement_checksum((u_short*) &iphdr->ip_src.s_addr, 4, sum);
  sum = nd_misc_ones_complement_checksum((u_short*) &iphdr->ip_dst.s_addr, 4, sum);
      
  addl_pseudo = (htons(IPPROTO_TCP) << 16) | htons((u_short) tcp_len);
  
  sum = nd_misc_ones_complement_checksum((u_short*) &addl_pseudo, 4, sum);
  sum = nd_misc_ones_complement_checksum((u_short*) tcphdr, tcp_len, sum);
  
  tcphdr->th_sum = old;
  result = ~sum;

  return result;
}


gboolean   
nd_tcp_csum_correct(const ND_Packet *packet, guint16 *correct_sum)
{
  struct tcphdr *tcphdr;
  guint16 old_sum, correct_sum_tmp;

  if (!packet)
    return FALSE;

  tcphdr = (struct tcphdr *) nd_packet_get_data(packet, nd_tcp_get(), 0);

  old_sum = tcphdr->th_sum;
  correct_sum_tmp = nd_tcp_checksum(packet);

  if (correct_sum)
    *correct_sum = correct_sum_tmp;

  return (old_sum == correct_sum_tmp);
}


/* Misc helper stuff below --------------------------------------------- */

void  
nd_tcp_set_gui_sport(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[0], DATA_TO_PTR(ntohs(tcphdr->th_sport)));
}

void  
nd_tcp_set_gui_dport(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[1], DATA_TO_PTR(ntohs(tcphdr->th_dport)));
}

void  
nd_tcp_set_gui_seq(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[2], DATA_TO_PTR(ntohl(tcphdr->th_seq)));
}

void  
nd_tcp_set_gui_ack(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[3], DATA_TO_PTR(ntohl(tcphdr->th_ack)));
}

void  
nd_tcp_set_gui_off(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[4], DATA_TO_PTR(tcphdr->th_off));
}

void  
nd_tcp_set_gui_unused(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[5], DATA_TO_PTR(((tcphdr->th_x2 << 2) | ((tcphdr->th_flags & 0xc0) >> 6))));
}

void  
nd_tcp_set_gui_flags(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[6],  DATA_TO_PTR(tcphdr->th_flags & TH_URG));
  nd_proto_field_set(pinf, &tcp_fields[7],  DATA_TO_PTR(tcphdr->th_flags & TH_ACK));
  nd_proto_field_set(pinf, &tcp_fields[8],  DATA_TO_PTR(tcphdr->th_flags & TH_PUSH));
  nd_proto_field_set(pinf, &tcp_fields[9],  DATA_TO_PTR(tcphdr->th_flags & TH_RST));
  nd_proto_field_set(pinf, &tcp_fields[10], DATA_TO_PTR(tcphdr->th_flags & TH_SYN));
  nd_proto_field_set(pinf, &tcp_fields[11], DATA_TO_PTR(tcphdr->th_flags & TH_FIN));
}

void  
nd_tcp_set_gui_win(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[12], DATA_TO_PTR(ntohs(tcphdr->th_win)));
}

void  
nd_tcp_set_gui_cksum(ND_ProtoInfo *pinf, struct ip *iphdr,
		     struct tcphdr *tcphdr, const ND_Packet *packet)
{
  nd_proto_field_set(pinf, &tcp_fields[13], DATA_TO_PTR(ntohs(tcphdr->th_sum)));

  if ((guchar *)iphdr + ntohs(iphdr->ip_len) > nd_packet_get_end(packet))
    {
      nd_proto_info_field_set_state(pinf,
				    &tcp_fields[13],
				    ND_FIELD_STATE_UNKN);
      return;
    }

  if (!nd_tcp_csum_correct(packet, NULL))
    nd_proto_info_field_set_state(pinf,
				  &tcp_fields[13],
				  ND_FIELD_STATE_ERROR);
  else
    nd_proto_info_field_set_state(pinf,
				  &tcp_fields[13],
				  ND_FIELD_STATE_NORMAL);
}

void  
nd_tcp_set_gui_urg(ND_ProtoInfo *pinf, struct tcphdr *tcphdr)
{
  nd_proto_field_set(pinf, &tcp_fields[14], DATA_TO_PTR(ntohs(tcphdr->th_urp)));
}


void
nd_tcp_set_gui_options(ND_ProtoInfo *pinf, struct tcphdr *tcphdr, const ND_Packet *packet)
{
  char         label[MAXPATHLEN];
  int          opts_len, o_len, opts_done, opts_done_old, i;
  guchar      *opts_p, *o_next;
  gboolean     is_error;

  nd_gui_proto_table_clear(packet->trace, pinf);

  opts_len  = (tcphdr->th_off*4) - 20;
  opts_done = 0;
  opts_done_old = -1;

  if (opts_len == 0)
    return;

  while (opts_done < opts_len && opts_done != opts_done_old)
    {
      opts_done_old = opts_done;
      is_error = FALSE;
      opts_p = ((guchar *) tcphdr) + 20 + opts_done;
  
      switch(*opts_p)
	{
	case TCPOPT_EOL:
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("EOL"), FALSE);
	  opts_p++;
	  opts_done = opts_len;
	  break;

	case TCPOPT_NOP:
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("NOP"), FALSE);
	  opts_p++;
	  opts_done++;
	  break;

	case TCPOPT_MAXSEG:
	  opts_p++;
	  o_len = *opts_p;
	  opts_done += o_len;

	  if (opts_done > opts_len)
	    break;
	  
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("MAXSEG"), is_error);
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[1], DATA_TO_PTR(o_len), is_error);
	  opts_p++;

	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[2], DATA_TO_PTR(htons(*(guint16*)opts_p)), is_error);
	  opts_p += 2;
	  break;

	case TCPOPT_WINDOW:
	  /*WSCALE is only allowed on SYN segments*/
	  opts_p++;
	  o_len = *opts_p;
	  opts_done += o_len;

	  if (opts_done > opts_len)
	    break;
	  
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("WINDOW"), is_error);
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[1], DATA_TO_PTR(o_len), is_error);
	  opts_p++;

	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[3], DATA_TO_PTR(*(guchar *) opts_p), is_error);
	  opts_p += 2;
	  break;

	case TCPOPT_TIMESTAMP:
	  opts_p++;
	  o_len = *opts_p;
	  opts_done += o_len;

	  if (opts_done > opts_len)
	    break;

	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("TS"), is_error);
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[1], DATA_TO_PTR(o_len), is_error);
	  opts_p++;

	  for (i = 0; i < 2; i++)
	    {
	      nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[5], DATA_TO_PTR(ntohl(* (guint32*) opts_p)), is_error);
	      opts_p += 4;
	    }
	  break;	  

	case TCPOPT_SACK_PERMITTED:
	  opts_p++;
	  o_len = *opts_p;
	  opts_done += o_len;

	  if (opts_done > opts_len)
	    break;

	  /* Sack-permitted is only allowed on SYNs,
	     so declare as erroneous when this is not a SYN: */
	  if ((tcphdr->th_flags & TH_SYN) == 0)
	    is_error = TRUE;
	  
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("SAck Perm"), is_error);
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[1], DATA_TO_PTR(o_len), is_error);
	  opts_p++;

	  if (o_len > 2)
	    {
	      tcp_opt_fields[4].bits = (o_len * 8) - 16;
	      g_snprintf(label, MAXPATHLEN, tcp_opt_fields[4].label, o_len - 2);
	      /* This data isn't supposed to be here, so show as error: */
	      nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[4], DATA_TO_PTR(o_len - 2), TRUE);
	    }
	  break;

	case TCPOPT_SACK:
	  opts_p++;
	  o_len = *opts_p;
	  o_next = opts_p + o_len - 1;
	  opts_done += o_len;

	  if (opts_done > opts_len)
	    break;
	  
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("SAck"), is_error);
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[1], DATA_TO_PTR(o_len), is_error);
	  opts_p++;

	  while (opts_p < o_next)
	    {
	      nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[6], DATA_TO_PTR(ntohl(*(guint32*) opts_p)), is_error);
	      opts_p += 4;
	      nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[7], DATA_TO_PTR(ntohl(*(guint32*) opts_p)), is_error);
	      opts_p += 4;
	    }
	  break;

	case 19: /* MD5 signature of BGP data, RFC 2385 */
	  opts_p++;
	  o_len = *opts_p;
	  opts_done += o_len;

	  if (opts_done > opts_len)
	    break;
	  
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], _("MD5"), is_error);
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[1], DATA_TO_PTR(o_len), is_error);
	  opts_p++;

	  tcp_opt_fields[4].bits = (o_len * 8) - 16;
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[4], DATA_TO_PTR(o_len - 2), is_error);
	  break;
	  
	default:
	  opts_p++;
	  o_len = *opts_p;
	  opts_done += o_len;

	  if (opts_done > opts_len)
	    break;

	  g_snprintf(label, MAXPATHLEN, "%u", *(opts_p - 1));
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[0], label, FALSE);
	  nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[1], DATA_TO_PTR(o_len), is_error);
	  
	  if (o_len > 2) 
	    {
	      tcp_opt_fields[4].bits = (o_len * 8) - 16;
	      nd_gui_proto_table_add(packet->trace, pinf, &tcp_opt_fields[4], DATA_TO_PTR(o_len - 2), is_error);
	    }
	}
	  
      if (opts_done > opts_len)
	D(("Warning -- bogus TCP options. ...\n"));
    }      
}


ND_Protocol *
nd_tcp_get(void)
{
  return tcp;
}


ND_Protocol *
nd_tcp_get_ip(void)
{
  static ND_Protocol *ip = NULL;
  
  if (!ip)
    {
      ip = nd_proto_registry_find(ND_PROTO_LAYER_NET, 0x0800);
      
      if (ip->id == 1)
	ip = NULL;
    }

  return ip;
}


gboolean
nd_tcp_get_first(const ND_Packet *packet, struct ip **iphdr, struct tcphdr **tcphdr)
{
  GList        *l;
  ND_Protocol  *ip;
  ND_ProtoData *pd, *pd_prev;

  if (!packet || !iphdr || !tcphdr)
    return FALSE;

  if (! (ip = nd_tcp_get_ip()))
    return FALSE;

  /* Walk the proto data and find the first TCP
     (we don't support nesting TCP) with a preceding IP
  */
  
  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      if (g_list_previous(l) && pd->inst.proto == tcp)
	{
	  pd_prev = (ND_ProtoData *) g_list_previous(l)->data;

	  if (pd_prev->inst.proto == ip)
	    {
	      *iphdr  = (struct ip *) pd_prev->data;
	      *tcphdr = (struct tcphdr *) pd->data;

	      return TRUE;
	    }
	}
    }
  
  return FALSE;
}
