/* Copyright 2009, UCAR/Unidata and OPeNDAP, Inc.
   See the COPYRIGHT file for more information. */

%{
#include "ncdap3.h"
#include "dapdump.h"

#define YYDEBUG 1

#define MAX_TOKEN_LENGTH 1024

/*! Specifies CEparsestate. */
typedef struct CEparsestate {
    NClist* projections;
    NClist* selections;
    struct NCsegment* segment; /*temporary */
    struct NCprojection* projection; /*temporary */
    NClist* selectionpath; /* temporary */
    struct NCselection* selection; /* temporary */
    int dapconstraint;
    struct CElexstate {
        char* input;
        char* next; /* next char in uri.query */
        char yytext[MAX_TOKEN_LENGTH];
        struct CElasttoken {
            char* text;
            int token;
        } lasttoken;
    } lexstate;
    char errorbuf[1024];
    int errorcode;
    NCbytes* tmp;
} CEparsestate;


union YYSTYPE; /* keep compiler quiet */

static int yylex(union YYSTYPE*, CEparsestate*);
static int yyerror(CEparsestate*,char*);

static int projectionlist(CEparsestate*);
static int pathsegment(CEparsestate* state);
static int segment(CEparsestate* state, char*);
static void range(CEparsestate*, int, size_t, size_t, size_t);
static int sel_clause(CEparsestate* state);
static void selectionvar(CEparsestate* state);
static void selectionvalue(CEparsestate* state, char* text, int tag);
static void selectionpath(CEparsestate* state, char* text);
static void function(CEparsestate* state, char* text);
static void rel_op(CEparsestate* state, int optoken);
static char* scanword(CEparsestate* state);
static char* scannumber(CEparsestate* state);
static char* scanstring(CEparsestate* state);

static void celexinit(char*,struct CElexstate*);
static void celexcleanup(struct CElexstate*);

%}

%pure-parser
%lex-param {CEparsestate* parsestate}
%parse-param {CEparsestate* parsestate}

%union {
    char* text;
    size_t index;
    int none;
}

%token  SCAN_WORD
%token  SCAN_STRING
%token  SCAN_NUMBER
%token  SCAN_EQ SCAN_NEQ SCAN_RE SCAN_GE SCAN_GT SCAN_LT SCAN_LE

%type <index> index
%type <text> ident string word number

%start constraints

%%
	
constraints:
	  projections
	| selections
	;

projections: /*empty*/ | projectionlist;

selections: selectionlist;

projectionlist:
	  projection
	    {projectionlist(parsestate);}
	| projectionlist ',' projection
	    {projectionlist(parsestate);}
	;

projection:
	  pathsegment
	| projection '.' pathsegment
	;

pathsegment: /* appends segment to state->projection */
	  segment
	     {pathsegment(parsestate);}
	| segment array_indices

	     {pathsegment(parsestate);}
	;

segment: /* creates state->segment and state->projection */
	word
	    {segment(parsestate,$1);}
	;


array_indices: /* appends indices to state->segment */
	  array_index
        | array_indices array_index
	;

array_index: 
	'[' range ']'
	;

range:
	  index
	    {range(parsestate,0,$1,0,0);}
	| index ':' index
	    {range(parsestate,1,$1,$3,0);}
	| index ':' index ':' index
	    {range(parsestate,2,$1,$3,$5);}
	;

selectionlist:
	  '&' sel_clause
	| selectionlist sel_clause
	;

sel_clause:
	  selectionvar rel_op '{' value_list '}'
	    {sel_clause(parsestate);}
	| selectionvar rel_op value
	    {sel_clause(parsestate);}
	| function
        ;

function:
	  functionname '(' ')'
	| functionname'(' value_list ')'
	;

functionname:
	ident {function(parsestate,$1);}
	;


value_list:
	  value
	| value_list '|' value
	;

selectionvar:
	selectionpath
	    {selectionvar(parsestate);}
	;

selectionpath:
	  ident
	    {selectionpath(parsestate,$1);}
	| pathsegment '.' ident
	    {selectionpath(parsestate,$3);}
	;

value:
	  selectionpath /* can be variable or an integer */
	    {selectionvalue(parsestate,NULL,SCAN_WORD);}
	| number
	    {selectionvalue(parsestate,$1,SCAN_NUMBER);}
	| string
	    {selectionvalue(parsestate,$1,SCAN_STRING);}
	;

rel_op:
	  SCAN_EQ  {rel_op(parsestate,SCAN_EQ);}
	| SCAN_NEQ {rel_op(parsestate,SCAN_NEQ);}
	| SCAN_RE  {rel_op(parsestate,SCAN_RE);}
	| SCAN_GE  {rel_op(parsestate,SCAN_GE);}
	| SCAN_GT  {rel_op(parsestate,SCAN_GT);}
	| SCAN_LT  {rel_op(parsestate,SCAN_LT);}
	| SCAN_LE  {rel_op(parsestate,SCAN_LE);}
	;

ident:  word
	    {$$ = $1;}
	;

index:  number
	    { unsigned long tmp = 0;
		if(sscanf($1,"%lu",&tmp) != 1) {
		    yyerror(parsestate,"Index is not an integer");
		}
		$$ = (size_t)tmp;
		efree($1);
	    }
	;

word:  SCAN_WORD
	    {$$ = scanword(parsestate);}
	;

number:  SCAN_NUMBER
	    {$$ = scannumber(parsestate);}
	;

string: SCAN_STRING
	    {$$ = scanstring(parsestate);}
	;

%%

static int
yyerror(CEparsestate* state, char* msg)
{
    strcpy(state->errorbuf,msg);
    state->errorcode=1;
    return 0;
}

/************************************************/

static int
projectionlist(CEparsestate* state)
{
    if(state->projections)
        nclistpush(state->projections,(ncelem)state->projection);
    else
	freencprojection1(state->projection);
    state->projection = NULL;
#ifdef DEBUG
fprintf(stderr,"parse.projectionlist: %s\n",
	dumpprojections(state->projections));
#endif
    return 0;
}

static int
pathsegment(CEparsestate* state)
{
    if(state->projection->segments == NULL)
        state->projection->segments = nclistnew();
    nclistpush(state->projection->segments,(ncelem)state->segment);
#ifdef DEBUG
fprintf(stderr,"parse.pathsegment: %s\n",
	dumpprojection1(state->projection));
#endif
    state->segment = NULL;
    return 0;
}


static int
segment(CEparsestate* state, char* segment)
{
    if(state->projection == NULL) {
	state->projection = createncprojection();
    }
    ASSERT((state->segment == NULL));
    state->segment = createncsegment();
    state->segment->segment = segment;
    state->segment->slicesdefined = 0;
    return 0;
}

static void
range(CEparsestate* state, int rangecase,
	    size_t field1, size_t field2, size_t field3)
{
    NCsegment* seg = state->segment;
    NCslice* slice = &seg->slices[seg->slicerank++];
    size_t first,last,stride;

    ASSERT(seg != NULL);
    /* Set values based on rangecase, dapconstraint flags */
    first = field1; /* always */
    stride = 1; /* mostly */
    switch (rangecase) {
    case 0: last = first; break;
    case 1: last = field2; break;
    case 2:
	if(state->dapconstraint) {
	    stride = field2; last = field3; break;
	} else {
	    stride = field3; last = field2; break;
	}
	break;
    default: break;
    }
    if(stride <= 0)
	yyerror(state,"Illegal index range stride");
    if(first < 0)
	yyerror(state,"Illegal index range first index");
    if(last < 0)
	yyerror(state,"Illegal index range last index");
    slice->first  = first;
    slice->stride = stride;
    slice->stop   = last + 1;
    slice->length  = slice->stop - slice->first;
    slice->count  = slice->length / slice->stride;
    state->segment->slicesdefined = 1;
}

/* Selection Procedures */

static int
sel_clause(CEparsestate* state)
{
    if(state->selections)
        nclistpush(state->selections,(ncelem)state->selection);
    else
	freencselection1(state->selection);
    state->selection = NULL;
    return 0;
}

static void
selectionvar(CEparsestate* state)
{
    NCselection* sel = createncselection();
    sel->operator = ST_NIL;
    sel->path = state->selectionpath;
    state->selectionpath = NULL;
    sel->node = NULL;
    sel->values = nclistnew();
    state->selection = sel;
}

static void
selectionpath(CEparsestate* state, char* text)
{
    if(state->selectionpath == NULL)
        state->selectionpath = nclistnew();
    ASSERT((text != NULL));
    nclistpush(state->selectionpath,(ncelem)text);
}

static void
selectionvalue(CEparsestate* state, char* text, int tag)
{
    NCvalue* value = createncvalue();
    switch (tag) {
    case SCAN_STRING:
	value->kind = ST_STR;
	value->value.text = text;
	break;
    case SCAN_NUMBER:
	if(sscanf(text,"%lld",&value->value.intvalue)==1)
	    value->kind = ST_INT;
	else if(sscanf(text,"%lg",&value->value.floatvalue)==1)
	    value->kind = ST_FLOAT;
	else {
	    sscanf(text,"%lG",&value->value.floatvalue);
	    value->kind = ST_FLOAT;
	}
	efree(text);
	break;
    case SCAN_WORD:
    default: {
#ifdef FIX
Why did I do this since text is ignored?
	char* text = NULL;
	/* Convert the current path to a single string */
	ncbytesclear(state->tmp);
	text = simplepathstring3(state->selectionpath,".");
#endif
	value->kind = ST_VAR;
	value->value.var.path = state->selectionpath; /* convert to cdfnode later */
	state->selectionpath = NULL;
    } break;
    }
    nclistpush(state->selection->values,(ncelem)value);
}

static void
rel_op(CEparsestate* state, int optoken)
{
    switch (optoken) {
    case SCAN_EQ: state->selection->operator = ST_EQ; break;
    case SCAN_NEQ: state->selection->operator = ST_NEQ; break;
    case SCAN_RE: state->selection->operator = ST_RE; break;
    case SCAN_GE: state->selection->operator = ST_GE; break;
    case SCAN_GT: state->selection->operator = ST_GT; break;
    case SCAN_LT: state->selection->operator = ST_LT; break;
    case SCAN_LE: state->selection->operator = ST_LE; break;
    default:
	yyerror(state,"rel_op: unknown token");
	break;
    }
}

static char*
scanword(CEparsestate* state)
{
    char* id = strdup(state->lexstate.yytext);
    state->lexstate.yytext[0] = '\0';
    return id;
}

static char*
scannumber(CEparsestate* state)
{
    char* id = strdup(state->lexstate.yytext);
    state->lexstate.yytext[0] = '\0';
    return id;
}

static char*
scanstring(CEparsestate* state)
{
    char* sc = strdup(state->lexstate.yytext);
    state->lexstate.yytext[0] = '\0';
    return sc;
}

static void
function(CEparsestate* state, char* fcnname)
{
    NCselection* sel = createncselection();
    sel->operator = ST_FCN;
    sel->path = nclistnew();
    nclistpush(sel->path,(ncelem)fcnname);
    sel->node = NULL;
    sel->values = nclistnew();
    state->selection = sel;
    state->selectionpath = NULL;
}

/**************************************************/
/*
Simple lexer
*/

static void
celexinit(char* input, struct CElexstate* lexstate)
{
    memset((void*)lexstate,0,sizeof(struct CElexstate));
    lexstate->input = input;
    lexstate->next = lexstate->input;
}

static void
celexcleanup(struct CElexstate* lexstate)
{
}

/* First character in identifier */
static char* idchars1=
"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-+_/%\\";
/* Non-first character in identifier */
static char* idcharsn=
"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-+_/%\\";
/* Number characters */
static char* numchars1="+-0123456789";
static char* numcharsn="Ee.+-0123456789";

static int
tohex(int c)
{
    if(c >= 'a' && c <= 'f') return (c - 'a') + 0xa;
    if(c >= 'A' && c <= 'F') return (c - 'A') + 0xa;
    if(c >= '0' && c <= '9') return (c - '0');
    return -1;
}

int
celex(YYSTYPE* lvalp, CEparsestate* state)
{
    int token;
    int c;
    int i;
    char* p=state->lexstate.next;
    token = 0;
    p=state->lexstate.next;
    while(token == 0 && (c=*p)) {
	if(c <= ' ' || c >= '\177') {p++; continue;}
	if(c == '"') {
	    int more = 1;
	    /* We have a SCAN_STRING */
	    i=0;
	    while(more && (c=*(++p))) {
		switch (c) {
		case '"': p++; more=0; break;
		case '\\':
		    c=*(++p);
		    switch (c) {
		    case 'r': c = '\r'; break;
		    case 'n': c = '\n'; break;
		    case 'f': c = '\f'; break;
		    case 't': c = '\t'; break;
		    case 'x': {
			int d1,d2;
			c = '?';
			++p;
		        d1 = tohex(*p++);
			if(d1 < 0) {
			    ceerror(state,"Illegal \\xDD in SCAN_STRING");
			} else {
			    d2 = tohex(*p++);
			    if(d2 < 0) {
			        ceerror(state,"Illegal \\xDD in SCAN_STRING");
			    } else {
				c=(((unsigned int)d1)<<4) | (unsigned int)d2;
			    }
			}
		    } break;
		    default: break;
		    }
		    break;
		default: break;
		}
		state->lexstate.yytext[i++]=c;
	    }
	    state->lexstate.yytext[i] = '\0';
	    token=SCAN_STRING;
	} else if(strchr(numchars1,c) != NULL) {
	    /* we might have a SCAN_NUMBER */
	    int isnumber = 0;
	    double number;
	    char* stoken = state->lexstate.yytext;
	    i=0;
	    stoken[i++]=c;
	    for(p++;(c=*p);p++) {
		if(strchr(numcharsn,c) == NULL) break;
		stoken[i++]=c;
	    }
	    stoken[i] = '\0';
	    /* See if this is a number */
	    if(sscanf(stoken,"%lg",&number) == 1
	       || sscanf(stoken,"%lG",&number) == 1)
		isnumber = 1; /* maybe */
	    /* A number followed by an id char is assumed to just be
		a funny id */	       
	    if(isnumber && (*p == '\0' || strchr(idcharsn,*p) == NULL))  {
	        token = SCAN_NUMBER;
	    } else {
		/* Now, if the funny word has a "." in it,
		   we have to back up to that dot */
		char* dotpoint = strchr(stoken,'.');
		if(dotpoint != NULL) {
		    p = dotpoint;
		    *dotpoint = '\0';
		}
		token = SCAN_WORD;
	    }
	} else if(strchr(idchars1,c) != NULL) {
	    /* we have a SCAN_WORD */
	    i=0;
	    state->lexstate.yytext[i++]=c;
	    for(p++;(c=*p);p++) {
		if(strchr(idcharsn,c) == NULL) break;
		state->lexstate.yytext[i++]=c;
	    }
	    state->lexstate.yytext[i] = '\0';
	    token=SCAN_WORD;
	} else {
	    /* we have a single or double char token */
	    int c1, char2 = 0;
	    state->lexstate.yytext[0] = c;
	    state->lexstate.yytext[1] = '\0';
	    token = c;
	    p++;
	    c1 = *p;
	    switch (c) {
	    case '=':
		if(c1 == '~') {char2 = 1; token = SCAN_RE;}
		else token = SCAN_EQ;
		break;
	    case '!':
		if(c1 == '=') {char2 = 1; token = SCAN_NEQ;}
		break;
	    case '>':
		if(c1 == '=') {char2 = 1; token = SCAN_GE;}
		else token = SCAN_GT;
		break;
	    case '<':
		if(c1 == '=') {char2 = 1; token = SCAN_LT;}
		else token = SCAN_LE;
		break;
	    default: break;
	    }
	    if(char2) {
		state->lexstate.yytext[1] = c1;
	        state->lexstate.yytext[2] = '\0';
		p++;
	    }
	}
    }
    state->lexstate.next = p;
    if(cedebug) fprintf(stderr,"TOKEN=|%s|\n",state->lexstate.yytext);
    return token;
}

/**************************************************/

static CEparsestate*
ce_parse_init(char* input, int dapconstraint,
		NClist* projections, NClist* selections)
{
    CEparsestate* state = NULL;
    if(input==NULL) {
        yyerror(state,"ce_parse_init: no input buffer");
    } else {
        state = (CEparsestate*)emalloc(sizeof(CEparsestate));
        MEMCHECK(state,(CEparsestate*)NULL);
        memset((void*)state,0,sizeof(CEparsestate)); /* Zero memory*/
        state->errorbuf[0] = '\0';
        state->errorcode = 0;
        celexinit(input,&state->lexstate);
	state->projections = projections;
	state->selections = selections;
	state->dapconstraint = dapconstraint;
	state->tmp = ncbytesnew();
    }
    return state;
}

static void
ce_parse_cleanup(CEparsestate* state)
{
    celexcleanup(&state->lexstate);
    ncbytesfree(state->tmp);
    free(state);
}

/* Wrap the parse process */
int
ncceparse(char* input, int dapconstraint,
	  NClist* projections, NClist* selections, char** errmsgp)
{
    CEparsestate* state;
    int errcode = 0;

    if(input != NULL) {
#ifdef DEBUG
fprintf(stderr,"ncceparse: input=%s\n",input);
#endif
        state = ce_parse_init(input,dapconstraint,projections,selections);
        if(!ceparse(state) == 0) {
	    if(errmsgp) *errmsgp = nulldup(state->errorbuf);
	}
	errcode = state->errorcode;
        ce_parse_cleanup(state);
    }
    return errcode;
}
