/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.php.editor.completion;

import java.io.IOException;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.swing.ImageIcon;
import javax.swing.text.Document;
import org.netbeans.api.gsf.CompletionProposal;
import org.netbeans.api.gsf.Element;
import org.netbeans.api.gsf.ElementKind;
import org.netbeans.api.gsf.HtmlFormatter;
import org.netbeans.api.gsf.Modifier;
import org.netbeans.api.lexer.Token;
import org.netbeans.api.lexer.TokenSequence;
import org.netbeans.modules.php.editor.TokenUtils;

/**
 * Implementation of the <code>CompletionResultProvider</code> for the new 
 * instruction context.
 * <p><b>Note that this implementation is not synchronized.</b></p> 
 * 
 * @author Victor G. Vasilyev 
 */
public class NewInstructionContext implements CompletionResultProvider {

    /**
     * Returns <code>true</code> iif the specified <code>context</code>
     * is applicable for inserting a new instruction.
     * E.g 
     * <p> 
     * <b>&lt;?php</b> 
     *     <span style="color: rgb(255, 0, 0);"><blink>|</blink></span>
     *     ... <b>?&gt;</b>
     * </p> 
     * or
     * <p>
     * <b>&lt;?php</b> <i>Instruction</i><b>;</b> 
     *    <span style="color: rgb(255, 0, 0);"><blink>|</blink></span>
     *     ... <b>?&gt;</b>
     * </p>
     * 
     * @param context
     * @return
     */
    public boolean isApplicable(CodeCompletionContext context) {
        assert context != null;
        myContext = context;
        try {
            TokenSequence ts = getTokenSequencePHP();
            Token t = TokenUtils.getEnteredToken(ts, getCaretOffset());
            assertWhitespace(t);
            boolean hasPreviousToken = ts.movePrevious();
            if(hasPreviousToken) {
                t = ts.token();
                assertOneOf(t, APPLICABLE_PREV_TOKENS);
            }
            return true;
        } catch (IOException ioe) {
            return false; // getDocument() fails
        } catch (ConcurrentModificationException cme) {
            // the token sequence is no longer valid because of an underlying 
            // mutable input source modification.
            return false;
        } catch (Exception e) {
            return false;
        }
    }

    public List<CompletionProposal> getProposals(final CodeCompletionContext context) {
        if (context != myContext) {
            throw new IllegalStateException("The isApplicable method MUST BE called before.");
        }
        List<CompletionProposal> proposalList = new ArrayList<CompletionProposal>();
        proposalList.add(new CompletionItem(getCaretOffset(), context.getFormatter()) {

                    public String getName() {
                        return "echo \"\"";
                    }

                    public String getInsertPrefix() {
                        return "echo \"\";\n";
                    }

                    public String getLhsHtml() {
                        HtmlFormatter formatter = getFormatter();
                        formatter.reset();
                        formatter.name(getKind(), true);
                        formatter.appendText("echo");
                        formatter.name(getKind(), false);
                        return formatter.getText();
                    }

                    public String getRhsHtml() {
                        return "language construct";
                    }

                    public ElementKind getKind() {
                        return ElementKind.KEYWORD;
                    }

                    public ImageIcon getIcon() {
                        return null;
                    }

                    public Set<Modifier> getModifiers() {
                        return null;
                    }

                    public boolean isSmart() {
                        return false;
                    }
                });
        return proposalList;
    }
    
    /**
     * Returns the <code>TokenSequence</code> of the PHP block pointed by 
     * the caret offset of the <code>CodeCompletionContext</code>.
     * @return the <code>TokenSequence</code> of the PHP block.
     * @throws java.io.IOException if {@link #getDocument()} fails
     * @throws java.lang.Exception if pointed code is not a PHP block or 
     * <code>TokenSequence</code> can't be returned for this code.
     */
    protected TokenSequence getTokenSequencePHP() throws IOException, Exception {
            assertPHPContext();
            Document doc = getDocument();
            int offset = getCaretOffset();
            TokenSequence ts = TokenUtils.getEmbeddedTokenSequence(doc, offset);
            if (ts == null) {
                throw new Exception();
            }
            return ts;
        
    }

    protected void assertPHPContext() throws IOException, Exception {
        Document doc = getDocument();
        int offset = getCaretOffset();
        if (!TokenUtils.checkPhp(doc, offset)) {
            throw new Exception();
        }
    }

    protected void assertWhitespace(Token t) throws Exception {
        if (!TokenUtils.PHPTokenName.WHITESPACE.value().equals(TokenUtils.getTokenType(t))) {
            throw new Exception();
        }
    }

    protected void assertOneOf(Token t, Set<String> tokenNameSet) throws Exception {
        if (!tokenNameSet.contains(TokenUtils.getTokenType(t))) {
            throw new Exception();
        }
    }

    protected Document getDocument() throws IOException {
        return myContext.getCompilationInfo().getDocument();
    }

    protected int getCaretOffset() {
        return myContext.getCaretOffset();
    }
    
    private CodeCompletionContext myContext;
    
    private static final Set<String> APPLICABLE_PREV_TOKENS = new HashSet<String>();
    static {
        APPLICABLE_PREV_TOKENS.add(TokenUtils.PHPTokenName.SEPARATOR.value());
        APPLICABLE_PREV_TOKENS.add(TokenUtils.PHPTokenName.BLOCK_COMMENT.value());
        APPLICABLE_PREV_TOKENS.add(TokenUtils.PHPTokenName.LINE_COMMENT.value());
        APPLICABLE_PREV_TOKENS.add(TokenUtils.PHPTokenName.WHITESPACE.value());
    }
}
