/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.scripting.php.dbginterface;

import java.util.ArrayList;
import java.util.List;
import org.netbeans.modules.scripting.php.dbginterface.models.Variable;

/**
 *
 * @author marcow
 */
public class DbgEvalValues {
    private List<Variable> vars;
    private char[]buffer;
    private int startIdx;
    
    /** Creates a new instance of DbgEvalValues */
    public DbgEvalValues(String str, String name) {
        if (str == null || str.length() == 0) {
            return;
        }
        
        buffer = str.toCharArray();
        startIdx = 0;
        vars = new ArrayList<Variable>();
        List<Variable> varList = new ArrayList<Variable>();
        
        parseFull(name, null, vars, varList, true);
    }
    
    public List<Variable> getVars() {
        return vars;
    }
    
    private boolean parseFull(String name, Variable parent, List<Variable> list,
            List<Variable> varList, boolean makePhpString) {
        if (startIdx >= buffer.length) {
            return false;
        }
        
        char ch = buffer[startIdx++];
        boolean ret_value = false;
        
        switch (ch) {
        case 'N':
            ret_value = ParseEvalNULL(parent, name, list, varList);
            break;
        case 'i':
            ret_value = ParseEvalInt(parent, name, list, varList);
            break;
        case 'd':
            ret_value = ParseEvalDouble(parent, name, list, varList);
            break;
        case 's':
            ret_value = ParseEvalString(parent, name, list, varList, makePhpString);
            break;
        case 'a':
            ret_value = ParseEvalArray(parent, name, list, varList, "", Variable.Type.ARRAY);
            break;
        case 'O':
            ret_value = ParseEvalObject(parent, name, list, varList);
            break;
        case 'b':
            ret_value = ParseEvalBool(parent, name, list, varList);
            break;
        case 'z':
            ret_value = ParseEvalResource(parent, name, list, varList);
            break;
        case 'R':
            ret_value = ParseEvalRef(parent, name, list, varList, false);
            break;
        case 'r':
            ret_value = ParseEvalRef(parent, name, list, varList, true);
            break;
        }
        
        if (!ret_value) {
            // try to recover
            int i = startIdx;
            
            while (i < buffer.length && buffer[i] != '{' && buffer[i]!= ';') {
                i++;
            }
            
            if (i < buffer.length && buffer[i] == '{') {
                int cnt = 1;
                
                i++;
                while (i < buffer.length && cnt != 0) {
                    if (buffer[i] == '{') {
                        cnt++;
                    }
                    else if (buffer[i] == '}') {
                        cnt--;
                    }
                    
                    i++;
                }
            }
            
            startIdx = i;
        }
        
        return ret_value;
    }

    private boolean ParseEvalNULL(Variable parent, String name, List<Variable> list,
            List<Variable> varList) {
        int idx = startIdx;
        
        if (idx >= buffer.length || buffer[idx] != ';') {
            return false;
        }
        
        startIdx++;
        
        Variable item = new Variable(parent, name, Variable.Type.UNDEFINED, "", "NULL");

        list.add(item);
        
        if (varList != null) {
            varList.add(item);
        }
        
        return true;
    }
    
    private boolean ParseEvalInt(Variable parent, String name, List<Variable> list,
            List<Variable> varList) {
        String subs;
   
        if ((subs = ExtractSubStr(':', ';')) == null) {
            return false;
        }
        
        Variable item = new Variable(parent, name, Variable.Type.LONG, "", subs);

        list.add(item);
        
        if (varList != null) {
            varList.add(item);
        }
                
        return true;
    }
    
    private boolean ParseEvalDouble(Variable parent, String name, List<Variable> list,
            List<Variable> varList) {
        String subs;
   
        if ((subs = ExtractSubStr(':', ';')) == null) {
            return false;
        }
        
        Variable item = new Variable(parent, name, Variable.Type.DOUBLE, "", subs);
        
        list.add(item);
        
        if (varList != null) {
            varList.add(item);
        }
        
        return true;
    }
 
    private boolean ParseEvalString(Variable parent, String name, List<Variable> list, 
            List<Variable> varList, boolean makePhpStr) {
        Integer slenI = ExtractInt(':', ':');
        
        if (slenI == null) {
            return false;
        }
        
        int slen = slenI.intValue();
        
        if (startIdx >= (buffer.length - 1) || buffer[startIdx] != '"') {
            return false;
        }
        
        startIdx++;
        
        String subs = new String(buffer, startIdx, slen);
        
        startIdx += slen;
        
        if (startIdx > (buffer.length - 2) || buffer[startIdx] != '"' ||
                buffer[startIdx + 1] != ';' || subs.length() != slen) {
            return false;
        }
        
        startIdx += 2;
        
        if (makePhpStr) {
            subs = ConvertToPhpString(subs);
        }
        
        Variable item = new Variable(parent, name, Variable.Type.STRING, "", subs);
        
        list.add(item);
        
        if (varList != null) {
            varList.add(item);
        }

        return true;
    }
    
    private boolean ParseEvalArray(Variable parent, String name, List<Variable> list,
            List<Variable> varList, String classname, Variable.Type atype) {
        Integer arritemsI = ExtractInt(':', ':');
        
        if (arritemsI == null) {
            return false;
        }
        
        int arritems = arritemsI.intValue();

        if (startIdx >= (buffer.length - 1) || buffer[startIdx] != '{') {
            return false;
        }
        
        startIdx++;

        Variable item = new Variable(parent, name, atype, classname);
        List<Variable> children = null;
        
        list.add(item);
        
        if (varList != null) {
            varList.add(item);
        }

        if (arritems > 0) {
            children = new ArrayList<Variable>();
            item.setChildren(children);
        }
        else if (buffer[startIdx] != '}') {
            return false;
        }

        while (startIdx < buffer.length && buffer[startIdx] != '}') {
            List<Variable> tmplst = new ArrayList<Variable>();
                // name
            if (!parseFull("", null, tmplst, null, false) || tmplst.size() != 1) {
                return false;
            }
            
                // value
            if (!parseFull(tmplst.get(0).getValue(), item, children, varList, true)) {
                return false;
            }
        }
        
        startIdx++;
        
        return true;
    }

    private boolean ParseEvalObject(Variable parent, String name, List<Variable> list,
            List<Variable> varList) {
        Integer slenI = ExtractInt(':', ':');
        
        if (slenI == null) {
            return false;
        }
        
        int slen = slenI.intValue();
        
        if (startIdx >= (buffer.length - 1)) {
            return false;
        }
        
        String classname;
        
        if ((classname = ExtractQuotedSubStr(slen)) == null) {
            return false;
        }
        
        if (classname.length() != slen) {
            return false;
        }
        
        return ParseEvalArray(parent, name, list, varList, classname, Variable.Type.OBJECT);
    }

    private boolean ParseEvalBool(Variable parent, String name, List<Variable> list,
            List<Variable> varList) {
        Integer bI = ExtractInt(':', ';');
        
        if (bI == null) {
            return false;
        }
        
        Variable item = new Variable(parent, name, Variable.Type.BOOLEAN, "",
                (bI.intValue() == 0 ? "FALSE" : "TRUE"));

        list.add(item);
        
        if (varList != null) {
            varList.add(item);
        }

        return true;
    }
    
    private boolean ParseEvalResource(Variable parent, String name, List<Variable> list,
            List<Variable> varList) {
        Integer slenI = ExtractInt(':', ':');
        
        if (slenI == null) {
            return false;
        }
        
        int slen = slenI.intValue();
        String restype;
        
        if ((restype = ExtractQuotedSubStr(slen)) == null) {
            return false;
        }
        
        if (restype.length() != slen) {
            return false;
        }

        Integer valI = ExtractInt(':', ';');
        
        if (valI == null) {
            return false;
        }
        
        Variable item = new Variable(parent, name, Variable.Type.RESOURCE, restype, valI.toString());

        list.add(item);
        
        if (varList != null) {
            varList.add(item);
        }

        return true;
    }

    private boolean ParseEvalRef(Variable parent, String name, List<Variable> list,
            List<Variable> varList, boolean isSoftRef) {
        Integer valI = ExtractInt(':', ';');
        
        if (valI == null) {
            return false;
        }
        
        int val = valI.intValue();

        Variable item = new Variable(parent, name,
                (isSoftRef ? Variable.Type.SOFT_REFERENCE : Variable.Type.REFERENCE), "");
        
        list.add(item);
        
        val--; // ref ID is 1-based, EvalList is 0-based

        if (varList == null || val < 0 || val >= varList.size()) {
            // Keep the ref to null
            // item.setRef(item); // self-resolving
        }
        else {
            // XXX I don't know what that's supposed to do right now.
            // So I store the info until I know more.
            item.setRef(varList.get(val));
        }
        
        return true;
        
    }
    
    private String ExtractSubStr(char chstart, char chend) {
        int idx = startIdx;
        
        if (idx >= (buffer.length - 1) || buffer[idx] != chstart) {
            return null;
        }
        
        int i = ++idx;
        
        while (i < buffer.length && buffer[i] != chend) {
            i++;
        }
        
        if (i == buffer.length) {
            return null;
        }
        
        String ret = new String(buffer, idx, i - idx);

        startIdx = i + 1;
        return ret;
    }

    String ExtractQuotedSubStr(int slen) {
        int idx = startIdx;
        
        if (idx + slen + 1 >= buffer.length || buffer[idx] != '"' ||
                buffer[idx + slen + 1] != '"') {
            return null;
        }
        
        String rslt = new String(buffer, idx + 1, slen);
        
        startIdx += slen + 2;
        
        return rslt;
    }

    Integer ExtractInt(char chstart, char chend) {
        String subs;
        
        if ((subs = ExtractSubStr(chstart, chend)) == null) {
            return null;
        }
        
        Integer ret = null;
        
        try {
            ret = Integer.decode(subs);
        }
        catch (NumberFormatException nfe) {
            // the ret == null is handled outside
        }

        return ret;
}

    private String ConvertToPhpString(String s) {
        char[] sIn = s.toCharArray();
        String rslt = "\"";
        
        for (int i = 0; i < sIn.length; i++) {
            char ch =sIn[i];
            
            switch (ch) {
            case '"':
            case '\\':
            case '$':
                rslt += '\\' + ch;
                break;
            case '\r':
                rslt += '\\' + 'r';
                break;
            case '\n':
                rslt += '\\' + 'n';
                break;
            case '\t':
                rslt += '\\' + 't';
                break;
            default:
                if (ch >= 0 && ch < 32) {
                    rslt += "\\x" + Integer.toHexString(ch);
                }
                else {
                    rslt += ch;
                }
            }            
        }
        
        rslt += "\"";
        
        return rslt;
    }
}