/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2007 Sun Microsystems, Inc.
 */

package org.netbeans.server.uihandler;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.prefs.AbstractPreferences;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.persistence.EntityManager;
import javax.persistence.Query;
import org.netbeans.modules.exceptions.entity.Logfile;
import org.netbeans.modules.exceptions.entity.Preference;

/** A preferences that do store objects into database.
 *
 * @author Jaroslav Tulach
 */
final class DbPreferences extends AbstractPreferences {
    private Statistics<?> statistics;
    private Logfile log;
    private String prefix;
    private EntityManager persist;
    private transient volatile List<Preference> preferences;
    
    private DbPreferences(
        AbstractPreferences parent, String name, 
        Statistics<?> statistics, Logfile log, String prefix,
        EntityManager em
    ) {
        super(parent, name);
        this.statistics = statistics;
        this.log = log;
        this.prefix = prefix;
        this.persist = em;
    }

    public static Preferences root(Logfile log, Statistics<?> stat, EntityManager em) {
        return new DbPreferences(null, "", stat, log, "/", em);
    }

    protected void putSpi(String key, String value) {
        Preference p = null;
        for (Preference current : getPreferences()) {
            if (key.equals(current.getKey())) {
                p = current;
                p.setValue(value);
                break;
            }
        }
        if (p == null) {
            p = new Preference(statistics.name, log, prefix, key, value);
            preferences.add(p);
            persist.persist(p);
        } else {
            persist.merge(p);
        }
    }

    protected String getSpi(String key) {
        for (Preference p : getPreferences()) {
            if (key.equals(p.getKey())) {
                return p.getValue();
            }
        }
        return null;
    }
    
    /** Loads current preferences from the database.
     * @return always non-null list of objects
     */
    private List<Preference> getPreferences() {
        List<Preference> prefs = preferences;
        if (prefs == null) {
            prefs = new ArrayList<Preference>();
            Query q = persist.createNamedQuery("Preference.findWithoutKey").
                setParameter("prefix", this.prefix).
                setParameter("statistic", this.statistics.name).
                setParameter("logfile", this.log);
            for (Object o : q.getResultList()) { // NOI18N
                Preference p = (Preference) o;
                prefs.add(p);
            }

            preferences = new CopyOnWriteArrayList<Preference>(prefs);
        }
        return prefs;
    }

    protected void removeSpi(String key) {
        for (Preference p : getPreferences()) {
            if (key.equals(p.getKey())) {
                persist.remove(p);
                getPreferences().remove(p);
            }
        }
    }

    protected void removeNodeSpi() throws BackingStoreException {
        for (Preference p : getPreferences()) { // NOI18N
            persist.remove(p);
        }
        preferences = null;
    }

    protected String[] keysSpi() throws BackingStoreException {
        List<String> keys = new ArrayList<String>();
        for (Preference p : getPreferences()) {
            keys.add(p.getKey());
        }
        
        return keys.toArray(new String[0]);
    }

    protected String[] childrenNamesSpi() throws BackingStoreException {
        Query q = persist.createNamedQuery("Preference.findAllPrefixes"). // NOI18N
            setParameter("prefix", this.prefix + "%"). // NOI18N
            setParameter("statistic", this.statistics.name). // NOI18N
            setParameter("logfile", this.log); // NOI18N
        
        Set<String> subnames = new LinkedHashSet<String>();
        for (Object o : q.getResultList()) { // NOI18N
            Preference p = (Preference)o;
            
            assert p.getPrefix().startsWith(this.prefix);
            String n = p.getPrefix().substring(this.prefix.length());
            int idx = n.indexOf('/');
            if (idx > 0) {
                n = n.substring(0, idx);
            }
            if (n.length() == 0) {
                continue;
            }
            subnames.add(n);
        }
        
        return subnames.toArray(new String[0]);
        
    }

    protected AbstractPreferences childSpi(String name) {
        return new DbPreferences(this, name, statistics, log, prefix + name + '/', persist);
    }

    protected void syncSpi() throws BackingStoreException {
    }

    protected void flushSpi() throws BackingStoreException {
        boolean active = persist.getTransaction().isActive();
        if (!active) {
            persist.getTransaction().begin();
        }
        persist.flush();
        if (!active) {
            persist.getTransaction().commit();
        }
    }

}
