/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Ralph Krueger.
 */

package org.netbeans.modules.changelog;


import java.util.*;
import org.openide.windows.*;
import org.openide.*;
import java.io.*;
import java.text.SimpleDateFormat;

import org.netbeans.modules.changelog.html.*;

/**
 * prints the processed groups to xml file.. (based on LogPrinter_Text)
 * @author  rbalada
 * @author  ralph
 */
public class LogPrinter_XML implements LogPrinter {

    private java.io.PrintWriter writer;
    private boolean includeSummary;

    public LogPrinter_XML(File file) {
        if (!file.exists()) {
            try {
                if (file.getParentFile() != null) {
                    file.getParentFile().mkdirs();
                }
                file.createNewFile();
            } catch (IOException exc) {
                 org.openide.ErrorManager.getDefault().notify(exc);
                 System.out.println("error while creating file..");
            }
        }
        writer = null;
        try {
            writer = new PrintWriter(new FileOutputStream(file));
        } catch (IOException exc) {
                 org.openide.ErrorManager.getDefault().notify(exc);
                 System.out.println("error while opening file..");
        }
    }
    
    public void printHeader(ChangeLogProcessor processor) {
        if (writer == null) {
            return;
        }
        writer.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        writer.println();
        writer.println("<changelog>");
        includeSummary = processor.isIncludeSummary();
        if (processor.isIncludeQueryDescription()) {
            writer.println();
            writer.println("<query>");
            if (processor.getDateRange() != null) {
                writer.println("<daterange>" + ChangeLogUtils.xmlescapeString(processor.getDateRange()) + "</daterange>");
            }
            if (processor.getRevisionRange() != null) {
                writer.println("<revisionfilter>" + ChangeLogUtils.xmlescapeString(processor.getRevisionRange()) + "</revisionfilter>");
            }
            if (processor.getMessageFilter() != null) {
                String messageType = "";
                if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_SUBSTRING) {
                    messageType = "substring";
                } else if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_SOME_WORDS) {
                    messageType = "anywords";
                } else if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_ALL_WORDS) {
                    messageType = "allwords";
                } else if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_REGEXP) {
                    messageType = "regexp";
                }
                writer.println("<messagefilter messagetype=\"" + messageType + "\">" + ChangeLogUtils.xmlescapeString(processor.getMessageFilter()) + "</messagefilter>");
            }
            if (processor.getFileFilter() != null) {
                String fileType = "";
                if (processor.getFileFilterType() == ChangeLogProcessor.FILE_FILTER_SUBSTRING) {
                    fileType = "substring";
                } else if (processor.getFileFilterType() == ChangeLogProcessor.FILE_FILTER_REGEXP) {
                    fileType = "regexp";
                }
                writer.println("<filefilter filetype=\"" + fileType + "\">" + ChangeLogUtils.xmlescapeString(processor.getFileFilter()) + "</filefilter>");
            }
            if (processor.getSortMode() == ChangeLogProcessor.SORT_BY_DATE) {
                writer.print("<sort orderby=\"date\"");
            } else if (processor.getSortMode() == ChangeLogProcessor.SORT_BY_USER) {
                writer.print("<sort orderby=\"user\"");
            }
            if (processor.isDescendingSort()) {
                writer.println(" direction=\"descending\"/>");
            } else {
                writer.println(" direction=\"ascending\"/>");
            }
            writer.println("</query>");
            writer.println();
        }
    }
    
    public void printGroupHeader(RevisionsGroup group) {
        if (writer == null) {
            return;
        } 
        writer.println("<entry>");
        writer.println("<author>" + ChangeLogUtils.xmlescapeString(group.getUser()) + "</author>");
        final String sdate = (new java.text.SimpleDateFormat("yyyy-MM-dd")).format(group.getStartingDate());
        final String stime = (new java.text.SimpleDateFormat("hh:mm:ss")).format(group.getStartingDate());
        writer.println("<date>" + ChangeLogUtils.xmlescapeString(sdate) + "</date>");
        writer.println("<time>" + ChangeLogUtils.xmlescapeString(stime) + "</time>");
        if ( group.getCommonBranch() != null) {
            writer.println("<commonbranch>" + ChangeLogUtils.xmlescapeString(group.getCommonBranch()) + "</commonbranch>");
        }
        writer.println("<msg>" + ChangeLogUtils.xmlescapeString(group.getMessage()).trim() + "</msg>");
    }
    
    public void printSingleRevision(LogInfoRevision revision) {
        if (writer == null ) {
            return;
        }
        String repoFileName = revision.getLogInfoHeader().getRepositoryFilename();
        repoFileName = repoFileName.substring(0, repoFileName.length() - 2);
        writer.println("<file>");
        writer.println("<name>" + ChangeLogUtils.xmlescapeString(repoFileName) + "</name>");
        writer.println("<revision>" + ChangeLogUtils.xmlescapeString(revision.getNumber()) + "</revision>");
        if ( revision.getBranch() != "" ) {
            writer.println("<branch>" + ChangeLogUtils.xmlescapeString(revision.getBranch()) + "</branch>");
        }
        writer.println("</file>");
    }
    
    public void printGroupFooter(RevisionsGroup group) {
        if (writer == null) {
            return;
        } 
        writer.println("</entry>");
        writer.println();
    }
    
    public void printSummary(SummaryProcessor processor) {
        if (writer == null) {
            return;
        }
        if (includeSummary) {
            writer.println("<summary>");
            writer.println("<changecount>" + ChangeLogUtils.xmlescapeString(String.valueOf(processor.getCommitCount())) + "</changecount>");
            writer.println("<developers>");
            String[] users = processor.getUserList();
            for (int i = 0; i < users.length; i++) {
                writer.println("<developer>" + ChangeLogUtils.xmlescapeString(users[i]) + "</developer>");
            }
            writer.println("</developers>");
            String[] mostChanged = processor.getMostChangedFiles();
            writer.println("<mostchangedfiles>");
            for (int j = 0; j < mostChanged.length; j++) {
                writer.println("<mostchangedfile>" + ChangeLogUtils.xmlescapeString(mostChanged[j]) + "</mostchangedfile>");
            }
            writer.println("</mostchangedfiles>");
            writer.println("<mostactivedevelopers>");
            String[] mostActive= processor.getMostActiveUsers();
            for (int k = 0; k < mostActive.length; k++) {
                writer.println("<mostactivedeveloper>" + ChangeLogUtils.xmlescapeString(mostActive[k]) + "</mostactivedeveloper>");
            }
            writer.println("</mostactivedevelopers>");
            writer.println("</summary>");
            writer.println();
        }
    }
    
    public void printFooter(ChangeLogProcessor processor) {
        //do nothing
        if (writer != null) {
            writer.println("</changelog>");
            writer.println();
            writer.flush();
            writer.close();
        }
    }
    
}
