/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Ralph Krueger.
 */


package org.netbeans.modules.changelog;

/**
 *
 * @author  Ralph Krueger
 */

import org.apache.regexp.RE;
import org.apache.regexp.RESyntaxException;


import java.util.*;
import java.io.*;
import java.lang.reflect.*;
import org.openide.filesystems.*;
import org.openide.*;
import org.openide.loaders.*;
import org.openide.util.*;
import org.netbeans.modules.changelog.html.*;
import org.netbeans.modules.changelog.settings.*;

public class ChangeLogProcessor {

    public static final int MESSAGE_FILTER_SUBSTRING = 0;
    public static final int MESSAGE_FILTER_REGEXP = 1;
    public static final int MESSAGE_FILTER_ALL_WORDS = 2;
    public static final int MESSAGE_FILTER_SOME_WORDS = 3;
    public static final int FILE_FILTER_SUBSTRING = 0;
    public static final int FILE_FILTER_REGEXP = 1;
    private static final String INITIAL_REV = "Initial revision"; //NOI18N
    private static final String WAS_ADDED_ON_BRANCH = "was initially added on branch"; //NOI18N
    
    public static final int SORT_BY_USER = 0;
    public static final int SORT_BY_DATE = 1;
    
    private long revisionsNumber = 0;
    
    private HashMap fileObjectMap;
    
    private LinkedList rawResults;
    
    private List groupedResults;
    
    private List printerList;
    
    // properties
    private String messageFilter;
    private int messageFilterType;
    private int minutesGap;
    
    private RE messageRE = null;

    // properties
    private String fileFilter;
    private int fileFilterType;
    
    private RE fileRE = null;
    
    /** Holds value of property dateRange. */
    private String dateRange;
    
    /** Holds value of property user. */
    private String user;
    
    private int sortMode;
    
    /** Holds value of property revisionRange. */
    private String revisionRange;
    
    /** Holds value of property toOutputWindow. */
    private boolean toOutputWindow;
    
    /** Holds value of property toTextFile. */
    private java.io.File toTextFile;
    
    /** Holds value of property toXmlFile. */
    private java.io.File toXmlFile;
    
    /** Holds value of property toHtmlFile. */
    private java.io.File toHtmlFile;
    
    /** Holds value of property toExplorer. */
    private boolean toExplorer;
    
    /** Holds value of property descendingSort. */
    private boolean descendingSort;
    
    private boolean viewInBrowser;
    
    /** Holds value of property htmlSiteProcessor. */
    private ChangeLogHTMLService htmlSiteProcessor;
    
    /** Holds value of property includeQueryDescription. */
    private boolean includeQueryDescription;
    
    /** Holds value of property includeSummary. */
    private boolean includeSummary;
    
    /** Holds value of property includeBranchNames. */
    private boolean includeBranchNames;
    
    /** Creates new ChangeLogDisplayer */
    public ChangeLogProcessor() {
        fileObjectMap = new HashMap(10);
        rawResults = new LinkedList();
        groupedResults = new LinkedList();
        setMaxDateGap(10);
        setSortMode(SORT_BY_DATE);
        printerList = new LinkedList();
        setViewInBrowser(true);
        setIncludeSummary(true);
        setIncludeQueryDescription(true);
        
        ChangeLogSettings settings = (ChangeLogSettings)SharedClassObject.findObject(ChangeLogSettings.class, true);
        setIncludeBranchNames(settings.isShowBranchesByDefault());
        String id = settings.getDefaultServerInfo();
        if (!id.equals("")) {
            Lookup.Template template = new Lookup.Template(ChangeLogHTMLService.class, id, null);
            Lookup.Item item = Lookup.getDefault().lookupItem(template);
            setHtmlSiteProcessor((ChangeLogHTMLService)item.getInstance());
        }
    }
    
    public void setViewInBrowser(boolean view) {
        viewInBrowser = view;
    }
    public boolean isViewInBrowser() {
        return viewInBrowser;
    }
    
    public void setMessageFilter(int type, String filter) {
        messageFilter = filter;
        messageFilterType = type;
        if (messageFilterType == MESSAGE_FILTER_REGEXP) {
            try {
                messageRE = new RE(messageFilter);
            } catch (RESyntaxException exc) {
                messageRE = null;
            }
        }
        if  (messageFilterType == MESSAGE_FILTER_SOME_WORDS ||
             messageFilterType == MESSAGE_FILTER_ALL_WORDS) 
        {
            try {
                String regExp = "";
                StringTokenizer tok = new StringTokenizer(messageFilter, " ");
                while (tok.hasMoreTokens()) {
                    regExp = regExp + tok.nextToken();
                    if (tok.hasMoreTokens()) {
                        if (messageFilterType == MESSAGE_FILTER_ALL_WORDS) {
                            regExp = regExp + '&';
                        } else {
                            regExp = regExp + '|';
                        }
                    }
                }
                messageRE = new RE(regExp);
            } catch (RESyntaxException exc) {
                messageRE = null;
            }
        }
    }
    
    public String getMessageFilter() {
        return messageFilter;
    }
    
    public int getMessageFilterType() {
        return messageFilterType;
    }

    public void setFileFilter(int type, String filter) {
        fileFilter = filter;
        fileFilterType = type;
        if (fileFilterType == FILE_FILTER_REGEXP) {
            try {
                fileRE = new RE(fileFilter);
            } catch (RESyntaxException exc) {
                fileRE = null;
            }
        }
    }
    
    public String getFileFilter() {
        return fileFilter;
    }
    
    public int getFileFilterType() {
        return fileFilterType;
    }
    
    
    public void setMaxDateGap(int numberOfMinutes) {
        minutesGap = numberOfMinutes;
    }
    
    public int getMaxDateGap() {
        return minutesGap;
    }
    
    private long getMaxDateGapInTime() {
        return (long)((minutesGap * 60) * 1000);
    }
    
    /**
     * adds fileobjects to a map of selected fileobjects for this action.
     * Is needed to convert correctly the resulting files back to
     * fileobjects.
     */ 
/*    public void addFileObjects(FileObject[] fos) {
        if (fos != null) {
            for (int i=0; i < fos.length; i++) {
                File file = FileSystemCommand.toFile(fos[i]);
                if (file != null) {
                    fileObjectMap.put(file, fos[i]);
                }
            }
        }
    }
 */

    public boolean messageMatchesFilterPattern(String message) {
        if (messageFilter == null || message == null) {
            return true;
        }
        if (messageFilterType == MESSAGE_FILTER_SUBSTRING) {
            if (message.indexOf(messageFilter) >= 0) {
                return true;
            } 
        }
        else if (messageFilterType == MESSAGE_FILTER_REGEXP ||
                 messageFilterType == MESSAGE_FILTER_SOME_WORDS ||
                 messageFilterType == MESSAGE_FILTER_ALL_WORDS) {
            if (messageRE != null) {
                if (messageRE.match(message)) {
                    return true;
                } 
            } else {
                return true;
            }
        }
        return false;
    }
    
    public boolean fileMatchesFilterPattern(String file) {
        if (fileFilter == null || file == null) {
            return true;
        }
        if (fileFilterType == FILE_FILTER_SUBSTRING) {
            if (file.indexOf(fileFilter) >= 0) {
                return true;
            } 
        }
        else if (fileFilterType == FILE_FILTER_REGEXP) {
            if (fileRE != null) {
                if (fileRE.match(file)) {
                    return true;
                } 
            } else {
                return true;
            }
        }
        return false;
    }    
   
    public void addRevision(LogInfoRevision rev, String message) {
        Iterator it = groupedResults.iterator();
        boolean found = false;
        while (it.hasNext()) {
            RevisionsGroup group = (RevisionsGroup)it.next();
            // I wonder if the message first and then time comparison is faster then
            // vice versa..
            if (message != null &&
                group.getMessage() != null &&
                message.equals(group.getMessage())) 
            {
                    
                long gap1 = Math.abs(rev.getDate().getTime() - group.getStartingDate().getTime());
                long gap2 = Math.abs(rev.getDate().getTime() - group.getTrailingDate().getTime());
                if (gap2 < getMaxDateGapInTime() || gap1 < getMaxDateGapInTime()) {
                    group.addRevision(rev, message);
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            RevisionsGroup newGroup = new RevisionsGroup();
            newGroup.addRevision(rev, message);
            groupedResults.add(newGroup);
        }
    }
    
    
    private List findFOsForFiles(List fileList) {
/**
 needs to be rewritten to make the recognition by partial ath as well..
 *
 Iterator it = fileList.iterator();
        List foList = new LinkedList();
        while (it.hasNext()) {
            StatusInformation info = (StatusInformation)it.next();
            FileObject fo = (FileObject)fileObjectMap.get(info.getFile());
            foList.add(fo);
        }
        return foList;
 *
 */
        return null;
    }
    

    /**
     * returns a list of RevisionsGroup instances.
     */
    public List getGroupsList() {
        return groupedResults;
    }
        
    /** Getter for property startingdate.
     * @return Value of property startingdate.
     */
    public String getDateRange() {
        return this.dateRange;
    }
    
    /** Setter for property startingdate.
     * @param startingdate New value of property startingdate.
     */
    public void setDateRange(String dateRange) {
        this.dateRange = dateRange;
    }
    
    /** Getter for property user.
     * @return Value of property user.
     */
    public String getUser() {
        return this.user;
    }
    
    /** Setter for property user.
     * @param user New value of property user.
     */
    public void setUser(String user) {
        this.user = user;
    }
    
    /** Getter for property branch.
     * @return Value of property branch.
     */
    public String getRevisionRange() {
        return this.revisionRange;
    }
    
    /** Setter for property branch.
     * @param branch New value of property branch.
     */
    public void setRevisionRange(String revisionRange) {
        this.revisionRange = revisionRange;
    }
    
    /** Getter for property toOutputWindow.
     * @return Value of property toOutputWindow.
     */
    public boolean isToOutputWindow() {
        return this.toOutputWindow;
    }
    
    /** Setter for property toOutputWindow.
     * @param toOutputWindow New value of property toOutputWindow.
     */
    public void setToOutputWindow(boolean toOutputWindow) {
        this.toOutputWindow = toOutputWindow;
    }
    
    /** Getter for property toTextFile.
     * @return Value of property toTextFile.
     * Null means the text file is not created.
     */
    public java.io.File getToTextFile() {
        return this.toTextFile;
    }
    
    /** Setter for property toTextFile.
     * @param toTextFile New value of property toTextFile.
     */
    public void setToTextFile(java.io.File toTextFile) {
        this.toTextFile = toTextFile;
    }
    
    /** Getter for property toXmlFile.
     * @return Value of property toXmlFile.
     * Null means the xml file will not be written.
     */
    public java.io.File getToXmlFile() {
        return this.toXmlFile;
    }
    
    /** Setter for property toXmlFile.
     * @param toXmlFile New value of property toXmlFile.
     */
    public void setToXmlFile(java.io.File toXmlFile) {
        this.toXmlFile = toXmlFile;
    }
    
    /** Getter for property toHtmlFile.
     * @return Value of property toHtmlFile.
     * Null means the html file will not be written.
     */
    public java.io.File getToHtmlFile() {
        return this.toHtmlFile;
    }
    
    /** Setter for property toHtmlFile.
     * @param toHtmlFile New value of property toHtmlFile.
     */
    public void setToHtmlFile(java.io.File toHtmlFile) {
        this.toHtmlFile = toHtmlFile;
    }
    
    /** Getter for property toExplorer.
     * @return Value of property toExplorer.
     */
    public boolean isToExplorer() {
        return this.toExplorer;
    }
    
    /** Setter for property toExplorer.
     * @param toExplorer New value of property toExplorer.
     */
    public void setToExplorer(boolean toExplorer) {
        this.toExplorer = toExplorer;
    }
    
    public void setSortMode(int mode) {
        this.sortMode = mode;
    }
    
    public int getSortMode() {
        return sortMode;
    }
    
    
    public void finishProcessing() {
        if (getFileFilter() != null) {
            //postprocess the file based filter. 
            // (or do anything else that cannot be achieved by the log command.
            postprocess();
        }
        sort();
        if (isToOutputWindow()) {
            printerList.add(new LogPrinter_Output());
        }
        if (getToTextFile() != null) {
            printerList.add(new LogPrinter_Text(getToTextFile()));
        }
        if (getToXmlFile() != null) {
            printerList.add(new LogPrinter_XML(getToXmlFile()));
        }
        if (getToHtmlFile() != null || isViewInBrowser()) {
            printerList.add(new LogPrinter_HTML(getToHtmlFile(), isViewInBrowser(), getHtmlSiteProcessor()));
        }
        printResults();
    }

    private void postprocess() {
        List groupedResults = getGroupsList();
        Iterator it = groupedResults.iterator();
        while (it.hasNext()) {
            RevisionsGroup group = (RevisionsGroup)it.next();
            List list = group.getList();
            boolean ok = false;
            Iterator it2 = list.iterator();
            while (it2.hasNext()) {
                LogInfoRevision rev = (LogInfoRevision)it2.next();
                if (fileMatchesFilterPattern(rev.getLogInfoHeader().getRepositoryFilename())) {
                    ok = true;
                    break;
                }
            }
            if (!ok) {
                it.remove();
            }
        }
        
    }
    
    private void sort() {
        Comparator comp;
        if (getSortMode() == SORT_BY_DATE) {
            comp = new RevisionsGroup.GroupDateComparator(isDescendingSort());
        } else if (getSortMode() == SORT_BY_USER) {
            comp = new RevisionsGroup.UserDateComparator(isDescendingSort());
        } else {
            comp = new RevisionsGroup.GroupDateComparator(false);
        }
        Collections.sort(getGroupsList(), comp);
    }
    
    private void processPrinters(String methodName, Class[] paramTypes, Object[] params) {
        Iterator it = printerList.iterator();
        while (it.hasNext()) {
            Object printer = it.next();
            try {
                Method method = printer.getClass().getMethod(methodName, paramTypes);
                method.invoke(printer, params);
            } catch (InvocationTargetException exc1) {
                org.openide.ErrorManager.getDefault().annotate(exc1.getTargetException(), "");
            } catch (Exception exc) {
                org.openide.ErrorManager.getDefault().annotate(exc, "");
            } 
        }
    }
    
    private void printResults() {
        List groupedResults = getGroupsList();
        SummaryProcessor sumProc = new SummaryProcessor();
        processPrinters("printHeader", new Class[] {ChangeLogProcessor.class}, new Object[] {this});
        Iterator it = groupedResults.iterator();
        while (it.hasNext()) {
            RevisionsGroup group = (RevisionsGroup)it.next();
            processPrinters("printGroupHeader", new Class[] {RevisionsGroup.class}, new Object[] {group});
            sumProc.processGroup(group);
            List list = group.getSortedList(RevisionsGroup.SORTING_BY_FILE_NAME);
            Iterator it2 = list.iterator();
            while (it2.hasNext()) {
                LogInfoRevision rev = (LogInfoRevision)it2.next();
                processPrinters("printSingleRevision", new Class[] {LogInfoRevision.class}, new Object[] {rev});
            }
            processPrinters("printGroupFooter", new Class[] {RevisionsGroup.class}, new Object[] {group});
        }
        processPrinters("printSummary", new Class[] {SummaryProcessor.class}, new Object[] {sumProc});
        processPrinters("printFooter", new Class[] {ChangeLogProcessor.class}, new Object[] {this});
    }

    
    
    /** Getter for property descendingSort.
     * @return Value of property descendingSort.
     */
    public boolean isDescendingSort() {
        return this.descendingSort;
    }    
    
    /** Setter for property descendingSort.
     * @param descendingSort New value of property descendingSort.
     */
    public void setDescendingSort(boolean descendingSort) {
        this.descendingSort = descendingSort;
    }
    
    /** Getter for property htmlSiteProcessor.
     * @return Value of property htmlSiteProcessor.
     */
    public ChangeLogHTMLService getHtmlSiteProcessor() {
        return this.htmlSiteProcessor;
    }
    
    /** Setter for property htmlSiteProcessor.
     * @param htmlSiteProcessor New value of property htmlSiteProcessor.
     */
    public void setHtmlSiteProcessor(ChangeLogHTMLService htmlSiteProcessor) {
        this.htmlSiteProcessor = htmlSiteProcessor;
    }
    
    /** Getter for property includeQueryDescription.
     * @return Value of property includeQueryDescription.
     */
    public boolean isIncludeQueryDescription() {
        return this.includeQueryDescription;
    }
    
    /** Setter for property includeQueryDescription.
     * @param includeQueryDescription New value of property includeQueryDescription.
     */
    public void setIncludeQueryDescription(boolean includeQueryDescription) {
        this.includeQueryDescription = includeQueryDescription;
    }
    
    /** Getter for property includeSummary.
     * @return Value of property includeSummary.
     */
    public boolean isIncludeSummary() {
        return this.includeSummary;
    }
    
    /** Setter for property includeSummary.
     * @param includeSummary New value of property includeSummary.
     */
    public void setIncludeSummary(boolean includeSummary) {
        this.includeSummary = includeSummary;
    }
    
    /** Getter for property includeBranchNames.
     * @return Value of property includeBranchNames.
     */
    public boolean isIncludeBranchNames() {
        return this.includeBranchNames;
    }
    
    /** Setter for property includeBranchNames.
     * @param includeBranchNames New value of property includeBranchNames.
     */
    public void setIncludeBranchNames(boolean includeBranchNames) {
        this.includeBranchNames = includeBranchNames;
    }
    
}
