/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Compiler.Parsetree;
using Nemerle.Compiler;

namespace Nemerle.Compiler
{
  public module ScanTypeHierarchy
  {
    /** Process toplevel type declaration.
    
        This function does the very first pass, after parsing.
        It creates [TypeBuilder] object for each type and registers it
        in namespace hierarchy.
        It also creates classes of delegates and macros.
     */
    public ProcessDeclaration (decl : TopDeclaration) : void
    {
      def handle_type (parent, ns_node, t : TopDeclaration)
      {
        def env = t.ParsedName.context;
        
        match (t) {
          | TopDeclaration.Macro as t =>
            when (parent != null)
              Message.Error (decl.Location, "macro declarations are allowed only at top level, "
                             "move it outside the class definition");
            MacroClasses.GenerateMacroClass (t, ns_node)
          
          | TopDeclaration.Delegate (header) =>
            _ = Delegates.GenerateDelegateClass (env, parent, t.modifiers, header)

          | _ =>
            def tb = NamespaceTree.AddType (parent, ns_node, t);
            def node_of_type = tb.NamespaceNode;

            match (t) {
              | TopDeclaration.Interface (methods = ds)
              | TopDeclaration.Class (decls = ds) =>
                foreach (ClassMember.TypeDeclaration (t) in ds)
                  handle_type (tb, node_of_type, t)

              | TopDeclaration.Variant (decls = decls) =>
                foreach (x in decls) {
                  | ClassMember.TypeDeclaration (d) =>
                    handle_type (tb, node_of_type, d)

                  | ClassMember.EnumOption =>
                    Message.Error (x.Location, "variant options cannot have value, they are not enums")

                  | _ => ()
                }

              | TopDeclaration.VariantOption
              | TopDeclaration.Enum 
              | TopDeclaration.Alias => ()
              | _ => assert (false)
            };
            // they are added in reversed order, so bring them to source order
            tb.ReverseChildren ();
        }
      }
      handle_type (null, decl.ParsedName.context.CurrentNamespace, decl); 
    }
  } // module ScanTypeHierarchy
} // N.C
