 /*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;
using Nemerle.Compiler.Typedtree;

using System.Reflection;

using PT = Nemerle.Compiler.Parsetree;

namespace Nemerle.Compiler {

public abstract class MemberBuilder : IMember
{
  protected loc : Location;
  [Accessor]
  protected body_location : Location;
  [Accessor]
  protected mutable name : string;
  protected accessibility : Accessibility;
  [Accessor]
  protected modifiers : Modifiers;
  [Accessor]
  protected declaring_type : TypeBuilder;
  id : int;
  protected kind : MemberKind;
  internal mutable ty : MType;
  protected mutable handle : MemberInfo;

  [Accessor (flags = WantSetter)]
  protected mutable attributes : NemerleAttributes;

  protected mutable m_has_been_used : bool;

  internal mutable disable_type_attr : bool;
  
  protected this (par : TypeBuilder, d : PT.ClassMember)
  {
    loc = d.loc;
    modifiers = d.modifiers;
    name = d.Name; 
    declaring_type = par;
    id = Util.next_id ();

    attributes = modifiers.mods;

    unless (attributes %&& NemerleAttributes.AccessModifiers)
      attributes |= NemerleAttributes.Private;

    accessibility = 
      match (attributes %& NemerleAttributes.AccessModifiers) {
        | NemerleAttributes.Private => Accessibility.Private
        | NemerleAttributes.Protected => Accessibility.Protected
        | NemerleAttributes.Internal => Accessibility.Internal
        | _ =>
          if (attributes %&& (NemerleAttributes.Protected %| NemerleAttributes.Internal))
            Accessibility.ProtectedOrInternal
          else
            Accessibility.Public
      }

    match ((par.Accessibility, accessibility)) {
      | (Accessibility.Private, _) =>
        accessibility = Accessibility.Private
      | (Accessibility.Public, _) => ()
      | (Accessibility.Internal, Accessibility.Public) =>
        accessibility = Accessibility.Internal
      | (Accessibility.Protected, Accessibility.Internal)
      | (Accessibility.Internal, Accessibility.Protected) =>
         accessibility = Accessibility.ProtectedAndInternal
      | _ => ()
    }

    // mark public and protected members explicitly used
    m_has_been_used =
      (IsPublic || IsProtected) && declaring_type.IsExternallyAccessible ||
      name.StartsWith ("_") || declaring_type.Name.StartsWith ("_") ||
      name == "value__"
  }

  // does current member can be accessed from inside given type
  public CanAccess (source : TypeInfo) : bool
  {
    IsPublic
    || IsInternal && source is TypeBuilder
    || ({
      mutable result = false;
      for (mutable tc = source; tc != null && !result; tc = tc.DeclaringType)
        result = tc.Equals (declaring_type) || IsProtected && Option.IsSome (tc.SuperType (declaring_type));
      result
    })
  }

  public OverloadName : string
  {
    get { Util.ice ("OverloadName not implemented for non-external members") }
  }

  public MemberType : System.Reflection.MemberTypes {
    get {
      match (kind) {
        | MemberKind.Field => MemberTypes.Field
        | MemberKind.Method (meth) =>
            match (meth.GetFunKind ()) {
              | FunKind.Constructor
              | FunKind.StaticConstructor => MemberTypes.Constructor
              | _ => MemberTypes.Method
            }
        | MemberKind.Property => MemberTypes.Property
        | MemberKind.Type => MemberTypes.NestedType
        | MemberKind.Event => MemberTypes.Event
      }
    }
  }

  public GetMemType () : MType
  {
    ty
  }

  [Nemerle.OverrideObjectEquals]
  public Equals (o : IMember) : bool
  {
    id == o.GetHashCode ()
  }

  public GetKind () : MemberKind
  {
    kind
  }

  public Location : Location
  {
    get { loc }
  }

  public GetModifiers () : Modifiers
  {
    modifiers
  }
  
  public IsStatic : bool
  {
    get { attributes %&& NemerleAttributes.Static }
  }

  public IsPublic : bool
  {
    get { attributes %&& NemerleAttributes.Public }
  }

  public IsPrivate : bool
  {
    get { attributes %&& NemerleAttributes.Private }
  }

  public IsProtected : bool
  {
    get { attributes %&& NemerleAttributes.Protected }
  }

  public IsInternal : bool
  {
    get { attributes %&& NemerleAttributes.Internal }
  }
  
  public virtual HasBeenUsed : bool
  {
    get { m_has_been_used }
    set {
      m_has_been_used = m_has_been_used || value;
      declaring_type.HasBeenUsed = value
    }
  }
  
  internal SetInstanceUsed () : void {
    m_has_been_used = true;
  }

  public override GetHashCode () : int
  {
    id
  }

  public virtual GetHandle () : System.Reflection.MemberInfo
  {
    assert (false)
  }

  public MarkWithSpecialName () : void
  {
    attributes |= NemerleAttributes.SpecialName
  }

  static count_access (attrs : NemerleAttributes) : int {
    if (attrs %&& NemerleAttributes.Public) 4
    else if (attrs %&& NemerleAttributes.Protected) 2
    else if (attrs %&& NemerleAttributes.Internal) 3
    else if (attrs %&& NemerleAttributes.Private) 1
    else 1 // default to private
  }
  
  protected static UpdateEmbeddedModifiers (parent : NemerleAttributes, child : ref NemerleAttributes) : void
  {
    when (child %&& NemerleAttributes.Static && !(parent %&& NemerleAttributes.Static))
      Message.Error ("accessor cannot be static if containing entity isn't");

    // copy access modifiers only if child has none
    child |=
      (if (child %&& NemerleAttributes.AccessModifiers)
         parent %& ~NemerleAttributes.AccessModifiers;
       else
         parent);

    when (count_access (parent) < count_access (child))
      Message.Error ($ "accessor is more accessible than containing entity")
  }

  public override ToString () : string
  {
    DescribeMember (this)
  }

  internal static DescribeMember (m : IMember) : string
  {
    def declaring = m.DeclaringType;
    
    if (declaring != null &&
        declaring.GetConstantObject () : object == m)
      "constructor of constant variant " + declaring.FullName
    else {
      def full_name =
        if (declaring == null)
          m.Name
        else
          declaring.FullName + "." + m.Name;
      
      def kind =
        match (m.GetKind ()) {
          | MemberKind.Field => "field"
          | MemberKind.Method => "method"
          | MemberKind.Property => "property"
          | MemberKind.Type => "type"
          | MemberKind.Event => "event"
        };
      match (m.GetKind ()) {
        | MemberKind.Method (m) =>
          def parms = List.Map (m.GetParameters (), 
                                fun (p : Fun_parm) { 
                                    p.name + " : " + p.ty.ToString () 
                                });
          def ret_ty = m.ReturnType.ToString ();
          kind + " " + full_name + "(" + NString.Concat (", ", parms) + ") : " + ret_ty
        | _ =>
          kind + " `" + full_name + "'"
      }
    }
  }

  public abstract CheckAttributes () : void;
  
  // make sure only a valid subset of NemerleAttributes has been used
  protected check_for_invalid_attr (attr : NemerleAttributes, attr_name : string) : void
  {
    when (attributes %&& attr)
      Message.FatalError (loc, "invalid attribute `" + attr_name +
                           "' specified for " + ToString ())                           
  }
  
  /**
   * Performs the attributes checks that are common to methods and properties
   */
  protected check_method_like_attributes (title : string,
                                          title_plural : string) : void
  {                                               
    check_for_invalid_attr (NemerleAttributes.Mutable, "mutable");
    check_for_invalid_attr (NemerleAttributes.Struct, "struct");
    check_for_invalid_attr (NemerleAttributes.Macro, "macro");
    check_for_invalid_attr (NemerleAttributes.Volatile, "volatile");

    // check the access attributes for consistency
    match (TypeBuilder.CheckAccessAttributes (attributes)) {
      | Some (msg) => Message.FatalError (loc, $"$msg for $title: $(this)")
      | _ => ()
    }
    
    def mem_is_sealed = attributes %&& NemerleAttributes.Sealed;
    def mem_is_override = attributes %&& NemerleAttributes.Override;
    def mem_is_virtual = attributes %&& NemerleAttributes.Virtual;    
    def mem_is_new = attributes %&& NemerleAttributes.New;
    def mem_is_abstract = attributes %&& NemerleAttributes.Abstract;                

    // make sure 'virtual', 'new' and 'override' never appear at once 
    when (mem_is_override && mem_is_new)
      Message.FatalError (loc, $"both `override' and `new' attributes specified"
                           " for $title: $(this)");

    // only allow 'override' methods/properties/events to be 'sealed'
    when (mem_is_sealed && !mem_is_override)
      Message.FatalError (loc, $"only `override' $title are allowed to be `sealed': $(this)");
                           
    // do not allow new virtual methods/properties/events in sealed clases
    when (declaring_type.IsSealed && mem_is_virtual && !mem_is_override && !declaring_type.IsDelegate)
      Message.FatalError (loc, Name +
                           " is a new virtual member in a sealed class " +
                           declaring_type.FullName);

    // do not allow private abstract methods
            
    when (mem_is_abstract && IsPrivate)
      Message.FatalError (loc, "abstract " + title_plural + " are not allowed to be private: " +
                           ToString ());      

    // do not allow private 'virtual' or 'override' methods/properties/events
    when (mem_is_virtual && IsPrivate)
      Message.FatalError (loc, "virtual " + title_plural + " are not allowed to be private: " +
                           ToString ());      
    
    when (mem_is_override && IsPrivate)
      Message.FatalError (loc, $"override $title_plural are not allowed to be private: $(this)");
                           
    // do not allow to mix abstract with the 'override' modifier
    when (mem_is_abstract && mem_is_override)
      Message.FatalError (loc, $"abstract $title_plural are not allowed to have the "
                           "'override' modifier: $(this)");      

    // static methods/properties cannot be abstract, virtual or override
    when ((mem_is_abstract || mem_is_virtual || mem_is_override) && IsStatic)
      Message.FatalError (loc, $"static $title_plural are not allowed to have the "
                           "`virtual' or `override' modifier: $(this)");      

    // do not allow protected and protected internal methods in structures
    when (declaring_type.IsStruct && IsProtected)
    {
      def msg = if (IsInternal) "protected internal" else "protected";
          
      Message.FatalError (loc, $"$title_plural defined in a struct are not"
                           " allowed to be $msg: $(this)")
    }
  }
  
  
  protected abstract MacroTarget : MacroTargets
  { get; }

  protected abstract MacroSelfParams : list [PT.SyntaxElement]
  { get; }
    
  public virtual AddMacroAttribute (expr : PT.PExpr) : void
  {
    modifiers.macro_attrs ::= (TypesManager.AttributeMacroExpansion.Suffix (MacroTarget, MacroPhase.WithTypedMembers), expr);
    def expansion = TypesManager.AttributeMacroExpansion (MacroTarget, MacroPhase.WithTypedMembers, expr, 
                                                          MacroSelfParams, declaring_type, null);
    declaring_type.Manager.AddMacroExpansion (expansion);                                                              
  }
  
  internal virtual ProcessMacroAttributes () : void {
    declaring_type.process_attributes (MacroSelfParams, MacroTarget, MacroPhase.WithTypedMembers, modifiers, null)
  }

  internal abstract CreateEmitBuilder (emit_tb : Emit.TypeBuilder) : void;
  internal abstract Compile () : void;
}

public partial class FieldBuilder : MemberBuilder, IField
{
  public mutable field_builder : Emit.FieldBuilder;
  public mutable const_value : Literal;
  mutable has_been_assigned = false;

  public mutable required_modifiers : list[System.Type] = [];
  public mutable optional_modifiers : list[System.Type] = [];
  
  public this (par : TypeBuilder, f : PT.ClassMember.Field) {
    base (par, f);
    
    // check if field's type is valid
    match (f.ty) {
      | <[ _ ]> => Message.Error ("type inference for fields is available only when assigning literal value to them") 
      | _ => ()
    }
    
    ty = par.MonoBindType (f.ty);
        
    kind = MemberKind.Field (this);
    
    when (ty.Equals (InternalType.Void))
      Message.Error ($"$this has void type, which is not allowed");
    
    ty.CheckAccessibility (this, accessibility);
  }

  public IsMutable : bool
  {
    get { attributes %&& NemerleAttributes.Mutable }
  }

  public IsVolatile : bool
  {
    get { attributes %&& NemerleAttributes.Volatile }
  }
  
  public IsLiteral : bool
  {
    get
    {
      !IsMutable && IsStatic && const_value != null
    }
  }

  public HasBeenAssigned : bool
  {
    get { has_been_assigned }
    set {
      HasBeenUsed = HasBeenUsed || value;
      has_been_assigned = has_been_assigned || value;
    }
  }
  
  public GetValue () : Literal
  {
    assert (IsLiteral, Name);
    const_value
  }

  public GetFieldInfo () : Emit.FieldBuilder
  {
    assert (field_builder != null, Name);
    field_builder
  }
  
  override public GetHandle () : System.Reflection.MemberInfo
  {
    assert (field_builder != null, Name);
    field_builder
  }
  
  /**
   * Performs attributes checks for field definition.
   */
  public override CheckAttributes () : void
  {
    when (declaring_type.IsInterface)
      Message.FatalError (loc, "fields cannot be defined inside interface");

    check_for_invalid_attr (NemerleAttributes.Extern, "extern");
    check_for_invalid_attr (NemerleAttributes.Abstract, "abstract");
    check_for_invalid_attr (NemerleAttributes.Virtual, "virtual");
    check_for_invalid_attr (NemerleAttributes.Sealed, "sealed");
    check_for_invalid_attr (NemerleAttributes.Override, "override");
    check_for_invalid_attr (NemerleAttributes.Struct, "struct");
    check_for_invalid_attr (NemerleAttributes.Macro, "macro");
    
    // check the access attributes for consistency
    match (TypeBuilder.CheckAccessAttributes (attributes)) {
      | Some (msg) => Message.FatalError (loc, $"$msg for $(this)")
      | _ => ()
    }

    // check for non-mutable / volatile consistency
    when (!IsMutable && IsVolatile)
      Message.FatalError (loc, $"only mutable fields are allowed to be volatile in $(this)");

    // do not allow protected and protected internal methods in structures
    when (declaring_type.IsStruct && IsProtected)
    {
      def msg = if (IsInternal) "protected internal" else "protected";
          
      Message.FatalError (loc, "fields defined in a struct are not allowed to be " +
                           msg + "in " + ToString ())
    }
  }

  protected override MacroTarget : MacroTargets
  {
    get { MacroTargets.Field }
  }
  
  protected override MacroSelfParams : list [PT.SyntaxElement]
  {
    get { [PT.SyntaxElement.TypeBuilder (declaring_type), PT.SyntaxElement.FieldBuilder (this)] }
  }
}

public partial class PropertyBuilder : MemberBuilder, IProperty
{
  is_mutable : bool;
  getter : MethodBuilder;
  setter : MethodBuilder;
  internal parms : list [MType];
  mutable parent_property : IProperty = null;
  
  internal mutable property_builder : Emit.PropertyBuilder;
  
  public this (par : TypeBuilder, f : PT.ClassMember.Property) {
    base (par, f);
    is_mutable = Option.IsSome (f.set);
    ty = par.MonoBindType (f.prop_ty);

    kind = MemberKind.Property (this);

    parms = 
      List.Map (f.dims, fun (parm : PT.Fun_parm) {
        par.MonoBindType (parm.ty)
      });

    def process_accessor (_ : option [PT.ClassMember]) {
      | Some (fn) =>
        def fn = fn :> PT.ClassMember.Function;
        Util.locate (fn.Location, UpdateEmbeddedModifiers (attributes, ref fn.modifiers.mods));
        def method = MethodBuilder (par, fn, is_property = true);
        method.MarkWithSpecialName ();
        declaring_type.AddMember (method);
        method
      | None => null
    }
    getter = process_accessor (f.get);
    setter = process_accessor (f.set);

    ty.CheckAccessibility (this, accessibility);

    // we'll check for the get/set usage methods instead
    m_has_been_used = true
  }

  public IsMutable : bool
  {
    get { is_mutable }
  }

  public IsIndexer : bool
  {
    get
    {
      !(parms is [])
    }
  }
  
  public override HasBeenUsed : bool {
    set { 
      base.HasBeenUsed = value;
      // FIXME: only one of them should be used, but typing knows it a 
      //        little bit too late
      when (setter != null) setter.HasBeenUsed = value;
      when (getter != null) getter.HasBeenUsed = value;
    }
  }
  
  public GetPropertyInfo () : Emit.PropertyBuilder
  {
    assert (property_builder != null);
    property_builder
  }

  public GetGetter () : IMethod
  {
    if (getter == null && parent_property != null)
      parent_property.GetGetter ()
    else
      getter
  }
  
  public GetSetter () : IMethod
  {
    if (setter == null && parent_property != null)
      parent_property.GetSetter ()
    else
      setter
  }

  public UpdateParentProperty (parent_property : IProperty) : void
  {
    this.parent_property = parent_property;
  }
  
  override public GetHandle () : System.Reflection.MemberInfo
  {
    assert (property_builder != null);
    property_builder
  }
  
  /**
   * Performs attributes checks for properties definitions.
   */
  public override CheckAttributes () : void
  {
    // make sure no static indexers get defined
    when (IsStatic && IsIndexer)
      Message.FatalError (loc, "indexer properties are not allowed to be static in " +
                           ToString ());
    
    // most of the checks are common with the methods and events:
    check_method_like_attributes ("property", "properties")
  }

  protected override MacroTarget : MacroTargets
  {
    get { MacroTargets.Property }
  }
  
  protected override MacroSelfParams : list [PT.SyntaxElement]
  {
    get { [PT.SyntaxElement.TypeBuilder (declaring_type), PT.SyntaxElement.PropertyBuilder (this)] }
  }
}

public partial class MethodBuilder : MemberBuilder, IMethod
{
  mutable fun_kind : FunKind;
  // FIXME: make these private
  public mutable fun_header : Fun_header;
  
  public mutable method_builder : System.Reflection.Emit.MethodBuilder;
  public mutable ctor_builder : System.Reflection.Emit.ConstructorBuilder;

  internal mutable overridden_method : IMethod;

  is_var_args : bool;

  public GetMethodBase () : MethodBase
  {
    def res =
      if (method_builder == null)
        (ctor_builder : MethodBase)
      else
        method_builder;
    assert (res != null);
    res
  }

  public override GetHandle () : MemberInfo
  {
    GetMethodBase ()
  }

  public GetConstructorInfo () : Emit.ConstructorBuilder 
  {
    assert (ctor_builder != null);
    ctor_builder
  }
  
  public GetMethodInfo () : Emit.MethodBuilder
  {
    assert (method_builder != null, declaring_type.FullName + "." + Name);
    method_builder
  }

  public RunBodyTyper () : void
  {
    match (fun_header.body) {
      | FunBody.Parsed => _ = Typer (this)
      | _ => ()
    }
  }
  
  public HasAbstractBody : bool
  {
    get { (fun_header.body is FunBody.Abstract) }
  }
  
  public Body : PT.PExpr {
    get {
      match (fun_header.body) {
        | FunBody.Parsed (e) => e
        | _ =>
          Message.FatalError ("No parsetree body accessible")
      }
    }
    
    set { fun_header.body = FunBody.Parsed (value) }
  }

  public IsFinal : bool
  {
    get { attributes %&& NemerleAttributes.Sealed }
  }
  
  public IsVarArgs : bool
  {
    get { is_var_args }
  }

  public this (par : TypeBuilder, f : PT.ClassMember.Function)
  {
    this (par, f, false)
  }
  
  public this (par : TypeBuilder, f : PT.ClassMember.Function, is_property : bool)
  {
    base (par, f);

    when (LexerBase.IsOperator (name))
      unless (IsStatic && IsPublic)
        Message.Error ($"User defined operator '$(DeclaringType).$(name)'"
                       " must be declared static and public");
    
    // convert +, -, /= name to op_Addition equivalents
    match (OperatorLongName (this.name, List.Length (f.header.parms))) {
      | Some (long) =>
        this.name = long
      | None => ()
    };
    
    // both override and abstract imply virtual
    when (attributes %&& (NemerleAttributes.Override 
                          %| NemerleAttributes.Abstract))
      attributes |= NemerleAttributes.Virtual;

    def (tenv', typarms) =
        declaring_type.BindTyparms (f.header.typarms);

    // HACK HACK HACK!!!
    def typarms =
      if (declaring_type.forced_typarms != null) {
        def tmp = declaring_type.forced_typarms;
        declaring_type.forced_typarms = null;
        tmp
      } else typarms;
   
    mutable types_to_check = [];

    foreach (tv in typarms)
      foreach (t in tv.Constraints)
        types_to_check = t :: types_to_check;
        
    def bind (t : PT.PExpr) {
      when (t is <[ _ ]>)
        Message.Error (t.Location, "type inference on global methods is not yet supported");
      def ty = par.MonoBindType (tenv', t);
      types_to_check ::= ty;
      ty
    }

    def mkparm (p : PT.Fun_parm) : Fun_parm {
      def name = p.ParsedName;
      def has_default =
        p.modifiers.custom_attrs.Exists (fun (x) {
          x is <[ System.ComponentModel.DefaultValueAttribute ($_) ]>
        });
      def (kind, ty) =
        match (p.ty) {
          | <[ ref $t ]> => (ParmKind.Ref, MType.Ref (bind (t)))
          | <[ out $t ]> => (ParmKind.Out, MType.Out (bind (t)))
          | <[ _ ]> when has_default => (ParmKind.Normal, null)
          | <[ $t ]> => (ParmKind.Normal, bind (t))
        }
      def fp =
        Fun_parm (loc = p.loc, name = name.Id, color = name.color, ty = ty, 
                  modifiers = p.modifiers, kind = kind);

      fp.GetDefaultValueFromModifiers (par);
      fp
    }

    mutable fun_body = f.body;

    match (fun_body) {
      | Parsed (e) =>
        body_location = e.Location;
      | _ => {}
    }
    
   /* We need to convert things like:
        foo (x : int * int) : int { bar }
            ===>
        foo (_1 : int, _2 : int) : int { def x = (_1, _2); bar }
      To be consistent. */

    def parms =
      match (f.header.parms) {
        | [ PT.Fun_parm where (ty = <[ @* (..$_) ]>) as parm ] when !is_property =>
          def types = (par.MonoBindType (tenv', parm.ty) :> MType.Tuple).args;
          mutable modifiers = parm.modifiers;
          def mkdummyparm (t : TyVar) {
            // make the first argument intercept custom attributes
            def mods = modifiers;
            modifiers = Modifiers.Empty;
            Fun_parm (loc = parm.loc, 
                      name = Util.tmpname ("tupled"),
                      color = MacroColorizer.Color,
                      kind = ParmKind.Normal,
                      ty = t, 
                      modifiers = mods)
          };
          def parms = List.Map (types, mkdummyparm);
          def refs = List.Map (parms, fun (fp : Fun_parm) {
            PT.PExpr.Ref (PT.Name (fp.name))
          });
          match (fun_body) {
            | FunBody.Parsed (body) =>
              fun_body = FunBody.Parsed (<[ { def $(parm.ParsedName : name) =
                ( .. $refs ); $body } ]>)
            | _ => Util.ice ()
          };
          parms

        | _ => List.Map (f.header.parms, mkparm)
      };

    when (! parms.IsEmpty) {
      def is_params (parm : Fun_parm) {
        List.Exists (parm.modifiers.custom_attrs, 
          fun (e) {
            | <[ System.ParamArrayAttribute ]>
            | <[ System.ParamArrayAttribute () ]>
            | <[ System.ParamArray ]>
            | <[ System.ParamArray () ]> => true
            | _ => false
          })
      }
      def (till_last, last) = List.DivideLast (parms);
      
      when (is_params (last)) {
        match (last.ty.Fix ()) {
          | MType.Array (_, 1) => is_var_args = true
          | _ =>
            Message.Error ("only single-dimensional arrays are allowed in `params' parameters")
        }
      }

      when (List.Exists (till_last, is_params))
        Message.Error ("`params' is allowed only on last parameter");
    }

    fun_kind =
      match ((f.kind, this.name)) {
        | (FunKind.Method ([]), ".ctor") when IsStatic =>
          this.m_has_been_used = true;
          this.name = ".cctor";
          FunKind.StaticConstructor ()
          
        | (FunKind.Method ([]), ".ctor")         =>
          when (par.IsValueType && parms.IsEmpty)
            Message.Error ("explicit parameterless constructor in value type is not allowed");
          FunKind.Constructor ()
                
        | (FunKind.Method ([]), _) when IsStatic => FunKind.Function ()
        | (FunKind.Constructor, _) when IsStatic =>
          this.m_has_been_used = true;
          this.name = ".cctor";
          FunKind.StaticConstructor ()
          
        | (FunKind.Method (_ :: _), ".ctor") =>
          Message.Error ("constructor cannot implement anything");
          FunKind.Constructor ()
          
        | (FunKind.Method (_ :: _), _) when IsStatic =>
          Message.Error ("static function cannot implement anything");
          FunKind.Function ()
          
        | (FunKind.Method (_ :: _ as impl), name) when is_property =>
          // update names of explicitly implemented accessor methods
          def prefix = if (name.StartsWith ("get_")) "get_" else "set_";
          def newimpl = List.Map (impl, fun (m) {
            | PT.PExpr.Member (obj, mem) =>
              def name = mem.GetName ();
              PT.PExpr.Member (m.Location, obj, PT.Splicable.Name (name.NewName (prefix + name.Id)))

            | x => Message.Error ("implemented property must be of Interface.ProprtyName form"); x
          });
          FunKind.Method (newimpl)

        | (x, _) => x
      };

    fun_header = Fun_header (
      ret_type = bind (f.header.ret_type),
      typarms = typarms,
      name = this.name,
      parms = parms,
      tenv = tenv',
      loc = f.header.loc);

    fun_header.body = fun_body;

    foreach (parm in parms) {
      when (parm.ty.Fix () is MType.Void)
        Message.Error ($ "method `$name' has void argument");
    }
      
    ty = MType.ConstructFunctionType (fun_header);
    
    kind = MemberKind.Method (this);

    // check it at the end, since error reporting routine there
    // needs us to be initialized to name us
    foreach (t in types_to_check)
      t.CheckAccessibility (this, accessibility);

    // don't warn about the unused `Main' methods...
    when (IsStatic && name == "Main")
      HasBeenUsed = true
  }

  public GetHeader () : Fun_header
  { fun_header }

  public GetParameters () : list [Fun_parm] 
  {
    fun_header.parms
  }

  public ReturnType : TyVar
  {
    get { fun_header.ret_type }
  }
  
  public GetFreshType () : MType * list [TyVar]
  {
    def (subst, vars) = Subst.Fresh (fun_header.typarms);
    (subst.MonoApply (GetMemType ()), vars)
  }

  public GetFunKind () : FunKind
  {
    fun_kind
  }

  internal SetFunKind (kind : FunKind) : void
  {
    fun_kind = kind
  }
  
  static OperatorLongName (op : string, params_amount : int) : option [string]
  {
    if (params_amount == 2) {
      match (op) {
        | "+" => Some ("op_Addition")
        | "-" => Some ("op_Subtraction")
        | "*" => Some ("op_Multiply")
        | "/" => Some ("op_Division")
        | "%" => Some ("op_Modulus")
        | "%^" => Some ("op_ExclusiveOr")
        | "%&" => Some ("op_BitwiseAnd")
        | "%|" => Some ("op_BitwiseOr")
        | "^" => Some ("op_ExclusiveOr")
        | "&" => Some ("op_BitwiseAnd")
        | "|" => Some ("op_BitwiseOr")
        | "&&" => Some ("op_LogicalAnd")
        | "||" => Some ("op_LogicalOr")
        | "=" => Some ("op_Assign")
        | "<<" => Some ("op_LeftShift")
        | ">>" => Some ("op_RightShift")
        // | ">>*" => Some ("op_SignedRightShift")
        // | ">>&" => Some ("op_UnsignedRightShift")
        | "==" => Some ("op_Equality")
        | ">" => Some ("op_GreaterThan")
        | "<" => Some ("op_LessThan")
        | "!=" => Some ("op_Inequality")
        | ">=" => Some ("op_GreaterThanOrEqual")
        | "<=" => Some ("op_LessThanOrEqual")
        | "*=" => Some ("op_MultiplicationAssignment")
        | "-=" => Some ("op_SubtractionAssignment")
        | "^=" => Some ("op_ExclusiveOrAssignment")
        | "<<=" => Some ("op_LeftShiftAssignment")
        | "%=" => Some ("op_ModulusAssignment")
        | "+=" => Some ("op_AdditionAssignment")
        | "&=" => Some ("op_BitwiseAndAssignment")
        | "|=" => Some ("op_BitwiseOrAssignment")
        | "," => Some ("op_Comma")
        | "/=" => Some ("op_DivisionAssignment")
        | _ => None ()
      }
    }
    else if (params_amount == 1) {
      match (op) {
        | ":" => Some ("op_Implicit")
        | ":>" => Some ("op_Explicit")
        | "+" when params_amount == 1 => Some ("op_UnaryPlus")
        | "-" => Some ("op_UnaryNegation")        
        | "--" => Some ("op_Decrement")
        | "++" => Some ("op_Increment")
        | "~" => Some ("op_OnesComplement")
        | "!" => Some ("op_LogicalNot")
        | _ => None ()
      }
    }
    else None ()
  }
  
  /**
   * Performs attributes checks for method definitions.
   */
  public override CheckAttributes () : void
  {
    when (declaring_type.IsInterface) {
      unless (HasAbstractBody)
        Message.Error (loc, "interface method cannot have body");
      when (GetFunKind () is FunKind.Method (_ :: _))        
        Message.Error (loc, "interface method cannot implement anything");
    }

    // only allow abstract and extern methods to have no bodies
    def bodyless = attributes %&& (NemerleAttributes.Abstract | NemerleAttributes.Extern);

    when (HasAbstractBody && !bodyless)
      Message.FatalError (loc, "missing body of a non-abstract and non-extern method in " +
                           ToString ());

    when (attributes %&& NemerleAttributes.Extern && !HasAbstractBody)
      Message.Error (loc, "method with `extern' modifier cannot have body");
                           
    // most of the checks are common with the properties and events:
    check_method_like_attributes ("method", "methods");
  }
      
      
  public BuiltinKind : BuiltinMethodKind
  {
    get { BuiltinMethodKind.NotBuiltin () }
  }
  
  protected override MacroTarget : MacroTargets
  {
    get { MacroTargets.Method }
  }
  
  protected override MacroSelfParams : list [PT.SyntaxElement]
  {
    get { [PT.SyntaxElement.TypeBuilder (declaring_type), PT.SyntaxElement.MethodBuilder (this)] }
  }
  
  internal override ProcessMacroAttributes () : void {
    base.ProcessMacroAttributes ();
    foreach (p : Fun_parm in GetParameters ())
      declaring_type.process_attributes ([PT.SyntaxElement.TypeBuilder (declaring_type), PT.SyntaxElement.MethodBuilder (this), 
                                          PT.SyntaxElement.ParameterBuilder (p)],
                                         MacroTargets.Parameter, MacroPhase.WithTypedMembers, p.modifiers, this);
  }
}

public partial class EventBuilder : MemberBuilder, IEvent
{
  adder : MethodBuilder;
  remover : MethodBuilder;
  
  internal mutable event_builder : Emit.EventBuilder;
  internal storage_field : FieldBuilder;
  
  public this (par : TypeBuilder, f : PT.ClassMember.Event) {
    base (par, f);
    
    ty = par.MonoBindType (f.ty);

    kind = MemberKind.Event (this);

    // prevent closurising `this' 
    def self = this;

    when (f.field != null) {
      def update_mfunction (meth : PT.ClassMember.Function, which_accessor : string) {
        if (declaring_type.IsInterface)
          meth.body = FunBody.Abstract ()
        else {
          def lock_expr =
            if (IsStatic) <[
              typeof ($(declaring_type.ParsedName : name))
            ]>
            else <[ this ]>;
          def field_name = f.field.ParsedName;

          // according to C# spec, we have to add locks
          // http://www.jaggersoft.com/csharp_standard/17.7.1.htm
          def bd =
            if (which_accessor == "add")
              <[ lock ($lock_expr) { $(field_name : name) += $(field_name.NewName ("value") : name) } ]>;
            else
              <[ lock ($lock_expr) { $(field_name : name) -= $(field_name.NewName ("value") : name) } ]>;
          meth.body = FunBody.Parsed (bd)
        }
      };
      
      unless (declaring_type.IsInterface) {
        when (IsStatic) f.field.modifiers.mods |= NemerleAttributes.Static;
        storage_field = FieldBuilder (par, f.field);
        declaring_type.AddMember (storage_field);
      }
      update_mfunction (f.add, "add");
      update_mfunction (f.remove, "remove");
    }
      
    def make_method (mfunc : PT.ClassMember.Function) {
      UpdateEmbeddedModifiers (self.attributes, ref mfunc.modifiers.mods);
      def meth' = MethodBuilder (self.declaring_type, mfunc);
      self.declaring_type.AddMember (meth');
      meth'
    };

    adder = make_method (f.add);
    remover = make_method (f.remove);
    remover.HasBeenUsed = true;

    ty.CheckAccessibility (this, accessibility);
  }

  public GetEventInfo () : EventInfo
  {
     assert (false);
  }

  public override GetHandle () : MemberInfo
  {
    // for some reason !(event_builder)...
    /// lame spec: SRE.EventBuilder is not MemberInfo or event EventInfo
    /// but we return null to do not spoil API
    null
  }

  public GetAdder () : MethodBuilder
  {
    adder
  }
  
  public GetRemover () : MethodBuilder
  {
    remover
  }
  
  /**
   * Performs attributes checks for event definition.
   */
  public override CheckAttributes () : void
  {
    // most of the checks are common with the properties and methods:
    check_method_like_attributes ("event", "events")
  }
  
  protected override MacroTarget : MacroTargets
  {
    get { MacroTargets.Event }
  }
  
  protected override MacroSelfParams : list [PT.SyntaxElement]
  {
    get { [PT.SyntaxElement.TypeBuilder (declaring_type), PT.SyntaxElement.EventBuilder (this)] }
  }
}
}
