/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

namespace Nemerle.Compiler
{
  class Typer3
  {
    #region Top typer
    // one instance is created for every source top level method
    // each nested typer3 has a reference to it later
    class TopTyper {
      // some variables are moved to other locations, this is used
      // to locate them
      internal redirects : Hashtable [LocalValue, LocalValue] = Hashtable ();

      // If given global static function is used as a first class
      // value a special proxy is created. These proxies can be shared
      // inside a class so we reuse them.
      internal static_proxies : Hashtable [IMethod, IField] = Hashtable ();
      
      // the type we're sitting in, it's the same for a method
      // and all its local functions
      internal current_type : TypeBuilder;

      internal this (t : TypeBuilder)
      {
        current_type = t;
      }
    }

    top_typer : TopTyper;

    Redirects : Hashtable [LocalValue, LocalValue]
    {
      get { top_typer.redirects }
    }

    StaticProxies : Hashtable [IMethod, IField]
    {
      get { top_typer.static_proxies }
    }

    CurrentType : TypeBuilder
    {
      get { top_typer.current_type }
    }
    #endregion

    #region Private fields
    // this is either CurrentType for toplevel method (or a local method
    // translated to static method), or _N_lambda736393 kind of stuff
    // for first class local functions
    local_function_type : TypeBuilder;

    // ID of the label marking beginning of the function, used in 
    // expansion of SelfTailCall.
    mutable start_label : int;

    // typer of our typer method or null for top-level methods
    parent : Typer3;
    
    // current method we're working on
    the_method : MethodBuilder;

    // this can be Fun_header of a function expanded to a loop:
    current_local_fun : Fun_header;
    // and this not, this always corresponds to a method in IL:
    current_method : Fun_header;

    // substitution applied to all types generated in a local function
    // this is done with FixType
    mutable current_subst : Subst;

    // if we have a closure, how to call it?
    mutable current_closure : LocalValue;

    // how to get our parent's closures, at least one of these is null
    // both are null if there are no parent closures
    // shall we go through this:
    mutable closure_fields : Hashtable [int, IField];
    // or just through parms
    mutable closure_parms : Hashtable [int, LocalValue];

    /* Given:
         static foo[A,B] () { 
           def bar[C,D] () {
             def qux[E,F]() { }
           }
         }
       for qux typer it will hold [E,F,C,D,A,B]
     */
    accumulated_typarms : list [StaticTyVar];
    #endregion

    
    #region Entry points
    public this (meth : MethodBuilder)
    {
      this (null, meth);
    }


    this (parent : Typer3, meth : MethodBuilder)
    {
      the_method = meth;
      this (parent, meth.GetHeader ());
      local_function_type = meth.DeclaringType;
      current_method = meth.GetHeader ();
    }


    this (parent : Typer3, fn : Fun_header)
    {
      current_local_fun = fn;
      
      if (parent != null) {
        local_function_type = parent.local_function_type;
        current_method = parent.current_method;

        if (fn.typarms.IsEmpty) {
          accumulated_typarms = parent.accumulated_typarms;
        } else {
          // we have already copied all accumulated_typarms,
          // so don't add copies
          def additional_typarms_count =
            fn.typarms.Length - parent.accumulated_typarms.Length;
          accumulated_typarms = 
            fn.typarms.FirstN (additional_typarms_count) + 
            parent.accumulated_typarms;
        }
        
        top_typer = parent.top_typer;
      } else {
        current_method = current_local_fun;
        accumulated_typarms = fn.typarms;
        top_typer = TopTyper (the_method.DeclaringType);
      }

      this.parent = parent;
    }


    public Run () : void
    {
      Util.locate (current_local_fun.loc, {
        //Message.Debug ($"T3::run: $(current_local_fun.name)");
        start_label = Util.next_id ();

        def initializers = PrepareProlog ();

        match (current_local_fun.body) {
          | FunBody.Typed (body) =>
            // FIXME?: Typed3 
            def body = BuildRevSequence (Walk (body) :: initializers);
            def expr = TExpr.Label (body.Type, start_label, body);
            def expr =
              if (current_subst == null || current_subst.IsEmpty)
                expr
              else
                SubstExpr (expr);
            current_local_fun.body = FunBody.Typed (expr);
            when (Options.ShouldDump (current_local_fun))
              Message.Debug ($ "after T3: $CurrentType.$(current_local_fun.name) "
                               "-> $(current_local_fun.ret_type) : $body\n");

            // run this for toplevel methods
            when (the_method != null) {
              def t4 = Typer4 (the_method);
              t4.Run ();
            }
            
          | _ => assert (false)
        }
      })
    }
    #endregion


    #region Utilities
    // expects reversed list
    internal static BuildRevSequence (exprs : list [TExpr]) : TExpr
    {
      match (exprs) {
        | [] =>
          TExpr.Literal (InternalType.Void, Literal.Void ())
        | x :: xs =>
          def loop (acc, l) {
            match (l) {
              | [] => acc
              | TExpr.DefValIn (name, val, null) :: xs =>
                loop (TExpr.DefValIn (acc.Type, name, val, acc), xs)
              | x :: xs =>
                loop (TExpr.Sequence (acc.Type, x, acc), xs)
            }
          }
          loop (x, xs)
      }
    }


    static internal SingleMemberLookup (tb : TypeInfo, name : string) : IMember
    {
      match (tb.LookupMember (name)) {
        | [mem] => mem
        | [] => Util.ice ()
        | lst =>
          match (lst.Filter (fun (mem) { mem.DeclaringType.Equals (tb) })) {
            | [mem] => mem
            | lst => Util.ice ($ "mulitple members $name in $tb $lst")
          }
      }
    }


    IsTopLevelFun : bool
    {
      get { local_function_type.Equals (CurrentType) }
    }


    /** Just a shorthand for TExpr.LocalRef.  */
    static PlainRef (decl : LocalValue) : TExpr
    {
      assert (decl != null);
      TExpr.LocalRef (decl.Type, decl)
    }


    static StaticRef (mem : IMember) : TExpr
    {
      assert (mem is IMethod || mem is IField);
      TExpr.StaticRef (mem.GetMemType (),
                       mem.DeclaringType.GetMemType (), 
                       mem, 
                       [])
    }

   
    static internal CheckedConversion (expr : TExpr, target_type : TyVar) : TExpr
    {
      TExpr.TypeConversion (target_type, expr, target_type,
                            ConversionKind.IL (true));
    }


    /** Given a call with [parms] to [fh] return a list of parameters
        that should be passed along with any initialization code.  */
    TupleParms (fh : Fun_header, parms : list [Parm], closure_parm_count = 0) 
               : list [Parm] * list [TExpr]
    {
      def fh_len = fh.parms.Length - closure_parm_count;
      if (parms.Length == 1 && fh_len > 1) {
        def tupled = List.Hd (parms).expr;
        def cache =
          LocalValue (current_local_fun, Util.tmpname ("tupl_cache"),
                      tupled.Type, LocalValue.Kind.Plain (),
                      is_mutable = false);
        cache.Register ();
        cache.UseFrom (current_local_fun);

        def types =
          match (tupled.MType) {
            | Tuple (lst) => lst
            | _ => Util.ice ()
          }

        def len = types.Length;
        Util.cassert (fh_len == len);
        mutable pos = -1;
        def parms = types.Map (fun (ty) {
          pos++;
          Parm (TExpr.TupleIndexer (ty, PlainRef (cache), pos, len))
        });
         
        (parms, 
         [TExpr.DefValIn (cache, tupled, null)])

      } else if (parms.Length > 1 && fh_len == 1) {
        def types = parms.Map (fun (fp) { 
          assert (fp.required_type != null);
          fp.required_type 
        });
        def exprs = parms.Map (fun (fp) { 
          Typer.ImplicitCast (fp.expr, fp.required_type) 
        });
        def parm = TExpr.Tuple (MType.Tuple (types), exprs);
        ([Parm (parm)], [])

      } else {
        Util.cassert (parms.Length == fh_len,
                      $ "parms length mismatch, $(fh.name) "
                        "$parms $(fh.parms)");
        (parms, [])
      }
    }


    WithCached (e : TExpr, f : TExpr -> TExpr) : TExpr
    {
      def needs_cache =
        match (e) {
          | TExpr.LocalRef             
          | TExpr.StaticRef            
          | TExpr.ConstantObjectRef    
          | TExpr.Literal              
          | TExpr.This => false
          | _ => true
        }

      if (needs_cache) {
        def cache =
          LocalValue (current_local_fun, Util.tmpname ("cache"),
                      e.Type, LocalValue.Kind.Plain (),
                      is_mutable = false);
        cache.Register ();
        cache.UseFrom (current_local_fun);
        def body = f (PlainRef (cache));
        TExpr.DefValIn (body.Type, cache, e, body)
      } else f (e)
    }


    static IsObject (t : TyVar) : bool
    {
      t.Fix ().Equals (InternalType.Object)
    }
    #endregion


    #region Typarms handling
    CopyFunTyparms (add_class_tyvars = false) : list [StaticTyVar] * Subst
    {
      mutable subst = null;
      mutable new_tp = [];

      def typarms =
        if (add_class_tyvars) 
          CurrentType.GetTyparms () + accumulated_typarms
        else 
          accumulated_typarms;

      when (! typarms.IsEmpty) {
        (subst, new_tp) = 
          StaticTyVar.CopyList (typarms);
      }

      def current_subst =
        if (true || add_class_tyvars)
          this.current_subst
        else {
          // skip any used-as-1st-class methods as they have additional
          // subst on type's tyvars
          def loop (typer) {
            if (typer.current_method.usage == FunctionUsage.UsedAsFirstClass)
              loop (typer.parent)
            else
              typer.current_subst
          }
          loop (this)
        }

      if (subst == null) 
        subst = current_subst;
      else if (current_subst == null) {}
      else {
        when (!add_class_tyvars)
          foreach (tv in CurrentType.GetTyparms ())
            subst.Add (tv, MType.TyVarRef (tv));
        //Message.Debug ($ "tp=$typarms -> $new_tp, subst=$subst,current_subst=$current_subst");
        subst.Combine (current_subst);
        //Message.Debug ($ "subst=$subst");
      }

      (new_tp, subst)
    }
    

    FunAndClassTyparmsRefs () : list [TyVar]
    {
      def local_tp = CurrentType.GetTyparms () + accumulated_typarms;
      local_tp.Map (fun (x) { SubstType (MType.TyVarRef (x)) })
    }


    static FixType (t : TyVar, subst : Subst) : MType
    {
      if (subst == null)
        t.DeepFix ()
      else
        subst.Apply (t.DeepFix ()).Fix ()
    }
    #endregion


    #region Local reference through closures
    /** Return code referencing closure of [hd].  */
    ClosureRef (hd : Fun_header) : TExpr
    {
      if (hd.usage == FunctionUsage.UsedJustOnce) {
        ClosureRef (hd.decl.DefinedIn)
      } else if (current_method.is_in_closure_of != 0) {
        // Message.Debug ($ "lookup $(hd.name) from $(current_method.name)");
        Util.cassert (hd.id == current_method.is_in_closure_of ||
                      (current_method.decl != null && 
                       current_method.decl.DefinedIn.id == hd.id));
        TExpr.This (local_function_type.GetMemType ())
      } else if (hd.id == current_method.id) {
        PlainRef (current_closure)
      } else if (closure_fields == null) {
        Util.cassert (closure_parms != null);
        Util.cassert (closure_parms.Contains (hd.id), $ "no clo parm for $(hd.name)");
        def decl = closure_parms [hd.id];
        decl.UseFrom (current_local_fun);
        PlainRef (decl)
      } else {
        Util.cassert (closure_fields != null);
        Util.cassert (closure_fields.Contains (hd.id), $ "no clo for $(hd.name) from $(current_local_fun.name)");
        def field = closure_fields [hd.id];
        // Message.Debug ($"cloref: $(hd.name) from $(current_local_fun.name) $(field.GetMemType()) $field");
        TExpr.FieldMember (field.GetMemType (),
                           TExpr.This (local_function_type.GetMemType ()),
                           field)
      }
    }


    LocalRef (decl : LocalValue, for_store : bool = false) : TExpr
    {
      // Message.Debug ($"local ref $decl $(decl.GetHashCode ()) $for_store");
      def decl =
        if (Redirects.Contains (decl))
          Redirects [decl]
        else decl;
      def is_this = decl.ValKind is LocalValue.Kind.ClosurisedThisPointer;

      def res =
        if (decl.InClosure)
          if (!for_store && is_this && IsTopLevelFun)
            TExpr.This ()
          else {
            assert (decl.ClosureField != null);
            def clo_ref = ClosureRef (decl.DefinedIn);
            TExpr.FieldMember (clo_ref.MType.TypeOfMember (decl.ClosureField),
                               clo_ref, decl.ClosureField)
          }
        else if (is_this)
          TExpr.This ()
        else
         PlainRef (decl);

      when (res.ty == null)
        res.ty = decl.Type;
      res
    }
    #endregion


    #region Proxies
    EmitStaticProxy (sr : TExpr.StaticRef) : TExpr
    {
      def meth = sr.mem :> IMethod;
      def can_cache =
        sr.type_parms.ForAll (IsObject) && sr.from.args.ForAll (IsObject);

      // in fact we should look for occurrences of typarms refs in
      // type_parms and from.args
      def need_method_typarms = 
        ! (can_cache || (current_local_fun.typarms.IsEmpty &&
                         CurrentType.typarms.IsEmpty));

      def need_new_type = ! can_cache || ! StaticProxies.Contains (meth);

      mutable subst = null : Subst;
      mutable builder = null;
      mutable new_tp = [];

      def fix_type (t) { FixType (t, subst) : TyVar }

      when (need_new_type) {
        def (parm_types, ret_type) = 
          Option.UnSome (sr.Type.Fix ().FunReturnTypeAndParms ());
        def fnty = FunctionType.Make (sr.Type);
        
        when (need_method_typarms) {
          (new_tp, subst) = 
            CopyFunTyparms (add_class_tyvars = true);
        }

        def name = Macros.NewSymbol ("static_proxy");
        
        // Message.Debug ($"$name : $(current_method.typarms) $(sr.Type) -> $(fix_type (fnty))");

        builder =
          CurrentType.DefineNestedType (<[ decl:
            private sealed class $(name : name) : $(fix_type (fnty) : typed)
            {
            }
          ]>, do_fixup = false);

        builder.DisableImplicitConstructor ();
        builder.forced_typarms = new_tp;
        builder.FixupDefinedClass ();
        def sinst_type = builder.GetMemType ();

        def sinst = builder.DefineAndReturn (<[ decl:
          public static single_instance : $(sinst_type : typed);
        ]>);

        builder.Define (<[ decl:
          private this () { }
        ]>);
        
        builder.Define (<[ decl:
          static this ()
          {
            $(TExpr.StaticRef (sinst_type, sinst_type, sinst, []) : typed)
              = $(name : name) ();
          }
        ]>);

        def (formal_parms, parm_refs) =
          List.Split (parm_types.Map (fun (ty) {
            def name = Macros.NewSymbol ("sp_parm");
            ( <[ parameter: $(name : name) : $(fix_type (ty) : typed) ]>,
              <[ $(name : name) ]> )
          }));

        def new_ref =
          TExpr.StaticRef (fix_type (sr.Type),
                           fix_type (sr.from) :> MType.Class,
                           meth,
                           sr.type_parms.Map (fix_type));

        def apply_name = 
          if (ret_type.Fix () is MType.Void)
            "apply_void" else "apply";
        
        builder.Define (<[ decl:
          public override $(apply_name : dyn) (.. $formal_parms) : $(fix_type (ret_type) : typed)
          {
            $(new_ref : typed) (.. $parm_refs)
          }
        ]>);

        builder.MarkWithSpecialName ();
        builder.Compile ();

        when (can_cache) {
          StaticProxies [meth] =
            SingleMemberLookup (builder, "single_instance") :> IField;
        }
      }

      if (can_cache) {
        def field = StaticProxies [meth];
        StaticRef (field)
      } else {
        def field = SingleMemberLookup (builder, "single_instance");

        if (need_method_typarms) {
          def spt = MType.Class (builder, FunAndClassTyparmsRefs ());
          TExpr.StaticRef (spt, spt, field, [])
        } else {
          StaticRef (field)
        }
      }
    }
    

    EmitDelegateProxy (expr : TExpr) : TExpr * TExpr.MethodAddress
    {
      def decl =
        match (expr.Type.Fix ()) {
          | MType.Fun (from, ret_type) as ty =>
            def parms =
              from.Fix ().GetFunctionArguments ().Map (fun (ty) {
                def name = Macros.NewSymbol ("parm");
                (<[ parameter: $(name : name) : $(ty : typed) ]>,
                 <[ $(name : name) ]>)
              });
            def (parms, parm_refs) = List.Split (parms);
            
            <[ decl: 
              private sealed class $(Macros.NewSymbol ("delegate_proxy") : name)
              {
                funptr : $(ty : typed);
                public this (fp : $(ty : typed))
                {
                  funptr = fp;
                }

                public InvokeDelegate (.. $parms) : $(ret_type : typed)
                {
                  funptr (.. $parm_refs)
                }
              }
            ]>

          | _ => assert (false)
        }

      def tb = CurrentType.DefineNestedType (decl);
      tb.MarkWithSpecialName ();
      tb.Compile ();

      def ctor = SingleMemberLookup (tb, ".ctor");

      def ctor_call =
        TExpr.Call (tb.GetMemType (), StaticRef (ctor), [Parm (expr)], false);

      def meth = SingleMemberLookup (tb, "InvokeDelegate") :> IMethod;
      
      (ctor_call, TExpr.MethodAddress (tb.GetMemType (), meth, false, []))
    }
    #endregion


    #region Function prolog
    PrepareClosureParms () : void
    {
      when (closure_parms != null &&
            (parent == null || parent.closure_parms : object != closure_parms)) {
        def len = current_local_fun.used_closures.Length;
        List.Iter2 (current_local_fun.parms.FirstN (len),
                    current_local_fun.used_closures, fun (parm, header) {
                      closure_parms [header.id] = parm.decl
                    })
      }
    }


    static PrepareEnumeratorObject (yield_type : TyVar) : Parsetree.ClassMember
    {
      <[ decl:
        class $(Macros.NewSymbol ("Enumerator") : name) :
          System.Collections.IEnumerator,
          System.Collections.Generic.IEnumerator [$(yield_type : typed)],
          System.IDisposable
        {
          public Current : $(yield_type : typed)
          {
            get { this._N_current }
          }

          public Dispose () : void
          {
            foobar
          }

          public Reset () : void
          {
            throw System.NotSupportedException ();
          }

          public MoveNext () : bool
          {
            foobar
          }
        }
      ]>
    }

    PrepareEnumerableObject (yield_type : TyVar) : Parsetree.ClassMember
    {
      def name = Macros.NewSymbol ("Enumerable");

      def has_mutable_parm = current_local_fun.parms.Exists (fun (fp) { fp.decl.IsMutable });

      def interlocked_stuff =
        if (has_mutable_parm)
          <[ true ]>
        else
          <[ System.Threading.Interlocked.CompareExchange 
             (ref _N_this_used, 1, 0) != 0 ]>;
           
      <[ decl:
        class $(name : name) :
          System.Collections.IEnumerator,
          System.Collections.Generic.IEnumerator [$(yield_type : typed)],
          System.IDisposable,
          System.Collections.IEnumerable,
          System.Collections.Generic.IEnumerable [$(yield_type : typed)]
        {
          public Current : $(yield_type : typed)
          {
            get { this._N_current }
          }

          public Dispose () : void
          {
            foobar
          }

          public Reset () : void
          {
            throw System.NotSupportedException ();
          }

          public MoveNext () : bool
          {
            foobar
          }

          mutable _N_this_used : int;
          
          public GetEnumerator () : System.Collections.Generic.IEnumerator [$(yield_type : typed)]
          {
            if ($interlocked_stuff) {
              def res = $(name : name) () : this;
              res.CopyFrom (this);
              res
            } else
              this
          }

          private NonGenericGetEnum () : System.Collections.IEnumerator 
            implements System.Collections.IEnumerable.GetEnumerator
          {
            if ($interlocked_stuff) {
              def res = $(name : name) () : this;
              res.CopyFrom (this);
              res
            } else
              this
          }

          CopyFrom (other : this) : void
          {
            _ = other; // avoid warning about unused other
            _N_this_used = 1;
          }
        }
      ]>
    }

    static RewriteTryFinally (clo_type : TypeBuilder, fh : Fun_header) : void
    {
      mutable dispose_expr = Typer.VoidLiteral ();

      def thisref = TExpr.This (clo_type.GetMemType ());

      def look_for_invalid_yield (expr : TExpr) {
        | Assign (LocalRef (decl), _) when decl.Name == "_N_current" =>
          Message.Error ("using `yield' is not allowed here");
          null
        | _ => null
      }

      def rewrite (expr : TExpr) {
        | TryFinally (body, handler) =>
          def fld = clo_type.DefineAndReturn (<[ decl: 
            mutable $(Macros.NewSymbol ("finally_needed") : name) : bool;
          ]>) :> IField;
          fld.HasBeenAssigned = true;
          fld.HasBeenUsed = true;
          def fldref = TExpr.FieldMember (InternalType.Boolean, thisref, fld);
          def for_dispose =
            TExpr.If (InternalType.Void, fldref, handler, Typer.VoidLiteral ());
          dispose_expr = TExpr.Sequence (dispose_expr.Type, for_dispose, dispose_expr);
          
          BuildRevSequence (
             [handler.Walk (look_for_invalid_yield), 
              TExpr.Assign (InternalType.Void, fldref, TExpr.FalseLiteral),
              body.Walk (rewrite),
              TExpr.Assign (InternalType.Void, fldref, TExpr.TrueLiteral)])

        | TryWith (body, _, handler) =>
          _ = body.Walk (look_for_invalid_yield);
          _ = handler.Walk (look_for_invalid_yield);
          expr

        | _ => null
      }
      
      match (fh.body) {
        | FunBody.Typed (expr) =>
          def expr = expr.Walk (rewrite);
          
          def dispose_meth =
            SingleMemberLookup (clo_type, "Dispose") :> IMethod;
          def dispose_call = 
            TExpr.Call (InternalType.Void,
                        TExpr.MethodRef (dispose_meth.GetMemType (),
                                         thisref, dispose_meth, [], true), 
                        [], false);
          def expr = TExpr.TryFault (expr.Type, expr, dispose_call);
          fh.body = FunBody.Typed (expr);
          
          def this_N_state =
            TExpr.FieldMember (InternalType.Int32, thisref,
                               SingleMemberLookup (clo_type, "_N_state") :> IField);
          dispose_expr =
            TExpr.Sequence (InternalType.Void,
              TExpr.Assign (InternalType.Void,
                            this_N_state, 
                            TExpr.Literal (InternalType.Int32,
                                           Literal.FromInt (-1))),
              dispose_expr);
          dispose_meth.GetHeader ().body = FunBody.Typed (dispose_expr);
          
        | _ => Util.ice ()
      }
    }


    SetEnumeratorBody (clo_type : TypeBuilder, subst : Subst, parm_field_names : list [string]) : void
    {
      match (current_local_fun.body) {
        | FunBody.Typed (DefValIn (_, _, DefValIn (_, _, DefFunctionsIn ([fh], _)))) =>
          def meth = SingleMemberLookup (clo_type, "MoveNext") :> MethodBuilder;
          RewriteTryFinally (clo_type, fh);
          fh.used_closures = fh.GetParents ();
          fh.is_in_closure_of = fh.id;
          PrepareForEmission (meth, fh, subst);
          def child = Typer3 (this, meth);
          child.current_subst = subst;
          child.Run ();

          def meth = SingleMemberLookup (clo_type, "Dispose") :> MethodBuilder;
          meth.GetHeader ().used_closures = fh.GetParents ();
          meth.GetHeader ().is_in_closure_of = fh.id;
          def child = Typer3 (this, meth);
          child.current_subst = subst;
          child.Run ();
          
          unless (parm_field_names.IsEmpty) {
            def meth = SingleMemberLookup (clo_type, "CopyFrom") :> MethodBuilder;
            def assigns = meth.Body :: parm_field_names.Map (fun (name) {
              <[ this . $(name : dyn) = other . $(name : dyn) ]>
            });
            meth.Body = <[ { .. $assigns } ]>;
          }

          current_local_fun.body = FunBody.Typed (PlainRef (current_closure));

        | FunBody.Typed (t)
        | _ with t = null => Util.ice ($ "oops, t=$t")
      }
    }

    
    PrepareClosure () : list [TExpr]
    {
      //Message.Debug ($"closure for $(current_local_fun.name) $(current_local_fun.closure_vars)");
      if (current_local_fun.closure_vars.IsEmpty) []
      else {
        Stats.FunctionClosures++;
        
        def (new_tp, subst) = CopyFunTyparms (add_class_tyvars = true);

        def fix_type (t) {
          if (subst == null)
            t.DeepFix ()
          else
            subst.Apply (t.DeepFix ())
        }

        def uses_yield = current_local_fun.yield_type != null;
        def is_enumerable =
          if (uses_yield)
            match (current_local_fun.ret_type.Fix ()) {
              | Class (tc, _) =>
                tc.Equals (InternalType.Generic_IEnumerable_tc) ||
                tc.Equals (InternalType.IEnumerable_tc)
              | _ => Util.ice ()
            }
          else false;

        def clo_decl = 
          if (!uses_yield)
            <[ decl:
              private sealed class $(Macros.NewSymbol ("closure") : name)
              {
                internal this () {}
              }
            ]>
          else 
            if (is_enumerable)
              PrepareEnumerableObject (fix_type (current_local_fun.yield_type));
            else 
              PrepareEnumeratorObject (fix_type (current_local_fun.yield_type));

        def clo_type = CurrentType.DefineNestedType (clo_decl, do_fixup = false);
        clo_type.forced_typarms = new_tp;
        clo_type.FixupDefinedClass ();
        
        //Message.Debug ($"acc_tp: $accumulated_typarms, cs=$current_subst");
        def clo_mtype = MType.Class (clo_type, FunAndClassTyparmsRefs ());
        current_local_fun.closure_type = clo_mtype;
        def closure_val =
          LocalValue (current_local_fun, "_N_closure", clo_mtype,
                      LocalValue.Kind.Plain (), is_mutable = false);
        closure_val.Register ();
        closure_val.UseFrom (current_local_fun);
        current_closure = closure_val;

        // FIXME: this should be moved up, so empty closures are not generated
        // but this causes ICE

        // skip non-1st class functional values
        def vars =
          current_local_fun.closure_vars.Filter (fun (decl : LocalValue) {
            match (decl.ValKind) {
              | Function (Fun_header where (usage = UsedAsFirstClass), _) => true
              | Function => false
              | _ => true
            }
          });

        mutable parm_field_names = [];

        foreach (decl in vars) {
          def name =
            if (decl.Name == "_N_current" || decl.Name == "_N_state")
              Parsetree.Name (decl.Name)
            else Macros.NewSymbol (decl.Name);
          //Message.Debug ($"clo field: $name $(fix_type (decl.Type)) tv=$new_tp");
          def ptdecl = <[ decl: 
            internal mutable 
              $(name : name) : $(fix_type (decl.Type) : typed);
          ]>;
          def fld = clo_type.DefineAndReturn (ptdecl);
          decl.ClosureField = fld :> IField;
          decl.ClosureField.HasBeenAssigned = true;
          when (is_enumerable && decl.ValKind is FunParm (_))
            parm_field_names ::= name.Id;
        }
        def ctor = SingleMemberLookup (clo_type, ".ctor");
        def ctor_ref = TExpr.StaticRef (ctor.GetMemType (), clo_mtype, ctor, []);
        def ctor_call = TExpr.Call (clo_mtype, ctor_ref, [], false);
                      
        clo_type.MarkWithSpecialName ();
        clo_type.HasBeenUsed = true;

        when (uses_yield)
          SetEnumeratorBody (clo_type, subst, parm_field_names);
        

        clo_type.Compile ();

        [TExpr.DefValIn (InternalType.Void, closure_val, ctor_call, null)]
      }
    }


    LoadParameters () : list [TExpr]
    {
      mutable initializers = [];
      
      foreach (fp in current_local_fun.parms) {
        def parm = fp.decl;
        assert (parm != null);

        def parmtype = parm.Type.Fix ();

        def need_cast =
          fp.ty.Fix ().IsSystemObject && ! parmtype.IsSystemObject;

        when (need_cast)
          parm.SetType (InternalType.Object); // modify its type

         //Message.Debug ($"handle fp $parm $(parm.Type)");
        when (parm.InClosure) {
          def expr =
            if (need_cast)
              CheckedConversion (PlainRef (parm), parmtype)
            else
              PlainRef (parm);
          def a = TExpr.Assign (InternalType.Void,
                                LocalRef (parm, for_store = true), 
                                expr);
          
          initializers = a :: initializers
        }
      }

      initializers
    }


    GetBaseCall () : option [TExpr]
    {
      // put base () / this () call before storing 'this' in closure inside constructor
      if (current_local_fun.name == ".ctor" && !CurrentType.IsValueType)
        match (current_local_fun.body) {
          | FunBody.Typed (TExpr.Sequence (TExpr.Call as basecall, rest)) =>
            current_local_fun.body = FunBody.Typed (rest);
            Some (Walk (basecall))

          | FunBody.Typed (TExpr.Call as basecall) =>
            current_local_fun.body = FunBody.Typed (BuildRevSequence ([]));
            Some (Walk (basecall))

          // ctor call is (if not, it's a bug) placed somewhere inside, but this case disallows using 'this' in closures
          | FunBody.Typed (_) => Some (null) 

          | _ => assert (false)
        }
      else None ()
    }
    

    // the result is reversed!
    PrepareProlog () : list [TExpr]
    {
      // if we have closures passed in as parameters, add entries
      // to closure_parms
      PrepareClosureParms ();
      
      // assigments of parameters and 'this' to closure fields
      // interleaved with base (..) call in constructor
      mutable initializers = [];

      // we build the initialization stuff of method:
      // 1. method's closure ctor()
      // 2. store parameters into closure
      // 3. base (..) / this (..) for constructors
      // 4. store 'this' in closure

      initializers = PrepareClosure ();

      initializers = LoadParameters () + initializers;

      mutable base_call_missing = false;
      match (GetBaseCall ()) {
        | Some (null) => base_call_missing = true;
        | Some (call) =>
          initializers = call :: initializers;
        | None => {}
      }

      // store 'this' into closure object
      foreach (d in current_local_fun.closure_vars) 
        when (d.InClosure &&
              d.ValKind is LocalValue.Kind.ClosurisedThisPointer) {
          when (base_call_missing)
            Message.Error ("closure utilizing 'this' reference is not allowed when base ctor call is not placed at the beginning of current ctor");
          def ini =
            TExpr.Assign (InternalType.Void,
                          LocalRef (d, for_store = true),
                          TExpr.This (CurrentType.GetMemType ()));
          initializers = ini :: initializers;
        }

      initializers
    }
    #endregion


    #region Local function generation
    ComputeUsedClosures (h : Fun_header) : void
    {
      def we_use (var) {
        // FIXME: doesn't work, because function have to be removed from external closures
        //var.id != h.decl.id && // call to our function shouldn't be closurised
        (h :: h.children_funs).Exists (fun (child) { 
          var.UsedIn.Contains (child) 
        })
      }
      
      when (h.used_closures == null) {
        h.used_closures = [];

        def needed = Hashtable ();
        
        foreach (fh in h.GetParents ())
          foreach (var in fh.closure_vars) {
            when (we_use (var)) {
              needed [fh.id] = true;
              match (var.ValKind) {
                | Function (ch, _) =>
                  ComputeUsedClosures (ch);
                  foreach (clo in ch.used_closures)
                    needed [clo.id] = true;
                | _ => {}
              }
            }
          }

          h.used_closures = h.GetParents ().Filter (fun (fh) { needed.Contains (fh.id) })
      }
    }


    static ClosureParmCount (fn : Fun_header) : int
    {
      Util.cassert (fn.used_closures != null || fn.decl == null, 
                    $ "closures not computed for $(fn.name)");
      if (fn.static_method == null) 0
      else fn.used_closures.Length
    }


    static PrepareForEmission (meth : MethodBuilder, fn : Fun_header, subst : Subst) : void
    {
      def new_header = meth.fun_header;
      meth.fun_header = fn;

      def clo_count = ClosureParmCount (fn);
      def new_parms = new_header.parms.ChopFirstN (clo_count);

      foreach (parm in new_header.parms.FirstN (clo_count)) {
        parm.decl = LocalValue (fn, parm.name, parm.ty, 
                                LocalValue.Kind.Plain (),
                                is_mutable = false);
        parm.decl.Register ();
        parm.decl.UseFrom (fn);
      }
      
      if (new_parms.Length > 1 && fn.parms.Length == 1) {
        match (fn.body) {
          | FunBody.Typed (body) =>
            def vals = new_parms.Map (fun (parm : Fun_parm) {
              def local = LocalValue (fn, parm.name, parm.ty, 
                                      LocalValue.Kind.Plain (), is_mutable = false);
              local.Register ();
              local.UseFrom (fn);
              parm.decl = local;
              local
            });
            def parm = fn.parms.Head;
            def expr =
              TExpr.DefValIn (body.Type,
                              parm.decl, 
                              TExpr.Tuple (parm.ty, vals.Map (PlainRef)),
                              body);
            fn.body = FunBody.Typed (expr);
            fn.parms = new_header.parms;
          | _ => assert (false)
        }
      } else {
        List.Iter2 (fn.parms, new_parms, fun (orig, copy : Fun_parm) {
          copy.decl = orig.decl;
          // FIXME: is it still needed?
          copy.ty = FixType (copy.ty, subst);
          copy.decl.SetType (FixType (copy.decl.Type, subst));
        });
        fn.parms = new_header.parms;
      }

      fn.ret_type = new_header.ret_type;
    }


    EmitStaticLocalFunction (fn : Fun_header, children : Queue [Typer3]) : void
    {
      def (new_tp, subst) = CopyFunTyparms ();
      fn.typarms = fn.typarms + new_tp;
      def closures = fn.used_closures;

      // Message.Debug ($ "emit static $(fn.name) $(fn.typarms) subst=$subst");

      def clo_parms =
        closures.Map (fun (hd) {
          <[ parameter: $(Macros.NewSymbol (hd.name + "_cp") : name) 
                      : $(FixType (hd.closure_type, subst) : typed) ]>
        });

      def parms = clo_parms + fn.parms.Map (fun (fp) {
        <[ parameter: $(fp.decl.Name : dyn) 
                    : $(FixType (fp.decl.Type, subst) : typed) ]>
      });

      def is_static =
        // mark method static if it do not have closurised this pointer
        is_static : {
          foreach (h in closures)
            foreach (val in h.closure_vars)
              when (val.ValKind is LocalValue.Kind.ClosurisedThisPointer)
                is_static (false);
          true
        }

      def ret_type = FixType (fn.ret_type, subst);

      def decl =
        if (is_static)
          <[ decl:
            private static $(Macros.NewSymbol (fn.name) : name) (.. $parms) 
                      : $(ret_type : typed)
            {
            }
          ]>
        else
          <[ decl:
            private $(Macros.NewSymbol (fn.name) : name) (.. $parms) 
                      : $(ret_type : typed)
            {
            }
          ]>;

      CurrentType.forced_typarms = fn.typarms;
      def meth = CurrentType.DefineAndReturn (decl) :> MethodBuilder;

      fn.static_method = meth;
      PrepareForEmission (meth, fn, subst);
      fn.typarms_to_pass =
        accumulated_typarms.Map (fun (x) { MType.TyVarRef (x) });

      def child = Typer3 (this, meth);
      child.current_subst = subst;
      // make it here, so Run will fill it up
      child.closure_parms = Hashtable ();
      children.Push (child);
    }


    EmitFunctionalValue (fn : Fun_header, children : Queue [Typer3]) : TExpr
    {
      Stats.FirstClassFunctions++;
      def closures = fn.used_closures;

      Util.cassert (fn.typarms.IsEmpty, $"should be handled in T2, $(fn.name)");

      // Message.Debug ($ "emit 1st class $(fn.name) acc=$(accumulated_typarms)");
      def (new_tp, subst) = CopyFunTyparms (add_class_tyvars = true);
       
      def fix_type (t) { FixType (t, subst) }

      def fn_mtype = MType.ConstructFunctionType (fn).DeepFix ();
      def fnty = FunctionType.Make (fn_mtype);
      def (parm_types, ret_type) =
        Option.UnSome (fn_mtype.FunReturnTypeAndParms ());
      
      def formal_parms =
        parm_types.Map (fun (ty) { <[ parameter: 
          $(Macros.NewSymbol () : name) : $(fix_type (ty) : typed)
        ]> });

      def apply_name = 
        if (ret_type.Fix () is MType.Void)
          "apply_void" else "apply";

      def lambda_name = Macros.NewSymbol (fn.name + "_");
      def lambda_base_type = fix_type (fnty);

      //Message.Debug ($"lambda: $lambda_name $new_tp : $lambda_base_type");
        
      def builder = CurrentType.DefineNestedType (<[ decl:
        private sealed class $(lambda_name : name) : $(lambda_base_type : typed)
        {
          public override $(apply_name : dyn) (.. $formal_parms) : $(fix_type (ret_type) : typed)
          {
          }
        }
      ]>, false);
      builder.DisableImplicitConstructor ();
      builder.forced_typarms = new_tp;
      builder.FixupDefinedClass ();

      def clo_fields = Hashtable ();

      def (parms, assigns) =
        List.Split (closures.Map (fun (hd) {
          def name = Macros.NewSymbol (hd.name + "_clo");
          Util.cassert (hd.closure_type != null, $ "null closure for $(hd.name)");
          def clo_type = hd.closure_type;
          //Message.Debug ($"add field $name to $lambda_name : $clo_type --> $(fix_type (clo_type)) new_tp=$new_tp $local_function_type->$(local_function_type.typarms) $current_method->$(current_method.typarms)");
          def field =
            builder.DefineAndReturn (<[ decl: 
              mutable $(name : name) : $(fix_type (clo_type) : typed) 
            ]>) :> IField;
          clo_fields [hd.id] = field;
          (<[ parameter: $(name : name) : $(fix_type (clo_type) : typed) ]>,
           <[ this . $(name : name) = $(name : name) ]>)
        }));

      def ctor =
        builder.DefineAndReturn (<[ decl:
          public this (.. $parms)
          {
            { .. $assigns }
          }
        ]>);
      
      def the_method = SingleMemberLookup (builder, apply_name) :> MethodBuilder;

      PrepareForEmission (the_method, fn, subst);

      def child = Typer3 (this, the_method);
      child.closure_fields = clo_fields;
      child.current_subst = subst;
      children.Push (child);

      builder.MarkWithSpecialName ();
      builder.Compile ();

      def ctor_parms = closures.Map (fun (hd) { Parm (ClosureRef (hd)) });

      def from_type =
        MType.Class (builder, FunAndClassTyparmsRefs ());
      def ctorty =
        match (from_type.TypeOfMember (ctor).Fix ()) {
          | MType.Fun (from, _) => MType.Fun (from, fnty)
          | _ => Util.ice ()
        }

      def ctor_ref = TExpr.StaticRef (ctorty, from_type, ctor, []);
      TExpr.Call (fnty, ctor_ref, ctor_parms, false)
    }

    
    HandleLocalFunctions (fns : list [Fun_header]) : list [TExpr]
    {
      mutable res = [];
      def q = Queue ();

      
      foreach (fn in fns) {
        ComputeUsedClosures (fn);
        //Message.Debug ($"hlf: $(fn.name) $(fn.usage) $closures");
        match (fn.usage) {
          | FunctionUsage.UsedJustOnce // handled in EmitLoop
          | FunctionUsage.NotUsed => // obvious
            {}

          | FunctionUsage.Used =>
            EmitStaticLocalFunction (fn, q);

          | FunctionUsage.UsedAsFirstClass =>
            def fval = EmitFunctionalValue (fn, q);
            assert (fn.decl != null);
            res ::= TExpr.DefValIn (fn.decl, fval, null);
        }
      }

      while (! q.IsEmpty)
        q.Take ().Run ();

      res
    }
    #endregion
    

    #region Language constructs
    EmitLoop (hd : Fun_header, parms : list [Parm]) : TExpr
    {
      def child = Typer3 (this, hd);
      // the child can reuse our closure
      child.current_closure = current_closure;
      child.closure_fields = closure_fields;
      child.closure_parms = closure_parms;
      child.current_subst = current_subst;
      child.Run ();

      def body =
        match (hd.body) {
          | FunBody.Typed (x) => x
          | _ => assert (false);
        }

      // need to pass parameters
      def (parms, ini) = TupleParms (hd, parms);
      def assigns = List.RevMap2 (parms, hd.parms, fun (actual, formal) {
        assert (actual.kind == ParmKind.Normal);
        TExpr.DefValIn (formal.decl, actual.expr, null)
      });

      BuildRevSequence (body :: (assigns + ini));
    }


    EmitDelegateCtor (ret_ty : TyVar, sr : TExpr, parm : TExpr) : TExpr
    {
      def (obj, meth) =
        match (parm) {
          | TExpr.StaticRef (from, meth is IMethod, typarms) =>
            (TExpr.Literal (InternalType.Object, Literal.Null ()),
             TExpr.MethodAddress (from, meth, false, typarms))
             
          | TExpr.MethodRef (obj, meth, typarms, nonvirt) =>
            (Walk (obj), TExpr.MethodAddress (obj.Type, meth, !nonvirt, typarms))

          | TExpr.DefFunctionsIn ([func], LocalFunRef (decl, typarms))
            when func.decl.Equals (decl) => // this is for sure lambda

            ComputeUsedClosures (func);
            match (func.used_closures) {
              | [] when func.parms.Length == 
                        Option.UnSome (parm.MType.FunReturnTypeAndParms ()) [0].Length =>
                // Message.Debug ($"empty, parms count = $(func.parms.Length)");
                def q = Queue ();
                EmitStaticLocalFunction (func, q);
                
                while (!q.IsEmpty)
                  q.Take ().Run ();
                
                (TExpr.Literal (InternalType.Object, Literal.Null ()),
                 TExpr.MethodAddress (CurrentType.GetMemType (), func.static_method, false, typarms))

              | _cls =>
                // Message.Debug (_cls.ToString ());
                EmitDelegateProxy (Walk (parm))
            }
            
          | _ =>
            EmitDelegateProxy (Walk (parm))
        }

      TExpr.Call (ret_ty, sr,
                  [Parm (obj), Parm (meth)], false)
    }


    EmitCall (ret_type : TyVar, func : TExpr, parms : list [Parm], is_tail : bool) : TExpr
    {
      foreach (p in parms)
        p.expr = Walk (p.expr);

      def just_call (meth, func, clo_parms) {
        def (parms, ini) = TupleParms (meth.GetHeader (), parms, clo_parms.Length);
        def call = TExpr.Call (ret_type, func, clo_parms + parms, is_tail);
        BuildRevSequence (call :: ini)
      }

      def plain_call () {
        def ft = InternalType.GetFunctionType (parms.Length);
        def meth =
          if (ret_type.Fix () is MType.Void)
            ft.ApplyVoidMethod
          else
            ft.ApplyMethod;
        just_call (meth, TExpr.MethodRef (func.ty, Walk (func), meth, [], false), [])
      }

      match (func) {
        | TExpr.MethodRef (obj, meth, tp, notvirt) =>
          just_call (meth,
                     TExpr.MethodRef (func.ty, Walk (obj), meth, tp, notvirt), [])
          
        | TExpr.ConstantObjectRef => Util.ice ()
        
        | TExpr.Base (meth)
        | TExpr.StaticRef (_, meth is IMethod, _) =>
          just_call (meth, func, [])
          
        | TExpr.OpCode => TExpr.Call (func, parms, false)

        | TExpr.LocalRef (LocalValue where (ValKind = LocalValue.Kind.Function)) =>
          Util.ice ()

        | TExpr.LocalFunRef (LocalValue where 
                          (ValKind = LocalValue.Kind.Function (hd, _)), type_parms) =>
          match (hd.usage) {
            | FunctionUsage.UsedJustOnce =>
              EmitLoop (hd, parms)

            | _ =>
              if (hd.static_method != null) {
                // first check if we're dealing with static method from current type
                // or from our enclosing type (so we're calling a local function
                // that got static but we're first class)
                def from_type = hd.static_method.DeclaringType;
                def is_local = local_function_type.Equals (from_type);
                
                def from_memty =
                  if (is_local)
                    from_type.GetMemType ()
                  else {
                    // we have the enclosing type's type parameters
                    // as prefix of our type parameters
                    def parm_count = from_type.Typarms.Length;
                    def parms = local_function_type.Typarms.FirstN (parm_count);
                    def parms = parms.Map (fun (x) { MType.TyVarRef (x) });
                    MType.Class (from_type, parms)
                  }

                def typarms =
                  type_parms +
                  hd.typarms_to_pass.Map (fun (x) { 
                    SubstType (x) 
                  });
                    
                def clo_parms = 
                  hd.used_closures.Map (fun (hd) { 
                    Parm (ClosureRef (hd)) 
                  });

                // check if we can use the current this pointer
                // and if not, lookup one in closure
                def this_ptr =
                  if (hd.static_method.IsStatic)
                    null
                  else if (is_local) {
                    TExpr.This (from_memty)
                  } else {
                    def lookup_this (t3) {
                      if (t3.parent == null) {
                        def decl = 
                          Option.UnSome (
                            t3.current_local_fun.closure_vars.Find (fun (decl) {
                              decl.ValKind is LocalValue.Kind.ClosurisedThisPointer
                            }));
                        LocalRef (decl)
                      } else {
                        lookup_this (t3.parent)
                      }
                    }
                    lookup_this (this)
                  }

                def (fnc_type, fresh_typarms)  =
                  from_memty.TypeOfMethodWithTyparms (hd.static_method);

                List.Iter2 (fresh_typarms, typarms, fun (a, b) {
                  a.ForceUnify (b)
                });
                  

                def fnc =
                  if (hd.static_method.IsStatic)
                    TExpr.StaticRef (fnc_type,
                                     from_memty, hd.static_method,
                                     typarms)
                  else
                    TExpr.MethodRef (fnc_type,
                                     this_ptr, hd.static_method, 
                                     typarms, false);

                just_call (hd.static_method, fnc, clo_parms);
              } else
                plain_call ()
          }

        | _ => plain_call ()
      }
    }
    #endregion

    
    #region Matching compilation
    CompileMatch (m : TExpr.Match) : TExpr
    {
      mutable vals = Set ();
      
      foreach (case in m.cases)
        foreach ((pat, _, assigns) in case.patterns) {
          _ = pat.Walk (fun (_) {
            | Pattern.As (_, decl) =>
              vals = vals.Replace (decl);
              null
            | _ => null
          });
          foreach ((decl, _) in assigns)
            vals = vals.Replace (decl);
        }
         

      def is_bool_pattern (pat) {
        | Pattern.Literal (Bool)
        | Wildcard => true
        | _ => false
      }

      def match_comp = DecisionTreeCompiler.Run (m.Type, _, m.cases);

      def expr =
        match (m.cases) {
          | [([(p1, TExpr.Literal (Bool (true)), [])], _, _),
             ([(p2, TExpr.Literal (Bool (true)), [])], _, _)] 
            when is_bool_pattern (p1) && is_bool_pattern (p2) =>
            // don't cache
            match_comp (m.expr)
          | _ => 
            WithCached (m.expr, fun (e) {
              match_comp (e)
            });
        }

      def expr =
        vals.Fold (expr, fun (decl, expr) {
          if (decl.InClosure)
            expr
          else {
            // Message.Debug ($"store $decl $(decl.Id)");
            decl.UseFrom (current_local_fun);
            TExpr.DefValIn (expr.Type, decl, 
                            TExpr.DefaultValue (decl.Type), expr)
          }
        });

      Walk (expr)
    }
    #endregion


    #region Top level stuff
    Walk (expr : TExpr) : TExpr
    {
      expr.Walk (DoWalk)
    }


    DoWalk (expr : TExpr) : TExpr
    {
      // Message.Debug ($ "dowalk: $(expr.GetType()) $(expr)");
      match (expr) {
        | TExpr.LocalFunRef (decl, _)
        | TExpr.LocalRef (decl) =>
          LocalRef (decl)
          
        | TExpr.StaticRef (_, _ is IField, _) =>
          null

        | TExpr.StaticRef (_, _ is IMethod, _) as sr =>
          EmitStaticProxy (sr)

        // we do not support any other staticrefs here
        // everything should be handled by Typer2 already
        | TExpr.StaticRef => assert (false)
          
        | TExpr.DefFunctionsIn (funs, body) =>
          def res = HandleLocalFunctions (funs);
          Walk (BuildRevSequence (body :: res))

        | TExpr.DefValIn (decl, val, body) when decl.InClosure =>
          if (val is TExpr.DefaultValue)
            Walk (body)
          else {
            def assign =
              TExpr.Assign (InternalType.Void, LocalRef (decl), Walk (val));
            TExpr.Sequence (assign, Walk (body))
          }

        // handled by Typer2.LambdaTransform
        | TExpr.MethodRef => assert (false)

        | TExpr.Call (TExpr.StaticRef (_, m is IMethod, _) as sr, [parm], _)
          when
            m.GetFunKind () is FunKind.Constructor && 
            m.DeclaringType.IsDelegate =>
          EmitDelegateCtor (expr.Type, sr, parm.expr)

        | TExpr.Call (func, parms, is_tail) =>
          EmitCall (expr.Type, func, parms, is_tail)

        | TExpr.SelfTailCall (parms) =>
          def clo_len = ClosureParmCount (current_local_fun);
          def (parms, ini) = TupleParms (current_local_fun, parms, clo_len);
          def assigns =
            List.Map2 (parms, current_local_fun.parms.ChopFirstN (clo_len), 
              fun (parm, fp) {
                assert (parm.kind == ParmKind.Normal);
                def ty = fp.ty;
                def conv = CheckedConversion (Walk (parm.expr), ty);
                (fp.decl, conv)
              });

          def beg =
            if (assigns.IsEmpty)
              ini
            else
              TExpr.MultipleAssign (InternalType.Void, assigns) :: ini;
              
          def goto = TExpr.Goto (InternalType.Void, start_label, 1);
          
          BuildRevSequence (goto :: beg)

        | TExpr.ConstantObjectRef (_, mem) =>
          // FIXME this doesn't seem to be the right place for such a message
          Message.Warning ("using a constant object reference directly");
          Message.Warning ("  you probably have meant to write `" +
                           mem.DeclaringType.FullName + " ()'");
          def meth = SingleMemberLookup (mem.DeclaringType, 
                                         "_N_constant_object_generator");
          Walk (StaticRef (meth))

        | TExpr.Match as m =>
          CompileMatch (m)

        // we cannot handle closurised values as placeholders for exceptions
        // so we use a fresh variable for exception and assign it using
        // regular DefValIn
        | TExpr.TryWith (body, orig, handler) =>
          if (orig.InClosure) {
            def val =
              LocalValue (current_local_fun, orig.Name,
                          orig.Type,
                          LocalValue.Kind.ExceptionValue (),
                          is_mutable = false);
            val.Register ();
            val.UseFrom (current_local_fun);
            def handler =
              TExpr.DefValIn (handler.Type, orig, PlainRef (val), handler);
            TExpr.TryWith (Walk (body), val, Walk (handler))
          } else null


        // optimize ifs
        | If (cond, e1, e2) => 
          match (Walk (cond)) {
            | Literal (Bool (lit)) => Walk (if (lit) e1 else e2)
            
            | Sequence (prep, Literal (Bool (lit))) =>
              def e = Walk (if (lit) e1 else e2);
              TExpr.Sequence (e.Type, prep, e)
              
            | cond =>
              match ((Walk (e1), Walk (e2))) {
                | (Literal (Bool (true)), Literal (Bool (false))) =>
                  cond
                | (e1, e2) =>
                  TExpr.If (cond, e1, e2)
              }
          }
          
        | _ => null
      }
    }
    #endregion

    
    #region current_subst handling
    SubstType (t : TyVar) : MType
    {
      if (current_subst != null)
        current_subst.Apply (t.Fix ()).Fix ()
      else t.Fix ()
    }


    SubstTypes (t : list [TyVar]) : list [TyVar]
    {
      t.Map (fun (x) { SubstType (x) })
    }
    

    SubstExpr (expr : TExpr) : TExpr
    {
      expr.Walk (DoSubstExpr)
    }
    

    DoSubstExpr (expr : TExpr) : TExpr
    {
      when (expr.ty != null)
        expr.ty = SubstType (expr.ty);

      match (expr) {
        | StaticRef (from, mem, tp) =>
          TExpr.StaticRef (SubstType (from) :> MType.Class, mem, SubstTypes (tp))
        | MethodRef (obj, meth, tp, nv) =>
          TExpr.MethodRef (SubstExpr (obj), meth, SubstTypes (tp), nv)
        | TypeConversion (e, t, k) =>
          TExpr.TypeConversion (SubstExpr (e), SubstType (t), k)
        | TypeOf (t) =>
          TExpr.TypeOf (SubstType (t))
        | HasType (e, t) =>
          TExpr.HasType (SubstExpr (e), SubstType (t))
          
        | DefValIn (name, _, _)
        | TryWith (_, name, _) =>
          name.SetType (SubstType (name.Type));
          null

        | LocalRef
        | ImplicitValueTypeCtor
        | FieldMember
        | Call
        | Assign
        | Throw
        | TryFinally
        | TryFault
        | Literal
        | This
        | Base
        | Sequence
        | Tuple
        | Array
        | ArrayIndexer
        | TupleIndexer
        | OpCode
        | MethodAddress
        | MultipleAssign
        | Label
        | Goto
        | DefaultValue
        | If
        | Switch =>
          null

        | LocalFunRef
        | PropertyMember
        | StaticPropertyRef
        | EventMember
        | StaticEventRef
        | ConstantObjectRef
        | Delayed
        | Error
        | DefFunctionsIn
        | Match
        | SelfTailCall 
        | Block =>
          Util.cassert (Message.SeenError);
          null
      }
    }
    #endregion

  }
}
