#include "tests.h"

#define TEST_STRING_UTF8 "\xe1\x84\x83\xe1\x85\xa1\xe1\x86\xbc\xe1\x84\x89"\
			 "\xe1\x85\xb5\xe1\x86\xab\xe1\x84\x8e\xe1\x85\xa5"\
			 "\xe1\x84\x85\xe1\x85\xa5\xe1\x86\xb7\x20\xe1\x84"\
			 "\x84\xe1\x85\xa5\xe1\x84\x87\xe1\x85\xa5\xe1\x86"\
			 "\xaf\xe1\x84\x8b\xe1\x85\xb5\x3f\x20\xe1\x84\x92"\
			 "\xe1\x85\xa1\xe1\x84\x91\xe1\x85\xb3\x20\xe1\x84"\
			 "\x86\xe1\x85\xa2\xe1\x86\xa8\xe1\x84\x8c\xe1\x85"\
			 "\xae\x21"
			 
#define TEST_STRING_UTF16LE "\x03\x11\x61\x11\xbc\x11\x09\x11\x75\x11\xab\x11"\
			    "\x0e\x11\x65\x11\x05\x11\x65\x11\xb7\x11\x20\x00"\
			    "\x04\x11\x65\x11\x07\x11\x65\x11\xaf\x11\x0b\x11"\
			    "\x75\x11\x3f\x00\x20\x00\x12\x11\x61\x11\x11\x11"\
			    "\x73\x11\x20\x00\x06\x11\x62\x11\xa8\x11\x0c\x11"\
			    "\x6e\x11\x21\x00\0"
			    
#define TEST_STRING_GRAPHEMES 16
#define TEST_STRING_CHARS 16

static void printUTF8(const unsigned char * s)
{
	while(s[0] != '\0')
	   {
		printf("%x ", s[0]);
		s++;
	   }
	printf("\n");
}

static void printUTF16(const unsigned char * s)
{
	while(!(s[0] == 0x00 && s[1] == 0x00))
	   {	   		
	   	if ((s[1] >= 0xd8) && (s[1] <= 0xdf))
	   	   {
	   	   	printf("%x %x %x %x ", s[0], s[1], 
	   	   			       s[2], s[3]);
	   	   	s += 4;
	   	   }
	  	else
	  	   {
	  	   	printf("%x %x ", s[0], s[1]);
	  	   	s += 2;
		   }	
	   }
	   
	printf("\n");
}

int
main()
{
	uint32_t cp;
	int32_t ecp;
	size_t len;
	size_t elen;
	uint8_t *c;
	char buf[4];
	char *buf2;
	char *s1;
	char *s2;
	int intval;
	float floatval;
	int nchar;
	int enchar;

	test_begin("string");

	test_msg("checking returned length from nacore_char_utf8_encode()");
	for (cp = 0; cp < 0x200000; cp++)
	  {
		if (((cp >= 0xd800) && (cp <= 0xdfff)) || (cp == 0xfeff)
		    || (cp == 0xfffe) || (cp >= 0x110000))
			elen = 0;
		else if (cp < 0x80)
			elen = 1;
		else if (cp < 0x800)
			elen = 2;
		else if (cp < 0x10000)
			elen = 3;
		else
			elen = 4;

		len = nacore_char_utf8_encode(NULL, cp);
		if (len != elen)
			test_err("code point: %" PRIx32 ", expected length: "
				 "%" NACORE_LIBC_SIZE_FORMAT_LM "u, returned "
				 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u", cp,
				 (NACORE_LIBC_SIZE_FORMAT_TYPE)elen,
				 (NACORE_LIBC_SIZE_FORMAT_TYPE)len);
	  }
	  
	
	test_msg("checking returned length from nacore_char_utf16le_encode()");
	for (cp = 0; cp < 0x200000; cp++)
	  {
	  	if (((cp >= 0xd800) && (cp <= 0xdfff)) || (cp == 0xfeff)
	  	    || (cp == 0xfffe) || (cp >= 0x110000))
	  	    	elen = 0;
	  	else if (cp <= 0xffff)
	  		elen = 2;
	  	else if ((cp >= 0x10000) && (cp <= 0x10ffff))
	  		elen = 4;
	  		
	  	len = nacore_char_utf16le_encode(NULL, cp);
	  	if (len != elen)
	  		test_err("code point: %" PRIx32 ", expected length: "
	  			 "%" NACORE_LIBC_SIZE_FORMAT_LM "u, returned "
	  			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u", cp,
	  			 (NACORE_LIBC_SIZE_FORMAT_TYPE)elen,
	  			 (NACORE_LIBC_SIZE_FORMAT_TYPE)len);
	  }
	  
	
	test_msg("checking returned unicode code point from "
		 "nacore_char_utf8_decode()");
	/* checking for general values */
	for (cp = 0; cp < 0x200000; cp++)
	  {
	  	elen = nacore_char_utf8_encode(buf, cp);
	  	if (elen == 0)
	  		continue;
	  		  	
	  	nacore_char_utf8_decode(buf, (uint32_t *)&ecp);
	  	if (cp != ecp)
	  		test_err("expected code point: %" PRIx32 ", returned "
	  			 "code point: %" PRId32, cp, ecp);
	  }
	/* checking for high surrogates values */
	c = (uint8_t*) buf;
	for (cp = 0xd800; cp <= 0xdfff; cp++)
	  {
	   	c[0] = (cp >> 12) | 0xe0;
		c[1] = ((cp >> 6) & 0x3f) | 0x80;
		c[2] = (cp & 0x3f) | 0x80;

		if (nacore_char_utf8_decode(buf, (uint32_t *)&ecp) > 0)
			test_err("expected no code point, returned code point: "
				 "%" PRIx32, ecp);
	  }
	/* checking for invalid values */
	c[0] = 0xef;
	c[1] = 0xbb;
	c[2] = 0xbf;
		
	if (nacore_char_utf8_decode(buf, (uint32_t *)&ecp) > 0)
		test_err("expected no code point, returned code point: %"
			 PRIx32, ecp);
				 
	c[0] = 0xef;
	c[1] = 0xbf;
	c[2] = 0xbe;
		
	if (nacore_char_utf8_decode(buf, (uint32_t *)&ecp) > 0)
		test_err("expected negative code point, returned code point: %"
			 PRIx32, ecp);
				 
				 
	test_msg("checking returned unicode code point from "
		 "nacore_char_utf16le_decode()");
	/* checking for general values */
	for (cp = 0; cp < 0x200000; cp++)
	  {
	  	elen = nacore_char_utf16le_encode(buf, cp);
	  	if (elen == 0)
	  		continue;
	  		  	
	  	nacore_char_utf16le_decode(buf, (uint32_t *)&ecp);
	  	if (cp != ecp)
	  		test_err("expected code point: %" PRIx32 ", returned "
	  			 "code point: %" PRIx32, cp, ecp);
	  }
	/* checking for invalid values */
	c[0] = 0xfe;
	c[1] = 0xff;
		
	if (nacore_char_utf16le_decode(buf, (uint32_t *)&ecp) > 0)
		test_err("expected no code point, returned code point: %"
			 PRIx32, ecp);
				 
	c[0] = 0xff;
	c[1] = 0xfe;
		
	if (nacore_char_utf16le_decode(buf, (uint32_t *)&ecp) > 0)
		test_err("expected negative code point, returned code point: %"
			 PRIx32, ecp);
	
	
	test_msg("checking returned encoding of a default message from "
		 "nacore_string_utf8_to_utf16le()");
	
	buf2 = nacore_string_utf8_to_utf16le(TEST_STRING_UTF8);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else
	   {
		if (strcmp(TEST_STRING_UTF16LE, buf2) != 0)
		   {
			test_err("expected equal strings, returned ");
			printUTF16((unsigned char *) buf2);
		   }
		free(buf2);
	   }
	
	
	test_msg("checking returned encoding of a default message from "
		 "nacore_string_utf16le_to_utf8()");
		 
	buf2 = nacore_string_utf16le_to_utf8(TEST_STRING_UTF16LE);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else
	   {
	   	if (strcmp(TEST_STRING_UTF8, buf2) != 0)
	   	   {
	   	   	test_err("expected equal strings, returned ");
	   	   	printUTF8((unsigned char *) buf2);
		   }
		free(buf2);
   	   }
	  
	
	test_msg("checking returned size of a string length from "
		 "nacore_strnlen()");
		 
	len = nacore_strnlen(TEST_STRING_UTF8, 3);
	
	if(len != 3)
		test_msg("expected len: 3, returned len: %"
			 NACORE_LIBC_SIZE_FORMAT_LM "u",
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)len);
		
	len = nacore_strnlen(TEST_STRING_UTF8, 0);
	
	if(len != 0)
		test_msg("expected len: 0, returned len: %"
			 NACORE_LIBC_SIZE_FORMAT_LM "u",
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)len);
		
	len = nacore_strnlen(TEST_STRING_UTF8, 500000);
	elen = sizeof(TEST_STRING_UTF8)-1;
	
	if(len != elen)
		test_msg("expected len: %" NACORE_LIBC_SIZE_FORMAT_LM "u, "
			 "returned len: %" NACORE_LIBC_SIZE_FORMAT_LM "u",
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)elen,
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)len);

	test_msg("checking returned number of graphemes of a test string from "
		 "nacore_strgraphemes()");
	
	len = nacore_strgraphemes(TEST_STRING_UTF8);
	
	if (len != (size_t) TEST_STRING_GRAPHEMES)
		test_err("expected equal number of graphemes, expected "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u, returned "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u", 
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)TEST_STRING_GRAPHEMES,
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)len); 
		 
	
	test_msg("checking returned number of graphemes of a test string from "
		 "nacore_strngraphemes()");
	
	/* checking exact length */	 
	len = nacore_strngraphemes(TEST_STRING_UTF8, TEST_STRING_GRAPHEMES);
	
	if (len != (size_t) TEST_STRING_GRAPHEMES)
		test_err("expected equal number of graphemes, expected "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u, returned "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u", 
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)TEST_STRING_GRAPHEMES,
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)len);
			 
	/* checking length with a lower margin */		 
	len = nacore_strngraphemes(TEST_STRING_UTF8, 10);
	
	if (len != (size_t) (TEST_STRING_GRAPHEMES-6) )
		test_err("expected equal number of graphemes, expected "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u, returned "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u", 
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)TEST_STRING_GRAPHEMES
			 - 6, (NACORE_LIBC_SIZE_FORMAT_TYPE)len); 
		 
	/* checking length with a higher margin */		 
	len = nacore_strngraphemes(TEST_STRING_UTF8, 20);
	
	if (len != (size_t) (TEST_STRING_GRAPHEMES) )
		test_err("expected equal number of graphemes, expected "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u, returned "
			 "length: %" NACORE_LIBC_SIZE_FORMAT_LM "u", 
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)TEST_STRING_GRAPHEMES,
			 (NACORE_LIBC_SIZE_FORMAT_TYPE)len);

	test_msg("checking returned string from nacore_astrcpy()");
	
	buf2 = nacore_astrcpy(TEST_STRING_UTF8, NULL);
	
	if(buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else
	   {
		
		if (strlen(buf2) < strlen(TEST_STRING_UTF8))
			test_err("expected equal length, returned a lower"
				 " one: %" NACORE_LIBC_SIZE_FORMAT_LM "u",
				 (NACORE_LIBC_SIZE_FORMAT_TYPE)strlen(buf2));
		
		else if (strlen(buf2) > strlen(TEST_STRING_UTF8))
			test_err("expected equal length, returned a higher"
				 " one: %" NACORE_LIBC_SIZE_FORMAT_LM "u",
				 (NACORE_LIBC_SIZE_FORMAT_TYPE)strlen(buf2));
		
		else if (strcmp(buf2, TEST_STRING_UTF8) != 0)
			test_err("expected equal strings, returned different"
				 " ones");
			  
		free(buf2);
   	   }


	test_msg("checking returned string from nacore_asprintf()");
	
	s1 = "Test message";
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%s", s1);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
	
	s1 = "\tTest1\nTest2\n";
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%s", s1);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "Test1, test2\t";
	s2 = "Test3, test4\n";
	enchar = strlen(s1) + strlen(s2) + 1;
	
	nchar = nacore_asprintf(&buf2, "%s %s", s1, s2);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, "Test1, test2\t Test3, test4\n") != 0)
		test_err("expected '%s %s', returned '%s'", s1, s2, buf2);
		
	if(buf2 != NULL)
		free(buf2);
	
	s1 = "\nTest1, test2\t";
	s2 = "\nTest3, test4\n";
	intval = 10;
	enchar = strlen(s1) + strlen(s2) + 4;
	
	nchar = nacore_asprintf(&buf2, "%s %s %d", s1, s2, intval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, "\nTest1, test2\t \nTest3, test4\n 10") != 0)
		test_err("expected '%s %s %d', returned '%s'", s1, s2, intval,
			 buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "\nTest1, test2\t";
	s2 = "\nTest3, test4\n";
	intval = 10;
	floatval = 10.5;
	enchar = strlen(s1) + strlen(s2) + 9;
	
	nchar = nacore_asprintf(&buf2, "%s %s %d %.1f", s1, s2, intval, 
				floatval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, "\nTest1, test2\t \nTest3, test4\n 10 10.5") != 0)
		test_err("expected '%s %s %d %.1f', returned '%s'", s1, s2, 
			 intval, floatval, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "\0";
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%s", s1);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "\0\n";
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%s", s1);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "%%\\\a\t";
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%s", s1);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	floatval = 5.555555555;
	enchar = 6;
	
	nchar = nacore_asprintf(&buf2, "%.*f", 4, floatval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, "5.5556") != 0)
		test_err("expected '%.4f', returned '%s'", floatval, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "4";
	intval = 4;
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%u", intval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "4";
	intval = 4;
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%o", intval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "12";
	intval = 10;
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%o", intval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "3ea";
	intval = 1002;
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%x", intval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "3EA";
	intval = 1002;
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%X", intval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = "2\n";
	intval = 50;
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%c\n", intval);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	s1 = TEST_STRING_UTF8;
	enchar = strlen(s1);
	
	nchar = nacore_asprintf(&buf2, "%s", s1);
	if (buf2 == NULL)
		test_err("expected non-null char pointer, returned a null one");
	else if (nchar != enchar)
		test_err("expected length: %d, returned length: %d",
			enchar, nchar);
	else if (strcmp(buf2, s1) != 0)
		test_err("expected '%s', returned '%s'", s1, buf2);
		
	if(buf2 != NULL)
		free(buf2);
		
	
		
	
		
	test_end();
}
